/*
 * inode routines
 *
 * Author:
 *   Dietmar Maurer (dm@vlsivie.tuwien.ac.at)
 *
 *
 */

#include "vefs.h"

guint32
efs_inode_create (EFS *efs)
{
	EFSDriver *d;

	g_return_val_if_fail (efs != NULL, 0);
  
	d = efs->driver;

	return d->iops->inode_create (efs);
}

gint
efs_inode_erase (EFS *efs, guint32 inode)
{
	GList *l;
	EFSDriver *d;
	EFSINodeLEntry *e;

	g_return_val_if_fail (efs != NULL, -1);
	g_return_val_if_fail (inode != 0, -1);

	d = efs->driver;

	l = efs->inode_list;
	while (l) {
		e = (EFSINodeLEntry *)l->data;
		if (e->inode == inode) {
			e->erase = TRUE;
			return 0;
		}
		l = l->next;
	}

	return d->iops->inode_erase (efs, inode);
}

gint
efs_inode_trunc (EFS *efs, guint32 inode, guint32 block)
{
	EFSDriver *d;

	g_return_val_if_fail (efs != NULL, -1);
	g_return_val_if_fail (inode != 0, -1);

	d = efs->driver;

	return d->iops->inode_trunc (efs, inode, block);
}

EFSCacheEntry*
efs_inode_map (EFS *efs, guint32 inode)
{
	EFSDriver *d;

	g_return_val_if_fail (efs != NULL, NULL);
	g_return_val_if_fail (inode != 0, NULL);

	d = efs->driver;

	return d->iops->inode_map (efs, inode);
}

EFSCacheEntry*
efs_inode_bmap  (EFS *efs, guint32 inode, guint32 block)
{
	EFSDriver *d;

	g_return_val_if_fail (efs != NULL, NULL);
	g_return_val_if_fail (inode != 0, NULL);

	d = efs->driver;

	return d->iops->inode_bmap (efs, inode, block);
}

gint
efs_inode_ref (EFS *efs, guint32 inode)
{
	GList *l;
	EFSINodeLEntry *e;

	l = efs->inode_list;
	while (l) {
		e = (EFSINodeLEntry *)l->data;
		if (e->inode == inode) {
			e->ref_count++;
			return e->ref_count;
		}
		l = l->next;
	}
	
	e = g_malloc0 (sizeof(EFSINodeLEntry));
	e->inode = inode;
	e->ref_count = 1;
	efs->inode_list = g_list_prepend (efs->inode_list, e);

	return 1;
}

gint
efs_inode_unref (EFS *efs, guint32 inode)
{
	GList *l;
	EFSINodeLEntry *e;
	EFSDriver *d;

	d = efs->driver;

	l = efs->inode_list;
	while (l) {
		e = (EFSINodeLEntry *)l->data;
		if (e->inode == inode) {
			e->ref_count--;
			if (!e->ref_count) {
				if (e->erase) {
					d->iops->inode_erase (efs, inode);
				}
				efs->inode_list =
					g_list_remove (efs->inode_list, e);
				g_free (e);
				return 0;
			} else
				return e->ref_count;
		}
		l = l->next;
	}
	
	return -1;
}

gint
efs_inode_refcount (EFS *efs, guint32 inode)
{
	GList *l;
	EFSINodeLEntry *e;

	l = efs->inode_list;
	while (l) {
		e = (EFSINodeLEntry *)l->data;
		if (e->inode == inode) return e->ref_count;
		l = l->next;
	}
	
	return -1;
}
