/*
 * directory routines
 *
 * Author:
 *   Dietmar Maurer (dm@vlsivie.tuwien.ac.at)
 *
 *
 */

#include "vefs.h"

/**
 * efs_dir_open:
 * @dir: parent directory
 * @path: filesystem path
 * @flags: access flags
 *
 * Description: Opens or creates a new #EFS within file @path. @path
 * is simply the name of the directory like "test_dir". "./test_dir"
 * is not valid and will always return NULL.
 *
 * @flags is %EFS_READ or %EFS_RDWR which request opening the #EFS read-only 
 * or read/write. flags  may  also  be  bitwise-or'd with one or more of the
 * following:
 *
 * %EFS_CREATE If the #EFS does not exist it will be created.
 *
 * %EFS_EXCL  When used with %EFS_CREATE, if the #EFS already exists it is an 
 * error and the open will fail.
 *
 * Returns: a pointer to the subdirectory, or zero if an error occurred.
 **/

EFSDir*
efs_dir_open (EFSDir *dir, const char *path, gint flags)
{
	EFSDriver *d;

	g_return_val_if_fail(dir != NULL, NULL);
	g_return_val_if_fail(path != NULL, NULL);

	d = dir->efs->driver;

	return d->fops->dir_open (dir, path, flags);
}

/**
 * efs_dir_close
 * @dir: the directory to close.
 * 
 * Description: Closes @dir. This function 
 * recurses its opened files/subdirs. This 
 * function can be called on the root #EFS
 * after having called efs_commit.
 *
 * Returns: 0 upon success. -1 upon failure.
 */

gint         
efs_dir_close (EFSDir *dir)
{
	EFSDriver *d;

	g_return_val_if_fail(dir != NULL, -1);

	d = dir->efs->driver;

	return d->fops->dir_close (dir);
}

/**
 * efs_dir_seek
 * @dir:
 * @offset:
 *
 * Description:
 *
 * Returns:
 */

gint32       
efs_dir_seek (EFSDir *dir, gint32 offset)
{
	EFSDriver *d;

	g_return_val_if_fail(dir != NULL, -1);

	d = dir->efs->driver;

	return d->fops->dir_seek (dir, offset);
}

/**
 * efs_dir_read
 * @dir: the directory to close.
 * 
 * Description: allows you to browse @dir content.
 *
 * Returns: it will return the next #EFSDirEntry 
 * till all of them have been returned. It will 
 * return after the last #EFSDirEntry has been 
 * returned.
 */
EFSDirEntry*
efs_dir_read (EFSDir *dir)
{
	EFSDriver *d;

	g_return_val_if_fail(dir != NULL, NULL);

	d = dir->efs->driver;

	return d->fops->dir_read (dir);
}
