/*
 * efs handling
 *
 * Author:
 *   Dietmar Maurer (dm@vlsivie.tuwien.ac.at)
 *
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/file.h>
#include <string.h>

#include "vefs.h"

EFSDriver *efs_driver_list[] = { 
	&efs_driver_simple, 
	&efs_driver_fsys,
	NULL 
};

static EFSDriver*
efs_find_driver (gchar *drivername)
{
	EFSDriver **h;
	gint i;

	h = efs_driver_list;

	i = 0; 
	while (h[i] && g_strcasecmp(drivername, h[i]->drivername)) i++;
			     
	return h[i];
}

static void
efs_free (EFS *efs)
{
	gint i;

	for (i=0;i<EFS_CACHE_SIZE;i++) {
		if (efs->cache[i].data) g_free (efs->cache[i].data);
	}
}

/**
 * efs_open:
 * @path: filesystem path
 * @flags: access flags
 * @mode: file permissions
 *
 * Description: Opens or creates a new #EFS within file @path. 
 *
 * @flags is %EFS_READ or %EFS_RDWR which request opening the #EFS read-only 
 * or read/write. flags  may  also  be  bitwise-or'd with one or more of the
 * following:
 *
 * %EFS_CREATE If the #EFS does not exist it will be created.
 *
 * %EFS_EXCL  When used with %EFS_CREATE, if the #EFS already exists it is an 
 * error and the open will fail.
 *
 * @mode  specifies  the  permissions to use if a new file is created.
 * You can specify them in octal with 0700 which gives 700 permission
 * to the file created.
 *
 * Returns: a pointer to the filesystem, or zero if an error occurred.
 **/

EFSDir *
efs_open (const char *path, gint flags, gint mode)
{
	EFSHeader *head;
	EFSDriver *d;
	EFS *efs;
	gint i, j;
	gint fd, pflags;
	gchar buf[512];
	struct stat sb;
	gchar drivername[1024];
	const gchar *filename;

	g_return_val_if_fail (path != NULL, NULL);

	/* extract driver name */
	i = 0; while ((i<1024)&&path[i]&&isalpha((int)path[i])) i++;
	filename = path;
	drivername[0] = 0;

	if (path[i]==':') {
		j=i; while (path[j]==':') j++;
		strncpy (drivername, path, i);
		drivername[i] = 0;
		filename = &path[j];
	}

	d = NULL;
	if (!(stat (filename, &sb))) {       /* exists */
		if ((flags&EFS_CREATE)&&(flags&EFS_EXCL)) return NULL;
		if (S_ISDIR(sb.st_mode)) {   /* use fsys: driver */
			if (drivername[0] && 
			    strcmp(drivername,"fsys")) return NULL;
			if (!(d = efs_find_driver("fsys"))) return NULL;
		} else {                     /* detect driver */
			if ((flags&EFS_CREATE)||
			    (flags&EFS_WRITE)) pflags = O_RDWR;
			else pflags = O_RDONLY;

			if (!(fd = open(filename, pflags, mode))) return NULL;
			if (!((read (fd, buf, 512))==512)) { 
				close(fd);
				return NULL;
			}
			close(fd);
			head = (EFSHeader *)buf;
			if (strncmp(head->efs_id,EFS_FILE_ID,4)) return NULL;

			strncpy (drivername, head->drivername, 12);
			drivername[12] = 0;
			
			if (!(d = efs_find_driver(drivername))) return NULL;
		}


	} else { /* file does not exist */		
		if (!(flags&EFS_CREATE)) return NULL; 
		if (drivername[0] && (!(d = efs_find_driver(drivername)))) {
			g_warning ("Unable to find driver!");
		}	
		if (!d) d = efs_driver_list[0];  /* default driver */  
	}

	efs = g_malloc0(sizeof(EFS));
	efs->driver = d;
	return d->sops->open (efs, filename, flags, mode);
}

/**
 * efs_close:
 * @root: reference to the #EFS root directory
 *
 * Description: Closes the #EFS and frees all resources.
 *
 * Returns: zero on success, or -1 if an error occurred.
 **/

gint
efs_close (EFSDir *root)
{
	EFSDriver *d;
	gint rval;
	g_return_val_if_fail (root != NULL, -1);

	d = root->efs->driver;

	if (!(rval = d->sops->close (root->efs))) {
		efs_free (root->efs);
		g_free (root);
	}

	return rval;
}

/**
 * efs_commit:
 * @root: reference to the #EFS root directory
 *
 * Description: Synchronize data on disk with memory.
 *
 * Returns: zero on success, or -1 if an error occurred.
 **/

gint
efs_commit (EFSDir *root)
{
	EFSDriver *d;

	g_return_val_if_fail (root != NULL, -1);

	d = root->efs->driver;

	if (!(root->efs->mode&EFS_WRITE)) return 0;

	return d->sops->commit (root->efs);
}

/**
 * efs_stat:
 * @dir: reference to a #EFS directory
 *
 * Description: get information about the #EFS file system. 
 *
 * Returns: a pointer to a #EFSStat structure, or zero on failure.
 **/

EFSStat*
efs_stat (EFSDir *dir)
{
	EFSDriver *d;

	g_return_val_if_fail (dir != NULL, NULL);

	d = dir->efs->driver;

	return d->sops->fsstat (dir->efs);
}

/**
 * efs_map:
 * @efs: #EFS reference
 * @ce: pointer to a cache entry
 * @block: block number
 *
 * Description: Load block number @block into the cache entry pointed by @ce. 
 *
 * Returns: zero on success, or -1 if an error occurred.
 **/

gint
efs_map (EFS *efs, EFSCacheEntry *ce, guint32 block)
{
	EFSDriver *d;

	g_return_val_if_fail (efs != NULL, -1);
	g_return_val_if_fail (ce != NULL, -1);

	d = efs->driver;

	return d->sops->map (efs, ce, block);
}

/**
 * efs_unmap:
 * @efs: #EFS reference
 * @ce: pointer to a cache entry
 *
 * Description: Writes cache entry @ce back to the file. 
 *
 * Returns: zero on success, or -1 if an error occurred.
 **/

gint
efs_unmap (EFS *efs, EFSCacheEntry *ce)
{
	EFSDriver *d;

	g_return_val_if_fail (efs != NULL, -1);
	g_return_val_if_fail (ce != NULL, -1);

	if (!(efs->mode&EFS_WRITE)) return -1;

	d = efs->driver;

	return d->sops->unmap (efs, ce);
}

