/*
 * Copyright 1999, 2000, Helix Code, Inc.
 *
 * Authors:
 *   Nat Friedman (nat@helixcode.com)
 *   Michael Meeks (michael@helixcode.com)
 */
#define BONOBO_UI_HANDLER_COMPILATION

#include <config.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkaccellabel.h>
#include <gtk/gtkitemfactory.h>
#include <gtk/gtkradiomenuitem.h>
#include <gtk/gtkmain.h>
#include <libgnome/libgnome.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-defs.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-app-helper.h>
#include <libgnomeui/gtkpixmapmenuitem.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-pixmap.h>
#include <libgnomeui/gnome-uidefs.h>

#include <bonobo/bonobo-ui-handler.h>
#include <bonobo/bonobo-widget.h>

#include "bonobo-uih-private.h"

/*
 *
 * Toolbars.
 *
 * The name of the toolbar is the first element in the toolbar path.
 * For example:
 *		"Common/Save"
 *		"Graphics Tools/Erase"
 * 
 * Where both "Common" and "Graphics Tools" are the names of 
 * toolbars.
 */



static UIRemoteAttributeData *
toolbar_item_remote_attribute_data_get (BonoboUIHandler *uih, const char *path)
{
	UIRemoteAttributeData *attrs = g_new0 (UIRemoteAttributeData, 1);
	CORBA_Environment        ev;

	CORBA_exception_init   (&ev);

	Bonobo_UIHandler_toolbar_item_get_attributes (uih->top_level_uih,
						     bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
						     path, &attrs->sensitive, &attrs->active,
						     &attrs->pos, &attrs->label, &attrs->hint,
						     &attrs->accelerator_key, &attrs->ac_mods,
						     &attrs->toggle_state, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);
		CORBA_exception_free (&ev);
		g_free (attrs);
		return NULL;
	}

	CORBA_exception_free (&ev);
	return attrs;
}


static gboolean
toolbar_item_remote_attribute_data_set (BonoboUIHandler *uih, const char *path,
					UIRemoteAttributeData *attrs)
{
	gboolean success = TRUE;
	CORBA_Environment ev;

	CORBA_exception_init   (&ev);

	Bonobo_UIHandler_toolbar_item_set_attributes (uih->top_level_uih,
						     bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
						     path, attrs->sensitive, attrs->active,
						     attrs->pos, attrs->label, attrs->hint,
						     attrs->accelerator_key, attrs->ac_mods, attrs->toggle_state, &ev);

	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);
		success = FALSE;
	}

	bonobo_ui_handler_remote_attribute_data_free (attrs);

	CORBA_exception_free   (&ev);

	return success;
}

static ToolbarRemoteAttributeData *
toolbar_remote_attribute_data_get (BonoboUIHandler *uih, const char *path)
{
	ToolbarRemoteAttributeData *attrs = g_new0 (ToolbarRemoteAttributeData, 1);
	CORBA_Environment        ev;

	CORBA_exception_init   (&ev);

	Bonobo_UIHandler_toolbar_get_attributes (uih->top_level_uih,
						bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
						path, &attrs->orientation, &attrs->style,
						&attrs->space_style, &attrs->relief_style,
						&attrs->space_size, &attrs->sensitive, &ev);

	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);
		CORBA_exception_free (&ev);
		g_free (attrs);
		return NULL;
	}

	CORBA_exception_free (&ev);
	return attrs;
}

static void
toolbar_remote_attribute_data_free (ToolbarRemoteAttributeData *attrs)
{
	g_return_if_fail (attrs != NULL);
	g_free (attrs);
}

/*
 * Convenience function to set attributes + free data.
 */
static gboolean
toolbar_remote_attribute_data_set (BonoboUIHandler *uih, const char *path,
				   ToolbarRemoteAttributeData *attrs)
{
	gboolean success = TRUE;
	CORBA_Environment ev;

	CORBA_exception_init   (&ev);

	Bonobo_UIHandler_toolbar_set_attributes (uih->top_level_uih,
						bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
						path, attrs->orientation, attrs->style,
						attrs->space_style, attrs->relief_style,
						attrs->space_size, attrs->sensitive, &ev);

	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);
		success = FALSE;
	}
	toolbar_remote_attribute_data_free (attrs);

	CORBA_exception_free   (&ev);

	return success;
}

static void
toolbar_item_remote_set_label (BonoboUIHandler *uih, const char *path,
			       const gchar *label_text)
{
	UIRemoteAttributeData *attrs;
	
	attrs = toolbar_item_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;
	CORBA_free (attrs->label);
	attrs->label = CORBA_string_dup (CORBIFY_STRING (label_text));
	toolbar_item_remote_attribute_data_set (uih, path, attrs);
}

static BonoboUIHandlerToolbarItem *
toolbar_make_item (const char                     *path,
		   BonoboUIHandlerToolbarItemType  type,
		   const char                     *label,
		   const char                     *hint,
		   int                             pos,
		   const Bonobo_Control            control,
		   BonoboUIHandlerPixmapType       pixmap_type,
		   gpointer                        pixmap_data,
		   guint                           accelerator_key,
		   GdkModifierType                 ac_mods,
		   BonoboUIHandlerCallbackFunc     callback,
		   gpointer                        callback_data)
{
	BonoboUIHandlerToolbarItem *item;

	item = g_new0 (BonoboUIHandlerToolbarItem, 1);

	item->path            = g_strdup (path);
	item->type            = type;
	item->label           = g_strdup (label);
	item->hint            = g_strdup (hint);
	item->pos             = pos;
	item->control         = control;
	item->pixmap_type     = pixmap_type;
	item->pixmap_data     = pixmap_data;
	item->accelerator_key = accelerator_key;
	item->ac_mods         = ac_mods;
	item->callback        = callback;
	item->callback_data   = callback_data;

	return item;
}

static BonoboUIHandlerToolbarItem *
toolbar_copy_item (BonoboUIHandlerToolbarItem *item)
{
	BonoboUIHandlerToolbarItem *copy;
	CORBA_Environment ev;

	copy = g_new0 (BonoboUIHandlerToolbarItem, 1);

	copy->path            = COPY_STRING (item->path);
	copy->type            = item->type;
	copy->hint            = COPY_STRING (item->hint);
	copy->pos             = item->pos;

	copy->label           = COPY_STRING (item->label);

	copy->pixmap_data     = bonobo_ui_handler_pixmap_copy_data (item->pixmap_type, item->pixmap_data);
	copy->pixmap_type     = item->pixmap_type;

	copy->accelerator_key = item->accelerator_key;
	copy->ac_mods         = item->ac_mods;

	copy->callback        = item->callback;
	copy->callback_data   = item->callback_data;

	CORBA_exception_init (&ev);
	copy->control         = CORBA_Object_duplicate (item->control, &ev);
	CORBA_exception_free (&ev);

	return copy;
}

static void
toolbar_free_data (BonoboUIHandlerToolbarItem *item)
{
	g_free (item->path);
	item->path = NULL;

	g_free (item->label);
	item->label = NULL;

	g_free (item->hint);
	item->hint = NULL;

	bonobo_ui_handler_pixmap_free_data (item->pixmap_type, item->pixmap_data);
}

static void
toolbar_free (BonoboUIHandlerToolbarItem *item)
{
	toolbar_free_data (item);
	g_free (item);
}

static Bonobo_UIHandler_ToolbarType
toolbar_type_to_corba (BonoboUIHandlerToolbarItemType type)
{
	switch (type) {
	case BONOBO_UI_HANDLER_TOOLBAR_END:
		g_warning ("Passing ToolbarTypeEnd through CORBA!");
		return Bonobo_UIHandler_ToolbarTypeEnd;

	case BONOBO_UI_HANDLER_TOOLBAR_ITEM:
		return Bonobo_UIHandler_ToolbarTypeItem;

	case BONOBO_UI_HANDLER_TOOLBAR_RADIOITEM:
		return Bonobo_UIHandler_ToolbarTypeRadioItem;

	case BONOBO_UI_HANDLER_TOOLBAR_RADIOGROUP:
		return Bonobo_UIHandler_ToolbarTypeRadioGroup;

	case BONOBO_UI_HANDLER_TOOLBAR_TOGGLEITEM:
		return Bonobo_UIHandler_ToolbarTypeToggleItem;

	case BONOBO_UI_HANDLER_TOOLBAR_SEPARATOR:
		return Bonobo_UIHandler_ToolbarTypeSeparator;

	case BONOBO_UI_HANDLER_TOOLBAR_CONTROL:
		return Bonobo_UIHandler_ToolbarTypeControl;

	default:
		g_warning ("toolbar_type_to_corba: Unknown toolbar type [%d]!", (gint) type);
	}

	return BONOBO_UI_HANDLER_TOOLBAR_ITEM;
}

static BonoboUIHandlerToolbarItemType
toolbar_corba_to_type (Bonobo_UIHandler_ToolbarType type)
{
	switch (type) {
	case Bonobo_UIHandler_ToolbarTypeEnd:
		g_warning ("Passing ToolbarTypeEnd through CORBA!");
		return BONOBO_UI_HANDLER_TOOLBAR_END;

	case Bonobo_UIHandler_ToolbarTypeItem:
		return BONOBO_UI_HANDLER_TOOLBAR_ITEM;

	case Bonobo_UIHandler_ToolbarTypeRadioItem:
		return BONOBO_UI_HANDLER_TOOLBAR_RADIOITEM;

	case Bonobo_UIHandler_ToolbarTypeRadioGroup:
		return BONOBO_UI_HANDLER_TOOLBAR_RADIOGROUP;

	case Bonobo_UIHandler_ToolbarTypeSeparator:
		return BONOBO_UI_HANDLER_TOOLBAR_SEPARATOR;

	case Bonobo_UIHandler_ToolbarTypeToggleItem:
		return BONOBO_UI_HANDLER_TOOLBAR_TOGGLEITEM;

	case Bonobo_UIHandler_ToolbarTypeControl:
		return BONOBO_UI_HANDLER_TOOLBAR_CONTROL;

	default:
		g_warning ("toolbar_corba_to_type: Unknown toolbar type [%d]!", (gint) type);
			
	}

	return BONOBO_UI_HANDLER_TOOLBAR_ITEM;
}


static ToolbarItemLocalInternal *
toolbar_local_get_item (BonoboUIHandler *uih, const char *path)
{
	GList *l;

	l = g_hash_table_lookup (uih->path_to_toolbar_callback, path);

	if (l == NULL)
		return NULL;

	return (ToolbarItemLocalInternal *) l->data;
}

inline static GtkWidget *
toolbar_toplevel_item_get_widget (BonoboUIHandler *uih, const char *path)
{
	return g_hash_table_lookup (uih->top->path_to_toolbar_item_widget, path);
}

inline static GtkWidget *
toolbar_toplevel_get_widget (BonoboUIHandler *uih, const char *name)
{
	return g_hash_table_lookup (uih->top->name_to_toolbar_widget, name);
}

static ToolbarToolbarLocalInternal *
toolbar_local_get_toolbar (BonoboUIHandler *uih, const char *path)
{
	GList *l;

	l = g_hash_table_lookup (uih->path_to_toolbar_toolbar, path);

	if (l == NULL)
		return NULL;

	return (ToolbarToolbarLocalInternal *) l->data;
}

static ToolbarToolbarInternal *
toolbar_toplevel_get_toolbar (BonoboUIHandler *uih, const char *name)
{
	GList *l;
	
	l = g_hash_table_lookup (uih->top->name_to_toolbar, name);

	if (l == NULL)
		return NULL;

	return (ToolbarToolbarInternal *) l->data;
}

static ToolbarItemInternal *
toolbar_toplevel_get_item (BonoboUIHandler *uih, const char *path)
{
	GList *l;

	l = g_hash_table_lookup (uih->top->path_to_toolbar_item, path);

	if (l == NULL)
		return NULL;

	return (ToolbarItemInternal *) l->data;
}

static ToolbarItemInternal *
toolbar_toplevel_get_item_for_containee (BonoboUIHandler *uih, const char *path,
					 Bonobo_UIHandler uih_corba)
{
	GList *l, *curr;

	l = g_hash_table_lookup (uih->top->path_to_toolbar_item, path);

	for (curr = l; curr != NULL; curr = curr->next) {
		ToolbarItemInternal *internal;
		CORBA_Environment ev;

		internal = (ToolbarItemInternal *) curr->data;

		CORBA_exception_init (&ev);

		if (CORBA_Object_is_equivalent (uih_corba, internal->uih_corba, &ev)) {
			CORBA_exception_free (&ev);

			return internal;
		}

		CORBA_exception_free (&ev);
	}

	return NULL;
}

static gboolean
toolbar_toplevel_item_is_head (BonoboUIHandler *uih, ToolbarItemInternal *internal)
{
	GList *l;

	l = g_hash_table_lookup (uih->top->path_to_toolbar_item, internal->item->path);

	if (l->data == internal)
		return TRUE;

	return FALSE;
}

static char *
toolbar_get_toolbar_name (const char *path)
{
	char *parent_name;
	char **toks;
	int i;

	toks = bonobo_ui_handler_path_tokenize (path);

	parent_name = g_strdup (toks [1]);

	for (i = 0; toks [i] != NULL; i ++) {
		g_free (toks [i]);
	}
	g_free (toks);

	return parent_name;
}

static void
toolbar_toplevel_toolbar_create_widget (BonoboUIHandler *uih, ToolbarToolbarInternal *internal)
{
	GtkWidget *toolbar_widget;

	/*
	 * Create the toolbar widget.
	 */
	toolbar_widget = gtk_toolbar_new (internal->orientation, internal->style);

	/*
	 * Store it in the widget hash table.
	 */
	g_hash_table_insert (uih->top->name_to_toolbar_widget, g_strdup (internal->name), toolbar_widget);

	/*
	 * Stuff it into the application.
	 */
	gnome_app_set_toolbar (uih->top->app, GTK_TOOLBAR (toolbar_widget));

	gtk_widget_show (toolbar_widget);
}

static void
toolbar_toplevel_item_remove_widgets (BonoboUIHandler *uih, const char *path)
{
	GtkWidget *toolbar_item_widget;
	gboolean found;
	char *orig_key;

	/*
	 * Get the toolbar item widget and remove its entry from the
	 * hash table.
	 */
	found = g_hash_table_lookup_extended (uih->top->path_to_toolbar_item_widget, path,
					      (gpointer *) &orig_key, (gpointer *) &toolbar_item_widget);

	if (found) {
		g_hash_table_remove (uih->top->path_to_toolbar_item_widget, path);
		g_free (orig_key);

		/*
		 * Destroy the widget.
		 */
		if (toolbar_item_widget != NULL)
			gtk_widget_destroy (toolbar_item_widget);
	}
}

static void
toolbar_toplevel_item_remove_widgets_recursive (BonoboUIHandler *uih, const char *path)
{
	ToolbarItemInternal *internal;

	internal = toolbar_toplevel_get_item (uih, path);

	if (internal->children != NULL) {
		GList *curr;

		for (curr = internal->children; curr != NULL; curr = curr->next)
			toolbar_toplevel_item_remove_widgets_recursive (
				uih, (char *) curr->data);
	}

	toolbar_toplevel_item_remove_widgets (uih, path);
}

static void
toolbar_local_toolbar_create (BonoboUIHandler *uih, const char *name)
{
	ToolbarToolbarLocalInternal *internal;
	GList *l;

	internal = g_new0 (ToolbarToolbarLocalInternal, 1);

	l = g_hash_table_lookup (uih->path_to_toolbar_toolbar, name);

	if (l == NULL) {
		l = g_list_prepend (l, internal);
		g_hash_table_insert (uih->path_to_toolbar_toolbar, g_strdup (name), l);
	} else {
		l = g_list_prepend (l, internal);
		g_hash_table_insert (uih->path_to_toolbar_toolbar, g_strdup (name), l);
	}
}

static void
toolbar_toplevel_toolbar_remove_widgets_recursive (BonoboUIHandler *uih, const char *name)
{
	ToolbarToolbarInternal *internal;
	GtkWidget *toolbar_widget;
	char *orig_key;
	GList *curr;

	internal = toolbar_toplevel_get_toolbar (uih, name);
	g_return_if_fail (internal != NULL);

	/*
	 * First destroy each of the toolbar's children.
	 */
	for (curr = internal->children; curr != NULL; curr = curr->next)
		toolbar_toplevel_item_remove_widgets (uih, (char *) curr->data);

	/*
	 * Get the toolbar widget and remove its entry from the hash table.
	 */
	g_hash_table_lookup_extended (uih->top->name_to_toolbar_widget, name,
				      (gpointer *) &orig_key, (gpointer *) &toolbar_widget);
	g_hash_table_remove (uih->top->name_to_toolbar_widget, name);
	g_free (orig_key);

	if (GNOME_IS_DOCK_ITEM (toolbar_widget->parent)) {
		/*
		 * Since GnomeDock's get left floating, and since there is
		 * no good api for removing toolbars we must assasinate our parent.
		 */
		gtk_widget_destroy (toolbar_widget->parent);
	} else {
		/*
		 * just destroy ourselfs then
		 */
		gtk_widget_destroy (toolbar_widget);
	}
}

static void
toolbar_toplevel_item_override_notify (BonoboUIHandler *uih, const char *path)
{
	ToolbarItemInternal *internal;
	CORBA_Environment ev;

	internal = toolbar_toplevel_get_item (uih, path);
	g_return_if_fail (internal != NULL);

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_toolbar_overridden (internal->uih_corba, path, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);
	}

	CORBA_exception_free (&ev);
}

static void
toolbar_toplevel_item_override_notify_recursive (BonoboUIHandler *uih,
						 const char *path)
{
	ToolbarItemInternal *internal;

	toolbar_toplevel_item_override_notify (uih, path);

	internal = toolbar_toplevel_get_item (uih, path);
	if (internal->children != NULL) {
		GList *curr;

		for (curr = internal->children; curr != NULL; curr = curr->next)
			toolbar_toplevel_item_override_notify_recursive
				(uih, (char *) curr->data);
	}
}

static void
toolbar_toplevel_toolbar_override_notify_recursive (BonoboUIHandler *uih, const char *name)
{
	ToolbarToolbarInternal *internal;
	CORBA_Environment ev;
	char *toolbar_path;
	GList *curr;

	internal = toolbar_toplevel_get_toolbar (uih, name);

	toolbar_path = g_strconcat ("/", name, NULL);

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_toolbar_overridden (internal->uih_corba, toolbar_path, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) internal->uih_corba, &ev);
	}

	CORBA_exception_free (&ev);

	g_free (toolbar_path);

	for (curr = internal->children; curr != NULL; curr = curr->next)
		toolbar_toplevel_item_override_notify_recursive (uih, (char *) curr->data);
}

static void
toolbar_toplevel_toolbar_check_override (BonoboUIHandler *uih, const char *name)
{
	ToolbarToolbarInternal *internal;

	/*
	 * Check for an existing toolbar by this name.
	 */
        internal = toolbar_toplevel_get_toolbar (uih, name);

	if (internal == NULL)
		return;

	/*
	 * There is a toolbar by this name, and so it must
	 * be overridden.
	 *
	 * We remove its widgets, its children's widgets, and notify
	 * the owner of each overridden item that it has been
	 * overridden.
	 */
	toolbar_toplevel_toolbar_remove_widgets_recursive (uih, name);

	/*
	 * Notification.
	 */
	toolbar_toplevel_toolbar_override_notify_recursive (uih, name);
}

static ToolbarToolbarInternal *
toolbar_toplevel_toolbar_store_data (BonoboUIHandler *uih, const char *name, Bonobo_UIHandler uih_corba)
{
	ToolbarToolbarInternal *internal;
	CORBA_Environment ev;
	GList *l;

	internal       = g_new0 (ToolbarToolbarInternal, 1);
	internal->name = g_strdup (name);

	internal->orientation = GTK_ORIENTATION_HORIZONTAL;
	internal->style       = GTK_TOOLBAR_BOTH;

	if (gnome_preferences_get_toolbar_lines ()) {
		internal->space_style = GTK_TOOLBAR_SPACE_LINE;
		internal->space_size  = GNOME_PAD * 2;
	} else {
		internal->space_style = GTK_TOOLBAR_SPACE_EMPTY;
		internal->space_size  = GNOME_PAD;
	}

	if (gnome_preferences_get_toolbar_relief_btn ())
		internal->relief      = GTK_RELIEF_NORMAL;
	else
		internal->relief      = GTK_RELIEF_NONE;

	if (gnome_preferences_get_toolbar_labels ())
		internal->style       = GTK_TOOLBAR_BOTH;
	else
		internal->style       = GTK_TOOLBAR_ICONS;

	CORBA_exception_init (&ev);
	internal->uih_corba = CORBA_Object_duplicate (uih_corba, &ev);
	CORBA_exception_free (&ev);

	/*
	 * Put this new toolbar at the front of the list of toolbars.
	 */
	l = g_hash_table_lookup (uih->top->name_to_toolbar, name);

	if (l != NULL) {
		l = g_list_prepend (l, internal);
		g_hash_table_insert (uih->top->name_to_toolbar, g_strdup (name), l);
	} else {
		l = g_list_prepend (NULL, internal);
		g_hash_table_insert (uih->top->name_to_toolbar, g_strdup (name), l);
	}

	return internal;
}

static void
toolbar_toplevel_toolbar_create (BonoboUIHandler *uih, const char *name,
				 Bonobo_UIHandler uih_corba)
{
	ToolbarToolbarInternal *internal;

	/*
	 * If there is already a toolbar by this name, notify its
	 * owner that it is being overridden.
	 */
	toolbar_toplevel_toolbar_check_override (uih, name);

	/*
	 * Store the internal data for this toolbar.
	 */
	internal = toolbar_toplevel_toolbar_store_data (uih, name, uih_corba);

	/*
	 * Create the toolbar widget.
	 */
	toolbar_toplevel_toolbar_create_widget (uih, internal);
}

static void
toolbar_remote_toolbar_create (BonoboUIHandler *uih, const char *name)
{
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_toolbar_create (uih->top_level_uih,
					bonobo_object_corba_objref (BONOBO_OBJECT (uih)), name,
					&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);
	}

	CORBA_exception_free (&ev);
}

void
impl_Bonobo_UIHandler_toolbar_create (PortableServer_Servant  servant,
				      const Bonobo_UIHandler  containee,
				      const CORBA_char       *name,
				      CORBA_Environment      *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));

	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}
	toolbar_toplevel_toolbar_create (uih, name, containee);
}

/**
 * bonobo_ui_handler_create_toolbar:
 * @uih: UI handle
 * @name: Name of toolbar eg. 'pdf', must not
 * include leading or trailing '/'s
 * 
 * creates a toolbar, at the path given by 'name'.
 * heirarchical toolbars are quite meaningless, hence
 * only a single level of toolbar path is needed.
 * 
 **/
void
bonobo_ui_handler_create_toolbar (BonoboUIHandler *uih, const char *name)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (name != NULL);
	g_return_if_fail (name[0] != '/');

	toolbar_local_toolbar_create (uih, name);

	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		toolbar_remote_toolbar_create (uih, name);
		return;
	}

	toolbar_toplevel_toolbar_create (uih, name, bonobo_object_corba_objref (BONOBO_OBJECT (uih)));
}

static void
toolbar_toplevel_toolbar_remove (BonoboUIHandler *uih, const char *name)
{
	g_warning ("toolbar remove unimplemented");
}

static void
toolbar_remote_toolbar_remove (BonoboUIHandler *uih, const char *name)
{
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_toolbar_remove (uih->top_level_uih,
					bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
					name, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);
	}

	CORBA_exception_free (&ev);
}

void
impl_Bonobo_UIHandler_toolbar_remove (PortableServer_Servant  servant,
				      Bonobo_UIHandler        containee,
				      const CORBA_char      *name,
		     CORBA_Environment     *ev)
{
/*	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));*/

	g_warning ("toolbar remove unimplemented");
}

/**
 * bonobo_ui_handler_remove_toolbar:
 */
void
bonobo_ui_handler_remove_toolbar (BonoboUIHandler *uih, const char *name)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (name != NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		toolbar_remote_toolbar_remove (uih, name);
	else
		toolbar_toplevel_toolbar_remove (uih, name);
}

/**
 * bonobo_ui_handler_set_toolbar:
 */
void
bonobo_ui_handler_set_toolbar (BonoboUIHandler *uih, const char *name,
			       GtkWidget *toolbar)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (name != NULL);
	g_return_if_fail (toolbar != NULL);
	g_return_if_fail (GTK_IS_TOOLBAR (toolbar));

	/* This is a toplevel-only function. */
	g_return_if_fail (uih->top_level_uih == CORBA_OBJECT_NIL);

	toolbar_toplevel_toolbar_check_override (uih, name);

	/*
	 * Store it in the widget hash table.
	 */
	g_hash_table_insert (uih->top->name_to_toolbar_widget, g_strdup (name), toolbar);

	toolbar_toplevel_toolbar_store_data (
		uih, name, bonobo_object_corba_objref (BONOBO_OBJECT (uih)));
}

/**
 * bonobo_ui_handler_get_toolbar_list:
 */
GList *
bonobo_ui_handler_get_toolbar_list (BonoboUIHandler *uih)
{
	g_return_val_if_fail (uih != NULL, NULL);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), NULL);

	return NULL;
}

static void
toolbar_local_do_path (const char *parent_path, BonoboUIHandlerToolbarItem *item)
{
	bonobo_ui_handler_local_do_path (parent_path, item->label, & item->path);
}

static void
toolbar_local_remove_parent_entry (BonoboUIHandler *uih, const char *path,
				   gboolean warn)
{
	ToolbarToolbarLocalInternal *parent;
	char *parent_path;
	GList *curr;

	parent_path = toolbar_get_toolbar_name (path);
	parent = toolbar_local_get_toolbar (uih, parent_path);
	g_free (parent_path);

	if (parent == NULL)
		return;

	for (curr = parent->children; curr != NULL; curr = curr->next) {
		if (! strcmp (path, (char *) curr->data)) {
			parent->children = g_list_remove_link (parent->children, curr);
			g_free (curr->data);
			g_list_free_1 (curr);
			return;
		}
	}

	if (warn)
		g_warning ("toolbar_local_remove_parent_entry: No entry in toolbar for item path [%s]!", path);
}

static void
toolbar_local_add_parent_entry (BonoboUIHandler *uih, const char *path)
{
	ToolbarToolbarLocalInternal *parent_internal_cb;
	char *parent_name;

	toolbar_local_remove_parent_entry (uih, path, FALSE);

	parent_name = toolbar_get_toolbar_name (path);
	parent_internal_cb = toolbar_local_get_toolbar (uih, parent_name);
	g_free (parent_name);

	if (parent_internal_cb == NULL) {
		/*
		 * If we don't have an entry for the parent,
		 * it doesn't matter.
		 */
		return;
	}

	parent_internal_cb->children = g_list_prepend (parent_internal_cb->children, g_strdup (path));
}

static void
toolbar_local_create_item (BonoboUIHandler *uih, const char *parent_path,
			   BonoboUIHandlerToolbarItem *item)
{
	ToolbarItemLocalInternal *internal_cb;
	GList *l, *new_list;

	/*
	 * Setup this item's path.
	 */
	toolbar_local_do_path (parent_path, item);

	/*
	 * Store it.
	 */
	internal_cb = g_new0 (ToolbarItemLocalInternal, 1);
	internal_cb->callback = item->callback;
	internal_cb->callback_data = item->callback_data;

	l = g_hash_table_lookup (uih->path_to_toolbar_callback, item->path);
	new_list = g_list_prepend (l, internal_cb);

	if (l == NULL)
		g_hash_table_insert (uih->path_to_toolbar_callback, g_strdup (item->path), new_list);
	else
		g_hash_table_insert (uih->path_to_toolbar_callback, item->path, new_list);

	/*
	 * Update the toolbar's child list.
	 */
	toolbar_local_add_parent_entry (uih, item->path);
}

void
impl_Bonobo_UIHandler_toolbar_overridden (PortableServer_Servant  servant,
					  const CORBA_char       *path,
					  CORBA_Environment      *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	ToolbarItemLocalInternal *internal_cb;
	char *parent_path;

	parent_path = bonobo_ui_handler_path_get_parent (path);

	internal_cb = toolbar_local_get_item (uih, path);

	if (internal_cb == NULL && strcmp (parent_path, "/")) {
		g_warning ("Received override notification for a toolbar item I don't own [%s]!", path);
		return;
	}

	gtk_signal_emit (GTK_OBJECT (uih), bonobo_ui_handler_signals [TOOLBAR_ITEM_OVERRIDDEN],
			 path, internal_cb ? internal_cb->callback_data : NULL);
}

static void
toolbar_toplevel_item_check_override (BonoboUIHandler *uih, const char *path)
{
	ToolbarItemInternal *internal;

	internal = toolbar_toplevel_get_item (uih, path);

	if (internal == NULL)
		return;

	toolbar_toplevel_item_remove_widgets_recursive (uih, path);

	toolbar_toplevel_item_override_notify_recursive (uih, path);
}

static gint
toolbar_toplevel_item_activated (GtkWidget *widget, ToolbarItemInternal *internal)
{
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_toolbar_activated (internal->uih_corba, internal->item->path, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (internal->uih),
			(CORBA_Object) internal->uih_corba, &ev);
	}
	
	CORBA_exception_free (&ev);

	return FALSE;
}

void
impl_Bonobo_UIHandler_toolbar_activated (PortableServer_Servant  servant,
					 const CORBA_char       *path,
					 CORBA_Environment      *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	ToolbarItemLocalInternal *internal_cb;

	internal_cb = toolbar_local_get_item (uih, path);

	if (internal_cb == NULL) {
		g_warning ("Received activation notification about a toolbar item I don't own [%s]!", path);
		return;
	}

	if (internal_cb->callback != NULL)
		internal_cb->callback (uih, internal_cb->callback_data, path);
	
	gtk_signal_emit (GTK_OBJECT (uih), bonobo_ui_handler_signals [MENU_ITEM_ACTIVATED], path, internal_cb->callback_data);
}

static GtkToolbarChildType
toolbar_type_to_gtk_type (BonoboUIHandlerToolbarItemType type)
{
	switch (type) {
	case BONOBO_UI_HANDLER_TOOLBAR_ITEM:
		return GTK_TOOLBAR_CHILD_BUTTON;

	case BONOBO_UI_HANDLER_TOOLBAR_RADIOITEM:
		return GTK_TOOLBAR_CHILD_RADIOBUTTON;

	case BONOBO_UI_HANDLER_TOOLBAR_SEPARATOR:
		return GTK_TOOLBAR_CHILD_SPACE;

	case BONOBO_UI_HANDLER_TOOLBAR_TOGGLEITEM:
		return GTK_TOOLBAR_CHILD_TOGGLEBUTTON;

	case BONOBO_UI_HANDLER_TOOLBAR_RADIOGROUP:
		g_warning ("toolbar_type_to_gtk_type: Trying to convert UIHandler radiogroup "
			   "to Gtk toolbar type!");
		return GTK_TOOLBAR_CHILD_SPACE;
		
	default:
		g_warning ("toolbar_type_to_gtk_type: Unkonwn UIHandler toolbar "
			   "item type [%d]!", type);
	}

	return GTK_TOOLBAR_CHILD_BUTTON;
}

static void
toolbar_toplevel_item_create_widgets (BonoboUIHandler *uih,
				      ToolbarItemInternal *internal)
{
	GtkWidget *toolbar_item;
	GtkWidget *toolbar;
	GtkWidget *pixmap;
	char *parent_name;

	g_return_if_fail (internal != NULL);
	g_return_if_fail (internal->item != NULL);

	parent_name = toolbar_get_toolbar_name (internal->item->path);
	toolbar = g_hash_table_lookup (uih->top->name_to_toolbar_widget, parent_name);
	g_free (parent_name);

	g_return_if_fail (toolbar != NULL);

	toolbar_item = NULL;
	switch (internal->item->type) {

	case BONOBO_UI_HANDLER_TOOLBAR_CONTROL:
		toolbar_item = bonobo_widget_new_control_from_objref (
			internal->item->control, CORBA_OBJECT_NIL);

		gtk_widget_show (toolbar_item);

		if (internal->item->pos >= 0)
			gtk_toolbar_insert_widget (GTK_TOOLBAR (toolbar),
						   toolbar_item,
						   internal->item->hint, NULL,
						   internal->item->pos);
		else
			gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar),
						   toolbar_item,
						   internal->item->hint, NULL);

		break;

	case BONOBO_UI_HANDLER_TOOLBAR_SEPARATOR:
		/*
		 * No toolbar_item in this case because GtkToolbar doesn't create a widget.
		 * FIXME: This separator cannot be removed; attempting to do so has no
		 * adverse effects but no positive ones either.
		 */
		gtk_toolbar_insert_space (GTK_TOOLBAR (toolbar), internal->item->pos);
		break;

	case BONOBO_UI_HANDLER_TOOLBAR_ITEM:
		pixmap = NULL;
		if (internal->item->pixmap_data != NULL && internal->item->pixmap_type != BONOBO_UI_HANDLER_PIXMAP_NONE)
			pixmap = bonobo_ui_handler_toplevel_create_pixmap (
				toolbar, internal->item->pixmap_type, internal->item->pixmap_data);


		if (internal->item->pos >= 0)
			toolbar_item = gtk_toolbar_insert_element (GTK_TOOLBAR (toolbar),
								   toolbar_type_to_gtk_type (internal->item->type),
								   NULL, internal->item->label, internal->item->hint,
								   NULL, pixmap,
								   GTK_SIGNAL_FUNC (toolbar_toplevel_item_activated),
								   internal, internal->item->pos);
		else
			toolbar_item = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
								   toolbar_type_to_gtk_type (internal->item->type),
								   NULL, internal->item->label, internal->item->hint,
								   NULL, pixmap,
								   GTK_SIGNAL_FUNC (toolbar_toplevel_item_activated),
								   internal);

		break;

	case BONOBO_UI_HANDLER_TOOLBAR_TOGGLEITEM:

		if (internal->item->pos >= 0)
			toolbar_item = gtk_toolbar_insert_element (GTK_TOOLBAR (toolbar),
								   toolbar_type_to_gtk_type (internal->item->type),
								   NULL, internal->item->label, internal->item->hint,
								   NULL, NULL,
								   GTK_SIGNAL_FUNC (toolbar_toplevel_item_activated),
								   internal, internal->item->pos);
		else
			toolbar_item = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
								   toolbar_type_to_gtk_type (internal->item->type),
								   NULL, internal->item->label, internal->item->hint,
								   NULL, NULL,
								   GTK_SIGNAL_FUNC (toolbar_toplevel_item_activated),
								   internal);

		break;

	case BONOBO_UI_HANDLER_TOOLBAR_RADIOITEM:
		g_warning ("Toolbar radio items/groups not implemented");
		return;
		
	default:
		g_warning ("toolbar_toplevel_item_create_widgets: Unkonwn toolbar item type [%d]!",
			   (gint) internal->item->type);
		return;
	}

	/* 
	 * toolbar_item is legitimately (?) NULL in at least the BONOBO_UI_HANDLER_TOOLBAR_SEPARATOR case. 
	 * Perhaps we should insert the NULL into the hash table anyway as a marker for the separator,
	 * so that later we can actually remove the separator.
	 */
	if (toolbar_item != NULL) {
		/* FIXME: Connect to signals and gtk_widget_add_accelerator */
	        gtk_object_ref (GTK_OBJECT (toolbar_item));
		g_hash_table_insert (uih->top->path_to_toolbar_item_widget, g_strdup (internal->item->path), toolbar_item);
	}
}

static void
toolbar_toplevel_item_remove_parent_entry (BonoboUIHandler *uih, const char *path,
					   gboolean warn)
{
	ToolbarToolbarInternal *parent;
	char *parent_name;
	GList *curr;

	parent_name = toolbar_get_toolbar_name (path);
	parent = toolbar_toplevel_get_toolbar (uih, parent_name);

	if (parent == NULL) {
		g_warning ("toolbar_toplevel_remove_parent_entry: Cannot find toolbar [%s] for path [%s]!",
			   path, parent_name);
		g_free (parent_name);
		return;
	}
	g_free (parent_name);

	for (curr = parent->children; curr != NULL; curr = curr->next) {
		if (! strcmp (path, (char *) curr->data)) {
			parent->children = g_list_remove_link (parent->children, curr);
			g_free (curr->data);
			g_list_free_1 (curr);
			return;
		}
	}

	if (warn)
		g_warning ("toolbar_toplevel_remove_parent_entry: No entry in parent for child path [%s]!", path);
}

static void
toolbar_toplevel_item_add_parent_entry (BonoboUIHandler *uih, const char *path)
{
	ToolbarToolbarInternal *internal;
	char *parent_name;

	toolbar_toplevel_item_remove_parent_entry (uih, path, FALSE);
	
	parent_name = toolbar_get_toolbar_name (path);
	internal = toolbar_toplevel_get_toolbar (uih, parent_name);
	g_free (parent_name);

	if (!internal)
		return;

	internal->children = g_list_prepend (internal->children, g_strdup (path));
}

static ToolbarItemInternal *
toolbar_toplevel_item_store_data (BonoboUIHandler *uih, Bonobo_UIHandler uih_corba,
				  BonoboUIHandlerToolbarItem *item)
{
	ToolbarItemInternal *internal;
	CORBA_Environment ev;
	GList *l;

	/*
	 * Create the internal representation of the toolbar item.
	 */
	internal = g_new0 (ToolbarItemInternal, 1);
	internal->item = toolbar_copy_item (item);
	internal->uih = uih;
	internal->sensitive = TRUE;

	CORBA_exception_init (&ev);
	internal->uih_corba = CORBA_Object_duplicate (uih_corba, &ev);
	CORBA_exception_free (&ev);

	/*
	 * Store it.
	 */
	l = g_hash_table_lookup (uih->top->path_to_toolbar_item, internal->item->path);

	if (l != NULL) {
		l = g_list_prepend (l, internal);
		g_hash_table_insert (uih->top->path_to_toolbar_item, internal->item->path, l);
	} else {
		l = g_list_prepend (NULL, internal);
		g_hash_table_insert (uih->top->path_to_toolbar_item, g_strdup (internal->item->path), l);
	}

	toolbar_toplevel_item_add_parent_entry (uih, item->path);

	return internal;
}

static void
toolbar_toplevel_item_set_sensitivity_internal (BonoboUIHandler *uih, ToolbarItemInternal *internal,
						gboolean sensitive)
{
	GtkWidget *toolbar_widget;

	internal->sensitive = sensitive;

	if (! toolbar_toplevel_item_is_head (uih, internal))
		return;

	toolbar_widget = toolbar_toplevel_item_get_widget (uih, internal->item->path);

	if (toolbar_widget != NULL) {
		/* toolbar_widget is legitimately (?) NULL in at least the BONOBO_UI_HANDLER_TOOLBAR_SEPARATOR case. */ 
		gtk_widget_set_sensitive (toolbar_widget, sensitive);
	}
}

static void
toolbar_toplevel_set_sensitivity_internal (BonoboUIHandler *uih, ToolbarToolbarInternal *internal,
					   gboolean sensitive)
{
	GtkWidget *toolbar_widget;

	toolbar_widget = toolbar_toplevel_get_widget (uih, internal->name);
	g_return_if_fail (toolbar_widget != NULL);

	gtk_widget_set_sensitive (toolbar_widget, sensitive);
}

static void
toolbar_toplevel_set_space_size_internal (BonoboUIHandler *uih, ToolbarToolbarInternal *internal,
					  gint size)
{
	GtkWidget *toolbar_widget;

	toolbar_widget = toolbar_toplevel_get_widget (uih, internal->name);
	g_return_if_fail (toolbar_widget != NULL);

	gtk_toolbar_set_space_size (GTK_TOOLBAR (toolbar_widget), size);
}

static void
toolbar_toplevel_set_style_internal (BonoboUIHandler *uih, ToolbarToolbarInternal *internal,
				     GtkToolbarStyle style)
{
	GtkWidget *toolbar_widget;

	toolbar_widget = toolbar_toplevel_get_widget (uih, internal->name);
	g_return_if_fail (toolbar_widget != NULL);

	gtk_toolbar_set_style (GTK_TOOLBAR (toolbar_widget), style);
}

static void
toolbar_toplevel_set_orientation (BonoboUIHandler *uih, ToolbarToolbarInternal *internal,
				  GtkOrientation orientation)
{
	GtkWidget *toolbar_widget;

	toolbar_widget = toolbar_toplevel_get_widget (uih, internal->name);
	g_return_if_fail (toolbar_widget != NULL);

	gtk_toolbar_set_orientation (GTK_TOOLBAR (toolbar_widget), orientation);
}

static void
toolbar_toplevel_set_button_relief (BonoboUIHandler *uih, ToolbarToolbarInternal *internal,
				    GtkReliefStyle relief_style)
{
	GtkWidget *toolbar_widget;

	toolbar_widget = toolbar_toplevel_get_widget (uih, internal->name);
	g_return_if_fail (toolbar_widget != NULL);

	gtk_toolbar_set_button_relief (GTK_TOOLBAR (toolbar_widget), relief_style);
}

static void
toolbar_toplevel_item_set_toggle_state_internal (BonoboUIHandler *uih, ToolbarItemInternal *internal,
						 gboolean active)
{
	GtkWidget *toolbar_widget;

	if (! toolbar_toplevel_item_is_head (uih, internal))
		return;

	toolbar_widget = toolbar_toplevel_item_get_widget (uih, internal->item->path);
	g_return_if_fail (toolbar_widget != NULL);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toolbar_widget), active);
}

static gboolean
toolbar_toplevel_item_get_toggle_state_internal (BonoboUIHandler *uih, ToolbarItemInternal *internal)
{
	GtkWidget *toolbar_widget;

	if (! toolbar_toplevel_item_is_head (uih, internal))
		return FALSE;

	toolbar_widget = toolbar_toplevel_item_get_widget (uih, internal->item->path);
	g_return_val_if_fail (toolbar_widget != NULL, FALSE);

	return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toolbar_widget));
}

static void
toolbar_toplevel_item_set_radio_state_internal (BonoboUIHandler *uih, ToolbarItemInternal *internal,
						gboolean active)
{
}

static void
toolbar_toplevel_create_item (BonoboUIHandler *uih, const char *parent_path,
			      BonoboUIHandlerToolbarItem *item,
			      Bonobo_UIHandler uih_corba)
{
	ToolbarItemInternal *internal;

	/*
	 * Check to see if this item is overriding an existing toolbar
	 * item.
	 */
	toolbar_toplevel_item_check_override (uih, item->path);

	/*
	 * Store an internal representation of the item.
	 */
	internal = toolbar_toplevel_item_store_data (uih, uih_corba, item);

	/*
	 * Create the toolbar item widgets.
	 */
	toolbar_toplevel_item_create_widgets (uih, internal);

	/*
	 * Set its sensitivity.
	 */
	toolbar_toplevel_item_set_sensitivity_internal (uih, internal, internal->sensitive);

	/*
	 * Set its active state.
	 */
	if (internal->item->type == BONOBO_UI_HANDLER_TOOLBAR_TOGGLEITEM)
		toolbar_toplevel_item_set_toggle_state_internal (uih, internal, FALSE);
	else if (internal->item->type == BONOBO_UI_HANDLER_TOOLBAR_RADIOITEM)
		toolbar_toplevel_item_set_radio_state_internal (uih, internal, FALSE);
}

static void
toolbar_remote_create_item (BonoboUIHandler *uih, const char *parent_path,
			    BonoboUIHandlerToolbarItem *item)
{
	Bonobo_UIHandler_iobuf *pixmap_buf;
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	pixmap_buf = bonobo_ui_handler_pixmap_data_to_corba (item->pixmap_type, item->pixmap_data);

	Bonobo_UIHandler_toolbar_create_item (uih->top_level_uih,
					     bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
					     item->path,
					     toolbar_type_to_corba (item->type),
					     CORBIFY_STRING (item->label),
					     CORBIFY_STRING (item->hint),
					     item->pos,
					     item->control,
					     bonobo_ui_handler_pixmap_type_to_corba (item->pixmap_type),
					     pixmap_buf,
					     (CORBA_unsigned_long) item->accelerator_key,
					     (CORBA_long) item->ac_mods,
					     &ev);

	bonobo_object_check_env (BONOBO_OBJECT (uih), (CORBA_Object) uih->top_level_uih, &ev);

	CORBA_exception_free (&ev);

	CORBA_free (pixmap_buf);
}

void
impl_Bonobo_UIHandler_toolbar_create_item (PortableServer_Servant        servant,
					   const Bonobo_UIHandler        containee_uih,
					   const CORBA_char             *path,
					   Bonobo_UIHandler_ToolbarType  toolbar_type,
					   const CORBA_char             *label,
					   const CORBA_char             *hint,
					   CORBA_long                    pos,
					   const Bonobo_Control          control,
					   Bonobo_UIHandler_PixmapType   pixmap_type,
					   const Bonobo_UIHandler_iobuf *pixmap_data,
					   CORBA_unsigned_long           accelerator_key,
					   CORBA_long                    modifier,
					   CORBA_Environment            *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	BonoboUIHandlerToolbarItem *item;
	char *parent_path;


	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}

	item = toolbar_make_item (path, toolbar_corba_to_type (toolbar_type),
				  UNCORBIFY_STRING (label),
				  UNCORBIFY_STRING (hint),
				  pos,
				  control,
				  bonobo_ui_handler_pixmap_corba_to_type (pixmap_type),
				  bonobo_ui_handler_pixmap_corba_to_data (pixmap_type, pixmap_data),
				  (guint) accelerator_key, (GdkModifierType) modifier,
				  NULL, NULL);

	parent_path = bonobo_ui_handler_path_get_parent (item->path);
	if (parent_path == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
	} else {
		toolbar_toplevel_create_item (uih, parent_path, item, containee_uih);
	}

	bonobo_ui_handler_pixmap_free_data (item->pixmap_type, item->pixmap_data);

	g_free (item);
	g_free (parent_path);
}

/**
 * bonobo_ui_handler_toolbar_add_one:
 */
void
bonobo_ui_handler_toolbar_add_one (BonoboUIHandler *uih, const char *parent_path,
				  BonoboUIHandlerToolbarItem *item)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (parent_path != NULL);
	g_return_if_fail (item != NULL);

	/*
	 * Store the toolbar item's local callback data.
	 */
	toolbar_local_create_item (uih, parent_path, item);

	/*
	 * Create the item.
	 */
	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		toolbar_remote_create_item  (uih, parent_path, item);
	else
		toolbar_toplevel_create_item  (uih, parent_path, item,
					       bonobo_object_corba_objref (BONOBO_OBJECT (uih)));
}

void
bonobo_ui_handler_toolbar_add_list (BonoboUIHandler *uih, const char *parent_path,
				   BonoboUIHandlerToolbarItem *item)
{
	BonoboUIHandlerToolbarItem *curr;

	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (parent_path != NULL);
	g_return_if_fail (item != NULL);

	for (curr = item; curr->type != BONOBO_UI_HANDLER_TOOLBAR_END; curr ++)
		bonobo_ui_handler_toolbar_add_tree (uih, parent_path, curr);
}

void
bonobo_ui_handler_toolbar_add_tree (BonoboUIHandler *uih, const char *parent_path,
				   BonoboUIHandlerToolbarItem *item)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (parent_path != NULL);
	g_return_if_fail (item != NULL);

	/*
	 * Add this toolbar item.
	 */
	bonobo_ui_handler_toolbar_add_one (uih, parent_path, item);

	/*
	 * Recursive add its children.
	 */
	if (item->children != NULL)
		bonobo_ui_handler_toolbar_add_list (uih, item->path, item->children);
}

void
bonobo_ui_handler_toolbar_new (BonoboUIHandler *uih, const char *path,
			      BonoboUIHandlerToolbarItemType type,
			      const char *label, const char *hint,
			      int pos, const Bonobo_Control control,
			      BonoboUIHandlerPixmapType pixmap_type,
			      gpointer pixmap_data, guint accelerator_key,
			      GdkModifierType ac_mods,
			      BonoboUIHandlerCallbackFunc callback,
			      gpointer callback_data)
{
	BonoboUIHandlerToolbarItem *item;
	char *parent_path;

	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	item = toolbar_make_item (path, type, label, hint, pos, control, pixmap_type,
				  pixmap_data, accelerator_key, ac_mods, callback, callback_data);

	parent_path = bonobo_ui_handler_path_get_parent (path);
	g_return_if_fail (parent_path != NULL);

	bonobo_ui_handler_toolbar_add_one (uih, parent_path, item);

	g_free (item);
	g_free (parent_path);
}

/**
 * bonobo_ui_handler_toolbar_new_item
 */
void
bonobo_ui_handler_toolbar_new_item (BonoboUIHandler *uih, const char *path,
				   const char *label, const char *hint, int pos,
				   BonoboUIHandlerPixmapType pixmap_type,
				   gpointer pixmap_data,
				   guint accelerator_key, GdkModifierType ac_mods,
				   BonoboUIHandlerCallbackFunc callback,
				   gpointer callback_data)
{
	bonobo_ui_handler_toolbar_new (uih, path,
				      BONOBO_UI_HANDLER_TOOLBAR_ITEM,
				      label, hint, pos, CORBA_OBJECT_NIL, pixmap_type,
				      pixmap_data, accelerator_key,
				      ac_mods, callback, callback_data);
}

/**
 * bonobo_ui_handler_toolbar_new_control:
 */
void
bonobo_ui_handler_toolbar_new_control (BonoboUIHandler *uih, const char *path,
				      int pos, Bonobo_Control control)
{
	bonobo_ui_handler_toolbar_new (uih, path,
				      BONOBO_UI_HANDLER_TOOLBAR_CONTROL,
				      NULL, NULL, pos, control, BONOBO_UI_HANDLER_PIXMAP_NONE,
				      NULL, 0, 0, NULL, NULL);
}


/**
 * bonobo_ui_handler_toolbar_new_separator:
 */
void
bonobo_ui_handler_toolbar_new_separator (BonoboUIHandler *uih, const char *path,
					int pos)
{
	bonobo_ui_handler_toolbar_new (uih, path,
				      BONOBO_UI_HANDLER_TOOLBAR_SEPARATOR,
				      NULL, NULL, pos, CORBA_OBJECT_NIL,
				      BONOBO_UI_HANDLER_PIXMAP_NONE,
				      NULL, 0, 0, NULL, NULL);
}

/**
 * bonobo_ui_handler_toolbar_new_radiogroup:
 */
void
bonobo_ui_handler_toolbar_new_radiogroup (BonoboUIHandler *uih, const char *path)
{
	bonobo_ui_handler_toolbar_new (uih, path, BONOBO_UI_HANDLER_TOOLBAR_RADIOGROUP,
				      NULL, NULL, -1, CORBA_OBJECT_NIL,
				      BONOBO_UI_HANDLER_PIXMAP_NONE,
				      NULL, 0, 0, NULL, NULL);
}

/**
 * bonobo_ui_handler_toolbar_new_radioitem:
 */
void
bonobo_ui_handler_toolbar_new_radioitem (BonoboUIHandler *uih, const char *path,
					const char *label, const char *hint, int pos,
					guint accelerator_key, GdkModifierType ac_mods,
					BonoboUIHandlerCallbackFunc callback,
					gpointer callback_data)
{
	bonobo_ui_handler_toolbar_new (uih, path, BONOBO_UI_HANDLER_TOOLBAR_RADIOITEM,
				      label, hint, pos, CORBA_OBJECT_NIL,
				      BONOBO_UI_HANDLER_PIXMAP_NONE,
				      NULL, accelerator_key, ac_mods, callback, callback_data);
}

/**
 * bonobo_ui_handler_toolbar_new_toggleitem:
 */
void
bonobo_ui_handler_toolbar_new_toggleitem (BonoboUIHandler *uih, const char *path,
					 const char *label, const char *hint, int pos,
					 guint accelerator_key, GdkModifierType ac_mods,
					 BonoboUIHandlerCallbackFunc callback,
					 gpointer callback_data)
{
	bonobo_ui_handler_toolbar_new (uih, path, BONOBO_UI_HANDLER_TOOLBAR_TOGGLEITEM,
				      label, hint, pos, CORBA_OBJECT_NIL,
				      BONOBO_UI_HANDLER_PIXMAP_NONE,
				      NULL, accelerator_key, ac_mods, callback,
				      callback_data);
}

static void
toolbar_local_remove_item (BonoboUIHandler *uih, const char *path)
{
	GList *l, *new_list;

	/*
	 * Remove the local data at the front of the list for this
	 * guy.
	 */
	l = g_hash_table_lookup (uih->path_to_toolbar_callback, path);

	if (l == NULL)
		return;

	/*
	 * Remove the list link.
	 */
	new_list = g_list_remove_link (l, l);
	g_list_free_1 (l);

	/*
	 * If there are no items left in the list, remove the hash
	 * table entry.
	 */
	if (new_list == NULL) {
		char *orig_key;

		g_hash_table_lookup_extended (uih->path_to_toolbar_callback, path, (gpointer *) &orig_key, NULL);
		g_hash_table_remove (uih->path_to_toolbar_callback, path);
		g_free (orig_key);

		toolbar_local_remove_parent_entry (uih, path, TRUE);
	} else
		g_hash_table_insert (uih->path_to_toolbar_callback, g_strdup (path), new_list);
	
}

static void
toolbar_toplevel_item_reinstate_notify (BonoboUIHandler *uih, const char *path)
{
	ToolbarItemInternal *internal;
	CORBA_Environment ev;

	internal = toolbar_toplevel_get_item (uih, path);

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_toolbar_reinstated (internal->uih_corba, internal->item->path, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) internal->uih_corba, &ev);
	}

	CORBA_exception_free (&ev);
}

void
impl_Bonobo_UIHandler_toolbar_reinstated (PortableServer_Servant  servant,
					  const CORBA_char       *path,
					  CORBA_Environment      *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	ToolbarItemLocalInternal *internal_cb;

	internal_cb = toolbar_local_get_item (uih, path);

	if (internal_cb == NULL) {
		g_warning ("Received reinstatement notification for toolbar item I don't own [%s]!", path);
		return;
	}

	gtk_signal_emit (
		GTK_OBJECT (uih), bonobo_ui_handler_signals [TOOLBAR_ITEM_REINSTATED],
			 path, internal_cb->callback_data);

}

static void
toolbar_toplevel_item_remove_notify (BonoboUIHandler *uih, ToolbarItemInternal *internal)
{
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_toolbar_removed (internal->uih_corba, internal->item->path, &ev);

	CORBA_exception_free (&ev);
}

void
impl_Bonobo_UIHandler_toolbar_removed (PortableServer_Servant  servant,
				       const CORBA_char       *path,
				       CORBA_Environment      *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	ToolbarItemLocalInternal *internal_cb;

	internal_cb = toolbar_local_get_item (uih, path);

	if (internal_cb == NULL) {
		g_warning ("Received removal notification for toolbar item I don't own [%s]!", path);
		return;
	}

	gtk_signal_emit (
		GTK_OBJECT (uih), bonobo_ui_handler_signals [TOOLBAR_ITEM_REMOVED],
		path, internal_cb->callback_data);

	toolbar_local_remove_item (uih, path);
}

static void
toolbar_toplevel_item_remove_data (BonoboUIHandler *uih, ToolbarItemInternal *internal)
{
	CORBA_Environment ev;
	char *orig_key;
	char *path;
	GList *l;

	g_assert (internal != NULL);

	path = g_strdup (internal->item->path);

	/*
	 * Get the list of toolbar items which match this path and remove
	 * the specified toolbar item.
	 */
	g_hash_table_lookup_extended (uih->top->path_to_toolbar_item, path,
				      (gpointer *) &orig_key, (gpointer *) &l);
	g_hash_table_remove (uih->top->path_to_toolbar_item, path);
	g_free (orig_key);

	l = g_list_remove (l, internal);

	if (l != NULL) {
		g_hash_table_insert (uih->top->path_to_toolbar_item, g_strdup (path), l);
	} else {
		/*
		 * Remove this path entry from the parent's child list.
		 */
		toolbar_toplevel_item_remove_parent_entry (uih, path, TRUE);
	}

	/*
	 * Free the internal data structures associated with this
	 * item.
	 */
	CORBA_exception_init (&ev);
	CORBA_Object_release (internal->uih_corba, &ev);
	CORBA_exception_free (&ev);

	toolbar_free (internal->item);

	g_free (internal);
	g_free (path);
}

void
bonobo_ui_handler_toolbar_toplevel_remove_item_internal (BonoboUIHandler     *uih,
							 ToolbarItemInternal *internal)
{
	gboolean is_head;
	char *path;

	is_head = toolbar_toplevel_item_is_head (uih, internal);

	path = g_strdup (internal->item->path);

	/*
	 * Destroy the widgets associated with this item.
	 */
	if (is_head)
		toolbar_toplevel_item_remove_widgets (uih, path);

	/*
	 * Notify the item's owner that the item has been removed.
	 */
	toolbar_toplevel_item_remove_notify (uih, internal);

	/*
	 * Remove the internal data structures associated with this
	 * item.
	 */
	toolbar_toplevel_item_remove_data (uih, internal);

	/*
	 * If we just destroyed an active toolbar item (one
	 * which was mapped to widgets on the screen), then
	 * we need to replace it.
	 */
	if (is_head) {
		ToolbarItemInternal *replacement;

		replacement = toolbar_toplevel_get_item (uih, path);

		if (replacement == NULL) {
			g_free (path);
			return;
		}

		/*
		 * Notify the replacement's owner that its item
		 * is being reinstated.
		 */
		toolbar_toplevel_item_reinstate_notify (uih, replacement->item->path);

		toolbar_toplevel_item_create_widgets (uih, replacement);
	}

	g_free (path);
}

static void
toolbar_toplevel_remove_item (BonoboUIHandler *uih, const char *path, Bonobo_UIHandler uih_corba)
{
	ToolbarItemInternal *internal;

	internal = toolbar_toplevel_get_item_for_containee (uih, path, uih_corba);
	g_return_if_fail (internal != NULL);

	bonobo_ui_handler_toolbar_toplevel_remove_item_internal (uih, internal);
}

static void
toolbar_remote_remove_item (BonoboUIHandler *uih, const char *path)
{
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_toolbar_remove_item (uih->top_level_uih,
					     bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
					     path,
					     &ev);

	CORBA_exception_free (&ev);
}

void
impl_Bonobo_UIHandler_toolbar_remove_item (PortableServer_Servant  servant,
					   Bonobo_UIHandler        containee_uih,
					   const CORBA_char       *path,
					   CORBA_Environment      *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	ToolbarItemInternal *internal;
	
	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}

        internal = toolbar_toplevel_get_item_for_containee (uih, path, containee_uih);
	if (internal == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
		return;
	}

	bonobo_ui_handler_toolbar_toplevel_remove_item_internal (uih, internal);
}

/**
 * bonobo_ui_handler_toolbar_remove:
 */
void
bonobo_ui_handler_toolbar_remove (BonoboUIHandler *uih, const char *path)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		toolbar_remote_remove_item (uih, path);
	else
		toolbar_toplevel_remove_item (uih, path, bonobo_object_corba_objref (BONOBO_OBJECT (uih)));
}

BonoboUIHandlerToolbarItem *
bonobo_ui_handler_toolbar_fetch_one (BonoboUIHandler *uih, const char *path)
{
	g_return_val_if_fail (uih != NULL, NULL);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), NULL);
	g_return_val_if_fail (path != NULL, NULL);

	g_warning ("Unimplemented toolbar method");
	return NULL;
}

GList *
bonobo_ui_handler_toolbar_fetch_by_callback (BonoboUIHandler *uih,
					    BonoboUIHandlerCallbackFunc callback)
{
	g_return_val_if_fail (uih != NULL, NULL);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), NULL);

	g_warning ("Unimplemented toolbar method");
	return NULL;
}

GList *
bonobo_ui_handler_toolbar_fetch_by_callback_data (BonoboUIHandler *uih,
						 gpointer callback_data)
{
	g_return_val_if_fail (uih != NULL, NULL);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), NULL);

	g_warning ("Unimplemented toolbar method");
	return NULL;
}

static BonoboUIHandlerMenuItemType
toolbar_uiinfo_type_to_uih (GnomeUIInfoType uii_type)
{
	switch (uii_type) {
	case GNOME_APP_UI_ENDOFINFO:
		return BONOBO_UI_HANDLER_TOOLBAR_END;

	case GNOME_APP_UI_ITEM:
		return BONOBO_UI_HANDLER_TOOLBAR_ITEM;

	case GNOME_APP_UI_TOGGLEITEM:
		return BONOBO_UI_HANDLER_TOOLBAR_TOGGLEITEM;

	case GNOME_APP_UI_RADIOITEMS:
		return BONOBO_UI_HANDLER_TOOLBAR_RADIOGROUP;

	case GNOME_APP_UI_SUBTREE:
		g_error ("Toolbar subtree doesn't make sense!");

	case GNOME_APP_UI_SEPARATOR:
		return BONOBO_UI_HANDLER_TOOLBAR_SEPARATOR;

	case GNOME_APP_UI_HELP:
		g_error ("Help unimplemented."); /* FIXME */

	case GNOME_APP_UI_BUILDER_DATA:
		g_error ("Builder data - what to do?"); /* FIXME */

	case GNOME_APP_UI_ITEM_CONFIGURABLE:
		g_warning ("Configurable item!");
		return BONOBO_UI_HANDLER_MENU_ITEM;

	case GNOME_APP_UI_SUBTREE_STOCK:
		g_error ("Toolbar subtree doesn't make sense!");

	default:
		g_warning ("Unknown UIInfo Type: %d", uii_type);
		return BONOBO_UI_HANDLER_TOOLBAR_ITEM;
	}
}

static void
toolbar_parse_uiinfo_one (BonoboUIHandlerToolbarItem *item, GnomeUIInfo *uii)
{
	item->path = NULL;

	if (uii->type == GNOME_APP_UI_ITEM_CONFIGURABLE)
		gnome_app_ui_configure_configurable (uii);

	item->type = toolbar_uiinfo_type_to_uih (uii->type);

	item->label = g_strdup (L_(uii->label));
	item->hint = g_strdup (L_(uii->hint));

	item->pos = -1;

	if (item->type == BONOBO_UI_HANDLER_TOOLBAR_ITEM ||
	    item->type == BONOBO_UI_HANDLER_TOOLBAR_RADIOITEM ||
	    item->type == BONOBO_UI_HANDLER_TOOLBAR_TOGGLEITEM)
		item->callback = uii->moreinfo;
	item->callback_data = uii->user_data;

	item->pixmap_type = bonobo_ui_handler_uiinfo_pixmap_type_to_uih (uii->pixmap_type);
	item->pixmap_data = bonobo_ui_handler_pixmap_copy_data (item->pixmap_type, uii->pixmap_info);
	item->accelerator_key = uii->accelerator_key;
	item->ac_mods = uii->ac_mods;
}

static void
toolbar_parse_uiinfo_tree (BonoboUIHandlerToolbarItem *tree, GnomeUIInfo *uii)
{
	toolbar_parse_uiinfo_one (tree, uii);

	if (tree->type == BONOBO_UI_HANDLER_TOOLBAR_RADIOGROUP) {
		tree->children = bonobo_ui_handler_toolbar_parse_uiinfo_list (uii->moreinfo);
	}
}

static void
toolbar_parse_uiinfo_one_with_data (BonoboUIHandlerToolbarItem *item, GnomeUIInfo *uii, void *data)
{
	toolbar_parse_uiinfo_one (item, uii);
	item->callback_data = data;
}

static void
toolbar_parse_uiinfo_tree_with_data (BonoboUIHandlerToolbarItem *tree, GnomeUIInfo *uii, void *data)
{
	toolbar_parse_uiinfo_one_with_data (tree, uii, data);

	if (tree->type == BONOBO_UI_HANDLER_TOOLBAR_RADIOGROUP) {
		tree->children = bonobo_ui_handler_toolbar_parse_uiinfo_list_with_data (uii->moreinfo, data);
	}
}

/**
 * bonobo_ui_handler_toolbar_parse_uiinfo_one:
 */
BonoboUIHandlerToolbarItem *
bonobo_ui_handler_toolbar_parse_uiinfo_one (GnomeUIInfo *uii)
{
	BonoboUIHandlerToolbarItem *item;

	g_return_val_if_fail (uii != NULL, NULL);

	item = g_new0 (BonoboUIHandlerToolbarItem, 1);

	toolbar_parse_uiinfo_one (item, uii);

	return item;
}

/**
 * bonobo_ui_handler_toolbar_parse_uiinfo_list:
 */
BonoboUIHandlerToolbarItem *
bonobo_ui_handler_toolbar_parse_uiinfo_list (GnomeUIInfo *uii)
{
	BonoboUIHandlerToolbarItem *list;
	BonoboUIHandlerToolbarItem *curr_uih;
	GnomeUIInfo *curr_uii;
	int list_len;

	g_return_val_if_fail (uii != NULL, NULL);

	/*
	 * Allocate the BonoboUIHandlerToolbarItem array.
	 */
	list_len = 0;
	for (curr_uii = uii; curr_uii->type != GNOME_APP_UI_ENDOFINFO; curr_uii ++)
		list_len ++;

	list = g_new0 (BonoboUIHandlerToolbarItem, list_len + 1);

	curr_uih = list;
	for (curr_uii = uii; curr_uii->type != GNOME_APP_UI_ENDOFINFO; curr_uii ++, curr_uih ++)
		toolbar_parse_uiinfo_tree (curr_uih, curr_uii);

	/* Parse the terminal entry. */
	toolbar_parse_uiinfo_one (curr_uih, curr_uii);

	return list;
}

/**
 * bonobo_ui_handler_toolbar_parse_uiinfo_tree:
 */
BonoboUIHandlerToolbarItem *
bonobo_ui_handler_toolbar_parse_uiinfo_tree (GnomeUIInfo *uii)
{
	BonoboUIHandlerToolbarItem *item_tree;

	g_return_val_if_fail (uii != NULL, NULL);

	item_tree = g_new0 (BonoboUIHandlerToolbarItem, 1);

	toolbar_parse_uiinfo_tree (item_tree, uii);

	return item_tree;
}

/**
 * bonobo_ui_handler_toolbar_parse_uiinfo_one_with_data:
 */
BonoboUIHandlerToolbarItem *
bonobo_ui_handler_toolbar_parse_uiinfo_one_with_data (GnomeUIInfo *uii, void *data)
{
	BonoboUIHandlerToolbarItem *item;

	g_return_val_if_fail (uii != NULL, NULL);

	item = g_new0 (BonoboUIHandlerToolbarItem, 1);

	toolbar_parse_uiinfo_one_with_data (item, uii, data);

	return item;
}

/**
 * bonobo_ui_handler_toolbar_parse_uiinfo_list_with_data:
 */
BonoboUIHandlerToolbarItem *
bonobo_ui_handler_toolbar_parse_uiinfo_list_with_data (GnomeUIInfo *uii, void *data)
{
	BonoboUIHandlerToolbarItem *list;
	BonoboUIHandlerToolbarItem *curr_uih;
	GnomeUIInfo *curr_uii;
	int list_len;

	g_return_val_if_fail (uii != NULL, NULL);

	/*
	 * Allocate the BonoboUIHandlerToolbarItem array.
	 */
	list_len = 0;
	for (curr_uii = uii; curr_uii->type != GNOME_APP_UI_ENDOFINFO; curr_uii ++)
		list_len ++;

	list = g_new0 (BonoboUIHandlerToolbarItem, list_len + 1);

	curr_uih = list;
	for (curr_uii = uii; curr_uii->type != GNOME_APP_UI_ENDOFINFO; curr_uii ++, curr_uih ++)
		toolbar_parse_uiinfo_tree_with_data (curr_uih, curr_uii, data);

	/* Parse the terminal entry. */
	toolbar_parse_uiinfo_one (curr_uih, curr_uii);

	return list;
}

/**
 * bonobo_ui_handler_toolbar_parse_uiinfo_tree_with_data:
 */
BonoboUIHandlerToolbarItem *
bonobo_ui_handler_toolbar_parse_uiinfo_tree_with_data (GnomeUIInfo *uii, void *data)
{
	BonoboUIHandlerToolbarItem *item_tree;

	g_return_val_if_fail (uii != NULL, NULL);

	item_tree = g_new0 (BonoboUIHandlerToolbarItem, 1);

	toolbar_parse_uiinfo_tree_with_data (item_tree, uii, data);

	return item_tree;
}

void
bonobo_ui_handler_toolbar_free_one (BonoboUIHandlerToolbarItem *item)
{
	g_return_if_fail (item != NULL);

	toolbar_free (item);
}

void
bonobo_ui_handler_toolbar_free_list (BonoboUIHandlerToolbarItem *array)
{
	BonoboUIHandlerToolbarItem *curr;

	g_return_if_fail (array != NULL);

	for (curr = array; curr->type != BONOBO_UI_HANDLER_TOOLBAR_END; curr ++) {
		if (curr->children != NULL)
			bonobo_ui_handler_toolbar_free_list (curr->children);

		toolbar_free_data (curr);
	}

	g_free (array);
}

void
bonobo_ui_handler_toolbar_free_tree (BonoboUIHandlerToolbarItem *tree)
{
	g_return_if_fail (tree != NULL);

	if (tree->type == BONOBO_UI_HANDLER_TOOLBAR_END)
		return;

	if (tree->children != NULL)
		bonobo_ui_handler_toolbar_free_list (tree->children);

	toolbar_free (tree);
}

static gint
toolbar_toplevel_item_get_pos (BonoboUIHandler *uih, const char *path)
{
	ToolbarItemInternal *internal;

	internal = toolbar_toplevel_get_item (uih, path);

	g_return_val_if_fail (internal != NULL, -1);

	return internal->item->pos;
}

static gint
toolbar_item_remote_get_pos (BonoboUIHandler *uih, const char *path)
{
	UIRemoteAttributeData *attrs;
	gint ans;

	attrs = toolbar_item_remote_attribute_data_get (uih, path);
	if (!attrs)
		return -1;

	ans = (gint)attrs->pos;

	bonobo_ui_handler_remote_attribute_data_free (attrs);

	return ans;
}

static void
toolbar_toplevel_item_set_sensitivity (BonoboUIHandler *uih, const char *path, gboolean sensitive)
{
	ToolbarItemInternal *internal;

	internal = toolbar_toplevel_get_item (uih, path);
	g_return_if_fail (internal != NULL);

	toolbar_toplevel_item_set_sensitivity_internal (uih, internal, sensitive);
}

static gboolean
toolbar_item_remote_get_sensitivity (BonoboUIHandler *uih, const char *path)
{
	UIRemoteAttributeData *attrs;
	gboolean                    ans;
	
	attrs = toolbar_item_remote_attribute_data_get (uih, path);

	if (! attrs)
		return TRUE;

	ans = (gboolean) attrs->sensitive;

	bonobo_ui_handler_remote_attribute_data_free (attrs);

	return ans;
}

static void
toolbar_item_remote_set_sensitivity (BonoboUIHandler *uih, const char *path,
				     gboolean sensitive)
{
	UIRemoteAttributeData *attrs;
	
	attrs = toolbar_item_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;
	attrs->sensitive = sensitive;
	toolbar_item_remote_attribute_data_set (uih, path, attrs);
}

void
bonobo_ui_handler_toolbar_item_set_sensitivity (BonoboUIHandler *uih, const char *path,
					       gboolean sensitive)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (path != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		toolbar_item_remote_set_sensitivity (uih, path, sensitive);
	else
		toolbar_toplevel_item_set_sensitivity (uih, path, sensitive);
}

static gboolean
toolbar_toplevel_item_get_sensitivity (BonoboUIHandler *uih, const char *path)
{
	ToolbarItemInternal *internal;

	internal = toolbar_toplevel_get_item (uih, path);
	g_return_val_if_fail (internal != NULL, FALSE);

	return internal->sensitive;
}

static gboolean
toolbar_toplevel_item_get_toggle_state (BonoboUIHandler *uih, const char *path)
{
	GtkWidget *widget;

	widget = toolbar_toplevel_item_get_widget (uih, path);
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_TOGGLE_BUTTON (widget), FALSE);

	return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
}

void
impl_Bonobo_UIHandler_toolbar_item_get_attributes (PortableServer_Servant   servant,
						   const Bonobo_UIHandler   containee,
						   const CORBA_char        *path,
						   CORBA_boolean           *sensitive,
						   CORBA_boolean           *active,
						   CORBA_long              *pos,
						   CORBA_char             **label,
						   CORBA_char             **hint,
						   CORBA_long              *accelerator_key,
						   CORBA_long              *ac_mods,
						   CORBA_boolean           *toggle_state,
						   CORBA_Environment       *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	ToolbarItemInternal *internal;

	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}
	internal = toolbar_toplevel_get_item (uih, path);

	if (internal == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
		return;
	}

	g_warning ("toolbar_item_get_attributes substantially unimplemented");

	*pos             = toolbar_toplevel_item_get_pos (uih, (char *) path);
	*sensitive       = toolbar_toplevel_item_get_sensitivity (uih, path);
	*toggle_state    = toolbar_toplevel_item_get_toggle_state (uih, path);

	/* FIXME.  The following is just to prevent uninitialized
           memory accesses.  (And, specifically, to prevent the stub
           from crashing if it tries to free the strings we are
           supposed to return.)  */
	*label           = CORBA_string_dup ("FIXME");
	*hint            = CORBA_string_dup ("FIXME");
	*accelerator_key = 0;
	*ac_mods         = 0;
}

void
impl_Bonobo_UIHandler_toolbar_item_set_attributes (PortableServer_Servant  servant,
						   const Bonobo_UIHandler  containee,
						   const CORBA_char       *path,
						   CORBA_boolean           sensitive,
						   CORBA_boolean           active,
						   CORBA_long              pos,
						   const CORBA_char       *label,
						   const CORBA_char       *hint,
						   CORBA_long              accelerator_key,
						   CORBA_long              ac_mods,
						   CORBA_boolean           toggle_state,
						   CORBA_Environment      *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	ToolbarItemInternal *internal;

	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}

	internal = toolbar_toplevel_get_item_for_containee (uih, path, containee);
	if (internal == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
		return;
	}

	toolbar_toplevel_item_set_sensitivity_internal  (uih, internal, sensitive);
	toolbar_toplevel_item_set_toggle_state_internal (uih, internal, toggle_state);

	g_warning ("toolbar_set_attributes substantially unimplemented");
}


/*
 * Ugly hack !, gtk_toolbar is insufficiently flexible for this by a long way !
 * In fact, this makes me want to puke.
 */
static void
toolbar_toplevel_item_set_pixmap_internal (BonoboUIHandler *uih, ToolbarItemInternal *internal,
					   BonoboUIHandlerPixmapType type, gpointer data)
{
	GtkToolbar *toolbar;
	GList      *l;
	int         i;

	/*
	 * Update the widgets.
	 */
	if (! toolbar_toplevel_item_is_head (uih, internal))
		return;

	toolbar = GTK_TOOLBAR (toolbar_toplevel_item_get_widget (uih, internal->item->path));

	i = 0;
	for (l = toolbar->children; l; l = l->next) {
		GtkToolbarChild *kid = l->data;

		if (kid->label && !strcmp (GTK_LABEL (kid->label)->label, internal->item->label))
			break;
		i++;
	}
	if (l) {
		GtkToolbarChild *kid = l->data;

		/* Ugly remove code */
		gtk_widget_destroy (kid->widget);
		toolbar->children = g_list_remove (toolbar->children, kid);

		/*
		 * Update the internal data for this toolbar item.
		 */
		bonobo_ui_handler_pixmap_free_data (internal->item->pixmap_type, internal->item->pixmap_data);
		
		internal->item->pixmap_type = type;
		internal->item->pixmap_data = bonobo_ui_handler_pixmap_copy_data (type, data);

		/* Now re-create it ... */
		internal->item->pos = i;
		toolbar_toplevel_item_create_widgets (uih, internal);
	} else
		g_warning ("Can't find toolbar to update pixmap");
}

static void
toolbar_toplevel_item_set_pixmap (BonoboUIHandler *uih, const char *path,
				  BonoboUIHandlerPixmapType type, gpointer data)
{
	ToolbarItemInternal *internal;

	internal = toolbar_toplevel_get_item_for_containee (uih, path,
							    bonobo_object_corba_objref (BONOBO_OBJECT (uih)));
	g_return_if_fail (internal != NULL);

	toolbar_toplevel_item_set_pixmap_internal (uih, internal, type, data);
}

static void
toolbar_item_remote_set_pixmap (BonoboUIHandler *uih, const char *path,
				BonoboUIHandlerPixmapType type, gpointer data)
{
	Bonobo_UIHandler_iobuf *pixmap_buff;
	CORBA_Environment ev;

	pixmap_buff = bonobo_ui_handler_pixmap_data_to_corba (type, data);
	
	CORBA_exception_init (&ev);

	Bonobo_UIHandler_toolbar_item_set_data (
		uih->top_level_uih, bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
		path, bonobo_ui_handler_pixmap_type_to_corba (type),
		pixmap_buff, &ev);

	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);
	}

	CORBA_exception_free (&ev);

	CORBA_free (pixmap_buff);
}

void
bonobo_ui_handler_toolbar_item_set_pixmap (BonoboUIHandler *uih, const char *path,
					  BonoboUIHandlerPixmapType type, gpointer data)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);
	g_return_if_fail (data != NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		toolbar_item_remote_set_pixmap (uih, path, type, data);
		return;
	}

	toolbar_toplevel_item_set_pixmap (uih, path, type, data);
}

void
bonobo_ui_handler_toolbar_item_get_pixmap (BonoboUIHandler *uih, const char *path,
					  BonoboUIHandlerPixmapType *type, gpointer *data)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	g_warning ("Unimplemented toolbar method");
}

void
impl_Bonobo_UIHandler_toolbar_item_set_data (PortableServer_Servant        servant,
					     Bonobo_UIHandler              containee_uih,
					     const CORBA_char             *path,
					     Bonobo_UIHandler_PixmapType   corba_pixmap_type,
					     const Bonobo_UIHandler_iobuf *corba_pixmap_data,
					     CORBA_Environment            *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	ToolbarItemInternal *internal;
	
	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}
	
	internal = toolbar_toplevel_get_item_for_containee (uih, path, containee_uih);
	
	if (internal == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
		return;
	}
	
	toolbar_toplevel_item_set_pixmap_internal (
		uih, internal, bonobo_ui_handler_pixmap_corba_to_type (corba_pixmap_type),
		bonobo_ui_handler_pixmap_corba_to_data (corba_pixmap_type, corba_pixmap_data));
}

void
impl_Bonobo_UIHandler_toolbar_item_get_data (PortableServer_Servant        servant,
					     Bonobo_UIHandler              containee_uih,
					     const CORBA_char             *path,
					     Bonobo_UIHandler_PixmapType  *corba_pixmap_type,
					     Bonobo_UIHandler_iobuf      **corba_pixmap_data,
					     CORBA_Environment            *ev)
{
	/* FIXME: Implement me! */
	g_warning ("Unimplemented: remote get pixmap");
}


inline static GtkOrientation
toolbar_corba_to_orientation (Bonobo_UIHandler_ToolbarOrientation o)
{
	switch (o) {
	case Bonobo_UIHandler_ToolbarOrientationHorizontal:
		return GTK_ORIENTATION_HORIZONTAL;
	case Bonobo_UIHandler_ToolbarOrientationVertical:
	default:
	        return GTK_ORIENTATION_VERTICAL;
	}
}

inline static Bonobo_UIHandler_ToolbarOrientation
toolbar_orientation_to_corba (GtkOrientation orientation)
{
	switch (orientation) {
	case GTK_ORIENTATION_VERTICAL:
		return Bonobo_UIHandler_ToolbarOrientationVertical;
	case GTK_ORIENTATION_HORIZONTAL:
	default:
		return Bonobo_UIHandler_ToolbarOrientationHorizontal;
	}
}

inline static GtkToolbarStyle
toolbar_corba_to_style (Bonobo_UIHandler_ToolbarStyle s)
{
	switch (s) {
	case Bonobo_UIHandler_ToolbarStyleIcons:
		return GTK_TOOLBAR_ICONS;
	case Bonobo_UIHandler_ToolbarStyleText:
		return GTK_TOOLBAR_TEXT;
	case Bonobo_UIHandler_ToolbarStyleBoth:
	default:
		return GTK_TOOLBAR_BOTH;
	}
}

inline static Bonobo_UIHandler_ToolbarStyle
toolbar_style_to_corba (GtkToolbarStyle s)
{
	switch (s) {
	case GTK_TOOLBAR_ICONS:
		return Bonobo_UIHandler_ToolbarStyleIcons;
	case GTK_TOOLBAR_TEXT:
		return Bonobo_UIHandler_ToolbarStyleText;
	case GTK_TOOLBAR_BOTH:
	default:
		return Bonobo_UIHandler_ToolbarStyleBoth;
	}
}

inline static GtkReliefStyle
relief_corba_to_style (Bonobo_UIHandler_ReliefStyle s)
{
	switch (s) {
	case Bonobo_UIHandler_ReliefHalf:
		return GTK_RELIEF_HALF;
	case Bonobo_UIHandler_ReliefNone:
		return GTK_RELIEF_NONE;
	case Bonobo_UIHandler_ReliefNormal:
	default:
		return GTK_RELIEF_NORMAL;
	}
}

inline static Bonobo_UIHandler_ReliefStyle
relief_style_to_corba (GtkReliefStyle s)
{
	switch (s) {
	case GTK_RELIEF_HALF:
		return Bonobo_UIHandler_ReliefHalf;
	case GTK_RELIEF_NONE:
		return Bonobo_UIHandler_ReliefNone;
	case GTK_RELIEF_NORMAL:
	default:
		return Bonobo_UIHandler_ReliefNormal;
	}
}


void
impl_Bonobo_UIHandler_toolbar_set_attributes (PortableServer_Servant                     servant,
					      const Bonobo_UIHandler                     containee,
					      const CORBA_char                          *name,
					      const Bonobo_UIHandler_ToolbarOrientation  orientation,
					      const Bonobo_UIHandler_ToolbarStyle        style,
					      const Bonobo_UIHandler_ToolbarSpaceStyle   space_style,
					      const Bonobo_UIHandler_ReliefStyle         relief_style,
					      const CORBA_long                           space_size,
					      const CORBA_boolean                        sensitive,
					      CORBA_Environment                         *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	ToolbarToolbarInternal *internal;

	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}
	internal = toolbar_toplevel_get_toolbar (uih, name);
	if (internal == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
		return;
	}

	toolbar_toplevel_set_orientation          (uih, internal, 
						   toolbar_corba_to_orientation (orientation));
	toolbar_toplevel_set_style_internal       (uih, internal,
						   toolbar_corba_to_style (style));
	toolbar_toplevel_set_space_size_internal  (uih, internal, space_size);
	toolbar_toplevel_set_sensitivity_internal (uih, internal, sensitive);
	toolbar_toplevel_set_button_relief        (uih, internal,
						   relief_corba_to_style (relief_style));
}

void
impl_Bonobo_UIHandler_toolbar_get_attributes (PortableServer_Servant               servant,
					      const Bonobo_UIHandler               containee,
					      const CORBA_char                    *path,
					      Bonobo_UIHandler_ToolbarOrientation *orientation,
					      Bonobo_UIHandler_ToolbarStyle       *style,
					      Bonobo_UIHandler_ToolbarSpaceStyle  *space_style,
					      Bonobo_UIHandler_ReliefStyle        *relief_style,
					      CORBA_long                          *space_size,
					      CORBA_boolean                       *sensitive,
					      CORBA_Environment                   *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	ToolbarItemInternal *internal;

	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}
	internal = toolbar_toplevel_get_item_for_containee (uih, path, containee);
	if (internal == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
		return;
	}

	g_warning ("Unimplemented");
}

void
bonobo_ui_handler_toolbar_set_orientation (BonoboUIHandler *uih, const char *path,
					  GtkOrientation orientation)
{
	ToolbarRemoteAttributeData *attrs;
	
	attrs = toolbar_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;

	attrs->orientation = toolbar_orientation_to_corba (orientation);
	toolbar_remote_attribute_data_set (uih, path, attrs);
}

GtkOrientation
bonobo_ui_handler_toolbar_get_orientation (BonoboUIHandler *uih, const char *path)
{
	ToolbarRemoteAttributeData *attrs;
	GtkOrientation              ans;
	
	attrs = toolbar_remote_attribute_data_get (uih, path);
	if (!attrs)
		return GTK_ORIENTATION_HORIZONTAL;

	ans = toolbar_orientation_to_corba (attrs->orientation);

	toolbar_remote_attribute_data_free (attrs);

	return ans;
}

gboolean
bonobo_ui_handler_toolbar_item_get_sensitivity (BonoboUIHandler *uih, const char *path)
{
	g_return_val_if_fail (uih != NULL, FALSE);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), FALSE);
	g_return_val_if_fail (path != NULL, FALSE);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		return toolbar_item_remote_get_sensitivity (uih, path);

	return toolbar_toplevel_item_get_sensitivity (uih, path);
}

void
bonobo_ui_handler_toolbar_item_set_label (BonoboUIHandler *uih, const char *path,
					 gchar *label)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (path != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));

	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		toolbar_item_remote_set_label (uih, path, label);
		return;
	}

/*	toolbar_toplevel_item_set_label (uih, path, label_text);*/
}

gchar *
bonobo_ui_handler_toolbar_item_get_label (BonoboUIHandler *uih, const char *path)
{
	g_return_val_if_fail (uih != NULL, NULL);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), NULL);
	g_return_val_if_fail (path != NULL, NULL);

	g_warning ("Unimplemented toolbar method");
	return NULL;
}

void
bonobo_ui_handler_toolbar_item_get_accel (BonoboUIHandler *uih, const char *path,
					 guint *accelerator_key, GdkModifierType *ac_mods)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	g_warning ("Unimplemented toolbar method");
}

void
bonobo_ui_handler_toolbar_item_set_callback (BonoboUIHandler *uih, const char *path,
					    BonoboUIHandlerCallbackFunc callback,
					    gpointer callback_data)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	g_warning ("Unimplemented toolbar method");
}

void
bonobo_ui_handler_toolbar_item_get_callback (BonoboUIHandler *uih, const char *path,
					    BonoboUIHandlerCallbackFunc *callback,
					    gpointer *callback_data)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	g_warning ("Unimplemented toolbar method");
}

static void
toolbar_item_remote_set_state (BonoboUIHandler *uih, const char *path,
			       gboolean state)
{
	UIRemoteAttributeData *attrs;
	
	attrs = toolbar_item_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;
	attrs->toggle_state = state;
	toolbar_item_remote_attribute_data_set (uih, path, attrs);
}


static void
toolbar_toplevel_item_set_state (BonoboUIHandler *uih, const char *path,
				 gboolean state)
{
	ToolbarItemInternal *internal;

	internal = toolbar_toplevel_get_item (uih, path);

	g_return_if_fail (internal != NULL);

	toolbar_toplevel_item_set_toggle_state_internal (uih, internal, state);
}

void
bonobo_ui_handler_toolbar_item_toggle_set_state (BonoboUIHandler *uih, const char *path,
						 gboolean state)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		toolbar_item_remote_set_state (uih, path, state);
		return;
	}

	toolbar_toplevel_item_set_state (uih, path, state);
}

static gboolean
toolbar_item_remote_get_state (BonoboUIHandler *uih, const char *path)
{
	UIRemoteAttributeData *attrs;
	gboolean ans;

	attrs = toolbar_item_remote_attribute_data_get (uih, path);
	if (!attrs)
		return FALSE;

	ans = (gboolean)attrs->toggle_state;

	bonobo_ui_handler_remote_attribute_data_free (attrs);

	return ans;
}

static gboolean
toolbar_toplevel_item_get_state (BonoboUIHandler *uih, const char *path)
{
	ToolbarItemInternal *internal;

	internal = toolbar_toplevel_get_item (uih, path);

	g_return_val_if_fail (internal != NULL, -1);

	return toolbar_toplevel_item_get_toggle_state_internal (uih, internal);
}

gboolean
bonobo_ui_handler_toolbar_item_toggle_get_state (BonoboUIHandler *uih, const char *path)
{
	g_return_val_if_fail (uih != NULL,FALSE);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), FALSE);
	g_return_val_if_fail (path != NULL, FALSE);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		return toolbar_item_remote_get_state (uih, path);

	return toolbar_toplevel_item_get_state (uih, path);
}

gboolean
bonobo_ui_handler_toolbar_item_radio_get_state (BonoboUIHandler *uih, const char *path)
{
	g_return_val_if_fail (uih != NULL, FALSE);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), FALSE);
	g_return_val_if_fail (path != NULL, FALSE);

	g_warning ("Unimplemented toolbar method");
	return FALSE;
}

void
bonobo_ui_handler_toolbar_item_radio_set_state (BonoboUIHandler *uih, const char *path,
					       gboolean state)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	g_warning ("Unimplemented toolbar method");
}

