/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/**
 * Bonobo Desktop Window Control implementation.
 * Enables applications to export their geometry control through CORBA.
 *
 * Author:
 *   Miguel de Icaza (miguel@kernel.org)
 *
 * Copyright 1999 Helix Code, Inc.
 */
#include <config.h>
#include <gtk/gtkplug.h>
#include <bonobo/bonobo-main.h>
#include <bonobo/bonobo-desktop-window.h>
#include <gdk/gdkprivate.h>
#include <gdk/gdkx.h>

/* Parent object class in GTK hierarchy */
static BonoboObjectClass *bonobo_desktop_window_parent_class;

/* The entry point vectors for the server we provide */
POA_Bonobo_Desktop_Window__vepv bonobo_desktop_window_vepv;

Bonobo_Desktop_Window
bonobo_desktop_window_corba_object_create (BonoboObject *object)
{
	POA_Bonobo_Desktop_Window *servant;
	CORBA_Environment ev;
	
	servant = (POA_Bonobo_Desktop_Window *) g_new0 (BonoboObjectServant, 1);
	servant->vepv = &bonobo_desktop_window_vepv;

	CORBA_exception_init (&ev);
	POA_Bonobo_Desktop_Window__init ((PortableServer_Servant) servant, &ev);
	if (ev._major != CORBA_NO_EXCEPTION){
		g_free (servant);
		CORBA_exception_free (&ev);
		return CORBA_OBJECT_NIL;
	}

	CORBA_exception_free (&ev);
	return (Bonobo_Desktop_Window) bonobo_object_activate_servant (object, servant);
}

/**
 * bonobo_desktop_window_construct:
 * @desktop_win: The BonoboDesktopWindow object to be initialized.
 * @corba_desktop_win: The CORBA Bonobo_Desktop_Window CORBA objeect.
 * @toplevel: Window we will have control over.
 *
 * Returns: the intialized BonoboDesktopWindow object.
 */
BonoboDesktopWindow *
bonobo_desktop_window_construct (BonoboDesktopWindow *desk_win,
				Bonobo_Desktop_Window corba_desktop_window,
				GtkWindow *toplevel)
{
	g_return_val_if_fail (desk_win != NULL, NULL);
	g_return_val_if_fail (BONOBO_IS_DESKTOP_WINDOW (desk_win), NULL);
	g_return_val_if_fail (corba_desktop_window != CORBA_OBJECT_NIL, NULL);
	g_return_val_if_fail (toplevel != NULL, NULL);
	g_return_val_if_fail (GTK_IS_WINDOW (toplevel), NULL);
	
	bonobo_object_construct (BONOBO_OBJECT (desk_win), corba_desktop_window);

	desk_win->window = toplevel;
	return desk_win;
}

/**
 * bonobo_desktop_window_new:
 * @toplevel: The toplevel Gtk window to control
 * container process.
 *
 */
BonoboDesktopWindow *
bonobo_desktop_window_new (GtkWindow *toplevel)
{
	BonoboDesktopWindow *desktop_window;
	Bonobo_Desktop_Window corba_desktop_window;
	
	g_return_val_if_fail (toplevel != NULL, NULL);
	g_return_val_if_fail (GTK_IS_WINDOW (toplevel), NULL);

	desktop_window = gtk_type_new (bonobo_desktop_window_get_type ());

	corba_desktop_window = bonobo_desktop_window_corba_object_create (BONOBO_OBJECT (desktop_window));
	if (corba_desktop_window == CORBA_OBJECT_NIL){
		gtk_object_destroy (GTK_OBJECT (desktop_window));
		return NULL;
	}
	
	return bonobo_desktop_window_construct (desktop_window, corba_desktop_window, toplevel);
}

static CORBA_char *
impl_desktop_window_get_title (PortableServer_Servant servant, CORBA_Environment * ev)
{
	BonoboDesktopWindow *desk_win = BONOBO_DESKTOP_WINDOW (bonobo_object_from_servant (servant));

	return CORBA_string_dup (desk_win->window->title);
}

static void
impl_desktop_window_set_title (PortableServer_Servant servant,
			       const CORBA_char *value,
			       CORBA_Environment * ev)
{
	BonoboDesktopWindow *desk_win = BONOBO_DESKTOP_WINDOW (bonobo_object_from_servant (servant));

	gtk_window_set_title (desk_win->window, value);
}

static Bonobo_Desktop_Window_Geometry
impl_desktop_window_get_geometry (PortableServer_Servant servant,
				  CORBA_Environment * ev)
{
	BonoboDesktopWindow *desk_win = BONOBO_DESKTOP_WINDOW (bonobo_object_from_servant (servant));
	Bonobo_Desktop_Window_Geometry geo;
	gint x, y;
	
	gdk_window_get_origin (GTK_WIDGET (desk_win->window)->window, &x, &y);
	geo.x = x;
	geo.y = y;
	geo.width = GTK_WIDGET (desk_win->window)->allocation.width;
	geo.height = GTK_WIDGET (desk_win->window)->allocation.height;

	return geo;
}

static void
impl_desktop_window_set_geometry (PortableServer_Servant               servant,
				  const Bonobo_Desktop_Window_Geometry *geo,
				  CORBA_Environment                   *ev)
{
	BonoboDesktopWindow *desk_win = BONOBO_DESKTOP_WINDOW (bonobo_object_from_servant (servant));

	gtk_widget_set_uposition (GTK_WIDGET (desk_win->window), geo->x, geo->y);
	gtk_widget_set_usize (GTK_WIDGET (desk_win->window), geo->width, geo->height);
}

static CORBA_unsigned_long
impl_desktop_window_get_window_id (PortableServer_Servant servant, CORBA_Environment * ev)
{
	BonoboDesktopWindow *desk_win = BONOBO_DESKTOP_WINDOW (bonobo_object_from_servant (servant));

	return GDK_WINDOW_XWINDOW (GTK_WIDGET (desk_win->window)->window);
}

/**
 * bonobo_desktop_window_get_epv:
 */
POA_Bonobo_Desktop_Window__epv *
bonobo_desktop_window_get_epv (void)
{
	POA_Bonobo_Desktop_Window__epv *epv;

	epv = g_new0 (POA_Bonobo_Desktop_Window__epv, 1);

	epv->_get_title = impl_desktop_window_get_title;
	epv->_set_title = impl_desktop_window_set_title;
	epv->get_geometry = impl_desktop_window_get_geometry;
	epv->set_geometry = impl_desktop_window_set_geometry;
	epv->get_window_id = impl_desktop_window_get_window_id;

	return epv;
}

static void
init_desktop_window_corba_class (void)
{
	/* Setup the vector of epvs */
	bonobo_desktop_window_vepv.Bonobo_Unknown_epv = bonobo_object_get_epv ();
	bonobo_desktop_window_vepv.Bonobo_Desktop_Window_epv = bonobo_desktop_window_get_epv ();
}

static void
bonobo_desktop_window_class_init (BonoboDesktopWindowClass *klass)
{
	bonobo_desktop_window_parent_class = gtk_type_class (bonobo_object_get_type ());
	init_desktop_window_corba_class ();
}

static void
bonobo_desktop_window_init (BonoboDesktopWindow *desktop_window)
{
}

/**
 * bonobo_desktop_window_get_type:
 *
 * Returns: The GtkType corresponding to the BonoboDesktopWindow class.
 */
GtkType
bonobo_desktop_window_get_type (void)
{
	static GtkType type = 0;

	if (!type){
		GtkTypeInfo info = {
			"BonoboDesktopWindow",
			sizeof (BonoboDesktopWindow),
			sizeof (BonoboDesktopWindowClass),
			(GtkClassInitFunc) bonobo_desktop_window_class_init,
			(GtkObjectInitFunc) bonobo_desktop_window_init,
			NULL, /* reserved 1 */
			NULL, /* reserved 2 */
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (bonobo_object_get_type (), &info);
	}

	return type;
}

/**
 * bonobo_desktop_window_control:
 * @object: Object to be aggregated.
 * @win: Window to be controled.
 *
 * Attaches a Bonobo::Desktop::Window corba handler to a Bonobo
 * object controlling the window @win.  
 */
void
bonobo_desktop_window_control (BonoboObject *object, GtkWindow *win)
{
	BonoboObject *win_control_object;
	
	g_return_if_fail (object != NULL);
	g_return_if_fail (win != NULL);
	g_return_if_fail (BONOBO_IS_OBJECT (object));
	g_return_if_fail (GTK_IS_WINDOW (win));

	win_control_object = BONOBO_OBJECT (bonobo_desktop_window_new (win));
	bonobo_object_add_interface (object, win_control_object);
}
