/*
 *  FileOps for zlib compressed files
 *
 * Author:
 *   Dietmar Maurer (dm@vlsivie.tuwien.ac.at)
 *
 *
 */

#include <string.h>
#include <sys/types.h>
#include <dirent.h>
#include <unistd.h>
#include <errno.h>

#include "vefs.h"
#include "simple.h"
#include "fsys.h"
#include "comp.h"

static EFSFile    *comp_file_open  (EFSDir *dir, const char *path, gint flags);
static EFSDir     *comp_dir_open   (EFSDir *efs, const char *path, gint flags);
static gint        comp_file_close (EFSFile *file);
static gint        comp_dir_close  (EFSDir *dir);
static gint32      comp_file_seek  (EFSFile *file, gint32 offset, gint whence);
static gint32      comp_dir_seek   (EFSDir *dir, gint32 offset);
static gint32      comp_file_read  (EFSFile *file, void *buf, gint32 count);
static gint32      comp_file_write (EFSFile *file, void *buf, gint32 count);
static gint        comp_file_trunc (EFSFile *file, guint32 size);
static EFSDirEntry *comp_dir_read   (EFSDir *dir);
static gint        comp_erase      (EFSDir *dir, const char *path); 
static gint        comp_rename     (EFSDir *dir, const char *old_path, 
				    const char *new_path);

EFSFileOps file_ops_comp = {
	comp_file_open,
	comp_dir_open,
	comp_file_close,
	comp_dir_close,
	comp_file_seek,
	comp_dir_seek,
	comp_file_read,
	comp_file_write,
	comp_file_trunc,
	comp_dir_read,
	comp_erase,
	comp_rename,
	NULL
};

EFSFileOps file_ops_comp_simple = {
	comp_file_open,
	comp_dir_open,
	comp_file_close,
	comp_dir_close,
	comp_file_seek,
	comp_dir_seek,
	comp_file_read,
	comp_file_write,
	comp_file_trunc,
	comp_dir_read,
	comp_erase,
	comp_rename,
	&file_ops_simple
};

EFSFileOps file_ops_comp_fsys = {
	comp_file_open,
	comp_dir_open,
	comp_file_close,
	comp_dir_close,
	comp_file_seek,
	comp_dir_seek,
	comp_file_read,
	comp_file_write,
	comp_file_trunc,
	comp_dir_read,
	comp_erase,
	comp_rename,
	&file_ops_fsys
};

/*
 * comp_file_open:
 *
 * see efs_file_open().
 */
static EFSFile*
comp_file_open (EFSDir *dir, const char *path, gint flags)
{
	EFSFileOps * base = dir->efs->driver->fops->base;
	EFSFile *file;
	COMPFile *cfile;
	//gint err;
	
	if (!(file = base->file_open (dir, path, flags))) return NULL;
	
	cfile = g_malloc0 (sizeof(COMPFile));
	((EFSFile *)cfile)->efs = file->efs;
	((EFSFile *)cfile)->mode = file->mode;
	cfile->of = file;
	
	if (file->mode&EFS_WRITE) {

	} else {


	}
	
	return ((EFSFile *)cfile);
}


/*
 * comp_dir_open:
 *
 * see efs_dir_open().
 */
static EFSDir*
comp_dir_open (EFSDir *dir, const char *path, gint flags)
{
	EFSFileOps * base = dir->efs->driver->fops->base;

	return base->dir_open (dir, path, flags);
}

/*
 * comp_file_close:
 *
 * see efs_file_close().
 */
static gint
comp_file_close (EFSFile *file)
{
	EFSFileOps * base = file->efs->driver->fops->base;
	EFSFile *of = ((COMPFile *)file)->of;

	return base->file_close (of);
}

/*
 * comp_dir_close:
 *
 * see efs_dir_close().
 */
static gint
comp_dir_close (EFSDir *dir)
{
	EFSFileOps * base = dir->efs->driver->fops->base;

	return base->dir_close (dir);
}

/*
 * comp_file_seek:
 *
 * see efs_file_seek().
 */
static gint32
comp_file_seek (EFSFile *file, gint32 offset, gint whence)
{
	EFSFileOps * base = file->efs->driver->fops->base;
	EFSFile *of = ((COMPFile *)file)->of;

	return base->file_seek (of, offset, whence);
}

/*
 * comp_dir_seek:
 *
 * see efs_dir_seek().
 */
static gint32
comp_dir_seek (EFSDir *dir, gint32 offset)
{
	EFSFileOps * base = dir->efs->driver->fops->base;

	return base->dir_seek (dir, offset);
}

/*
 * comp_file_read:
 *
 * see efs_file_read().
 */
static gint32
comp_file_read (EFSFile *file, void *buf, gint32 count)
{
	EFSFileOps * base = file->efs->driver->fops->base;
	EFSFile *of = ((COMPFile *)file)->of;

	return base->file_read (of, buf, count);
}

/*
 * comp_file_write:
 *
 * see efs_file_write().
 */
static gint32
comp_file_write (EFSFile *file, void *buf, gint32 count)
{
	EFSFileOps * base = file->efs->driver->fops->base;
	EFSFile *of = ((COMPFile *)file)->of;

	return base->file_write (of, buf, count);
}

/*
 * comp_dir_read:
 *
 * see efs_dir_read().
 */
static EFSDirEntry*
comp_dir_read (EFSDir *dir)
{
	EFSFileOps * base = dir->efs->driver->fops->base;

	return base->dir_read (dir);
}

/*
 * comp_file_trunc:
 *
 * see efs_file_trunc().
 */
static gint
comp_file_trunc (EFSFile *file, guint32 size)
{
	EFSFileOps * base = file->efs->driver->fops->base;
	EFSFile *of = ((COMPFile *)file)->of;

	return base->file_trunc (of, size);
}

/*
 * comp_erase:
 *
 * see efs_erase().
 */
static gint
comp_erase (EFSDir *dir, const char *path) 
{
	EFSFileOps * base = dir->efs->driver->fops->base;

	return base->erase (dir, path);
}

/*
 * comp_rename:
 *
 * see efs_rename().
 */
static gint         
comp_rename (EFSDir *dir, const char *old_path, const char *new_path)
{
	EFSFileOps * base = dir->efs->driver->fops->base;

	return base->rename (dir, old_path, new_path);
}


