//
// Author: 
//   Mikael Hallendal <micke@imendio.com>
//
// (C) 2004 Imendio HB
// 

using System;
using System.Collections;
using System.Linq;
using System.IO;
using System.Reflection;
using System.Threading;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Serialization;

namespace Imendio.Blam {

    public class ChannelCollection {

	public event ChannelEventHandler ChannelAdded;
	public event ChannelGroupEventHandler ChannelGroupAdded;
	public event ChannelEventHandler ChannelUpdated;
	public event ChannelEventHandler ChannelRemoved;
	
	public event ChannelEventHandler ChannelRefreshStarted;
	public event ChannelEventHandler ChannelRefreshFinished;

	public string FileName;

	private Object clock = new Object();

	private bool mDirty = false;
	private uint mTimeoutId = 0;
	private static uint WRITE_TIMEOUT = 5 * 60 * 1000; // Every 5 minutes
	
	static XmlSerializer serializer = new XmlSerializer (typeof (ChannelCollection));
	
	private ArrayList mChannels;
	[XmlElement ("Channel", typeof (Channel))]
	public ArrayList Channels {
	    get {
		return mChannels;
	    }
	    set {
		mChannels = value;
	    }
	}

	[XmlElement("Group", typeof(ChannelGroup))] public ArrayList Groups;

	public int NrOfUnreadItems {
	    get {
		int unread = 0;

		foreach (Channel channel in mChannels) {
		    unread += channel.NrOfUnreadItems;
		}

        foreach (ChannelGroup changrp in Groups) {
            unread += changrp.NrOfUnreadItems;
        }
		
		return unread;
	    }
	}

        public int NrOfNewItems {
            get {
                int new_items = 0;

                foreach(Channel channel in mChannels){
                    new_items += channel.NrOfNewItems;
                }

                foreach(ChannelGroup changrp in Groups){
                    new_items += changrp.NrOfNewItems;
                }

                return new_items;
            }
        }

	public ChannelCollection ()
	{
	}

	public static ChannelCollection LoadFromFile (string file)
	{
	    ChannelCollection collection;

	    try {
		collection = RealLoadFromFile (file);
	    } catch {
		try {
		    collection = RealLoadFromFile (Defines.APP_DATADIR + "/collection.xml");
		} catch {
		    collection = new ChannelCollection ();
		    collection.mChannels = new ArrayList ();
            collection.Groups = new ArrayList ();
		}
	    }

	    collection.FileName = file;
            /*
             * FIXME: Each chan should listen to each of its items
             */
            foreach(ChannelGroup chan in collection.Groups){
                chan.Updated += collection.Updated;
                chan.Setup();
            }
            foreach(Channel chan in collection.mChannels){
                chan.Updated += collection.Updated;
                chan.Setup();
            }

            return collection;
	}

	private static ChannelCollection RealLoadFromFile (string file)
	{
	    ChannelCollection collection;
	    XmlTextReader reader = null;

	    try {
		reader = new XmlTextReader (file);
	    } catch (Exception) {
		reader = new XmlTextReader (GetTempFile (file));
	    }
	    
            collection = (ChannelCollection) serializer.Deserialize (reader);
            
	    reader.Close ();

            if (collection.Channels == null) {
                collection.mChannels = new ArrayList ();
            }

	    return collection;
	}
	
	public void SaveToFile ()
	{
	    lock (clock) {
		if (!mDirty) {
		    return;
		}

		string tmpFile = GetTempFile (this.FileName);

		try {
		    Stream writer = new FileStream (tmpFile, FileMode.Create);
		    serializer.Serialize (writer, this);
		    writer.Close();
		} catch (Exception) {
		    Console.Error.WriteLine ("Failed to save to temporary file");
		    return;
		}

		// Move the file to the real one
		try {
			File.Copy(tmpFile, this.FileName, true);
			File.Delete(tmpFile);
		} catch (Exception e) {
		    Console.Error.WriteLine ("File replace error: " + e.Message);
		    return;
		}
		
		MarkAsDirty (false);
	    }
	}

        public void Add (IChannel channel)
        {
            // If we already have this feed, simply return
            if (mChannels.Cast<Channel>().Any(channel.Url.Equals))
                return;

        if(channel.Name == null){
            channel.Name = channel.Url;
        }

        mChannels.Add (channel);
            channel.RefreshAsync(updated => {
                MarkAsDirty(true);
                if (ChannelAdded != null)
                    ChannelAdded (channel);
            });
	}

        public void Add(ChannelGroup group, IChannel channel)
        {
            group.Add(channel);
            channel.RefreshAsync(updated => {
                channel.Updated += Updated;

                if(ChannelGroupAdded != null){
                    ChannelGroupAdded(group, channel);
                }

                MarkAsDirty(true);
            });
        }

	public void Updated (IChannel channel)
	{
	    MarkAsDirty (true);

	    if (ChannelUpdated != null) {
		ChannelUpdated (channel);
	    }
	}

	public void Remove (IChannel channel)
	{
        /* Try to find out from which list we need to remove. */
        if(mChannels.Contains(channel)){
            mChannels.Remove (channel);
        } else if(Groups.Contains(channel)){
            Groups.Remove(channel);
        } else {
            /* It's not a first-level channel or group. Dig deeper. */
            foreach(ChannelGroup group in Groups){
                if(group.Channels.Contains(channel)){
                    group.Channels.Remove(channel);
                    break;
                }
            }
        }

	    if (ChannelRemoved != null) {
		ChannelRemoved (channel);
	    }

        MarkAsDirty(true);
	}

        delegate void RefreshAllDelegate();

        public void RefreshAllAsync()
        {
            var fun = new RefreshAllDelegate(RefreshAll);
            fun.BeginInvoke(fun.EndInvoke, null);
        }

        public void RefreshAll ()
        {
            var all = mChannels.Cast<Channel>();
            foreach(ChannelGroup group in Groups){
                all.Union(group.Channels.Cast<Channel>());
            }

            Parallel.ForEach(mChannels.Cast<Channel>(), c => {
                bool updated = c.Refresh();
                if (updated) {
                    MarkAsDirty (true);
                }
                new MainloopEmitter (this.ChannelRefreshFinished, c).Emit ();
            });
        }

	private void EmitChannelRefreshStarted (IChannel channel)
	{
	    if (ChannelRefreshStarted != null) {
		ChannelRefreshStarted (channel);
	    }
	}
	
	private bool WriteTimeoutHandler ()
	{
	    SaveToFile ();
            ItemStore.Save();
	    
	    return false;
	}

	private void MarkAsDirty (bool dirty)
	{
	    lock (clock) {
		if (dirty) {
		    if (mTimeoutId != 0) {
			GLib.Source.Remove (mTimeoutId);
		    } 
		
		    mTimeoutId = GLib.Timeout.Add (WRITE_TIMEOUT, new GLib.TimeoutHandler (WriteTimeoutHandler));
		} else {
		    if (mTimeoutId != 0) {
			GLib.Source.Remove (mTimeoutId);
			mTimeoutId = 0;
		    }
		}

		mDirty = dirty;
	    }
	}

	private static string GetTempFile (string fileName) 
	{
	    return fileName + ".tmp";
	}
    }
}

