import sys, logging, base64

import gobject, gconf, dbus, os, copy

from ddm import DataModel
import bigboard.globals as globals
import bigboard.keyring as keyring
import libbig.logutil
from libbig.logutil import log_except
from bigboard.libbig.gutil import *

_logger = logging.getLogger("bigboard.Accounts")

_accounts_schema_dir = "/schemas/apps/bigboard/accounts/TEMPLATE"

# This class is immutable. Make sure to change the __copy__ and __deepcopy__ methods, and add an __eq__ method
# if you add setters to the class.
class AccountKind(object):
    def __init__(self, account_id, name, provided_by_server):
        super(AccountKind, self).__init__()        
        self.__id = account_id
        self.__name = name   
        self.__provided_by_server = provided_by_server

    def __copy__(self):
        return self 

    def __deepcopy__(self, memo):
        return self

    def get_id(self):
        return self.__id

    def get_name(self):
        return self.__name

    def get_provided_by_server(self):
        return self.__provided_by_server

    def __str__(self):
        return "<AccountKind:%s, provided_by_server:%s>" % (self.get_id(), self.get_provided_by_server())

KIND_GOOGLE = AccountKind("google", "Google", True)
# KIND_TWITTER = AccountKind("twitter", "Twitter", True)
# KIND_RTM = AccountKind("rtm", "Remember The Milk", False)
# KIND_LIBRARY = AccountKind("library", "LibraryThing", False)

ALL_ACCOUNT_KINDS = [KIND_GOOGLE]

def kind_from_string(s):
    for kind in ALL_ACCOUNT_KINDS:
        if s == kind.get_id():
            return kind
    return None

class Account(gobject.GObject):
    __gsignals__ = {
        "changed" : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ())
        }

    def __init__(self, kind, username='', password='', url='', enabled=True, gconf_dir=None):
        super(Account, self).__init__()
        
        self.__kind = kind
        self.__username = username
        self.__password = password
        self.__url = url   ## url is used for Google Apps For Your Domain
        self.__enabled = enabled
        self.__gconf_dir = gconf_dir

        _logger.debug("Created account %s" % (str(self)))

    def __copy__(self):
        return Account(self.__kind, self.__username, self.__password, self.__url, self.__enabled, self.__gconf_dir)

    def __deepcopy__(self, memo):
         return Account(copy.deepcopy(self.__kind, memo), copy.deepcopy(self.__username, memo), copy.deepcopy(self.__password, memo), \
                        copy.deepcopy(self.__url, memo), copy.deepcopy(self.__enabled, memo), copy.deepcopy(self.__gconf_dir, memo))

    def get_kind(self):
        return self.__kind

    def get_username(self):
        return self.__username

    def get_username_as_google_email(self):
        if self.__username == '':
            return self.__username
        elif not self.__url or self.__url == '':
            return self.__username + '@gmail.com'
        else:
            return self.__username + '@' + self.__url

    def get_password(self):
        return self.__password

    def get_url(self):
        return self.__url

    def get_enabled(self):
        return self.__enabled

    def _get_gconf_dir(self):
        return self.__gconf_dir

    def _set_gconf_dir(self, gconf_dir):
        if self.__gconf_dir != gconf_dir:
            self.__gconf_dir = gconf_dir
            self.emit('changed')              

    def _set_enabled(self, enabled):
        if self.__enabled != enabled:  
            self.__enabled = enabled
            self.emit('changed')              

    def _set_password(self, password):
        if self.__password != password:
            self.__password = password
            self.emit('changed') 

    def _update_from_origin(self, new_props):
        """This is the only way to modify an Account object. It should be invoked only on change notification or refreshed data from the original origin of the account."""
        _logger.debug("updating account from origin")
        ## check it out!
        changed = False
        for (key,value) in new_props.items():
            if value is None:
                value = ''

            if key != "password":
                _logger.debug("key %s, value %s" % (key, value))
            else:
                _logger.debug("key %s, value length %d" % (key, len(value)))

            old = getattr(self, '_Account__' + key)
            if old != value:
                setattr(self, '_Account__' + key, value)
                changed = True

        if changed:
            self.emit('changed')


    def __str__(self):
        return "<Account userame:%s, url:%s, kind:%s, gconf_dir:%s, enabled:%s>" % (self.get_username(), self.get_url(), str(self.get_kind()), self._get_gconf_dir(), self.get_enabled())

def associate_schema(schema_dir, new_dir):
    """

    This function is used when dynamically creating a directory in
    GConf that should be associated with a fixed schema.

    schema_dir -- the directory where the schema is stored
    new_dir -- the directory to associate the schema with

    """
    
    client = gconf.client_get_default()
    engine = gconf.engine_get_default()
    _logger.debug("associating schema schema_dir %s" % schema_dir)
    for entry in client.all_entries(schema_dir):
        _logger.debug("entry key %s" % entry.get_key())
        # There is a schema for each individual key, 
        key = os.path.join(new_dir, os.path.basename(entry.get_key()))
        engine.associate_schema(key, entry.get_key())

class Accounts(gobject.GObject):
    __gsignals__ = {
        "account-added" : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, (gobject.TYPE_PYOBJECT,)), 
        "account-removed" : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, (gobject.TYPE_PYOBJECT,))
        }

    def __init__(self, *args, **kwargs):
        super(Accounts, self).__init__(*args, **kwargs)

        self.__received_response_from_server = set()

        self.__model = globals.get_data_model()
        self.__model.add_ready_handler(self.__on_ready)

        self.__server_accounts = set()
        self.__gconf_accounts = set()
        self.__weblogin_accounts = set()
        self.__enabled_accounts = set()

        try:
            self.__weblogindriver_proxy = dbus.SessionBus().get_object('org.gnome.WebLoginDriver', '/weblogindriver')
            self.__weblogindriver_proxy.connect_to_signal("SignonChanged",
                                                           self.__on_signon_changed)
        except dbus.DBusException, e:
            _logger.debug("weblogindriver not available")
            self.__weblogindriver_proxy = None
        if self.__weblogindriver_proxy:
            self.__recheck_signons()

        self.__gconf = gconf.client_get_default()
        self.__gconf.add_dir('/apps/bigboard/accounts', gconf.CLIENT_PRELOAD_RECURSIVE)
        self.__gconf.notify_add('/apps/bigboard/accounts', self.__on_gconf_change)

        ## a dict from gconf directory name underneath /apps/bigboard/accounts to
        ## a dict of the gconf values underneath that account directory
        self.__gconf_info = {}

        self.__reload_from_gconf()

        if self.__model.ready:
            self.__on_ready()

    def __find_weblogin_account(self, kind, username, url = None):
        for a in self.__weblogin_accounts:
            _logger.debug("comparing to a weblogin account %s bool %s a.get_kind() %s kind %s" % (a, a.get_kind() == kind, a.get_kind(), kind))
            if a.get_kind() == kind and a.get_username() == username \
               and (url == None or a.get_url() == url or 
                    (a.get_kind() == KIND_GOOGLE and url == "gmail.com" and (a.get_url() == None or a.get_url() == ''))):
                return a

        return None

    def __find_account_by_gconf_dir(self, gconf_dir):
        for a in self.__gconf_accounts:
            if a._get_gconf_dir() == gconf_dir:
                return a

        return None

    def __get_gconf_accounts_by_kind(self, kind):
        gconf_accounts_by_kind = set()
        for a in self.__gconf_accounts:
            if a.get_kind() == kind:
                gconf_accounts_by_kind.add(a)
        return gconf_accounts_by_kind 

    def __get_server_accounts_by_kind(self, kind):
        server_accounts_by_kind = set()
        for a in self.__server_accounts:
            if a.get_kind() == kind:
                server_accounts_by_kind.add(a)
        return server_accounts_by_kind 

    @log_except(_logger)
    def __update_account(self, account):

        _logger.debug("Updating account %s" % (str(account)))

        ## note that "kind" is never updated (not allowed to change without
        ## making a new Account object)

        was_enabled = account in self.__enabled_accounts

        fields = {}

        gconf_dir = account._get_gconf_dir()
        if account.get_kind().get_provided_by_server() and (account in self.__server_accounts):
            self.__ensure_account_in_gconf(account)
        elif account.get_kind().get_provided_by_server() and (account not in self.__server_accounts) and (account.get_kind() in self.__received_response_from_server):
            #remove gconf dir if one exists, set enabled field to False
            if gconf_dir:
                self.__remove_gconf_dir(gconf_dir)  

        ## first, look for the info from our local gconf storage
        if gconf_dir and gconf_dir in self.__gconf_info:
            gconf_info = self.__gconf_info[gconf_dir]

            if 'username' in gconf_info:
                fields['username'] = gconf_info['username']

            if 'enabled' in gconf_info:
                fields['enabled'] = gconf_info['enabled']

            if 'url' in gconf_info:
                fields['url'] = gconf_info['url']
        elif gconf_dir and gconf_dir not in self.__gconf_info:
            ## Account object originating with a gconf item
            ## apparently now deleted, so disable. This
            ## does mean if you create a gconf item corresponding
            ## to something from weblogin driver, then delete the
            ## gconf item, the weblogin account doesn't come back
            ## until you restart the process. Clunky.
            ## deleting a gconf dir isn't what we'd normally do though,
            ## we'd normally set enabled=False in gconf, which would
            ## persist across restarts.
            fields['enabled'] = False
            self.__gconf_accounts.remove(account)

        ## after compositing all this information, update our account object
        account._update_from_origin(fields)

        ## second, look at weblogin driver, though we don't want to prefer
        ## its password over keyring, so there's some trickiness
        weblogin_password = None
        _logger.debug("will look for a weblogin account with kind %s username %s url %s" % (account.get_kind(), account.get_username(), account.get_url()))
        weblogin_account = self.__find_weblogin_account(account.get_kind(), account.get_username(), account.get_url()) 
        if weblogin_account:
            _logger.debug("weblogin account found")
            weblogin_password = weblogin_account.get_password()

        ## use updated information to find password
        fields = {}
        
        ## third, look for password in keyring
        k = keyring.get_keyring()                
        password = k.get_password(kind=account.get_kind().get_id(),
                                  username=account.get_username(),
                                  url=account.get_url())
        if password and 'password' not in fields:
            _logger.debug("using password from keyring")
            fields['password'] = password
            
        ## fourth, if no password in keyring, use the weblogin one
        if weblogin_password and 'password' not in fields:
            _logger.debug("using password from weblogin")
            fields['password'] = weblogin_password

        ## if no password found, the password has to be set to empty
        if 'password' not in fields:
            fields['password'] = ''

        ## update account object again if we might have the password
        if 'password' in fields:
            account._update_from_origin(fields)

        ## now add or remove the account from the set of enabled accounts
        if was_enabled and not account.get_enabled():
            self.__enabled_accounts.remove(account)
            self.emit('account-removed', account)
        elif not was_enabled and account.get_enabled():
            self.__enabled_accounts.add(account)
            self.emit('account-added', account) 
           
    @log_except(_logger)
    def __remove_gconf_dir(self, gconf_dir):
        base_key = '/apps/bigboard/accounts/' + gconf_dir
        success = self.__gconf.recursive_unset(base_key, gconf.UNSET_INCLUDING_SCHEMA_NAMES)
        _logger.debug("removed gconf dir %s success %s" % (base_key, success))
        self.__gconf.suggest_sync() 
        if self.__gconf_info.has_key(gconf_dir):
            del self.__gconf_info[gconf_dir]

    def __try_ensure_and_update_account_for_gconf_dir(self, gconf_dir):         
        account = self.__find_account_by_gconf_dir(gconf_dir)
        _logger.debug("ensuring account for %s gconf key %s", account, gconf_dir);
        if account:
            self.__update_account(account)
            return
        
        if gconf_dir not in self.__gconf_info:
            _logger.error("trying to create Account for a gconf dir that doesn't exist")
            return

        gconf_info = self.__gconf_info[gconf_dir]
        if 'kind' not in gconf_info:
            _logger.error("gconf account has no kind setting")
            self.__remove_gconf_dir(gconf_dir)
            return
        
        kind = kind_from_string(gconf_info['kind'])
        if not kind:
            _logger.error("unknown account kind in gconf")
            self.__remove_gconf_dir(gconf_dir)
            return

        # account = self.__find_weblogin_account_by_kind(kind)
        # if account:
        #    account._set_gconf_dir(gconf_dir)
        # else:
        account = Account(kind, gconf_dir=gconf_dir, enabled=False)

        self.__gconf_accounts.add(account)
        
        self.__update_account(account)

    def __remove_dirname(self, gconf_key):
        i = gconf_key.rfind('/')
        return gconf_key[i+1:]

    def __on_ready(self):
        if self.__model.self_resource != None:
            _logger.debug("will get online desktop accounts")
            # TODO: get Twitter accounts here too, in the future, have the
            # stocks dictate what we get here 
            query = self.__model.query_resource(self.__model.self_resource, "googleEnabledEmails +")
            query.add_handler(self.__on_google_enabled_emails)
            query.add_error_handler(self.__on_datamodel_error)        
            query.execute()
        
    def __on_datamodel_error(self, code, str):
        _logger.error("datamodel error %s: %s", code, str)        
        
    def __on_google_enabled_emails(self, myself):    
        _logger.debug("received some google enabled emails")
        myself.connect(self.__update_google_enabled_emails, 'googleEnabledEmails') 
        self.__update_google_enabled_emails(myself)

    def __update_google_enabled_emails(self, myself):
        new_google_accounts = set() 
        if not hasattr(myself, 'googleEnabledEmails'):
            _logger.debug("No googleEnabledEmails in DDM identity")
        elif len(myself.googleEnabledEmails) == 0:
            _logger.debug("DDM identity has 0 googleEnabledEmails")
        else:
            for email in myself.googleEnabledEmails:                        
                (username, url) = email.split('@', 1)
                username = str(username)
                url = str(url) 
                _logger.debug("got a googleEnabledEmail %s username: %s url: %s", email, username, url)        
                new_google_accounts.add(Account(KIND_GOOGLE, username=username, url=url))
            
        self.update_accounts_from_server(KIND_GOOGLE, new_google_accounts)
     
    def update_accounts_from_server(self, account_kind, new_accounts):
        if not account_kind.get_provided_by_server():
            _logger.error("update_accounts_from_server was called with an account kind that we don't get from server: %s" % account_kind)
            return

        existing_accounts = self.__get_server_accounts_by_kind(account_kind) 

        for new_account in new_accounts:                        
            # if acccount already in the list of server_accounts, don't do anything
            # if it is new, add it, and update information for it
            # if it is no longer reurned by the server, remove it and remove it from other
            # lists, including gconf
            account_found = False
            for existing_account in existing_accounts:
                if existing_account.get_username() == new_account.get_username() and \
                   existing_account.get_url() == new_account.get_url():
                   # we found it, we don't need to change anything about it
                   account_found = True
                   existing_accounts.remove(existing_account)
                   break
             
            if not account_found:             
                new_account_to_add = self.__find_account_in_gconf(new_account.get_kind(), new_account.get_username(), new_account.get_url())
                if new_account_to_add is None:
                    new_account_to_add = copy.deepcopy(new_account)
                self.__server_accounts.add(new_account_to_add)
                # this will add the account to gconf and enabled accounts, and check if 
                # we have a password for it
                self.__update_account(new_account_to_add)

        # clear out accounts that are no longer found in the list of accounts of this kind from the server
        for existing_account in existing_accounts:
            self.__server_accounts.remove(existing_account)
            # this should remove the account from gconf and gconf_accounts
            self.__update_account(existing_account)

        self.__received_response_from_server.add(account_kind)

        # make sure that all accounts in gconf correspond to the ones returned from the server;
        # this will remove old accounts from gconf 
        existing_accounts_in_gconf = self.__get_gconf_accounts_by_kind(account_kind) 
        _logger.debug("existing_accounts_in_gconf of type %s: %d" % (account_kind, len(existing_accounts_in_gconf)))
        for gconf_account in existing_accounts_in_gconf:
            self.__update_account(gconf_account) 

    def __get_gconf_info(self, gconf_dir):
        base_key = "/apps/bigboard/accounts/" + gconf_dir
        gconf_info = {}
        def get_account_prop(gconf, gconf_info, base_key, prop):
            try:
                value = gconf.get_value(base_key + '/' + prop)
            except ValueError:
                value = None
            if value:
                # _logger.debug("key %s, value %s" % (prop, value))  
                gconf_info[prop] = value
        get_account_prop(self.__gconf, gconf_info, base_key, 'kind')
        get_account_prop(self.__gconf, gconf_info, base_key, 'username')
        get_account_prop(self.__gconf, gconf_info, base_key, 'url')
        get_account_prop(self.__gconf, gconf_info, base_key, 'enabled')
        # GConf returns None for get_value if the vale is False  
        if 'enabled' not in gconf_info:
            gconf_info['enabled'] = False   
        return gconf_info

    @log_except(_logger)            
    def __reload_from_gconf(self):
        gconf_dirs = self.__gconf.all_dirs('/apps/bigboard/accounts')

        _logger.debug("Reloading %s from gconf" % (str(gconf_dirs)))

        new_gconf_infos = {}
        for gconf_dir in gconf_dirs:
            base_key = gconf_dir
            gconf_dir = self.__remove_dirname(gconf_dir)            
            new_gconf_infos[gconf_dir] = self.__get_gconf_info(gconf_dir)

            # GConf might already have some settings that did not have a schema.
            # This portion is only for bootstrapping these settings to have a schema.
            # We check if one of the entries that must be there (username) has a schema,
            # and associate a schema with the whole directory if it doesn't.
            username_entry = self.__gconf.get_entry(base_key + '/username', '', True)
            if username_entry and not username_entry.get_schema_name():
                associate_schema(_accounts_schema_dir, base_key)              
            elif not username_entry:
                _logger.warn("We expected username entry to be in %s, but it was not found" % base_key)
 
        self.__gconf_info = new_gconf_infos

        ## create any new accounts
        gconf_info_keys = copy.copy(self.__gconf_info.keys())  
        for gconf_dir in gconf_info_keys:
            self.__try_ensure_and_update_account_for_gconf_dir(gconf_dir)

        ## now update any old accounts that are no longer in gconf,
        ## which should result in enabled=False
        for a in self.__gconf_accounts:
            gconf_dir = a._get_gconf_dir()
            _logger.debug("processing gconf account for %s" % gconf_dir)
            if gconf_dir and gconf_dir not in self.__gconf_info:
                self.__update_account(a)
        
    @defer_idle_func(timeout=400)        
    def __on_gconf_change(self, *args):
        _logger.debug("gconf change notify for accounts")
        self.__reload_from_gconf()

    @log_except(_logger)
    def __check_signons(self, signons):
        # this processes signons for a specific hostname
        # TODO: extend this to store signons for different types of accounts once we are ready to support them;
        #       make sure we store signon information for GAFYD accounts
        for signon in signons: 
            if 'hint' not in signon: continue
            if signon['hint'] == 'GMail':
                username = signon['username']
                password = base64.b64decode(signon['password'])
                _logger.debug("got signon information %s" % str(signon))                
                account = self.__find_weblogin_account(KIND_GOOGLE, username)
                _logger.debug("found weblogin account %s" % str(account))
                if not account:
                    account = Account(KIND_GOOGLE, enabled=False, username=str(username), password=password)
                    self.__weblogin_accounts.add(account)
                    self.__update_account(account)
                elif password != account.get_password():
                    account._set_password(password)
                    self.__update_account(account)
            
    def __recheck_signons(self):
        self.__weblogindriver_proxy.GetSignons(reply_handler=self.__on_get_signons_reply,
                                               error_handler=self.__on_dbus_error)

    @log_except(_logger)
    def __on_get_signons_reply(self, signondata):
        _logger.debug("got signons reply")
        for hostname,signons in signondata.iteritems():
            self.__check_signons(signons)

    @log_except(_logger)
    def __on_signon_changed(self, signons):
        # TODO: __on_signon_changed is called with all signons for a given hostname.
        # However, we need the hostname to be passed in as an argument, since it is needed
        # for the case when no more signons remain for a given hostname.
        # We should make this change, but for now, we just never remove existing weblogin accounts.
        # (The list will be re-created when the bigboard is restarted.) We will update a password
        # for an existing weblogin account if the updated signon information is passed in here.
        # Once we make this change, we'll need to make sure to associate the right hostnames with
        # weblogin accounts, in case they don't match what we store in the url field.
        _logger.debug("signons changed: %s", signons)
        self.__check_signons(signons)

    @log_except(_logger)
    def __on_dbus_error(self, err):
        self.__logger.error("D-BUS error: %s", err)    

    def __find_unused_gconf_dir(self, kind):
        ## find an unused gconf dir
        i = 0
        while True:
            gconf_dir = kind.get_id() + "_" + str(i)
            if not self.__find_account_by_gconf_dir(gconf_dir):
                return gconf_dir
            else:
                i = i + 1

    def __find_account_in_gconf(self, kind, username, url):
        for a in self.__gconf_accounts:
            if a.get_kind() == kind and a.get_username() == username and a.get_url() == url:  
                return a
        return None 

    # new_properties is optional, it should be passed in if there are changes
    # to an existing account and we will update other data constructs that contain
    # this account based on the information from gconf
    #
    # an account object passed in to this method must have the gconf_dir set or must not be found in gconf
    def __ensure_account_in_gconf(self, account, new_properties={}):       
        gconf_dir = account._get_gconf_dir()
        new_gconf_dir = False
                     
        # create a new GConf dir if it is a completely new account    
        if not gconf_dir:
            if self.__find_account_in_gconf(account.get_kind(), account.get_username(), account.get_url()):
                _logger.error("found an account in gconf that matches the passed in account object %s, but the passed in account object is missing a gconf_dir" % str(account))
                return
 
            gconf_dir = self.__find_unused_gconf_dir(account.get_kind())
            new_gconf_dir = True
            account._set_gconf_dir(gconf_dir)
            self.__gconf_accounts.add(account)
            
        base_key = '/apps/bigboard/accounts/' + gconf_dir
        
        def set_account_prop(gconf, base_key, prop, value):
            _logger.debug("prop %s value %s" % (prop, str(value)))
            if isinstance(value, AccountKind):
                value = value.get_id()

            if isinstance(value, bool):
                self.__gconf.set_bool(base_key + '/' + prop, value)
            elif isinstance(value, str):
                self.__gconf.set_string(base_key + '/' + prop, value)
            else:
                _logger.error("prop %s with value %s has an unexpected type %s" % (prop, str(value), type(value)))

        if new_gconf_dir:
            set_account_prop(self.__gconf, base_key, 'kind', account.get_kind())

        if 'username' in new_properties:
            set_account_prop(self.__gconf, base_key, 'username', new_properties['username'])
        elif new_gconf_dir:
            set_account_prop(self.__gconf, base_key, 'username', account.get_username())

        if 'url' in new_properties:
            set_account_prop(self.__gconf, base_key, 'url', new_properties['url'])
        elif new_gconf_dir:
            set_account_prop(self.__gconf, base_key, 'url', account.get_url())
  
        ## enable it last, so we ignore the other settings until we do this
        if 'enabled' in new_properties:
            _logger.debug("setting new enabled property to be %s" % new_properties['enabled'])
            set_account_prop(self.__gconf, base_key, 'enabled', new_properties['enabled'])
        elif new_gconf_dir:
             set_account_prop(self.__gconf, base_key, 'enabled', account.get_enabled())

        if new_gconf_dir:
            associate_schema(_accounts_schema_dir, base_key)
            self.__gconf_info[gconf_dir]=self.__get_gconf_info(gconf_dir)

    def save_account_changes(self, account, new_properties):

        _logger.debug("Saving new props for account %s: %s" % (str(account), str(new_properties.keys())))

        ## special-case handling of password since it goes in the keyring
        if 'password' in new_properties:
            if 'username' in new_properties:
                username = new_properties['username']
            else:
                username = account.get_username()

            if 'url' in new_properties:
                url = new_properties['url']
            else:
                url = account.get_url()

            if not url:
                url=''          

            k = keyring.get_keyring()

            k.store_login(kind=account.get_kind().get_id(),
                          username=username,
                          url=url,
                          password=new_properties['password'])

        ## now do everything else by stuffing it in gconf
        self.__ensure_account_in_gconf(account, new_properties)
      
        ## keyring doesn't have change notification so we have to do the work for it
        ## if the password was the only thing that got updated, otherwise we should wait 
        ## till gconf is reloaded
        if len(new_properties) == 1 and 'password' in new_properties:
            ## this should notice a new password
            self.__update_account(account)

    # return a tuple with an account as the first element, and a boolean indicating if a new account was created
    def get_or_create_account(self, kind, username):        
        account = self.__find_account_in_gconf(kind, username, "")
        if account:         
            return (account, False)
          
        account = Account(kind, username=username)
        self.__ensure_account_in_gconf(account)        
        return (account, True)

    def remove_account(self, account):
        _logger.debug("will remove account with gconf_dir %s" % account._get_gconf_dir())  
        self.__remove_gconf_dir(account._get_gconf_dir())
        self.__update_account(account)

    def get_enabled_accounts(self):
        return self.__enabled_accounts

    def get_all_accounts(self):
        return self.__gconf_accounts

    def has_account(self, account):
        return account in self.__gconf_accounts

    def get_accounts_with_kind(self, kind):
        accounts = set()
        for a in self.__enabled_accounts:
            if a.get_kind() == kind:
                accounts.add(a)
        return accounts
    
__accounts = None

def get_accounts():
    global __accounts
    if not __accounts:
        __accounts = Accounts()

    return __accounts
