//
// FileNameFilter.cs
//
// Copyright (C) 2004 Novell, Inc.
//

//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//

using System;
using System.Collections;
using System.Text.RegularExpressions;
using System.IO;

using Beagle.Util;

namespace Beagle.Daemon.FileSystemQueryable {

	public class FileNameFilter {
		
		static string home_dir;
		private FileSystemModel model = null;

		static FileNameFilter ()
		{
			home_dir = PathFinder.HomeDir;
		}

		private class Pattern {
			private string exactMatch;
			private string prefix;
			private string suffix;
			private Regex  regex;

			public Pattern (string pattern)
			{
				if (pattern.StartsWith ("/") && pattern.EndsWith ("/")) {
					regex = new Regex (pattern.Substring (1, pattern.Length - 2));
					return;
				}

				int i = pattern.IndexOf ('*');
				if (i == -1) {
					exactMatch = pattern;
				} else {
					if (i > 0)
						prefix = pattern.Substring (0, i);
					if (i < pattern.Length-1)
						suffix = pattern.Substring (i+1);
				}
			}

			public bool IsMatch (string name)
			{
				if (exactMatch != null)
					return name == exactMatch;
				if (prefix != null && ! name.StartsWith (prefix))
					return false;
				if (suffix != null && ! name.EndsWith (suffix))
					return false;
				if (regex != null && ! regex.IsMatch (name))
					return false;
				return true;
			}

			public override string ToString ()
			{
				string str = "[pattern:";
				if (exactMatch != null)
					str += " exact=" + exactMatch;
				if (prefix != null)
					str += " prefix=" + prefix;
				if (suffix != null)
					str += " suffix=" + suffix;
				if (regex != null)
					str += " regex=" + regex.ToString ();
				str += "]";
				return str;
			}
		}

		//////////////////////////////////////////////////////////////////////
		
		private Hashtable patterns_to_ignore;

		private void SetupDefaultPatternsToIgnore ()
		{
			// Add our default skip patterns.
			// FIXME: This probably shouldn't be hard-wired.  Or should it?
			AddPatternToIgnore (new string [] {
						   ".*",
						   "*~",
						   "#*#",
						   "*.o",
						   "*.a",
						   "*.S",
						   "*.la",
						   "*.lo",
						   "*.loT",
						   "*.so",
						   "*.exe",
						   "*.dll",
						   "*.mdb",
						   "*.com",
						   "*.csproj",
						   "*.dsp",
						   "*.dsw",
						   "*.m4",
						   "*.pc",
						   "*.pc.in",
						   "*.in.in",
						   "*.omf",
						   "*.aux",
						   "*.tmp",
						   "autom4te.cache",
						   "po",
						   "aclocal",
						   "Makefile",
						   "Makefile.am",
						   "Makefile.in",
						   "CVS",
						   "SCCS",
						   // Garbage generated by the autotools
						   "conftest",
						   "confdefs.h",
						   "conftest.*",
						   "confstat*",
						   "/conf[0-9]+.sh/",
						   "/conf[0-9]+.file/"
			});
		}

		public void AddPatternToIgnore (string pattern)
		{
			if (! patterns_to_ignore.ContainsKey (pattern))
				patterns_to_ignore.Add (pattern, new Pattern (pattern));
		}
		
		public void AddPatternToIgnore (IEnumerable patterns)
		{
			foreach (string pattern in patterns)
				AddPatternToIgnore (pattern);
		}
		
		/////////////////////////////////////////////////////////////

		public FileNameFilter (FileSystemModel model)
		{
			this.model = model;
			patterns_to_ignore = new Hashtable ();
			SetupDefaultPatternsToIgnore ();
		}

		/////////////////////////////////////////////////////////////

		// FIXME: We could make this more efficient by storing more information.
		// In particular, if ~/foo is an IgnoreAll directory, we could store
		// that info in the PerDirectoryInfo of subdir ~/foo/bar so that
		// we could avoid walking up the chain of directories.
		public bool Ignore (string path)
		{
			if (! Path.IsPathRooted (path))
				path = Path.GetFullPath (path);

			foreach (FileSystemModel.Directory root in model.Roots)
				if (root.FullName == path)
					return false;

			string name = Path.GetFileName (path);

			foreach (Pattern pattern in patterns_to_ignore.Values)
				if (pattern.IsMatch (name))
					return true;

			string dir = Path.GetDirectoryName (path);

			// A file should be ignored if any of its parent directories
			// is ignored.
			return Ignore (dir);
		}

		public bool Ignore (FileSystemInfo info)
		{
			return Ignore (info.FullName);
		}
	}		


#if false
	class Test {
		static void Main (string [] args)
		{
			FileNameFilter filter = new FileNameFilter ();

			foreach (string arg in args) {
				FileSystemInfo info = null;
				if (Directory.Exists (arg))
					info = new DirectoryInfo (arg);
				else if (File.Exists (arg))
					info = new FileInfo (arg);

				if (info != null)
					Console.WriteLine ("{0} {1}",
							   filter.Ignore (info) ? "IGNORE" : "      ",
							   info.FullName);
			}
		}
	}
#endif
}
