/* Atomix -- a little mind game about atoms and molecules.
 * Copyright (C) 1999 Jens Finke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include <gnome.h>
#include "global.h"
#include "interface.h"
#include "support.h"
#include "board.h"
#include "playfield.h"
#include "preferences.h"
#include "main.h"
#include "goal.h"
#include "level.h"
#include "gtk-time-limit.h"
#include "util.h"

/****************************
 *    global variables
 */
struct _LevelData
{
	gint no_level;   /* number of level */
	Level *level;    /* actual level */
	gdouble score;   /* players score */
	Theme *theme;    /* actual used theme */
};

typedef struct _LevelData LevelData;

static GtkWidget *app;              /* main application */
static LevelData *level_data;       /* all infos about a level */
gint game_state;                    /* current state of the game */


/**
 *  dialog callbacks
 */
gint next_level_dlg_cb(gpointer data);
void save_score(gdouble score);
void game_clean_up(void);

void game_load_level(LevelData *data);
void update_menu_item_state(gint state);
void update_player_info(LevelData *data);
void clear_player_info(void);
void create_user_config_dir(void);
GtkWidget* get_time_limit_widget(void);

void game_bonus_level_timeout(GtkWidget *widget, gpointer user_data);

/***************************
 *    functions
 */
GtkWidget* get_app() 
{
	return app;
}

Level* get_actual_level()
{
	return level_data->level;
}

Theme* get_actual_theme()
{
	return level_data->theme;
}

gint get_game_state()
{
	return game_state;
}

GtkWidget* get_time_limit_widget(void)
{
	GtkWidget *alignment;
	alignment = lookup_widget(get_app(), "clock");
	return GTK_BIN(alignment)->child;
}

void game_init()
{
	GtkWidget* clock;

	/* init level data struct */
	level_data = g_malloc(sizeof(LevelData));
	level_data->no_level = 1;
	level_data->level = NULL;
	level_data->score = 0.0;
	level_data->theme = NULL;

	/* init the board */
	board_init();

	/* init goal */
	goal_init();
    
	/* connect the timeout function */
	clock = get_time_limit_widget();
	gtk_signal_connect(GTK_OBJECT(clock), "timeout",
			   GTK_SIGNAL_FUNC(game_level_timeout),NULL);

	/* enable only usable menu items according to state */
	game_state = GAME_NOT_RUNNING;
	update_menu_item_state(game_state);

	/* clear the level info */
	clear_player_info();
}

void game_new()
{
	level_data->level = level_load_xml(level_get_first_level());

	if(level_data->level) 
	{
		board_hide_message(BOARD_MSG_NEW_GAME);

		/* init level data */
		level_data->no_level = 1;
		level_data->score = 0.0;

		/* update game state */
		game_state = GAME_RUNNING;
		update_menu_item_state(game_state);

		/* load the level */
		game_load_level(level_data);    
		
	}
	else
	{
		/* show level not found dialog */
		GtkWidget *dlg;
		dlg = create_dlg_level_not_found();
		gtk_widget_show(GTK_WIDGET(dlg));
		set_appbar_temporary(_("Level not found."));
	}
}

void game_undo_move()
{
	if(board_undo_move())
	{
		set_appbar_temporary(_("Undo last move. This costs you 5% of your score."));
		level_data->score = level_data->score * 0.95;
		update_player_info(level_data);
	}
	else
	{
		set_appbar_temporary(_("No more undo levels."));
	}
}

void game_skip_level(void)
{
	if(game_state == GAME_RUNNING)
	{
		if(level_is_last_level(level_data->level))
		{
			set_appbar_temporary(_("This is the last level. You can't skip any further."));
		}
		else
		{
			gchar *next;
			GtkWidget *clock;

			clock = get_time_limit_widget();
			gtk_time_limit_stop(GTK_TIME_LIMIT(clock));
			
			next = g_strdup(level_data->level->next);    
			level_destroy(level_data->level);
			level_data->level = level_load_xml(next);
			level_data->no_level++;
			g_free(next);
			
			/* load level */
			game_load_level(level_data);
		}
	}
}

void game_load_level(LevelData *data)
{
	g_return_if_fail(data != NULL);
	g_return_if_fail(data->level != NULL);

	/* load theme if necessary */
	if(data->theme)
	{
		if(g_strcasecmp(data->theme->name, data->level->theme_name) != 0)
		{
			/* the theme changed for this level */
			data->theme = theme_load_xml(data->level->theme_name);
		}
	}    
	else
	{
		data->theme = theme_load_xml(data->level->theme_name);    
	}

	/* init the level */
	board_clear();
	board_init_level(data->level);

	update_player_info(data);

	if((data->level->time > 0) &&  preferences_get()->score_time_enabled)
	{			
		gtk_time_limit_start(GTK_TIME_LIMIT(get_time_limit_widget()));
	}
}


void game_reload_level()
{
	game_load_level(level_data);
}

void game_level_timeout(GtkWidget *widget, gpointer data)
{
	GtkWidget *dlg;
	GtkWidget* clock;
	gint button_nr;

	/* stop the clock */
	clock = get_time_limit_widget();
	gtk_time_limit_stop(GTK_TIME_LIMIT(clock));      
        gtk_clock_set_seconds(GTK_CLOCK(clock), 0);

	/* show the cursor */
	board_show_normal_cursor();

	/* handle bonus level if neccessary */
	if(level_data->level->bonus_level)
	{
		game_bonus_level_timeout(widget, data);
		return;
	}

	/* update score */
	level_data->score = level_data->score * 0.9;
	update_player_info(level_data);

	/* show dialog "Do you want to try again?" */
	dlg = create_dlg_timeout();
	gtk_widget_show(GTK_WIDGET(dlg));
	button_nr = gnome_dialog_run(GNOME_DIALOG(dlg));

	if(button_nr == 0)
	{
		/* restart level */
		game_reload_level();
	}
	else
	{	
		/* player want to quit */
		game_clean_up();
		save_score(level_data->score);
		game_state = GAME_NOT_RUNNING;
		update_menu_item_state(game_state);
		board_view_message(BOARD_MSG_NEW_GAME);
	}
}

void game_bonus_level_timeout(GtkWidget *widget, gpointer user_data)
{
        GtkWidget *dlg;
	
	/* show dialog to inform the player */
	dlg = create_dlg_bonus_timeout();
	gtk_widget_show(GTK_WIDGET(dlg));
	gnome_dialog_run(GNOME_DIALOG(dlg));

	if(!level_is_last_level(level_data->level))
	{
		gchar *next;

		/* retrieve next level */
		next = g_strdup(level_data->level->next);    
		
		/* update level data */
		level_destroy(level_data->level);
		level_data->level = level_load_xml(next);
		level_data->no_level++;
		g_free(next);

		/* load level */
		game_load_level(level_data);
	}
	else
	{
		game_finished();
	}
}

void game_level_finished(void)
{
	GtkWidget* clock;
	gchar* next;
	gint secs;
	GtkWidget* dlg = NULL;
	NextLevelDlgInfo* dlg_info;

	/* stop clock */
	clock = get_time_limit_widget();
	gtk_time_limit_stop(GTK_TIME_LIMIT(clock));

	/* show cursur */
	board_show_normal_cursor();

	/* retrieve next level */
	next = g_strdup(level_data->level->next);    
    
	if(!level_is_last_level(level_data->level))
	{
		/* prepare next level */

		if(preferences_get()->score_time_enabled)
		{
			/* show statistics dialog */     
			secs = gtk_time_limit_get_seconds(GTK_TIME_LIMIT(clock));
			dlg = create_dlg_next_level ();
			dlg_info = g_malloc(sizeof(NextLevelDlgInfo));
			dlg_info->timer_id = -1;
			dlg_info->secs = secs;
			dlg_info->score = level_data->score;
			dlg_info->lb_secs = lookup_widget(dlg, "time");
			dlg_info->lb_score = lookup_widget(dlg, "score");
			dlg_info->timer_id = gtk_timeout_add(60, next_level_dlg_cb, 
							     dlg_info);
	    
			gtk_widget_show(GTK_WIDGET(dlg));
			gnome_dialog_run(GNOME_DIALOG(dlg));
			gtk_widget_destroy(GTK_WIDGET(dlg));
	    
			if(dlg_info->timer_id != -1)
			{
				gtk_timeout_remove(dlg_info->timer_id);
			}
			g_free(dlg_info);
    
			/* set new score */
			level_data->score = level_data->score + secs;
		}

		/* update level data */
		level_destroy(level_data->level);
		level_data->level = level_load_xml(next);
		level_data->no_level++;
       
		/* load level */
		game_load_level(level_data);
	}
	else
	{
		/* player reached game end */
		game_finished();
	}
	g_free(next);
}

void game_user_quits()
{
	GtkWidget *dlg;
	gint button_nr;
	game_pause();
	
	dlg = create_dlg_quit_game();
	button_nr = gnome_dialog_run(GNOME_DIALOG(dlg));
	
	game_continue();
	if(button_nr == 0 /* yes */)
	{
		GtkWidget *clock;
		/* stop the clock */
		clock = get_time_limit_widget();
		gtk_time_limit_stop(GTK_TIME_LIMIT(clock));
		gtk_clock_set_seconds(GTK_CLOCK(clock), 0);

		save_score(level_data->score);
		game_clean_up();
		game_state = GAME_NOT_RUNNING;
		board_view_message(BOARD_MSG_NEW_GAME);
		update_menu_item_state(game_state);
	}
}

gint next_level_dlg_cb(gpointer data)
{
	NextLevelDlgInfo* dlg_info = (NextLevelDlgInfo*) data;
	gint minutes, secs;
	gchar* time_str = g_malloc(6*sizeof(gchar));
	gchar* score_str = g_malloc(10*sizeof(gchar));
    
	if(dlg_info->secs > 0)
	{
		dlg_info->secs--;
		dlg_info->score = dlg_info->score + 1.0;
	}
	else
	{
		gtk_timeout_remove(dlg_info->timer_id);
		dlg_info->timer_id = -1;
	}

	minutes = (dlg_info->secs)/60;
	secs = (dlg_info->secs)%60;

	if(secs < 10)
	{
		g_snprintf(time_str, 6, "%i:0%i", minutes, secs);
	}
	else
	{
		g_snprintf(time_str, 6, "%i:%i", minutes, secs);
	}
	g_snprintf(score_str, 10, "%.2f", dlg_info->score);
	
	gtk_label_set_text(GTK_LABEL(dlg_info->lb_secs), time_str);
	gtk_label_set_text(GTK_LABEL(dlg_info->lb_score), score_str);
    

	return TRUE;
}


void game_finished(void)
{
	GtkWidget* clock;
	gint secs;
	GtkWidget* dlg = NULL;
	NextLevelDlgInfo* dlg_info;
	gint button_nr = -1;

	/* stop clock */
	clock = get_time_limit_widget();
	
	if(preferences_get()->score_time_enabled)
	{

		/* show statistics dialog */     
		secs = gtk_time_limit_get_seconds(GTK_TIME_LIMIT(clock));
		dlg = create_dlg_last_level ();
		dlg_info = g_malloc(sizeof(NextLevelDlgInfo));
		dlg_info->timer_id = -1;
		dlg_info->secs = gtk_time_limit_get_seconds(GTK_TIME_LIMIT(clock));
		dlg_info->score = level_data->score;
		dlg_info->lb_secs = lookup_widget(dlg, "time");
		dlg_info->lb_score = lookup_widget(dlg, "score");       
		dlg_info->timer_id = gtk_timeout_add(60, next_level_dlg_cb, dlg_info);
	
		gtk_widget_show(GTK_WIDGET(dlg));
		button_nr = gnome_dialog_run(GNOME_DIALOG(dlg));
		gtk_widget_destroy(GTK_WIDGET(dlg));

		if(dlg_info->timer_id != -1)
		{
			gtk_timeout_remove(dlg_info->timer_id);
		}
		g_free(dlg_info);

		/* calculate final score */
		level_data->score = level_data->score + secs;
		save_score(level_data->score);
	}

	if(button_nr == 0)
	{
		/* player wants new game */
		game_new();
	}
	else
	{
		/* no new game */
		game_clean_up();
		game_state = GAME_NOT_RUNNING;
		board_view_message(BOARD_MSG_NEW_GAME);
		update_menu_item_state(game_state);				
	}
}

void game_clean_up(void)
{
	clear_player_info();

	if(level_data->level) level_destroy(level_data->level);
	level_data->level = NULL;

	board_clear();
}

void game_pause(void)
{
	if(game_state == GAME_RUNNING)
	{
		if((level_data->level->time > 0) &&
		   preferences_get()->score_time_enabled)
		{
			GtkWidget *tl;
			tl = get_time_limit_widget();
			gtk_time_limit_stop(GTK_TIME_LIMIT(tl));
		}
	
		board_hide();
		board_view_message(BOARD_MSG_GAME_PAUSED);
		game_state = GAME_PAUSED;
		update_menu_item_state(game_state);
	}
}


void game_continue(void)
{
	if(game_state == GAME_PAUSED)
	{
		board_hide_message(BOARD_MSG_GAME_PAUSED);
		board_show();
	
		if((level_data->level->time > 0) && 
		   preferences_get()->score_time_enabled)
		{
			GtkWidget *tl;
			tl = get_time_limit_widget();
			gtk_time_limit_start(GTK_TIME_LIMIT(tl));
		}
	
		game_state = GAME_RUNNING;
		update_menu_item_state(game_state);
	}
}

void update_player_info(LevelData *data)
{
	GtkWidget *lb_number;
	GtkWidget *lb_name;
	GtkWidget *lb_score;
	GtkWidget *clock;
	gchar *str_buffer;
    
	g_return_if_fail(data != NULL);
	g_return_if_fail(data->level != NULL);

	str_buffer = g_malloc(10*sizeof(str_buffer));
	
	/* set level number */
	lb_number = GTK_WIDGET(lookup_widget(app, "lb_number"));
	g_snprintf(str_buffer, 10, "%i", data->no_level);
	gtk_label_set_text(GTK_LABEL(lb_number), str_buffer);

	/* set levelname */
	lb_name = GTK_WIDGET(lookup_widget(app, "lb_level"));
	gtk_label_set_text(GTK_LABEL(lb_name), data->level->name);

	/* set score and time*/
	lb_score = GTK_WIDGET(lookup_widget(app, "lb_score"));
	clock = get_time_limit_widget();
	gtk_widget_show(clock);
	if(preferences_get()->score_time_enabled)
	{
		g_snprintf(str_buffer, 10, "%.2f", data->score);
		gtk_label_set_text(GTK_LABEL(lb_score), str_buffer);

		gtk_clock_set_seconds(GTK_CLOCK(clock), data->level->time);
	}
	else
	{
		gtk_label_set_text(GTK_LABEL(lb_score), "");
		gtk_clock_set_seconds(GTK_CLOCK(clock), 0);
	}

	g_free(str_buffer);
}

void clear_player_info(void)
{
	GtkWidget *lb_number;
	GtkWidget *lb_name;
	GtkWidget *lb_score;
	GtkWidget *clock;

	/* level number */
	lb_number = GTK_WIDGET(lookup_widget(app, "lb_number"));
	gtk_label_set_text(GTK_LABEL(lb_number), "");

	/* level name */
	lb_name = GTK_WIDGET(lookup_widget(app, "lb_level"));
	gtk_label_set_text(GTK_LABEL(lb_name), "");
    
	/* level score */
	lb_score = GTK_WIDGET(lookup_widget(app, "lb_score"));
	gtk_label_set_text(GTK_LABEL(lb_score), "");
    
	/* time */
	clock = get_time_limit_widget();
	gtk_widget_hide(clock);
	gtk_clock_set_seconds(GTK_CLOCK(clock), 0);
}

void update_menu_item_state(gint state)
{
	GtkWidget *widget;
	GSList *enable_item = NULL;
	GSList *disable_item = NULL;
	
	switch(state)
	{
	case GAME_NOT_RUNNING:
		enable_item = g_slist_append(enable_item, "new_game");
		enable_item = g_slist_append(enable_item, "preferences");
		enable_item = g_slist_append(enable_item, "leveleditor");

		disable_item = g_slist_append(disable_item, "skip_level");
		disable_item = g_slist_append(disable_item, "undo_move");
		disable_item = g_slist_append(disable_item, "pause_game");
		disable_item = g_slist_append(disable_item, "continue_game");
		disable_item = g_slist_append(disable_item, "end_game");
		break;

	case GAME_RUNNING:
		enable_item = g_slist_append(enable_item, "skip_level");
		enable_item = g_slist_append(enable_item, "undo_move");
		enable_item = g_slist_append(enable_item, "pause_game");
		enable_item = g_slist_append(enable_item, "end_game");

		disable_item = g_slist_append(disable_item, "continue_game");
		disable_item = g_slist_append(disable_item, "new_game");
		break;

	case GAME_PAUSED:
		enable_item = g_slist_append(enable_item, "continue_game");

		disable_item = g_slist_append(disable_item, "new_game");
		disable_item = g_slist_append(disable_item, "skip_level");
		disable_item = g_slist_append(disable_item, "undo_move");
		disable_item = g_slist_append(disable_item, "pause_game");
		break;
	default:
	}
	
	while(enable_item != NULL)
	{
		widget = lookup_widget(app, (gchar*) enable_item->data);
		gtk_widget_set_sensitive(widget, TRUE);
		enable_item = enable_item->next;
	}

	while(disable_item != NULL)
	{
		widget = lookup_widget(app, (gchar*) disable_item->data);
		gtk_widget_set_sensitive(widget, FALSE);
		disable_item = disable_item->next;
	}

	g_slist_free(enable_item);
	g_slist_free(disable_item);
}

void save_score(gdouble score)
{
	if(score > 0.0)
	{
		gint highscore_pos;
		highscore_pos = gnome_score_log(score, NULL, TRUE);
		gnome_scores_display("Atomix", "atomix", NULL, highscore_pos);	    
	}
}

void create_user_config_dir(void)
{
	DIR* ds = NULL;
	char* home_dir; 
	char* atomix_dir;
	char* themes_dir;
	char* levels_dir;

	home_dir = g_get_home_dir();
 
	atomix_dir = g_strjoin( "/", home_dir, ".atomix", NULL);
	themes_dir = g_strjoin( "/", atomix_dir, "themes", NULL);
	levels_dir = g_strjoin( "/", atomix_dir, "levels", NULL);

	/* first, check whether .atomix exists */    
	ds = opendir(atomix_dir);
	if(ds==NULL)
	{
		/* try to create one */
		if(mkdir(atomix_dir, 0755 )!=0)
		{
			g_print("An error occured creating your .atomix dir!\n");
			return;
		}
		g_print(".atomix created.\n");
	}
	else
	{
		// g_print(".atomix already exists.\n");
		closedir(ds);
	}

	/* check if .atomix/themes exists */
	ds = opendir(themes_dir);
	if(ds==NULL)
	{
		/* try to create one */
		if(mkdir(themes_dir, 0755)!=0)
		{
			g_print("An error occured creating your .atomix/themes dir!\n");
			return;
		}
		g_print(".atomix/themes created.\n");
	}
	else
	{
		// g_print(".atomix/themes already exists.\n");
		closedir(ds);
	}

	/* check if .atomix/levels exists */
	ds = opendir(levels_dir);
	if(ds==NULL)
	{
		/* try to create one */
		if(mkdir(levels_dir, 0755)!=0)
		{
			g_print("An error occured creating your .atomix/levels dir!\n");
			return;
		}
		g_print(".atomix/levels created.\n");
	}
	else
	{
		// g_print(".atomix/levels already exists.\n");
		closedir(ds);
	}
    
	g_free(atomix_dir);
	g_free(themes_dir);
	g_free(levels_dir);
}

int
main (int argc, char *argv[])
{
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	textdomain (PACKAGE);

	gnome_init ("atomix", VERSION, argc, argv);

	/* init preferences */
	preferences_init();
	gnome_score_init("atomix");

	/* make a few initalisations here */
	create_user_config_dir();
	level_create_hash_table();
	theme_create_hash_table();

	app = create_app ();
	game_init();

	gtk_widget_show (app);

	gtk_main ();
	return 0;
}

