/* Atomixed -- the level-/themeeditor for atomix.
 * Copyright (C) 1999-2000 Jens Finke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "level_ctrl.h"
#include "level.h"
#include "main.h"
#include "support.h"
#include "interface.h"

/*=================================================================
 
  Declaration of global variables

  ---------------------------------------------------------------*/
Level *selected_level;     /* actual selected level */
GList *loaded_levels;      /* the list of all loaded levels */

/*=================================================================
 
  Declaration of internal functions

  ---------------------------------------------------------------*/
void level_ctrl_update_view(void);

void update_list_item(Level *level, ItemUpdateInfo *info);

void level_ctrl_select_list_row(gint pos);

void level_ctrl_save_level(gpointer data, gpointer user_data);

void level_ctrl_set_level_modified(Level *level);

void level_ctrl_remove_level(Level *level);

void destroy_level_item(Level *level, gpointer user_data);

gint equalName(Level* level, gchar *name);

/*=================================================================
 
  Level Control initialisation and clean up

  ---------------------------------------------------------------*/
void level_ctrl_init(void)
{
	Level *level;
	gchar *first_level;
	
	// create list of loaded levels
	first_level = level_get_first_level();
	level = level_load_xml(first_level);

	if(level != NULL)
	{
		selected_level = level;
		
		while(level != NULL)		
		{
			gchar *next;
			
			loaded_levels = g_list_append(loaded_levels, level);
			
			next = g_strdup(level->next);		
			level = level_load_xml(next);
			
			g_free(next);
		}	
		
		// update graphical representation
		level_ctrl_update_view();

		// select first level
		level_ctrl_select_list_row(0);
	}
}

void level_ctrl_update_view(void)
{
	GtkCList *clist;
	ItemUpdateInfo *info;

	clist = GTK_CLIST(lookup_widget(get_app(), "level_list"));
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);

	info = g_malloc(sizeof(ItemUpdateInfo));
	info->counter = 1;
	info->clist = clist;

	g_list_foreach(loaded_levels, (GFunc)update_list_item,
		       info);

	g_free(info);

	gtk_clist_thaw(clist);
}

void update_list_item(Level *level, ItemUpdateInfo *info)
{
	gchar *str_buffer;
	gchar *text[2];
	gint length;
	gint clist_row;

	str_buffer = g_malloc(3*sizeof(gchar));
	length = g_snprintf(str_buffer, 3, "%i", info->counter++);
	text[0] = str_buffer;
	if(level->modified)
	{
		text[1] = g_strconcat(level->name, "*", NULL);
	}
	else
	{
		text[1] = g_strdup(level->name);
	}
	
	clist_row = gtk_clist_append(info->clist, text);
	gtk_clist_set_row_data(info->clist, clist_row, level);	
}

void level_ctrl_select_list_row(gint pos)
{
	GtkCList *clist;
	clist = GTK_CLIST(lookup_widget(get_app(), "level_list"));

	gtk_clist_select_row(clist, pos, 0);
}

void level_ctrl_destroy(void)
{
	g_list_foreach(loaded_levels, (GFunc)destroy_level_item, NULL);
	g_list_free(loaded_levels);
}

void destroy_level_item(Level *level, gpointer user_data)
{
	level_destroy(level);
}

/*=================================================================
 
  Level Control functions

  ---------------------------------------------------------------*/

void level_ctrl_select_row(gint row)
{
	selected_level = level_ctrl_get_level(row);
}

void level_ctrl_move_up(void)
{
	g_return_if_fail(selected_level != NULL);
	
	if(selected_level != g_list_first(loaded_levels)->data)
	{
		/* level is not first level */
		gint pos;

		pos = g_list_index(loaded_levels, selected_level);		
		loaded_levels = g_list_remove(loaded_levels, selected_level);
		loaded_levels = g_list_insert(loaded_levels, selected_level, pos-1);

		level_ctrl_update_view();
		level_ctrl_select_list_row(pos-1);
	}
}

void level_ctrl_move_down(void)
{
	g_return_if_fail(selected_level != NULL);

	if(selected_level != g_list_last(loaded_levels)->data)
	{
		/* level is not last level */
		gint pos;

		pos = g_list_index(loaded_levels, selected_level);		
		loaded_levels = g_list_remove(loaded_levels, selected_level);
		loaded_levels = g_list_insert(loaded_levels, selected_level, pos+1);

		level_ctrl_update_view();
		level_ctrl_select_list_row(pos+1);
	}	
}

void level_ctrl_save_all_levels(void)
{
	if(level_ctrl_check_for_modifications())
	{
		gint length = g_list_length(loaded_levels);
		gint index;

		for(index = 0; index < length; index++)
		{
			Level *level = (Level*)g_list_nth_data(loaded_levels, index);
			if(level->modified)
			{
				level_ctrl_save_level(level, NULL);
				level->modified = FALSE;
			}
		}
		
		level_ctrl_update_view();		
		index = g_list_index(loaded_levels, selected_level);
		level_ctrl_select_list_row(index);
	}
}

/* 
 * functions which work on actual selected level 
 */

Level* level_ctrl_get_actual_level(void)
{
	return selected_level;
}

void level_ctrl_save_actual_level(void)
{
	level_ctrl_save_level((gpointer)selected_level, NULL);
}

void level_ctrl_set_actual_level_modified(void)
{
	level_ctrl_set_level_modified(selected_level);
}

void level_ctrl_remove_actual_level(void)
{
	level_ctrl_remove_level(selected_level);
}

void level_ctrl_actual_level_renamed(void)
{
	level_ctrl_set_actual_level_modified();      
	level_ctrl_update_view();
	level_ctrl_select_list_row(g_list_index(loaded_levels, selected_level));
}

/*
 * functions which work on arbitrary levels 
 */
void level_ctrl_remove_level(Level *level)
{
	if(level->first_level)
	{
		level->first_level = FALSE;
		level_ctrl_save_level(level, NULL);
	}

	loaded_levels  = g_list_remove(loaded_levels, level);

	/* destroy level */
	level_destroy(level);

	level_ctrl_update_view();
	
	level_ctrl_select_list_row(0);
}


Level* level_ctrl_get_level(gint row)
{
	Level *level;
	level = (Level*)g_list_nth_data(loaded_levels, row);
	return level;
}

void level_ctrl_save_level(gpointer data, gpointer user_data)
{
	/* FIXME: check whether user directory exists */
	gchar *file;
	Level *level = (Level*) data;

	g_return_if_fail(level != NULL);

	file = g_strconcat(g_get_home_dir(),"/.atomix/levels/",
			   level->file_name, NULL);
	level_save_xml(level, file);
	g_free(file);

	g_print("Save level %s\n", level->name);

       	/* remove modified mark */
	level->modified = FALSE;
}

gboolean level_ctrl_check_for_modifications(void)
{
	gboolean modified = FALSE;
	Level *level;
	gint index;
	gint length; 

	length = g_list_length(loaded_levels);
	
        for(index = 0; index < length; index++)
	{
		level = (Level*)g_list_nth_data(loaded_levels, index);
		
		if(level->modified) modified = TRUE;
		
		if(index == (length-1))
		{
			/* check if its really the last level */
			if(!level_is_last_level(level))
			{
				level_set_last_level(level);
				modified = TRUE;
				level->modified = TRUE;
			}			
		}
		else
		{
			/* is not last level */
			Level *next_level;

			if((index == 0) && (!level->first_level))
			{
				level->first_level = TRUE;
				level->modified = TRUE;
				modified = TRUE;
			}
			else if((index != 0) && (level->first_level))
			{
				level->first_level = FALSE;
				level->modified = TRUE;
				modified = TRUE;
			}

			
			next_level = (Level*)g_list_nth_data(loaded_levels, index+1);
			
			/* check whether the name of the next level is the same name
			   as the successor in the list has. If not, set it. */ 
			if(level->next != NULL)
			{
				if(g_strcasecmp(level->next, next_level->name) != 0)
				{
					modified = TRUE;
					
					g_free(level->next);
					level->next = g_strdup(next_level->name);
					level->modified = TRUE;
				}
			}
			else
			{
				modified = TRUE;
				level->next = g_strdup(next_level->name);
				level->modified = TRUE;
			}
		}
	}

	return modified;
}

void level_ctrl_set_level_modified(Level *level)
{
	gint row;

	g_return_if_fail(level != NULL);

	if(!level->modified)
	{
		level->modified = TRUE;
		
		level_ctrl_update_view();
		row = g_list_index(loaded_levels, selected_level);
		level_ctrl_select_list_row(row);
	}
}

void level_ctrl_new_level(const gchar *name, const gchar *file, const gchar *theme_name,
			  guint n_rows, guint n_cols,
			  guint goal_n_rows, guint goal_n_cols)
{
	Level *level;
	gint row;

	g_return_if_fail(name!=NULL);
	g_return_if_fail(file!=NULL);
	g_return_if_fail(theme_name!=NULL);

	/* init level struct */
	level = level_new();
	level->name = g_strdup(name);
	level->theme_name = g_strdup(theme_name);
	level->file_name = g_strdup(file);
	level->playfield = playfield_new();
	level->modified = TRUE;
	playfield_set_matrix_size(level->playfield, n_rows, n_cols);
	level->goal = playfield_new();
	playfield_set_matrix_size(level->goal, goal_n_rows, goal_n_cols);

	/* add it to the list */
	loaded_levels = g_list_append(loaded_levels, level);

	level_ctrl_update_view();

	row = g_list_length(loaded_levels) - 1;
	level_ctrl_select_list_row(row);
}

gboolean level_ctrl_does_name_exists(gchar *name)
{
	return (g_list_find_custom(loaded_levels, name, (GCompareFunc)equalName) != NULL);
}

gint equalName(Level* level, gchar *name)
{
	g_return_val_if_fail(level!=NULL,1);
	g_return_val_if_fail(name!=NULL,1);

	return g_strcasecmp(level->name, name);
}

void level_ctrl_load_level(gchar *file_path)
{
	Level *level;

	level = level_load_xml_file(file_path);
	
	if(level != NULL)
	{
		gboolean cancel_pressed = FALSE;

		// check whether the level name is already in use
		while(level_ctrl_does_name_exists(level->name) && !cancel_pressed)
		{
			GnomeDialog *dlg;
			GtkWidget *current_name;
			gint button;

			dlg = GNOME_DIALOG(create_change_level_name());
			current_name = GTK_WIDGET(lookup_widget(GTK_WIDGET(dlg), "current_name"));
			gtk_entry_set_text(GTK_ENTRY(current_name), level->name);

			button = gnome_dialog_run(dlg);
			if(button == 0 /* ok */)
			{
				GtkWidget *new_name;
				GtkWidget *file_name;
				
				new_name = GTK_WIDGET(lookup_widget(GTK_WIDGET(dlg), "new_name"));
				file_name = GTK_WIDGET(lookup_widget(GTK_WIDGET(dlg), "file_name"));
				
				level->name = gtk_editable_get_chars(GTK_EDITABLE(new_name), 0, -1);
				level->file_name = gtk_editable_get_chars(GTK_EDITABLE(file_name), 0, -1);
			}
			else
			{
				cancel_pressed = TRUE;
			}

			gtk_widget_destroy(GTK_WIDGET(dlg));
		}
		
		if(!cancel_pressed)
		{
			gint row;

			loaded_levels = g_list_append(loaded_levels, level);

			level_ctrl_update_view();

			row = g_list_index(loaded_levels, level);
			level_ctrl_select_list_row(row);
		}
		else
		{
			level_destroy(level);
		}
	}
	else
	{
		g_print("Error while loading level.\n");
	}
}
