/*
 * main.c: the main program / app list view
 *
 * Author:
 *    Michael Meeks
 *
 * Copyright 2002 Sun Microsystems, Inc.
 */

/*
 * TODO: listen for childen-changed on
 * the desktop object ... impl. a
 * 'Global state' object that we can sync
 *  stuff with
 */
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <cspi/spi.h>
#include <libgnomeui/gnome-ui-init.h>
#include <popt.h>
#include <gconf/gconf-client.h>

#include "graphics.h"
#include "child-listener.h"
#include "accessible-listener.h"

extern int max_children;
extern gboolean dontdisplaychildren;
extern gboolean use_table_if;

static const struct poptOption options[] = {
 	{ "maxchildren", 'm', POPT_ARG_INT, &max_children, 1,
	  "Specify the maximum number of children of an Accessible to be returned; or maximum ch ildren in a row or column if table is specified.\n", "INT"},
 	{ "dontdisplaychildren", 'd', POPT_ARG_NONE, &dontdisplaychildren, 1,
	  "Do not display children for object which has state MANAGES_DESCENDANTS\n", NULL},
 	{ "table", 't', POPT_ARG_NONE, &use_table_if, 1,
	  "Use AccessibleTable_getAccessibleAt to get children of Accessible which implements AccessibleTable\n", NULL},
	{NULL, '\0', 0, NULL, 0, NULL, NULL}
};

extern void poke (Accessible *accessible);
extern Accessible *get_accessible_at_index (GtkListStore *list_store, int i);

#define APP_COL_NAME   0
#define APP_COL_DESCR  1
#define APP_COL_HANDLE 2
#define APP_COL_LAST   3

#define GCONF_ACCESSIBILITY_KEY "/desktop/gnome/interface/accessibility"

typedef struct {
	GladeXML     *app_xml;

	GtkWidget    *window;
	
	GtkListStore *list_store;
	GtkTreeView  *list_view;

	GtkLabel     *toolkit_name;
	GtkLabel     *toolkit_version;
	GtkLabel     *app_id;

	Accessible   *selected;
} AppWindow;

static void
update_app_display (AppWindow  *app_window,
		    Accessible *accessible)
{
	char *txt;
	AccessibleApplication *application;

	if (!accessible ||
	    !(application = Accessible_getApplication (accessible))) {
		gtk_label_set_text (app_window->toolkit_name, "<invalid>");
		gtk_label_set_text (app_window->toolkit_version, "<invalid>");
		gtk_label_set_text (app_window->app_id, "<invalid>");
		return;
	}

	txt = AccessibleApplication_getToolkitName (application);
	gtk_label_set_text (app_window->toolkit_name, txt);
	SPI_freeString (txt);

	txt = AccessibleApplication_getVersion (application);
	gtk_label_set_text (app_window->toolkit_version, txt);
	SPI_freeString (txt);

	txt = g_strdup_printf ("%ld", AccessibleApplication_getID (application));
	gtk_label_set_text (app_window->app_id, txt);
	g_free (txt);

	AccessibleApplication_unref (application);
}

static void
app_list_selection_changed (GtkTreeSelection *selection,
			    AppWindow        *app_window)
{
	GtkTreeIter iter;
	Accessible *accessible;
	GtkTreeModel *model;
	
	if (!gtk_tree_selection_get_selected (selection, &model, &iter)) {
		return;
	}

	gtk_tree_model_get (model, &iter,
			    APP_COL_HANDLE, &accessible,
			    -1);
	update_app_display (app_window, accessible);

	app_window->selected = accessible;
}

static void
validate_up_down_linkage (Accessible *accessible)
{
	Accessible *child, *aa;
	Accessible *parent;

	aa = Accessible_queryInterface (accessible, "IDL:Accessibility/Accessible:1.0");

	g_assert (aa == accessible);

	Accessible_unref (aa);
	child = Accessible_getChildAtIndex (accessible, 0);
	if (!child)
		return;

	parent = Accessible_getParent (child);

	g_assert (parent == accessible);
	Accessible_unref (child);
	Accessible_unref (parent);
}

static void
app_list_row_activated (GtkTreeView       *tree_view,
			GtkTreePath       *path,
			GtkTreeViewColumn *column,
			AppWindow         *app_window)
{
	validate_up_down_linkage (app_window->selected);
	poke (app_window->selected);
}

static void
application_clicked (GtkButton *button,
		     AppWindow *app_window)
{
	if (!app_window->selected)
		gdk_beep ();
	else {
		validate_up_down_linkage (app_window->selected);
		poke (app_window->selected);
	}
}

Accessible *
get_accessible_at_index (GtkListStore *list_store, int i)
{
	GtkTreeIter iter;
	GtkTreeModel *model = GTK_TREE_MODEL (list_store);
	Accessible *accessible;
	gboolean ret;

	ret = gtk_tree_model_iter_nth_child (model, &iter, NULL, i);
	g_return_val_if_fail (ret, NULL);

	gtk_tree_model_get (model, &iter,
			    APP_COL_HANDLE, &accessible,
			    -1);
	return accessible;
}

static void
list_store_clear (GtkListStore *list_store)
{
	GtkTreeIter iter;
	GtkTreeModel *model = GTK_TREE_MODEL (list_store);
	Accessible *accessible;
	gboolean ret;

	ret = gtk_tree_model_get_iter_first (model, &iter);

	while (ret) {
		gtk_tree_model_get (model, &iter,
				    APP_COL_HANDLE, &accessible,
				    -1);
		Accessible_unref (accessible);
		ret = gtk_tree_model_iter_next (model, &iter);
	}
	gtk_list_store_clear (list_store);
}

static void
populate_app_list (AppWindow *app_window)
{
	int i, apps;
	Accessible *desktop;
	GtkTreeIter iter;

	desktop = SPI_getDesktop (0);

	apps = Accessible_getChildCount (desktop);

	list_store_clear (app_window->list_store);
	for (i = 0; i < apps; i++) {
		char *name, *descr;
		Accessible *child;

		child = Accessible_getChildAtIndex (desktop, i);
		if (!child)
			continue;

		name = Accessible_getName (child);
		descr = Accessible_getDescription (child);
		gtk_list_store_append (app_window->list_store, &iter);
		gtk_list_store_set (app_window->list_store,
				    &iter,
				    APP_COL_NAME,   name,
				    APP_COL_DESCR,  descr,
				    APP_COL_HANDLE, child,
				    -1);
		SPI_freeString (descr);
		SPI_freeString (name);
	}

	if (gtk_tree_model_get_iter_root (
		GTK_TREE_MODEL (app_window->list_store), &iter))
		gtk_tree_selection_select_iter (
			gtk_tree_view_get_selection (app_window->list_view), &iter);
	else
		app_window->selected = NULL;

	Accessible_unref (desktop);
}

static void
app_list_changed_cb (ChildListener *listener,
		     AppWindow     *window)
{
	populate_app_list (window);
}

static void
window_destroyed (GtkObject *obj)
{
	gtk_main_quit ();
}

static void
create_app_list (AppWindow *app_window)
{
	app_window->list_store = 
		gtk_list_store_new (APP_COL_LAST,
				    G_TYPE_STRING,
				    G_TYPE_STRING,
				    G_TYPE_POINTER);

	app_window->list_view = GTK_TREE_VIEW (
		glade_xml_get_widget (
			app_window->app_xml, "application_list_view"));

	gtk_tree_view_set_model (app_window->list_view,
				 GTK_TREE_MODEL (app_window->list_store));

	gtk_tree_view_insert_column_with_attributes
		(app_window->list_view,
		 APP_COL_NAME, "Application",
		 gtk_cell_renderer_text_new (),
		 "text", 0,
		 NULL);
	gtk_tree_view_insert_column_with_attributes
		(app_window->list_view,
		 APP_COL_DESCR, "Description",
		 gtk_cell_renderer_text_new (),
		 "text", 0,
		 NULL);
	gtk_tree_view_columns_autosize (app_window->list_view);
	gtk_tree_selection_set_mode (
		gtk_tree_view_get_selection (
			app_window->list_view),
		GTK_SELECTION_BROWSE);

	g_signal_connect (
		gtk_tree_view_get_selection (
			app_window->list_view),
		"changed",
		G_CALLBACK (app_list_selection_changed),
		app_window);

	g_signal_connect (
		GTK_TREE_VIEW (app_window->list_view),
		"row_activated",
		G_CALLBACK (app_list_row_activated),
		app_window);
}

static void
application_window (void)
{
	GClosure  *closure;
	GladeXML  *app_xml;
	AppWindow *app_window = g_new0 (AppWindow, 1);
	ChildListener *cl;

	app_xml = get_glade_xml ();
	app_window->app_xml = app_xml;

	if (!app_xml) {
		g_error ("Can't find at-poke.glade2");
	}

	app_window->window = glade_xml_get_widget (
		app_xml, "application_window");
	g_object_add_weak_pointer (G_OBJECT (app_window->window), (gpointer *)&(app_window->window));

	gtk_widget_show (app_window->window);

	create_app_list (app_window);

	app_window->toolkit_name = GTK_LABEL (
		glade_xml_get_widget (app_xml, "application_toolkit_name"));
	app_window->toolkit_version = GTK_LABEL (
		glade_xml_get_widget (app_xml, "application_toolkit_version"));
	app_window->app_id = GTK_LABEL (
		glade_xml_get_widget (app_xml, "application_id"));

	populate_app_list (app_window);

	g_signal_connect (
		glade_xml_get_widget (
			app_xml, "application_poke_button"),
		"clicked",
		G_CALLBACK (application_clicked),
		app_window);

	g_signal_connect_closure (
		glade_xml_get_widget (
			app_xml, "application_refresh_button"),
		"clicked",
		g_cclosure_new_swap (
			G_CALLBACK (populate_app_list),
			app_window, NULL),
		0);

	closure = g_cclosure_new (
		G_CALLBACK (app_list_changed_cb),
		app_window, NULL);
	g_object_watch_closure (
		G_OBJECT (app_window->window),
		closure);
	cl = child_listener_create (app_window->list_store);
	g_signal_connect_closure (
		cl,
		"app_list_changed",
		closure, FALSE);

	g_signal_connect (app_window->window,
			  "destroy",
			  G_CALLBACK (window_destroyed),
			  NULL);

	gtk_main ();

	if (GTK_IS_WIDGET (app_window->window))
		gtk_widget_destroy (app_window->window);

	list_store_clear (app_window->list_store);
	g_object_unref (app_xml);
	g_free (app_window);
	g_object_unref (cl);
	/*
	 * Deregister listeners which have been registered.
	 */
	accessible_listener_shutdown ();
}

static gboolean
no_accessible_apps (void)
{
	int num_apps;
	Accessible *desktop;

	desktop = SPI_getDesktop (0);

	num_apps = Accessible_getChildCount (desktop);
	
	Accessible_unref (desktop);

	return (num_apps == 0);
}

int
main (int argc, char **argv)
{
	int leaked;
	int init_error;

	init_error = SPI_init ();
	if (init_error) {
	        g_warning ("Error initialising SPI");
	        return init_error;
	}

	/* We don't want to show ourself */
	putenv ("GTK_MODULES=");
	putenv ("GNOME_ACCESSIBILITY=0");

	gnome_program_init ("at-poke", VERSION,
			    LIBGNOMEUI_MODULE,
			    argc, argv, 
			    GNOME_PARAM_POPT_TABLE, options, NULL);
	if (max_children < 0)
		max_children = 0;
	if (argc % 2  == 0 || no_accessible_apps ()) {
		gboolean a11y_enabled;
		gchar *prog_name;
		GError *error = NULL;

		a11y_enabled = gconf_client_get_bool (gconf_client_get_default (), GCONF_ACCESSIBILITY_KEY, &error);

		if (!a11y_enabled) {
			GtkWidget *dialog;

			dialog = gtk_message_dialog_new (NULL,
							 GTK_DIALOG_MODAL,
							 GTK_MESSAGE_INFO,
							 GTK_BUTTONS_OK,
							 "GConf accessibility key is not set. at-poke will launch gtk-demo or the program you specified on the command line with accessibility enabled");
			gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog);
		}

		if (argc % 2 == 0)
			prog_name = argv[argc -1];
		else
			prog_name = "gtk-demo";
		if (!fork ()) {
			putenv ("GTK_MODULES=gail:atk-bridge");
			putenv ("GNOME_ACCESSIBILITY=1");
			execlp (prog_name, prog_name, NULL);
			g_warning ("Unable to exec %s; check your PATH environment variable", prog_name);
			_exit (0);
		}
	}

	application_window ();

	if ((leaked = SPI_exit ()))
		g_error ("Leaked %d SPI handles", leaked);

	g_assert (!SPI_exit ());

	if (g_getenv ("_MEMPROF_SOCKET")) {
		fprintf (stderr, "Waiting for memprof\n");
		gtk_main ();
	}

	putenv ("AT_BRIDGE_SHUTDOWN=1");

	return 0;
}
