/*
 * project_dbase.c
 * Copyright (C) 2000  Kh. Naba Kumar Singh
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <dirent.h>

#include <gnome.h>
#include "anjuta.h"
#include "project_dbase.h"
#include "mainmenu_callbacks.h"
#include "project_dbase_cbs.h"
#include "utilities.h"
#include "messagebox.h"
#include "fileselection.h"
#include "support.h"
#include "controls.h"
#include "utilities.h"

#include "../pixmaps/bfoldo.xpm"
#include "../pixmaps/bfoldc.xpm"
#include "../pixmaps/file_file.xpm"
#include "../pixmaps/file_c.xpm"
#include "../pixmaps/file_cpp.xpm"
#include "../pixmaps/file_h.xpm"
#include "../pixmaps/file_pix.xpm"

#define COMPATIBILITY_LEVEL    0

GdkPixmap *opened_folder_pix,
  *closed_folder_pix,
  *file_h_pix, *file_c_pix, *file_cpp_pix, *file_pix_pix, *file_file_pix;

GdkBitmap *opened_folder_mask,
  *closed_folder_mask,
  *file_h_mask, *file_c_mask, *file_cpp_mask, *file_pix_mask, *file_file_mask;

void get_pixmask_on_file_ext (FileExtType t, GdkPixmap ** p, GdkBitmap ** m);

int select_only_file (const struct dirent *e);

ProjectDBase *
project_dbase_new ()
{
  ProjectDBase *p;
  GtkStyle *style;

  p = g_malloc (sizeof (ProjectDBase));
  if (p)
  {
    /* Must declare static, because it will be used forever */
    static FileSelData fsd1 = { N_("Open Project"), NULL,
      on_open_prjfilesel_ok_clicked,
      on_open_prjfilesel_cancel_clicked,
      NULL
    };

    /* Must declare static, because it will be used forever */
    static FileSelData fsd2 = { N_("Add To Project"), NULL,
      on_add_prjfilesel_ok_clicked,
      on_add_prjfilesel_cancel_clicked,
      NULL
    };

    /* Cannot assign p to static fsd.data, so doing it now */
    fsd1.data = p;
    fsd2.data = p;

    create_project_dbase_gui (p);
    gtk_window_set_title (GTK_WINDOW (p->widgets.window), _("Project: None"));
    p->fileselection_open = create_fileselection_gui (&fsd1);
    p->fileselection_add = create_fileselection_gui (&fsd2);

    p->widgets.root_node = NULL;
    p->widgets.current_node = NULL;
    p->project_is_open = FALSE;
    p->project_has_gui = FALSE;
    p->overwrite_makefiles = FALSE;
    p->gettext_support = FALSE;
    p->set_cflags = FALSE;
    p->is_saved = TRUE;
    p->src_makefile_saved = TRUE;
    p->doc_makefile_saved = TRUE;
    p->potfile_saved = TRUE;
    p->is_showing = TRUE;
    p->is_docked = TRUE;
    p->win_pos_x = 100;
    p->win_pos_y = 80;
    p->win_width = 200;
    p->win_height = 400;
    p->top_proj_dir = NULL;
    p->proj_filename = NULL;
    p->proj_name = NULL;
    p->prog_name = NULL;
    p->version = NULL;

    p->src_files = NULL;
    p->doc_files = NULL;
    p->pix_files = NULL;

    p->src_dir = NULL;
    p->pix_dir = NULL;
    p->doc_dir = NULL;

    p->cur_filename = NULL;
    p->includes = NULL;
    p->libraries = NULL;

    style = gtk_widget_get_style (p->widgets.window);
    closed_folder_pix =
      gdk_pixmap_create_from_xpm_d (p->widgets.window->window,
				    &closed_folder_mask,
				    &style->bg[GTK_STATE_NORMAL], bfoldc_xpm);
    opened_folder_pix =
      gdk_pixmap_create_from_xpm_d (p->widgets.window->window,
				    &opened_folder_mask,
				    &style->bg[GTK_STATE_NORMAL], bfoldo_xpm);
    file_h_pix =
      gdk_pixmap_create_from_xpm_d (p->widgets.window->window,
				    &file_h_mask,
				    &style->bg[GTK_STATE_NORMAL], file_h_xpm);
    file_c_pix =
      gdk_pixmap_create_from_xpm_d (p->widgets.window->window,
				    &file_c_mask,
				    &style->bg[GTK_STATE_NORMAL], file_c_xpm);
    file_cpp_pix =
      gdk_pixmap_create_from_xpm_d (p->widgets.window->window,
				    &file_cpp_mask,
				    &style->bg[GTK_STATE_NORMAL],
				    file_cpp_xpm);
    file_pix_pix =
      gdk_pixmap_create_from_xpm_d (p->widgets.window->window,
				    &file_pix_mask,
				    &style->bg[GTK_STATE_NORMAL],
				    file_pix_xpm);
    file_file_pix =
      gdk_pixmap_create_from_xpm_d (p->widgets.window->window,
				    &file_file_mask,
				    &style->bg[GTK_STATE_NORMAL],
				    file_file_xpm);
  }
  return p;
}

void
project_dbase_destroy (ProjectDBase * p)
{
  if (p)
  {
    project_dbase_clear (p);
    gtk_widget_unref (p->widgets.window);
    gtk_widget_unref (p->widgets.client_area);
    gtk_widget_unref (p->widgets.client);
    gtk_widget_unref (p->widgets.scrolledwindow);
    gtk_widget_unref (p->widgets.ctree);
    gtk_widget_unref (p->widgets.menu);
    gtk_widget_unref (p->widgets.menu_add_src);
    gtk_widget_unref (p->widgets.menu_add_doc);
    gtk_widget_unref (p->widgets.menu_add_pix);
    gtk_widget_unref (p->widgets.menu_add_other);
    gtk_widget_unref (p->widgets.menu_add_dir);
    gtk_widget_unref (p->widgets.menu_remove);
    gtk_widget_unref (p->widgets.menu_view);
    gtk_widget_unref (p->widgets.menu_configure);
    gtk_widget_unref (p->widgets.menu_info);

    if (p->widgets.window)
      gtk_widget_destroy (p->widgets.window);
    if (p->fileselection_open)
      gtk_widget_destroy (p->fileselection_open);
    if (p->fileselection_add)
      gtk_widget_destroy (p->fileselection_add);
    g_free (p);
    p = NULL;
  }
}

void
project_dbase_clear (ProjectDBase * p)
{
  gint i;
  if (p)
  {
    if (p->widgets.root_node)
    {
      gtk_ctree_remove_node (GTK_CTREE (p->widgets.ctree),
			     p->widgets.root_node);
      p->widgets.root_node = NULL;
    }
    if (p->top_proj_dir)
    {
      g_free (p->top_proj_dir);
      p->top_proj_dir = NULL;
    }
    if (p->proj_name)
    {
      g_free (p->proj_name);
      p->proj_name = NULL;
    }
    if (p->prog_name)
    {
      g_free (p->prog_name);
      p->prog_name = NULL;
    }
    if (p->src_dir)
    {
      g_free (p->src_dir);
      p->src_dir = NULL;
    }
    if (p->pix_dir)
    {
      g_free (p->pix_dir);
      p->pix_dir = NULL;
    }
    if (p->doc_dir)
    {
      g_free (p->doc_dir);
      p->doc_dir = NULL;
    }
    if (p->includes)
    {
      g_free (p->includes);
      p->includes = NULL;
    }
    if (p->libraries)
    {
      g_free (p->doc_dir);
      p->libraries = NULL;
    }
    if (p->cur_filename)
    {
      g_free (p->cur_filename);
      p->cur_filename = NULL;
    }

    for (i = 0; i < g_list_length (p->src_files); i++)
      g_free (g_list_nth_data (p->src_files, i));
    g_list_free (p->src_files);
    p->src_files = NULL;

    for (i = 0; i < g_list_length (p->doc_files); i++)
      g_free (g_list_nth_data (p->doc_files, i));
    g_list_free (p->doc_files);
    p->doc_files = NULL;

    for (i = 0; i < g_list_length (p->pix_files); i++)
      g_free (g_list_nth_data (p->pix_files, i));
    g_list_free (p->pix_files);
    p->pix_files = NULL;

    if (p->src_files)
    {
      g_list_free (p->src_files);
      p->src_files = NULL;
    }
    gtk_window_set_title (GTK_WINDOW (p->widgets.window), _("Project: None"));
    p->project_is_open = FALSE;
    p->project_has_gui = FALSE;
    p->is_saved = TRUE;
    p->src_makefile_saved = TRUE;
    p->doc_makefile_saved = TRUE;
    p->potfile_saved = TRUE;
    extended_toolbar_update ();
  }
}

void
project_dbase_show (ProjectDBase * p)
{
  if (p)
  {
    if (p->is_showing)
    {
      if (p->is_docked == FALSE)
	gdk_window_raise (p->widgets.window->window);
      return;
    }
    if (p->is_docked)
    {
      project_dbase_attach (p);
    }
    else			/* Is not docked */
    {
      gtk_widget_set_uposition (p->widgets.window, p->win_pos_x,
				p->win_pos_y);
      gtk_window_set_default_size (GTK_WINDOW (p->widgets.window),
				   p->win_width, p->win_height);
      gtk_widget_show (p->widgets.window);
    }
    p->is_showing = TRUE;
  }
}

void
project_dbase_hide (ProjectDBase * p)
{
  if (p)
  {
    if (p->is_showing == FALSE)
      return;
    if (p->is_docked == TRUE)
    {
      project_dbase_detach (p);
    }
    else			/* Is not docked */
    {
      gdk_window_get_root_origin (p->widgets.window->window, &p->win_pos_x,
				  &p->win_pos_y);
      gdk_window_get_size (p->widgets.window->window, &p->win_width,
			   &p->win_height);
      gtk_widget_hide (p->widgets.window);
    }
    p->is_showing = FALSE;
  }
}

void
project_dbase_open_project (ProjectDBase * p)
{
  chdir (app->preferences->all_projects_dir);
  gtk_file_selection_complete(GTK_FILE_SELECTION(p->fileselection_open),
        app->preferences->all_projects_dir);
  gtk_widget_show (p->fileselection_open);
}

gboolean
project_dbase_load_project (ProjectDBase * p)
{
  FILE *fp;
  gchar buff[256], *temp;
  gint tmp, i;

  anjuta_set_busy ();
  if (p->project_is_open == TRUE)
    project_dbase_clean_left (p);
  p->project_is_open = TRUE;
  p->proj_filename =
    gtk_file_selection_get_filename (GTK_FILE_SELECTION
				     (p->fileselection_open));
  fp = fopen (p->proj_filename, "r");
  if (fp == NULL)
  {
    anjuta_error (_
		  ("Error: Cannot load project.\nMake sure the Project file exists."));
    p->proj_filename = NULL;
    p->project_is_open = FALSE;
    anjuta_set_active ();
    return FALSE;
  }
  if (fscanf (fp, "# Anjuta Version %s \n", buff) < 1)
    goto error_show;
  if (fscanf (fp, "Compatibility Level: %d\n", &tmp) < 1)
    goto error_show;
  if (tmp < COMPATIBILITY_LEVEL)
  {
    temp =
      g_strdup_printf (_
		       ("Error: You need Anjuta version %s or later to open this project.\n"
			"Please upgrade Anjuta to the latest version (Help for more information).\n"
			"For the time being, I am too old to load it. :-("),
buff);
    anjuta_error (temp);
    g_free (temp);
    p->proj_filename = NULL;
    fclose (fp);
    p->project_is_open = FALSE;
    anjuta_set_active ();
    return FALSE;
  }

  if (fscanf (fp, "Project Type: %d \n", &tmp) < 1)
    goto error_show;
  p->type = (ProjectType) tmp;
  if (p->type > gnome_full)
    goto error_show;

  if (fscanf (fp, "Project Name: %s \n", buff) < 1)
    goto error_show;
  p->proj_name = g_strdup (buff);

  if (fscanf (fp, "Program Name: %s \n", buff) < 1)
    goto error_show;
  p->prog_name = g_strdup (buff);

  if (fscanf (fp, "Version: %s \n", buff) < 1)
    goto error_show;
  p->version = g_strdup (buff);

  fscanf (fp, "Author: ");
  read_line (fp, &p->author);

  p->top_proj_dir = g_dirname (p->proj_filename);
  if (fscanf (fp, "Source Dir: %s \n", buff) < 1)
    goto error_show;
  if (strcmp (buff, ".") == 0)
    p->src_dir = g_strdup (p->top_proj_dir);
  else
    p->src_dir = g_strdup_printf ("%s/%s", p->top_proj_dir, buff);
  if (fscanf (fp, "Pixmap Dir: %s \n", buff) < 1)
    goto error_show;
  p->pix_dir = g_strdup_printf ("%s/%s", p->top_proj_dir, buff);
  if (fscanf (fp, "Document Dir: %s \n", buff) < 1)
    goto error_show;
  p->doc_dir = g_strdup_printf ("%s/%s", p->top_proj_dir, buff);

  if (!read_string (fp, "Includes", &temp))
    goto error_show;
  p->includes = g_strdup (temp);
  g_free (temp);

  if (!read_string (fp, "Libraries", &temp))
    goto error_show;
  p->libraries = g_strdup (temp);
  g_free (temp);

  if (fscanf (fp, "Project Has Gui: %d\n", &tmp) < 1)
    goto error_show;
  p->project_has_gui = (gboolean) tmp;

  if (fscanf (fp, "Gettext Support: %d\n", &tmp) < 1)
    goto error_show;
  p->gettext_support = (gboolean) tmp;

  if (fscanf (fp, "Manage Makefiles: %d\n", &tmp) < 1)
    goto error_show;
  p->overwrite_makefiles = (gboolean) tmp;

  if (fscanf (fp, "Set Cflags: %d\n", &tmp) < 1)
    goto error_show;
  p->set_cflags = (gboolean) tmp;

  compiler_options_save (app->compiler_options, NULL);
  if (compiler_options_load (app->compiler_options, fp) < 1)
    goto error_show;

  src_paths_save (app->src_paths, NULL);
  if (src_paths_load (app->src_paths, fp) < 1)
    goto error_show;

  if (fscanf (fp, "No of Source Files: %d\n", &tmp) < 1)
    goto error_show;
  for (i = 0; i < tmp; i++)
  {
    if (fscanf (fp, "Src: %s \n", buff) < 1)
      goto error_show;
    p->src_files = g_list_append (p->src_files, g_strdup (buff));
  }
  if (fscanf (fp, "No of Document Files: %d \n", &tmp) < 1)
    goto error_show;
  for (i = 0; i < tmp; i++)
  {
    if (fscanf (fp, "Doc: %s \n", buff) < 1)
      goto error_show;
    p->doc_files = g_list_append (p->doc_files, g_strdup (buff));
  }
  project_dbase_scan_pix_files (p);
  sprintf (buff, "Project: %s-%s", p->proj_name, p->version);
  gtk_window_set_title (GTK_WINDOW (p->widgets.window), buff);
  extended_toolbar_update ();
  fclose (fp);
  project_dbase_update_tree (p);
  project_dbase_show (p);
  p->is_saved = TRUE;
  tags_manager_load (app->tags_manager);
  if (app->preferences->tags_update)
    tags_manager_update_image (app->tags_manager, p->src_files);
  anjuta_status (_("Project loaded successfully."));
  p->is_saved = TRUE;
  p->src_makefile_saved = TRUE;
  p->doc_makefile_saved = TRUE;
  p->potfile_saved = TRUE;

  anjuta_set_active ();
  return TRUE;

error_show:
  anjuta_error (_("ERROR: Unable to load the project.\n" \
		  "It seems you have tempered with the file. Do not do it again."));
  project_dbase_clean_left (p);
  p->proj_filename = NULL;
  p->project_is_open = FALSE;
  p->is_saved = TRUE;
  p->src_makefile_saved = TRUE;
  p->doc_makefile_saved = TRUE;
  p->potfile_saved = TRUE;

  fclose (fp);
  anjuta_set_active ();
  return FALSE;
}

void
project_dbase_update_tree (ProjectDBase * p)
{
  GtkCTreeNode *top, *node[3];
  gint i;
  GdkPixmap *pix;
  GdkBitmap *mask;
  gchar *dummy[1];
  gchar *title[] =
    { N_("Source Files"), N_("Document Files"), N_("Pixmap Files") };

  if (p->project_is_open == FALSE)
    return;
  gtk_clist_freeze (GTK_CLIST (p->widgets.ctree));
  if (p->widgets.root_node)
    gtk_ctree_remove_node (GTK_CTREE (p->widgets.ctree),
			   p->widgets.root_node);

  dummy[0] = g_strdup_printf ("%s-%s", p->proj_name, p->version);
  top = gtk_ctree_insert_node (GTK_CTREE (p->widgets.ctree),
			       NULL, NULL,
			       dummy,
			       5,
			       closed_folder_pix, closed_folder_mask,
			       opened_folder_pix, opened_folder_mask, FALSE,
			       TRUE);
  p->widgets.root_node = top;
  g_free (dummy[0]);
  for (i = 0; i < 3; i++)
  {
    gboolean expand;
    switch (i)
    {
    case 0:
      if (strcmp (p->src_dir, p->top_proj_dir) != 0)
      {
	dummy[0] =
	  g_strconcat (extract_filename (p->src_dir), "(", _(title[i]), ")",
		       NULL);
	if (g_list_length (p->src_files) > 0)
	  expand = TRUE;
	else
	  expand = FALSE;
	node[i] = gtk_ctree_insert_node (GTK_CTREE (p->widgets.ctree),
					 top, NULL,
					 dummy,
					 5,
					 closed_folder_pix,
					 closed_folder_mask,
					 opened_folder_pix,
					 opened_folder_mask, FALSE, expand);
      }
      else
      {
	dummy[0] = g_strdup ("");
	node[i] = top;
      }
      break;
    case 1:
      dummy[0] =
	g_strconcat (extract_filename (p->doc_dir), "(", _(title[i]), ")",
		     NULL);
      if (g_list_length (p->doc_files) > 0)
	expand = TRUE;
      else
	expand = FALSE;
      node[i] = gtk_ctree_insert_node (GTK_CTREE (p->widgets.ctree),
				       top, NULL,
				       dummy,
				       5,
				       closed_folder_pix, closed_folder_mask,
				       opened_folder_pix, opened_folder_mask,
				       FALSE, expand);

      break;
    case 2:
      dummy[0] =
	g_strconcat (extract_filename (p->pix_dir), "(", _(title[i]), ")",
		     NULL);
      if (g_list_length (p->pix_files) > 0)
	expand = TRUE;
      else
	expand = FALSE;
      node[i] = gtk_ctree_insert_node (GTK_CTREE (p->widgets.ctree),
				       top, NULL,
				       dummy,
				       5,
				       closed_folder_pix, closed_folder_mask,
				       opened_folder_pix, opened_folder_mask,
				       FALSE, expand);

      break;
    default:
      /* This will never reach. Only to suppress compiler warning */
      expand = FALSE;
    }
    g_free (dummy[0]);
  }
  for (i = 0; i < g_list_length (p->src_files); i++)
  {
    dummy[0] = g_strdup (g_list_nth_data (p->src_files, i));
    top = node[0];
    get_pixmask_on_file_ext (get_file_ext_type (dummy[0]), &pix, &mask);
    gtk_ctree_insert_node (GTK_CTREE (p->widgets.ctree),
			   top, NULL,
			   dummy, 5, pix, mask, pix, mask, TRUE, FALSE);
    g_free (dummy[0]);
  }
  for (i = 0; i < g_list_length (p->doc_files); i++)
  {
    dummy[0] = g_strdup (g_list_nth_data (p->doc_files, i));
    top = node[1];
    get_pixmask_on_file_ext (get_file_ext_type (dummy[0]), &pix, &mask);
    gtk_ctree_insert_node (GTK_CTREE (p->widgets.ctree),
			   top, NULL,
			   dummy, 5, pix, mask, pix, mask, TRUE, FALSE);
    g_free (dummy[0]);
  }
  for (i = 0; i < g_list_length (p->pix_files); i++)
  {
    dummy[0] = g_strdup (g_list_nth_data (p->pix_files, i));
    top = node[2];
    get_pixmask_on_file_ext (get_file_ext_type (dummy[0]), &pix, &mask);
    gtk_ctree_insert_node (GTK_CTREE (p->widgets.ctree),
			   top, NULL,
			   dummy, 5, pix, mask, pix, mask, TRUE, FALSE);
    g_free (dummy[0]);
  }
  gtk_clist_thaw (GTK_CLIST (p->widgets.ctree));
  p->project_is_open = TRUE;
}

void
get_pixmask_on_file_ext (FileExtType t, GdkPixmap ** p, GdkBitmap ** m)
{
  switch (t)
  {
  case FILE_EXT_TYPE_C:
    *p = file_c_pix;
    *m = file_c_mask;
    return;
  case FILE_EXT_TYPE_CPP:
    *p = file_cpp_pix;
    *m = file_cpp_mask;
    return;
  case FILE_EXT_TYPE_H:
    *p = file_h_pix;
    *m = file_h_mask;
    return;
  case FILE_EXT_TYPE_XPM:
  case FILE_EXT_TYPE_JPG:
  case FILE_EXT_TYPE_BMP:
  case FILE_EXT_TYPE_PNG:
  case FILE_EXT_TYPE_TIFF:
  case FILE_EXT_TYPE_GIF:
    *p = file_pix_pix;
    *m = file_pix_mask;
    return;
  default:
    *p = file_file_pix;
    *m = file_file_mask;
    return;
  }
}

void
project_dbase_close_project (ProjectDBase * p)
{
  if (p->project_is_open == FALSE)
    return;
  tags_manager_save (app->tags_manager);
  if (p->is_saved == FALSE)
    project_dbase_save_project (p);
  project_dbase_update_menu (p);
  project_dbase_hide (p);
  project_dbase_clean_left (p);
}

gboolean
project_dbase_save_project (ProjectDBase * p)
{
  FILE *fp;
  gchar *buff;
  gint i;

  tags_manager_save(app->tags_manager);
  if (p->is_saved)
    return TRUE;
  anjuta_set_busy ();
  if (p->project_is_open == FALSE)
  {
    anjuta_set_active ();
    return TRUE;
  }
  buff = g_strdup_printf ("%s.bak", p->proj_filename);
  rename (p->proj_filename, buff);
  g_free (buff);
  fp = fopen (p->proj_filename, "w");
  if (fp == NULL)
    goto error_show;
  if (fprintf (fp, "# Anjuta Version %s \n", VERSION) < 1)
    goto error_show;
  if (fprintf (fp, "Compatibility Level: %d \n", COMPATIBILITY_LEVEL) < 1)
    goto error_show;
  if (fprintf (fp, "Project Type: %d \n", (int) p->type) < 1)
    goto error_show;
  if (fprintf (fp, "Project Name: %s \n", p->proj_name) < 1)
    goto error_show;
  if (fprintf (fp, "Program Name: %s \n", p->prog_name) < 1)
    goto error_show;
  if (fprintf (fp, "Version: %s \n", p->version) < 1)
    goto error_show;
  fprintf (fp, "Author: ");
  write_line (fp, p->author);

  if (strcmp (p->src_dir, p->top_proj_dir) == 0)
    fprintf (fp, "Source Dir: . \n");
  else
    if (fprintf (fp, "Source Dir: %s \n", extract_filename (p->src_dir)) <
	1) goto error_show;
  if (fprintf (fp, "Pixmap Dir: %s \n", extract_filename (p->pix_dir)) < 1)
    goto error_show;
  if (fprintf (fp, "Document Dir: %s \n", extract_filename (p->doc_dir)) < 1)
    goto error_show;
  if (!write_string (fp, "Includes", p->includes))
    goto error_show;
  if (!write_string (fp, "Libraries", p->libraries))
    goto error_show;
  if (fprintf (fp, "Project Has Gui: %d \n", (int) p->project_has_gui) < 1)
    goto error_show;
  if (fprintf (fp, "Gettext Support: %d \n", (int) p->gettext_support) < 1)
    goto error_show;
  if (fprintf (fp, "Manage Makefiles: %d \n", (int) p->overwrite_makefiles) <
      1) goto error_show;
  if (fprintf (fp, "Set Cflags: %d \n", (int) p->set_cflags) < 1)
    goto error_show;

  if (compiler_options_save (app->compiler_options, fp) < 1)
    goto error_show;
  if (src_paths_save (app->src_paths, fp) < 1)
    goto error_show;

  if (fprintf (fp, "No of Source Files: %d \n", g_list_length (p->src_files))
      < 1)
    goto error_show;
  anjuta_status ("Saving Project ...");

  for (i = 0; i < g_list_length (p->src_files); i++)
  {
    if (fprintf
	(fp, "Src: %s \n", (gchar *) g_list_nth_data (p->src_files, i)) < 1)
      goto error_show;
  }

  if (fprintf
      (fp, "No of Document Files: %d \n", g_list_length (p->doc_files)) < 1)
    goto error_show;
  for (i = 0; i < g_list_length (p->doc_files); i++)
  {
    if (fprintf
	(fp, "Doc: %s \n", (gchar *) g_list_nth_data (p->doc_files, i)) < 1)
      goto error_show;
  }
  project_dbase_write_src_makefile (p);
  project_dbase_write_doc_makefile (p);
  project_dbase_write_potfile (p);
  tags_manager_save (app->tags_manager);

  p->is_saved = TRUE;
  fclose (fp);
  anjuta_status (_("Project Saved Successfully"));
  p->is_saved = TRUE;
  anjuta_set_active ();
  return TRUE;

error_show:
  anjuta_error (_("ERROR: Unable to save the project.\n" \
		  "Either You do not have enough free space " \
		  "or it is a bug. Use the backup file. :-("));
  fclose (fp);
  anjuta_status (_("Error while saving project"));
  anjuta_set_active ();
  return FALSE;
}

gboolean
project_dbase_save_yourself (ProjectDBase * p, FILE * stream)
{
  if (!p)
    return FALSE;

  if (fprintf (stream, "Project Dbase:\n") < 0)
    return FALSE;
  if (p->is_docked)
  {
    if (fprintf (stream, "%d\n", 1) < 1)
      return FALSE;
  }
  else if (fprintf (stream, "%d\n", 0) < 1)
    return FALSE;
  if (p->is_showing)
  {
    if (!p->is_docked)
    {
      gdk_window_get_root_origin (p->widgets.window->window,
				  &p->win_pos_x, &p->win_pos_y);
      gdk_window_get_size (p->widgets.window->window,
			   &p->win_width, &p->win_height);
    }
  }
  if (fprintf (stream, "%d %d %d %d\n",
	       p->win_pos_x, p->win_pos_y, p->win_width, p->win_height) < 4)
    return FALSE;
  return TRUE;
}

gboolean
project_dbase_load_yourself (ProjectDBase * p, FILE * stream)
{
  gboolean dock_flag;
  if (!p)
    return FALSE;

  if (fscanf (stream, "Project Dbase:\n") < 0)
    return FALSE;
  if (fscanf (stream, "%d\n", &dock_flag) < 1)
    return FALSE;
  if (fscanf (stream, "%d %d %d %d\n",
	      &p->win_pos_x, &p->win_pos_y, &p->win_width,
	      &p->win_height) < 4)
    return FALSE;

  if (dock_flag)
    project_dbase_dock (p);
  else
    project_dbase_undock (p);
  return TRUE;
}

int
select_only_file (const struct dirent *e)
{
  gchar *filename;
  int ret;
  filename = g_strconcat (app->project_dbase->pix_dir, "/", e->d_name, NULL);
  ret = file_is_regular (filename);
  g_free (filename);
  return ret;
}

void
project_dbase_scan_pix_files (ProjectDBase * p)
{
  struct dirent **namelist;
  int n, i;
  n = scandir (p->pix_dir, &namelist, select_only_file, alphasort);
  if (n >= 0)
  {
    for (i = 0; i < g_list_length (p->pix_files); i++)
      g_free (g_list_nth_data (p->pix_files, i));
    g_list_free (p->pix_files);
    p->pix_files = NULL;
    for (i = 0; i < n; i++)
      p->pix_files =
	g_list_append (p->pix_files, g_strdup (namelist[i]->d_name));
    free (namelist);
  }
}

void
project_dbase_write_src_makefile (ProjectDBase * p)
{
  FILE *fp;
  gchar *filename, *buff, *buff1;
  gint i;

  if (!p)
    return;

  if (p->project_is_open == FALSE ||
      p->overwrite_makefiles == FALSE ||
      p->type == blank || p->src_makefile_saved == TRUE)
    return;

  filename = g_strdup_printf ("%s/Makefile.am", p->src_dir);
  fp = fopen (filename, "w");
  if (fp)
  {
    fprintf (fp,
	     _
	     ("## Makefile.am : Created by anjuta and will be overwritten\n"));
    fprintf (fp,
	     _
	     ("## If you do not want it to be overwritten then please Un-check\n"));
    fprintf (fp,
	     _
	     ("## the button \"Overwrite Makefiles\" in the compiler options\n"));

    buff = compiler_options_get_include_paths (app->compiler_options);
    buff1 = compiler_options_get_library_paths (app->compiler_options);
    fprintf (fp, "INCLUDES = %s %s %s \n\n", p->includes, buff, buff1);
    g_free (buff);
    if (p->set_cflags)
    {
      buff = compiler_options_get_cflags (app->compiler_options);
      fprintf (fp, "CFLAGS = %s\n\n", buff);
      g_free (buff);
    }
    fprintf (fp, "bin_PROGRAMS = %s \n\n", p->prog_name);
    fprintf (fp, "%s_SOURCES = ", p->prog_name);
    for (i = 0; i < g_list_length (p->src_files); i++)
    {
      fprintf (fp, "\\\n\t %s ", (gchar *) g_list_nth_data (p->src_files, i));
    }
    fprintf (fp, "\n\n");
    buff = compiler_options_get_libraries (app->compiler_options);
    fprintf (fp, "%s_LDADD = %s %s\n\n", p->prog_name, p->libraries, buff);
    g_free (buff);
    fclose (fp);
  }
}

void
project_dbase_write_doc_makefile (ProjectDBase * p)
{
  FILE *fp;
  gchar *filename;
  gint i;
  if (!p)
    return;

  if (p->project_is_open == FALSE ||
      p->overwrite_makefiles == FALSE ||
      p->type == blank || p->doc_makefile_saved == TRUE)
    return;

  filename = g_strdup_printf ("%s/Makefile.am", p->doc_dir);
  fp = fopen (filename, "w");
  if (fp)
  {
    fprintf (fp,
	     _
	     ("## Makefile.am : Created by anjuta and will be overwritten\n"));
    fprintf (fp,
	     _
	     ("## If you do not want it to be overwritten then please Un-check\n"));
    fprintf (fp,
	     _
	     ("## the button \"Overwrite Makefiles\" in the compiler options\n"));
    fprintf (fp, "%sdocdir = $(prefix)/doc/%s\n\n", p->prog_name,
	     p->prog_name);
    fprintf (fp, "%sdoc_DATA = ", p->prog_name);
    for (i = 0; i < g_list_length (p->doc_files); i++)
    {
      fprintf (fp, "\\\n\t %s ", (gchar *) g_list_nth_data (p->doc_files, i));
    }
    fprintf (fp, "\n\n");
    fprintf (fp, "EXTRA_DIST = $(%sdoc_DATA) \n\n", p->prog_name);
    fclose (fp);
  }
}

void
project_dbase_write_potfile (ProjectDBase * p)
{
  FILE *fp;
  gchar *filename, *src_dir;
  gint i;

  if (!p)
    return;

  if (p->project_is_open == FALSE ||
      p->overwrite_makefiles == FALSE ||
      p->type == blank || p->potfile_saved == TRUE)
    return;

  if (p->gettext_support == FALSE)
    return;
  filename = g_strdup_printf ("%s/po/POTFILES.in", p->top_proj_dir);
  fp = fopen (filename, "w");
  if (fp)
  {
    fprintf (fp,
	     _
	     ("## POTFILES.in : Created by anjuta and will be overwritten\n"));
    fprintf (fp,
	     _
	     ("## If you do not want it to be overwritten then please Un-check\n"));
    fprintf (fp,
	     _
	     ("## the button \"Overwrite POTFILES.in\" in the compiler options\n"));
    fprintf (fp,
	     "# List of source files containing translatable strings.\n\n");
    fprintf (fp, "# Source files:\n");
    src_dir = extract_filename (p->src_dir);
    for (i = 0; i < g_list_length (p->src_files); i++)
    {
      fprintf (fp, "%s/%s\n", src_dir,
	       (gchar *) g_list_nth_data (p->src_files, i));
    }
    fclose (fp);
  }
}


void
project_dbase_dock (ProjectDBase * p)
{
  if (p)
  {
    if (p->is_docked)
      return;
    if (p->is_showing)
    {
      project_dbase_hide (p);
      p->is_docked = TRUE;
      project_dbase_show (p);
      return;
    }
    else
    {
      p->is_docked = TRUE;
      return;
    }
  }
}
void
project_dbase_undock (ProjectDBase * p)
{
  if (p)
  {
    if (!p->is_docked)
      return;
    if (p->is_showing)
    {
      project_dbase_hide (p);
      p->is_docked = FALSE;
      project_dbase_show (p);
      return;
    }
    else
    {
      p->is_docked = FALSE;
      return;
    }
  }
}

/*******************************
 * Private functions: Do not use  *
 *******************************/
void
project_dbase_detach (ProjectDBase * p)
{
  gtk_container_remove (GTK_CONTAINER
			(app->widgets.project_dbase_win_container),
			app->project_dbase->widgets.client);
  gtk_container_add (GTK_CONTAINER (app->project_dbase->widgets.client_area),
		     app->project_dbase->widgets.client);

  if (app->widgets.the_client == app->widgets.vpaned)
  {
    gtk_container_remove (GTK_CONTAINER (app->widgets.hpaned),
			  app->widgets.notebook);
    gtk_container_remove (GTK_CONTAINER (app->widgets.vpaned),
			  app->widgets.hpaned);
    gtk_container_add (GTK_CONTAINER (app->widgets.vpaned),
		       app->widgets.notebook);
    app->widgets.hpaned_client = app->widgets.notebook;
  }
  else
  {
    gtk_container_remove (GTK_CONTAINER (app->widgets.hpaned),
			  app->widgets.notebook);
    gtk_container_remove (GTK_CONTAINER (app->widgets.client_area),
			  app->widgets.hpaned);
    gtk_container_add (GTK_CONTAINER (app->widgets.client_area),
		       app->widgets.notebook);
    app->widgets.hpaned_client = app->widgets.notebook;
    app->widgets.the_client = app->widgets.notebook;
  }
  gtk_widget_show (app->project_dbase->widgets.client);
  gtk_widget_show (app->widgets.notebook);
}

void
project_dbase_attach (ProjectDBase * p)
{
  gtk_container_remove (GTK_CONTAINER
			(app->project_dbase->widgets.client_area),
			app->project_dbase->widgets.client);
  gtk_container_add (GTK_CONTAINER (app->widgets.project_dbase_win_container),
		     app->project_dbase->widgets.client);

  if (app->widgets.the_client == app->widgets.vpaned)
  {
    gtk_container_remove (GTK_CONTAINER (app->widgets.vpaned),
			  app->widgets.notebook);
    gtk_container_add (GTK_CONTAINER (app->widgets.hpaned),
		       app->widgets.notebook);
    gtk_container_add (GTK_CONTAINER (app->widgets.vpaned),
		       app->widgets.hpaned);
    app->widgets.hpaned_client = app->widgets.hpaned;
  }
  else
  {
    gtk_container_remove (GTK_CONTAINER (app->widgets.client_area),
			  app->widgets.notebook);
    gtk_container_add (GTK_CONTAINER (app->widgets.hpaned),
		       app->widgets.notebook);
    gtk_container_add (GTK_CONTAINER (app->widgets.client_area),
		       app->widgets.hpaned);
    app->widgets.hpaned_client = app->widgets.hpaned;
    app->widgets.the_client = app->widgets.hpaned;
  }
  gtk_widget_show (app->widgets.the_client);
}

void
project_dbase_update_menu (ProjectDBase * p)
{
  GtkWidget *submenu;
  if (p->project_is_open)
  {
    app->recent_projects =
      update_string_list (app->recent_projects, p->proj_filename,
			  app->preferences->max_recent_prjs);
    submenu =
      create_submenu (_("Recent Projects "), app->recent_projects,
		      GTK_SIGNAL_FUNC
		      (on_recent_projects_menu_item_activate));
    gtk_menu_item_set_submenu (GTK_MENU_ITEM
			       (app->widgets.menubar.file.recent_projects),
			       submenu);
  }
}

void
project_dbase_clean_left (ProjectDBase * p)
{
  TextEditor *te;
  gint i;
  project_dbase_clear (p);
  tags_manager_freeze (app->tags_manager);
  tags_manager_clear (app->tags_manager);
  for (i = 0; i < g_list_length (app->text_editor_list); i++)
  {
    te = g_list_nth_data (app->text_editor_list, i);
    if (te->full_filename)
      tags_manager_update (app->tags_manager, te->full_filename);
  }
  tags_manager_thaw (app->tags_manager);
  compiler_options_load (app->compiler_options, NULL);
  src_paths_load (app->src_paths, NULL);
}

gboolean
project_dbase_file_is_src (ProjectDBase * p, gchar * file)
{
  gint i;
  gchar *dir, *fname;

  if (!p || !file)
    return FALSE;
  if (p->project_is_open == FALSE)
    return FALSE;
  dir = g_dirname (file);
  fname = extract_filename (file);
  if (dir)
    if (strcmp (p->src_dir, dir) != 0)
    {
      if (dir)
	g_free (dir);
      return FALSE;
    }
  g_free (dir);
  for (i = 0; i < g_list_length (p->src_files); i++)
    if (strcmp ((gchar *) g_list_nth_data (p->src_files, i), fname) == 0)
      return TRUE;
  return FALSE;
}

void
project_dbase_show_info (ProjectDBase * p)
{
  gchar *str[14];
  gint i;

  if (!p)
    return;
  if (p->project_is_open == FALSE)
    return;
  str[0] = g_strdup (p->proj_name);
  str[1] = g_strdup (p->version);
  str[2] = g_strdup (p->author);
  if (p->project_has_gui)
    str[3] = g_strdup (_("Yes"));
  else
    str[3] = g_strdup (_("No"));
  if (p->overwrite_makefiles)
    str[4] = g_strdup (_("Yes"));
  else
    str[4] = g_strdup (_("No"));
  if (p->gettext_support)
    str[5] = g_strdup (_("Yes"));
  else
    str[5] = g_strdup (_("No"));
  str[6] = g_strdup (p->prog_name);
  if (strcmp (p->src_dir, p->top_proj_dir) == 0)
    str[7] = g_strdup (_("< Top level dir >"));
  else
    str[7] = g_strdup (extract_filename (p->src_dir));
  str[8] = g_strdup (extract_filename (p->doc_dir));
  str[9] = g_strdup (extract_filename (p->pix_dir));
  str[10] = g_strdup_printf ("%d", g_list_length (p->src_files));
  str[11] = g_strdup_printf ("%d", g_list_length (p->doc_files));
  str[12] = g_strdup_printf ("%d", g_list_length (p->pix_files));
  switch (p->type)
  {
  case blank:
    str[13] = g_strdup (_("Blank Application (Custom)"));
    break;
  case console:
    str[13] = g_strdup (_("Console Application"));
    break;
  case gnome_blank:
    str[13] = g_strdup (_("Gnome Blank Application"));
    break;
  case gnome_full:
    str[13] = g_strdup (_("Gnome Full Application"));
    break;
  default:
    str[13] = g_strdup (_("Unknown Application type"));
    break;
  }
  gtk_widget_show (create_project_dbase_info_gui (str));
  for (i = 0; i < 14; i++)
    g_free (str[i]);
  return;
}

void
project_dbase_update_controls (ProjectDBase * pd)
{
  gboolean P;
  P = pd->project_is_open;
  gtk_widget_set_sensitive (pd->widgets.menu_add_src, P);
  gtk_widget_set_sensitive (pd->widgets.menu_add_doc, P);
  gtk_widget_set_sensitive (pd->widgets.menu_add_pix, P);
  gtk_widget_set_sensitive (pd->widgets.menu_add_other, P);
  gtk_widget_set_sensitive (pd->widgets.menu_add_dir, P);
  gtk_widget_set_sensitive (pd->widgets.menu_remove, P);
  gtk_widget_set_sensitive (pd->widgets.menu_configure, P);
  gtk_widget_set_sensitive (pd->widgets.menu_info, P);
}
