/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#define HAVE_GNU_REGEXP

#ifdef EDITOR_DEBUG
#include <stdio.h>
#endif

#include <stdlib.h>

#include <sys/types.h>
#include <regex.h>
#include <string.h>

#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>
#include "gtkaneditor.h"

/* hilite flags */
#define BASIC         0x01
#define IN_COMMENT    0x02
#define IN_STRING     0x04
#define IN_KEYWORD    0x08

#define HIDDEN        0x10
#define LINK          0x20

#define TIME_DELAY    600	/* delay in milisec. before hilite */

#ifdef EDITOR_DEBUG_MATCH
#define MATCH(a,b,c) (g_print ("\nMATCH (%d,%d,%d)\n", (a), (b), (c)), \
		      (MAX (MAX ((a),(b)), (c)) > -1))
#define FIRST_LEAST(a,b,c) (g_print ("\nFL (%d,%d,%d)\n", (a), (b), (c)), \
			    (((b)>-1 ? (a) <= (b) : 1) \
			    && ((c)>-1 ? (a) <= (c) : 1) \
			    && ((a)>-1)))
#else
#define MATCH(a,b,c) (MAX (MAX ((a),(b)), (c)) > -1)
#define FIRST_LEAST(a,b,c) (((b)>-1 ? (a) <= (b) : 1) \
			    && ((c)>-1 ? (a) <= (c) : 1) \
			    && ((a)>-1))
#endif

typedef struct _MarkData MarkData;

#ifdef HAVE_GNU_REGEXP
/* GNU REGEXP LIBRARY -- more regexp stuff. */
typedef struct _Regex {
  struct re_pattern_buffer buf;
  struct re_registers reg;
} Regex;
#else  /* use posix */
typedef regex_t Regex;
#endif

#ifdef HAVE_GNU_REGEXP
/* GNU REGEXP LIBRARY -- more regexp stuff. */

static GtkAnEditorHilitePatterns default_patterns =
{ "/\\*",			/* comment_start */
  "\\*/",			/* comment_end */
  "\"\\|'",			/* string_start */
  "\"\\|'",			/* string_end */
  "\\b\\(if\\|else\\|for\\|while\\|do\\|break\\|"\
  "continue\\|switch\\|case\\|goto\\|return\\|"\
  "int\\|unsigned\\|short\\|long\\|char\\|float\\|"\
  "double\\|gchar\\|gint\\|guint\\|glong\\|gboolean\\|gulong\\|"\
  "gchar\\|gfloat\\|gdouble\\|typedef\\|struct\\|"\
  "enum\\|union\\|static\\|auto\\|extern\\|asm\\|"\
  "void\\|include\\|define\\|ifdef\\|ifndef\\|endif\\)\\b"
};

#else

static GtkAnEditorHilitePatterns default_patterns =
{ "/\\*",			/* comment_start */
  "\\*/",			/* comment_end */
  "\"\\|'",			/* string_start */
  "\"\\|'",			/* string_end */
  "if|else|for|while|do|break|continue|switch|"\
  "case|goto|return|int|unsigned|short|long|char|"\
  "float|double|gchar|gint|guint|glong|gulong|gchar|gboolean|"\
  "gfloat|gdouble|typedef|struct|enum|union|static|"\
  "auto|extern|asm|void|include|define|ifdef|ifndef|endif"
};

#endif /* HAVE_GNU_REGEXP */

typedef struct _Match {
  gint from;
  gint to;
} Match;

struct _GtkAnEditorHiliteRegexps {
  Regex comment_start_regex;
  Regex comment_end_regex;
  Regex string_start_regex;
  Regex string_end_regex;
  Regex keywords_regex;
};

struct _MarkData {
  guint8 state;
  gchar *hidden;
};

/* widget stuff */
static void   gtk_aneditor_class_init          (GtkAnEditorClass   *klass);
static void   gtk_aneditor_init                (GtkAnEditor        *editor);
static void   gtk_aneditor_destroy             (GtkObject        *object);

/* text property data */
static gboolean cmp_prop_data                (MarkData         *d1,
					      MarkData         *d2);
static MarkData *clone_prop_data             (MarkData         *d);
static void free_prop_data                   (MarkData         *d);
static MarkData *new_mark_data               (guint8            state,
					      gchar            *comment);

/* search stuff */
static gint   search                         (GtkAnEditor        *editor,
					      guint             pos,
					      gchar            *string,
					      gboolean          casein);
static gint   search_back                    (GtkAnEditor        *editor,
					      guint             pos,
					      gchar            *string,
					      gboolean          casein);

/* regex stuff */
static gboolean compile_regex                (char             *pattern,
					      Regex            *regex);
/* I don't use the following 2 functions right now, but I don't dare
 * throw them away yet...who knows when they will be usefull again? */
static gint   regex_search_string            (Regex            *regex,
					      char             *string,
					      Match            *m);
static gint   regex_match_string             (Regex            *regex,
					      char             *string);
static gint   regex_search                   (GtkAnText          *text,
					      guint             pos,
					      Regex            *regex,
					      gboolean          forward,
					      Match            *m);
static gint   regex_match                    (GtkAnText          *text,
					      guint             pos,
					      Regex            *regex);

/* hilit */
static GtkAnEditorHiliteColors *
              new_colors                     (GdkColormap      *cmap);

static GtkAnEditorHiliteFonts *
              new_fonts                     (void);

static void   hilite_interval                (GtkAnEditor        *editor,
					      guint             from,
					      guint             to);
static gboolean hilite_when_idle             (GtkAnEditor        *editor);

/* indent */
static void   default_one_line_ind           (GtkAnEditor        *editor);
static void   indent_region                  (GtkAnEditor        *editor);


/* events */
static gint   gtk_aneditor_key_press           (GtkWidget        *widget,
					      GdkEventKey      *event);
static void   gtk_aneditor_changed_pre         (GtkAnEditor        *editor);
static void   gtk_aneditor_changed_post        (GtkAnEditor        *editor);


/* --<local data>--------------------------------------------------------- */
static GtkWidgetClass *parent_class = NULL;
static GtkAnEditorHiliteRegexps *default_regexps = NULL;

/* --<widget initialization, constructor and destructor>------------------ */
guint
gtk_aneditor_get_type ()
{
  static guint editor_type = 0;

  if (!editor_type)
    {
      GtkTypeInfo editor_info =
      {
	"GtkAnEditor",
	sizeof (GtkAnEditor),
	sizeof (GtkAnEditorClass),
	(GtkClassInitFunc) gtk_aneditor_class_init,
	(GtkObjectInitFunc) gtk_aneditor_init,
	(GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL,
      };

      editor_type = gtk_type_unique (gtk_antext_get_type (), &editor_info);
    }

  return editor_type;
}

static void
gtk_aneditor_class_init (GtkAnEditorClass *class)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) class;

#ifdef HAVE_GNU_REGEXP
  re_set_syntax (RE_SYNTAX_EMACS);
#endif

  /* init local data */
  parent_class = gtk_type_class (gtk_antext_get_type ());
  default_regexps = gtk_aneditor_new_regexps (&default_patterns);
  g_assert (default_regexps);	/* the defaults shouldn't fail */

  /* setup signals */
  object_class->destroy = gtk_aneditor_destroy;
}

static void
gtk_aneditor_init (GtkAnEditor *editor)
{
  GtkAnText *text = GTK_ANTEXT (editor);

  gtk_antext_set_property_data_functions (text,
					(gboolean(*)(gpointer,gpointer))cmp_prop_data,
					(gpointer(*)(gpointer))clone_prop_data,
					(void(*)(gpointer))free_prop_data);

  /* hilite */
  editor->regexps = default_regexps;
  editor->colors = new_colors (gtk_widget_get_colormap (GTK_WIDGET (editor)));

  /* I put the fonts -Naba */
  editor->fonts = new_fonts ();

  editor->hilite_when_idle = TRUE;
  editor->hilite_time = 0;
  editor->changed_from = editor->changed_to = 0;

  /* to keep track of changes, for hilite_when_idle */
  gtk_signal_connect (GTK_OBJECT (editor), "insert_text",
		      GTK_SIGNAL_FUNC (gtk_aneditor_changed_pre),
		      NULL);
  gtk_signal_connect (GTK_OBJECT (editor), "delete_text",
		      GTK_SIGNAL_FUNC (gtk_aneditor_changed_pre),
		      NULL);
  gtk_signal_connect (GTK_OBJECT (editor), "changed",
		      GTK_SIGNAL_FUNC (gtk_aneditor_changed_post),
		      NULL);

  /* indent */
  editor->one_line_ind = default_one_line_ind;
  gtk_signal_connect (GTK_OBJECT (editor), "key_press_event",
		      GTK_SIGNAL_FUNC (gtk_aneditor_key_press),
		      NULL);
}

GtkWidget*
gtk_aneditor_new (GtkAdjustment *hadj,
		GtkAdjustment *vadj)
{
  GtkAnEditor *editor;

  editor = gtk_type_new (gtk_aneditor_get_type ());

  gtk_antext_set_adjustments (GTK_ANTEXT (editor), hadj, vadj);

  return GTK_WIDGET (editor);
}

static void
gtk_aneditor_destroy (GtkObject *object)
{
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_ANEDITOR (object));

  if (GTK_ANEDITOR (object)->regexps != default_regexps) {
    gtk_aneditor_destroy_regexps (GTK_ANEDITOR (object)->regexps);
  }
  g_free (GTK_ANEDITOR (object)->colors);

  if(GTK_ANEDITOR (object)->fonts->plain)
    gdk_font_unref(GTK_ANEDITOR (object)->fonts->plain);
  if(GTK_ANEDITOR (object)->fonts->comment)
    gdk_font_unref(GTK_ANEDITOR (object)->fonts->comment);
  if(GTK_ANEDITOR (object)->fonts->string)
    gdk_font_unref(GTK_ANEDITOR (object)->fonts->string);
  if(GTK_ANEDITOR (object)->fonts->keyword)
    gdk_font_unref(GTK_ANEDITOR (object)->fonts->keyword);
  g_free (GTK_ANEDITOR (object)->fonts);
  
  GTK_OBJECT_CLASS(parent_class)->destroy (object);
}

/* --<text property data>------------------------------------------------- */
static gboolean
cmp_prop_data (MarkData *d1, MarkData *d2)
{
  if (d1 == d2) return TRUE;

  if (d1 && d2 && (d1->state == d2->state)) {
    /* states equal */
    if (d1->hidden == d2->hidden) {
      /* should only be possible if both are NULL */
      return TRUE;
    } else {
      return (d1->hidden && d2->hidden
	      && (strcmp (d1->hidden, d2->hidden) == 0));
    }
  }
      
  return FALSE;
}

static MarkData *
clone_prop_data (MarkData *d)
{
  MarkData *new;
  if (!d) return NULL;
  new = g_new (MarkData, 1);
  new->state = d->state;
  if(d->hidden) new->hidden = g_strdup (d->hidden);
  else  new->hidden = NULL;
  return new;
}

static void
free_prop_data (MarkData *d)
{
  if (!d) return;
  if( d->hidden) g_free (d->hidden);
  g_free (d);
}

static MarkData *
new_mark_data (guint8 state, gchar *hidden)
{
  MarkData *new = g_new (MarkData, 1);
  new->state = state;
  new->hidden = hidden;
  return new;
}

/* --<hide and show stuff>------------------------------------------------ */
/* gtk_aneditor_hide_text -- removes the text in the interval [from,to[,
 * and insert subs as a substitute.  The old text is saved in the
 * property data, and can be called forth with a following show call.
 * The substitute will have properties corresponding to
 * (font,fore,back). The hidden text will not keep any properties
 * (since the interval migth contain several, and there is no clear
 * way of knowing which to choose.  Any other text hidden in the
 * interval will be lost.  FIXME: This probably shouldn't be so. */
/* FIXME: crashes!  I think it's a bug in gtktext...I'll try to
 * reproducet with only the text widget. */
void
gtk_aneditor_hide_text (GtkAnEditor *editor, guint from,
		      guint to, gchar *subs,
		      GdkFont *font, GdkColor *fore, GdkColor *back)
{
  gchar *tmp;
  MarkData *mdata, *old;

  g_return_if_fail (from >= 0);
  g_return_if_fail (to <= gtk_antext_get_length (GTK_ANTEXT (editor)));
  g_return_if_fail (subs != NULL);

  g_print ("gtk_aneditor_hide_text [%u, %u[\n", from, to);

  tmp = gtk_editable_get_chars (GTK_EDITABLE (editor), from, to);
  gtk_editable_set_position (GTK_EDITABLE (editor), from);
  old = gtk_antext_get_property_data (GTK_ANTEXT (editor),
				    gtk_antext_get_point (GTK_ANTEXT (editor)));
  if (old) {
    mdata = new_mark_data (old->state | HIDDEN, tmp);
  } else {
    mdata = new_mark_data (BASIC | HIDDEN, tmp);
  }
  gtk_editable_delete_text (GTK_EDITABLE (editor), from, to);
  gtk_antext_insert_with_data (GTK_ANTEXT (editor),
			     font, fore, back, mdata,
			     subs, strlen (subs));
}

/* --<search'n'stuff>----------------------------------------------------- */
/* NB! There is really no reason why all this searching stuff couldn't
 * be moved to the text widget...the only reason I can think of is
 * that, the both gtktext and gtkeditor would have to deal with
 * regexps, but that doesn't seem too discouraging. We will have to
 * make them more 'public' though...the editor _needs_ some of the
 * static functions, so we can't hide them in the text widget. */

/* search -- searches in editor, from pos and forward, for string.  If
 * string is found, returns position, otherwise returns -1.
 * If casein is true, search is case insensitive. */
/* FIXME: This could and should be optimised. There is no need to
 * get_text all the time, is there?  We just have to be carefull with
 * the gap. */
static gint
search (GtkAnEditor *editor, guint pos, gchar *string, gboolean casein)
{
  gchar *buf = NULL;
  gint match = -1;
  gint text_len = gtk_antext_get_length (GTK_ANTEXT (editor));
  gint len = strlen (string);

  g_return_val_if_fail (pos <= text_len, -1);
  if (pos == text_len) return -1; /* not illegal, but nothing to do */

  gtk_antext_freeze (GTK_ANTEXT (editor));
  for (; pos < (text_len - len)+1; pos++) {
    buf = gtk_editable_get_chars (GTK_EDITABLE (editor), pos, pos + len);
    if (!buf) return match;	/* somethings wrong */
    if (casein) {
      if (strcasecmp (buf, string) == 0) {
	match = pos;
	g_free (buf);
	break;
      }
    } else {
      if (strcmp (buf, string) == 0) {
	match = pos;
	g_free (buf);
	break;
      }
    }
    g_free (buf);
  }
  gtk_antext_thaw (GTK_ANTEXT (editor));

  return match;
}

/* gtk_aneditor_search_from_point -- searches for string, if found, goto
 * and select match. If casein is true, search is case insensitive. If
 * string is found it returns TRUE, otherwise FALSE. */
gboolean
gtk_aneditor_search_from_point (GtkAnEditor *editor, gchar *string, gboolean casein)
{
  gint new_pos;

  g_return_val_if_fail (editor != NULL, FALSE);
  g_return_val_if_fail (string != NULL, FALSE);

  new_pos = search (editor, gtk_antext_get_point (GTK_ANTEXT (editor)), string, casein);
  if (new_pos > -1) {
    gtk_editable_set_position (GTK_EDITABLE (editor), new_pos + strlen (string));
    gtk_antext_set_point (GTK_ANTEXT (editor), new_pos + strlen (string));
    gtk_editable_select_region (GTK_EDITABLE (editor), new_pos,
				new_pos + strlen (string));
    return TRUE;
  } else {
    return FALSE;
  }
}

/* search_back -- searches in editor, from pos and backward, for
 * string.  If string is found, returns position, otherwise returns
 * -1.  If casein is true, search is case insensitive. */
/* FIXME: This could and should be optimised. There is no need to
 * get_text all the time, is there?  We just have to be carefull with
 * the gap. */
static gint
search_back (GtkAnEditor *editor, guint pos, gchar *string, gboolean casein)
{
  gchar *buf = NULL;
  gint match = -1;
  gint text_len = gtk_antext_get_length (GTK_ANTEXT (editor));
  gint len = strlen (string);

  if (pos == 0) return -1; /* not illegal, but nothing to do */
  if (pos > text_len - len) pos = text_len - len; /* we must be reasonable */

  gtk_antext_freeze (GTK_ANTEXT (editor));
  pos -= strlen(string);
  if(pos <= 0) return -1;
  for (; pos >= 0; pos--) {
    buf = gtk_editable_get_chars (GTK_EDITABLE (editor), pos, pos + len);
    if (!buf) return match;	/* somethings wrong */
    if (casein) {
      if (strcasecmp (buf, string) == 0) {
	match = pos;
	break;
      }
    } else {
      if (strcmp (buf, string) == 0) {
	match = pos;
	break;
      }
    }
    g_free (buf);
  }
  g_free (buf);
  gtk_antext_thaw (GTK_ANTEXT (editor));

  return match;
}

/* gtk_aneditor_search_back_from_point -- searches backward for string,
 * if found, goto and select match. If casein is true, search is case
 * insensitive. If string is found it returns TRUE, otherwise it
 * returns FALSE. */
gboolean
gtk_aneditor_search_back_from_point (GtkAnEditor *editor, 
				   gchar *string,
				   gboolean casein)
{
  gint new_pos;

  g_return_val_if_fail (editor != NULL, FALSE);
  g_return_val_if_fail (string != NULL, FALSE);

  new_pos = search_back (editor, gtk_antext_get_point (GTK_ANTEXT (editor)),
			 string, casein);
  if (new_pos > -1) {
    gtk_editable_set_position (GTK_EDITABLE (editor), new_pos);
    gtk_antext_set_point (GTK_ANTEXT (editor), new_pos);
    gtk_editable_select_region (GTK_EDITABLE (editor), new_pos,
				new_pos + strlen (string));
    return TRUE;
  } else {
    return FALSE;
  }
}

/* compile_regex -- compiles pattern into a regex structure that can
 * be used in regex_search and regex_match.  Returns TRUE if
 * succesfull, FALSE if not.*/
static gboolean
compile_regex (gchar *pattern, Regex *regex)
{
#ifdef HAVE_GNU_REGEXP
  regex->buf.translate = NULL;
  regex->buf.fastmap = NULL;
  regex->buf.allocated = 0;
  regex->buf.buffer = NULL;
  if (re_compile_pattern (pattern, strlen (pattern), &regex->buf) == 0) {
    return TRUE;
  } else {
    return FALSE;
  }
#else
  if (regcomp (regex, pattern, REG_EXTENDED) == 0) {
    return TRUE;
  } else {
    return FALSE;
  }
#endif /* HAVE_GNU_REGEXP */
}

/* regex_free -- wrapper around regfree, to abstract from the differen
 * regex libs.  This function does *not* free the Regex structure!  If
 * this has to be done call g_free on the structure after calling
 * regex_free. */
static void
regex_free (Regex *regex)
{
#ifdef HAVE_GNU_REGEXP
  if (regex->buf.regs_allocated) {
    free (regex->reg.start);
    free (regex->reg.end);
  }
  regfree (&regex->buf);
#else
  regfree (regex);
#endif
}

/* regex_search_string -- searches for regex in string. If found,
 * returns index of beginning of the match, otherwise returns < 0.  If
 * found the match interval is [m.from,m.to[.  m.from is allways equal
 * to the return value, but m.to is undefined if no match. */
static gint
regex_search_string (Regex *regex, char *string, Match *m)
{
#ifdef HAVE_GNU_REGEXP
  gint len = strlen (string);
  m->from = re_search (&regex->buf, string, len, 0, len, &regex->reg);
  if (m->from > -1) m->to = regex->reg.end[0];
  return m->from;
#else
  regmatch_t pm[1];		/* Why array? So it's easier to extend! */

  if (regexec (regex, string, 1, pm, 0) == 0) {
    m->from = pm[0].rm_so;
    m->to = pm[0].rm_eo;
  } else {
    m->from = -1;
  }
  return m->from;
#endif /* HAVE_GNU_REGEXP */
}

/* regex_match_string -- tries to match regex in beginning of
 * string. It returns the number of chars matched, or -1 if no match.
 * Warning!  The number matched can be 0, if the regex matches the
 * empty string. */
static gint
regex_match_string (Regex *regex, char *string)
{
#ifdef HAVE_GNU_REGEXP
  gint len = strlen (string);
  return re_match (&regex->buf, string, len, 0, NULL);
#else
  /* FIXME: This is *very* inefficient since it searches forward until
   * it finds a match, and *then* checks wether it was in the
   * beginning of the string.  This *has* to be reimplemented! */
  regmatch_t pm;

  if (regexec (regex, string, 1, &pm, 0) == 0) {
    if (pm.rm_so == 0) {
      return pm.rm_eo - pm.rm_so;
    }
  }
  return -1;			/* no match */
#endif /* HAVE_GNU_REGEXP */
}

/* regex_search -- searches for regex in text from position
 * 'start'. If 'forward' is TRUE, it searches forward, otherwise
 * backwards.  If found, returns index of beginning of the match,
 * otherwise returns < 0.  If found the match interval is
 * [m.from,m.to[.  m.from is always equal to the return value, but
 * m.to is undefined if no match.  I search in GtkAnText, since I don't
 * really need an editor for this function. This will also make it
 * easier to move this function to the text widget, should we want
 * that. */
static gint
regex_search (GtkAnText *text, guint start, Regex *regex,
	      gboolean forward, Match *m)
{
  g_return_val_if_fail (start <= text->text_end - text->gap_size, -1);
  g_return_val_if_fail (regex != NULL, -1);
  g_return_val_if_fail (m != NULL, -1);

#ifdef HAVE_GNU_REGEXP
  m->from = re_search_2 (&regex->buf,
			 /* text before gap */
			 text->text.ch, text->gap_position,
			 /* text after gap */
			 text->text.ch + text->gap_position + text->gap_size,
			 text->text_end - (text->gap_position + text->gap_size),
			 /* from 'start' and forward to the end */
			 start,
			 (forward ? text->text_end - text->gap_size - start
			  : -start),
			 &regex->reg,
			 text->text_end - text->gap_size - start);

  if (m->from > -1) m->to = regex->reg.end[0];
  return m->from;    
#else
  g_warning ("%s (%u): regex_search () not implemented yet!", __FILE__, __LINE__);
  return -1;
#endif
}

/* select_regex -- a wrapper around _regex_search_ and
 * _regex_search_back_.  They behave the same besides the search
 * direction.  If forward is TRUE, searches forward, otherwise
 * searches backwards. */
gint
select_regex (GtkAnEditor *editor, gchar *regex, gboolean forward)
{
  Regex re;
  Match m;

  g_return_val_if_fail (editor != NULL, -1);
  g_return_val_if_fail (regex != NULL, -1);

  if (!compile_regex (regex, &re)) {
    regex_free (&re);
    return -1;
  }
  if (regex_search (GTK_ANTEXT (editor), GTK_ANTEXT (editor)->point.index,
		    &re, forward, &m) > -1) {
    if(forward)
    {
        gtk_editable_set_position (GTK_EDITABLE (editor), m.to);
        gtk_antext_set_point (GTK_ANTEXT (editor), m.to);
    }
    else
    {
        gtk_editable_set_position (GTK_EDITABLE (editor), m.from);
        gtk_antext_set_point (GTK_ANTEXT (editor), m.from);
    }
    gtk_editable_select_region (GTK_EDITABLE (editor), m.from, m.to);
    regex_free (&re);
    return 1;
  } else {
    regex_free (&re);
    return 0;
  }
}

/* gtk_aneditor_search_regex_from_point -- searches for string matching
 * regex, if found, goto and select match.  If the regex doesn't
 * compile (or a serious error occurs) it returns -1, if a match is
 * found it returns 1, and if no match is found it returns 0. */
gint
gtk_aneditor_regex_search_from_point (GtkAnEditor     *editor,
				    gchar         *regex)
{
  return select_regex (editor, regex, TRUE);
}

/* gtk_aneditor_search_regex_back_from_point -- searches for string
 * matching regex, if found, goto and select match.  If the regex
 * doesn't compile (or a serious error occurs) it returns -1, if a
 * match is found it returns 1, and if no match is found it returns
 * 0. */
gint
gtk_aneditor_regex_search_back_from_point (GtkAnEditor     *editor,
					 gchar         *regex)
{
  return select_regex (editor, regex, FALSE);
}


/* regex_match -- tries to match regex at the 'pos' position in the
 * text. It returns the number of chars matched, or -1 if no match.
 * Warning!  The number matched can be 0, if the regex matches the
 * empty string.  The reason for workin on GtkAnText is the same as in
 * regex_search. */
static gint
regex_match (GtkAnText *text, guint pos, Regex *regex)
{
  g_return_val_if_fail (pos <= text->text_end - text->gap_size, -1);
  g_return_val_if_fail (regex != NULL, -1);

#ifdef HAVE_GNU_REGEXP
  return re_match_2 (&regex->buf,
		     /* text before gap */
		     text->text.ch, text->gap_position,
		     /* text after gap */
		     text->text.ch + text->gap_position + text->gap_size,
		     text->text_end - (text->gap_position + text->gap_size),
		     /* from pos and not after the end */
		     pos, NULL, text->text_end - text->gap_size);
#else
  g_warning ("%s (%u): regex_match () not implemented yet!", __FILE__, __LINE__);
  return -1;
#endif
}



/* --<hilite stuff>------------------------------------------------------- */
/* gtk_aneditor_new_regexps -- returns compiled regexps from patterns, if all
 * compilations are succesfull. If an error occurs, returns NULL. This
 * function is useful if you want to remember different kinds of
 * 'modes', otherwise you can use gtk_aneditor_set_regexps. */
GtkAnEditorHiliteRegexps *
gtk_aneditor_new_regexps (GtkAnEditorHilitePatterns *patterns)
{
  GtkAnEditorHiliteRegexps *regexps = g_new (GtkAnEditorHiliteRegexps, 1);

  if (!compile_regex (patterns->comment_start,
		      &regexps->comment_start_regex)) {
    g_free (regexps);
    return NULL;
  }
  if (!compile_regex (patterns->comment_end,
		      &regexps->comment_end_regex)) {
    regex_free (&regexps->comment_start_regex);
    g_free (regexps);
    return NULL;
  }
  if (!compile_regex (patterns->string_start,
		      &regexps->string_start_regex)) {
    regex_free (&regexps->comment_start_regex);
    regex_free (&regexps->comment_end_regex);
    g_free (regexps);
    return NULL;
  }
  if (!compile_regex (patterns->string_end,
		      &regexps->string_end_regex)) {
    regex_free (&regexps->comment_start_regex);
    regex_free (&regexps->comment_end_regex);
    regex_free (&regexps->string_start_regex);
    g_free (regexps);
    return NULL;
  }
  if (!compile_regex (patterns->keywords,
		      &regexps->keywords_regex)) {
    regex_free (&regexps->comment_start_regex);
    regex_free (&regexps->comment_end_regex);
    regex_free (&regexps->string_start_regex);
    regex_free (&regexps->string_end_regex);
    g_free (regexps);
    return NULL;
  }

  return regexps;
}

/* gtk_aneditor_set_regexps -- sets all regexps to the new patterns in
 * patterns.  If the patterns cannot compile, it returns FALSE, and
 * the editors regexps remain unchanged.  Otherwise it returns TRUE.
 * This is just a wrapper around gtk_new_regexps */
gboolean
gtk_aneditor_set_patterns (GtkAnEditor *editor,
			 GtkAnEditorHilitePatterns *patterns)
{
  GtkAnEditorHiliteRegexps *tmp;

  g_return_val_if_fail (editor != NULL, FALSE);
  g_return_val_if_fail (patterns != NULL, FALSE);
  
  if ((tmp = gtk_aneditor_new_regexps (patterns))) {
    if (editor->regexps != default_regexps) {
      gtk_aneditor_destroy_regexps (editor->regexps);
    }
    editor->regexps = tmp;
    return TRUE;
  } else {
    return FALSE;
  }
}

/* gtk_aneditor_set_regexps -- like gtk_aneditor_set_patterns, but working
 * on regex structs and not patterns. */
void
gtk_aneditor_set_regexps (GtkAnEditor *editor,
			GtkAnEditorHiliteRegexps *regexps)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (regexps != NULL);

  if (editor->regexps != default_regexps) {
    gtk_aneditor_destroy_regexps (editor->regexps);
  }
  editor->regexps = regexps;
  
}

void
gtk_aneditor_set_colors (GtkAnEditor *editor,
	GdkColor *pc,
	GdkColor *cc,
	GdkColor *sc,
	GdkColor *kc)
{
  GtkAnEditorHiliteColors *colors;

  g_return_if_fail (editor != NULL); 

  colors = editor->colors;
  if(pc)
  {
  colors->plain.pixel = pc->pixel;
  colors->plain.red = pc->red;
  colors->plain.green = pc->green;
  colors->plain.blue = pc->blue;
  }
  if(cc)
  {
  colors->comment.pixel = cc->pixel;
  colors->comment.red = cc->red;
  colors->comment.green = cc->green;
  colors->comment.blue = cc->blue;
  }
  if(sc)
  {
  colors->string.pixel = sc->pixel;
  colors->string.red = sc->red;
  colors->string.green = sc->green;
  colors->string.blue = sc->blue;
  }
  if(kc)
  {
  colors->keywords.pixel = kc->pixel;
  colors->keywords.red = kc->red;
  colors->keywords.green = kc->green;
  colors->keywords.blue = kc->blue;
  }
  gdk_color_alloc(colors->cmap, &(colors->plain));
  gdk_color_alloc(colors->cmap, &(colors->comment));
  gdk_color_alloc(colors->cmap, &(colors->string));
  gdk_color_alloc(colors->cmap, &(colors->keywords));
}

void
gtk_aneditor_set_fonts (GtkAnEditor *editor,
	GdkFont *pf,
	GdkFont *cf,
	GdkFont *sf,
	GdkFont *kf)
{
  GtkAnEditorHiliteFonts *fonts;

  g_return_if_fail (editor != NULL); 
  fonts = editor->fonts;

  if(pf) gdk_font_ref(pf);
  if(cf) gdk_font_ref(cf);
  if(sf) gdk_font_ref(sf);
  if(kf) gdk_font_ref(kf);

  if(fonts->plain) gdk_font_unref(fonts->plain);
  if(fonts->comment) gdk_font_unref(fonts->comment);
  if(fonts->string) gdk_font_unref(fonts->string);
  if(fonts->keyword) gdk_font_unref(fonts->keyword);

  fonts->plain = pf;
  fonts->comment = cf;
  fonts->string = sf;
  fonts->keyword = kf;
}

/* gtk_aneditor_upd_regexps -- Changes the regexps corresponding to the
 * non-NULL patterns in patterns.  If successful, it returns TRUE,
 * otherwise FALSE.  If not successful, some of the regexps might be
 * changed, so beware, there is not guaranty of consistency!  If you
 * want guaranty of consistency use gtk_aneditor_set_regexps. */
gboolean
gtk_aneditor_upd_regexps (GtkAnEditor *editor,
			GtkAnEditorHilitePatterns *patterns)
{
  Regex tmp;
  g_return_val_if_fail (editor != NULL, FALSE);
  g_return_val_if_fail (patterns != NULL, FALSE);

  if (editor->regexps == default_regexps) {
    /* we cannot begin changing the defaults, so we get our own copy. */
    editor->regexps = gtk_aneditor_new_regexps (&default_patterns);
  }

  if (patterns->comment_start) {
    if (compile_regex (patterns->comment_start, &tmp)) {
      regex_free (&editor->regexps->comment_start_regex);
      editor->regexps->comment_start_regex = tmp;
    } else {
      regex_free (&tmp);
      return FALSE;
    }
  }

  if (patterns->comment_end) {
    if (compile_regex (patterns->comment_end, &tmp)) {
      regex_free (&editor->regexps->comment_end_regex);
      editor->regexps->comment_end_regex = tmp;
    } else {
      regex_free (&tmp);
      return FALSE;
    }
  }

  if (patterns->string_start) {
    if (compile_regex (patterns->string_start, &tmp)) {
      regex_free (&editor->regexps->string_start_regex);
      editor->regexps->string_start_regex = tmp;
    } else {
      regex_free (&tmp);
      return FALSE;
    }
  }

  if (patterns->string_end) {
    if (compile_regex (patterns->string_end, &tmp)) {
      regex_free (&editor->regexps->string_end_regex);
      editor->regexps->string_end_regex = tmp;
    } else {
      regex_free (&tmp);
      return FALSE;
    }
  }

  if (patterns->keywords) {
    if (compile_regex (patterns->keywords, &tmp)) {
      regex_free (&editor->regexps->keywords_regex);
      editor->regexps->keywords_regex = tmp;
    } else {
      regex_free (&tmp);
      return FALSE;
    }
  }
  return TRUE;
}

void
gtk_aneditor_destroy_regexps (GtkAnEditorHiliteRegexps *regexps)
{
  regex_free (&regexps->comment_start_regex);
  regex_free (&regexps->comment_end_regex);
  regex_free (&regexps->string_start_regex);
  regex_free (&regexps->string_end_regex);
  regex_free (&regexps->keywords_regex);

  g_free (regexps);
}

static GtkAnEditorHiliteFonts *
new_fonts (void)
{
  GtkAnEditorHiliteFonts *fonts = g_new (GtkAnEditorHiliteFonts, 1);
  fonts->plain = NULL;
  fonts->comment = NULL;
  fonts->string = NULL;
  fonts->keyword = NULL;
  return fonts;
}

static GtkAnEditorHiliteColors *
new_colors (GdkColormap *cmap)
{
  GtkAnEditorHiliteColors *colors = g_new (GtkAnEditorHiliteColors, 1);

  colors->cmap = cmap;

  /* FIXME: these colors should of cause be specified/configured in
   * .gtkrc and in the tool using the editor */
  colors->plain.pixel = 0;
  colors->plain.red = 0;
  colors->plain.green = 0;
  colors->plain.blue = 0;

  colors->comment.pixel = 0;
  colors->comment.red = 65535;
  colors->comment.green = 0;
  colors->comment.blue = 0;

  colors->string.pixel = 0;
  colors->string.red = 65535;
  colors->string.green = 43908;
  colors->string.blue = 16383;

  colors->keywords.pixel = 0;
  colors->keywords.red = 0;
  colors->keywords.green = 0;
  colors->keywords.blue = 65535;

  gdk_color_alloc(cmap, &(colors->plain));
  gdk_color_alloc(cmap, &(colors->comment));
  gdk_color_alloc(cmap, &(colors->string));
  gdk_color_alloc(cmap, &(colors->keywords));

  return colors;
}

/* set_props_on_interval -- updates interval [from,to[ to the text
 * properties associatet with state.  Legal values for state is BASIC,
 * IN_COMMENT, IN_STRING and IN_KEYWORD.  This function is used to
 * wrap around handling of links and hidden text in the interval. */

/* FIXME: there should be special properties for text hiding other
 * text. Also, we only save hidden text in one level. */

/* FIXME: the font and back property is there for future expansion */
static void
set_props_on_interval (GtkAnEditor *editor, guint from, guint to, guint8 state)
{
  GdkFont *font = NULL;
  GdkColor *fore = NULL, *back = NULL;
#if 0
  MarkData *mdata, *old;
  GList *tmp;
  gint f, t;
#endif

  /* setup properties */
  switch (state) {
  case BASIC:
    font = editor->fonts->plain;
    fore = &editor->colors->plain;
    back = NULL;
    break;

  case IN_COMMENT:
    font = editor->fonts->comment;;
    fore = &editor->colors->comment;
    back = NULL;
    break;

  case IN_STRING:
    font = editor->fonts->string;
    fore = &editor->colors->string;
    back = NULL;
    break;

  case IN_KEYWORD:
    font = editor->fonts->keyword;
    fore = &editor->colors->keywords;
    back = NULL;
    break;

  default:
    g_warning ("should not happen!");
  }

#if 0
  tmp = GTK_ANTEXT (editor)->point.property;
  if (!tmp) {
    gtk_antext_set_property (GTK_ANTEXT (editor), from, to,
			   font, fore, back,
			   new_mark_data (state, NULL));
  } else {
    f = from;
    t = from - GTK_ANTEXT (editor)->point.offset;
    while (t < to - 1 && tmp) {
      g_print ("\n f1 = %d, t1 = %d\n", f, t);
      t += gtk_antext_get_property_length ((GtkAnTextProperty*)tmp->data);
      g_print (" f1 = %d, t1b = %d\n", f, t);
      old = gtk_antext_get_property_data ((GtkAnTextProperty*)tmp->data);
      g_print ("old = ");
      if (old) g_print ("{ %#x, %p = \"%s\" }\n", old->state,
			old->hidden, old->hidden ? old->hidden : "");
      else g_print ("NULL \n");
      if (old && (old->state & HIDDEN)) {
	g_print (" f2 = %d, t2 = %d\n", f, t);
	mdata = new_mark_data (state | HIDDEN, g_strdup (old->hidden));
      } else {
	mdata = new_mark_data (state, NULL);
	/* get properties until next hidden */
	if (tmp->next) {
	  old = gtk_antext_get_property_data ((GtkAnTextProperty*)tmp->next->data);
	}
	while ((t < to -1) && !(old && old->state & HIDDEN) && tmp->next) {
	  g_print ("old = ");
	  if (old) g_print ("{ %#x, %p = \"%s\" }\n", old->state,
			    old->hidden, old->hidden ? old->hidden : "");
	  else g_print ("NULL \n");
	  t += gtk_antext_get_property_length ((GtkAnTextProperty*)tmp->next->data);
	  g_print (" f3 = %d, t3 = %d\n", f, t);
	  old = gtk_antext_get_property_data ((GtkAnTextProperty*)tmp->next->data);
	  tmp = tmp->next;
	}
      }
      
      g_print (" fn = %d, tn = %d\n", f, t);
      g_print ("mdata = { %#x, %p = \"%s\" }\n", mdata->state,
	       mdata->hidden, mdata->hidden ? mdata->hidden : "");
      gtk_antext_set_property (GTK_ANTEXT (editor), f, MIN (t,to),
			     font, fore, back, mdata);
      tmp = tmp->next;
      f = t;
    }
  }
#else 
  gtk_antext_set_property (GTK_ANTEXT (editor), from, to, font, fore, back,
			 new_mark_data (state, NULL));
#endif
}

/* hilite_interval -- yep! That's what it does. It gets the state in
 * position 'from', and hightlights according to the regexps forward
 * to 'to'.  More precise, it hilites from the beginning of 'from's
 * textproperty and forward to 'to', and if need be (to get to a
 * consistent state) a little further. 
 */
static void
hilite_interval (GtkAnEditor *editor, guint from, guint to)
{
  MarkData *mdata;
  gint8 state;
  guint pos;
  guint match_from = 0, insert_from = 0;
  Match next_comment, next_string, next_keyword;

  g_return_if_fail (editor != NULL);
  g_return_if_fail (from <= to);
  g_return_if_fail (to <= gtk_antext_get_length (GTK_ANTEXT (editor)));

  pos = gtk_editable_get_position (GTK_EDITABLE (editor));

  gtk_antext_freeze (GTK_ANTEXT (editor));

  /* Go to beginning of state. If we are on offset 1, then the state
   * should really be the previous state.  It's a wonder to me, why it
   * shouldn't be offset 0...but this works, it doesn't with offset 0.
   * It's probably because this is to fix deleting the last char of a
   * state, and this can only be done with backsspace, which puts us
   * in offset 1, or something. */
  gtk_antext_set_point (GTK_ANTEXT (editor), from);
  if (GTK_ANTEXT (editor)->point.offset == 1) {
    GList *prev = GTK_ANTEXT (editor)->point.property->prev;
    if (prev) {
      /* it's a little hacky to get the length of the previous prop. */
      gint len = gtk_antext_get_property_length (GTK_ANTEXT (editor),
					       from - 2);
      /* goto previous frame...the 1 is for the offset==1 thingy */
      from -= len + 1;
      if ((gint)from < 0) from = 0;	/* not before beginning of text */
    } else {
      from = 0;
    }
  } else {
    from -= GTK_ANTEXT (editor)->point.offset; /* go to beginning of property. */
  }
  /* and correct point */
  gtk_antext_set_point (GTK_ANTEXT (editor), from);

  mdata = gtk_antext_get_property_data (GTK_ANTEXT (editor),
				      gtk_antext_get_point (GTK_ANTEXT (editor)));
  if (!mdata) {			/* happens first time only */
    /* assume basic */
    mdata = new_mark_data (BASIC, NULL);
    gtk_antext_set_property_data (GTK_ANTEXT (editor),
				gtk_antext_get_point (GTK_ANTEXT (editor)), mdata);
  }
  state = mdata->state;

#ifdef EDITOR_DEBUG
  g_print ("hilite_interval [%u,%u](%#x)\n", from, to, state);
#endif

  /* fix up start conditions */
  if (state & BASIC) {
    /* nothing to do really */
    insert_from = match_from = from;

  } else if (state & IN_COMMENT) {
    /* make sure it's still a comment, and if it is, set match_from to
     * end of 'comment_start' match. */
    match_from = regex_match (GTK_ANTEXT (editor), from,
			      &editor->regexps->comment_start_regex);
    if ((gint)match_from == -1) {
      /* no longer a comment */
      state = BASIC;
      insert_from = match_from = from;
    } else {
      /* a comment, match from end of start_comment match, which is at
       * from + match_from */
      insert_from = from;
      match_from += from;
    }

  } else if (state & IN_STRING) {
    /* make sure it's still a string, and if it is, set match_from to
     * end of 'string_start' match. */
    match_from = regex_match (GTK_ANTEXT (editor), from,
			      &editor->regexps->string_start_regex);
    if ((gint)match_from == -1) {
      /* no longer a string */
      state = BASIC;
      insert_from = match_from = from;
    } else {
      /* a string, match from end of start_string match, which is at
       * from + match_from */
      insert_from = from;
      match_from += from;
    }

  } else if (state & IN_KEYWORD) {
    /* Make sure it's still a keyword, and if it is jump to end of the
     * keyword. */
    match_from = regex_match (GTK_ANTEXT (editor), from,
			      &editor->regexps->keywords_regex);
    if ((gint)match_from == -1) {
      /* no longer keyword */
      state = BASIC;
      insert_from = match_from = from;
    } else {
      /* it was a keyword, jump forward to end of property */
      state = BASIC;		/* we jump to the next state, which must be BASIC */
      insert_from = from + match_from;
      match_from = insert_from;
    }

  } else {
    g_warning ("can't happen!");
  }

#ifdef EDITOR_DEBUG
  g_print ("hilite_interval corrected [%u,%u](%#x)\n", from, to, state);
#endif

#ifdef EDITOR_DEBUG
  g_print ("insert: ");
#endif    

  /* scan text and insert hilite'ed text */
  while (insert_from < to) {
    if (state & BASIC) {
      /* FIXME: This can be optimised (I think) by combining these
       * regexps into a single regexp, searching for that, and then
       * find out which regex matched, with regex_match (). */
      regex_search (GTK_ANTEXT (editor), match_from,
		    &editor->regexps->comment_start_regex,
		    TRUE, &next_comment);
      regex_search (GTK_ANTEXT (editor), match_from,
		    &editor->regexps->string_start_regex,
		    TRUE, &next_string);
      regex_search (GTK_ANTEXT (editor), match_from,
		    &editor->regexps->keywords_regex,
		    TRUE, &next_keyword);
      if (MATCH (next_comment.from, next_string.from, next_keyword.from)) {
	if (FIRST_LEAST (next_comment.from, next_string.from,
			 next_keyword.from)) { /* found start of comment */

#ifdef EDITOR_DEBUG
	  g_print ("(b->c)[%u,%u] ",  insert_from, next_comment.from);
#endif

	  set_props_on_interval (editor, insert_from, next_comment.from, BASIC);
	  insert_from = next_comment.from;
	  match_from = next_comment.to;
	  state = IN_COMMENT;

	} else if (FIRST_LEAST (next_string.from, next_comment.from,
				next_keyword.from)) { /* found start of string */

#ifdef EDITOR_DEBUG
	  g_print ("(b->s)[%u,%u] ", insert_from, next_string.from);
#endif
	  
	  set_props_on_interval (editor, insert_from, next_string.from, BASIC);
	  insert_from = next_string.from;
	  match_from = next_string.to;
	  state = IN_STRING;

	} else {		/* found keyword */

#ifdef EDITOR_DEBUG
	  g_print ("(b->k)[%u,%u] ", insert_from, next_keyword.from);
#endif

	  /* set plain */
	  set_props_on_interval (editor, insert_from, next_keyword.from, BASIC);

#ifdef EDITOR_DEBUG
	  g_print ("(k)[%u,%u] ", next_keyword.from, next_keyword.to);
#endif

	  /* then keyword */
	  set_props_on_interval (editor, next_keyword.from, next_keyword.to, IN_KEYWORD);
	  match_from = insert_from = next_keyword.to;
	}

      } else {
	/* no match */

	/* Since we match in entire widget, no match means *no*
	 * match...so we must insert this state in the rest of the
	 * buffer. */
	to = gtk_antext_get_length (GTK_ANTEXT (editor));

#ifdef EDITOR_DEBUG
	g_print ("(rb)[%u,%u] ", insert_from, to);
#endif

	set_props_on_interval (editor, insert_from, to, BASIC);
	insert_from = to;	/* we're done */
      }

    } else if (state & IN_COMMENT) {
      if (regex_search (GTK_ANTEXT (editor), match_from,
			&editor->regexps->comment_end_regex,
			TRUE, &next_comment) > -1) {

#ifdef EDITOR_DEBUG
	g_print ("(c)[%u,%u] ", insert_from, next_comment.to);
#endif

	set_props_on_interval (editor, insert_from, next_comment.to, IN_COMMENT);
	match_from = insert_from = next_comment.to;
	state = BASIC;

      } else {
	/* no match */

	/* Since we match in entire widget, no match means *no*
	 * match...so we must insert this state in the rest of the
	 * buffer. */
	to = gtk_antext_get_length (GTK_ANTEXT (editor));

#ifdef EDITOR_DEBUG
	g_print ("(rc)[%u,%u] ", insert_from, to);
#endif

	set_props_on_interval (editor, insert_from, to, IN_COMMENT);
	insert_from = to;	/* we're done */
      }

    } else if (state & IN_STRING) {
      if (regex_search (GTK_ANTEXT (editor), match_from,
			&editor->regexps->string_end_regex,
			TRUE, &next_string) > -1) {

#ifdef EDITOR_DEBUG
	g_print ("(s)[%u,%u] ", insert_from, next_string.to);
#endif

	set_props_on_interval (editor, insert_from, next_string.to, IN_STRING);
	match_from = insert_from = next_string.to;
	state = BASIC;

      } else {
	/* no match */

	/* Since we match in entire widget, no match means *no*
	 * match...so we must insert this state in the rest of the
	 * buffer. */
	to = gtk_antext_get_length (GTK_ANTEXT (editor));

#ifdef EDITOR_DEBUG
	g_print ("(rs)[%u,%u] ", insert_from, to);
#endif

	set_props_on_interval (editor, insert_from, to, IN_STRING);
	insert_from = to;	/* we're done */

      }

    } else if (state & IN_KEYWORD) {
      /* Can't happen! We can start hiliting in a keyword, (we *have*
       * taken care of that), but we cannot enter a keyword
       * during. (See match in BASIC state). */
      g_warning ("can't happen!");
    } else {
      g_warning ("can't happen!");
    }
  }

#ifdef EDITOR_DEBUG
  g_print ("\n");
#endif

/*    gtk_editable_set_position (GTK_EDITABLE (editor), pos); */
  gtk_antext_thaw (GTK_ANTEXT (editor));
}

void
gtk_aneditor_hilite_buffer (GtkAnEditor *editor)
{
  hilite_interval (editor, 0,
		   GTK_ANTEXT (editor)->text_end - GTK_ANTEXT (editor)->gap_size);
  /* and just a little extra service */
  editor->changed_from = editor->changed_to = GTK_ANTEXT (editor)->point.index;
}

void
gtk_aneditor_flush_properties(GtkAnEditor *editor)
{
  MarkData *mdata, *nmdata;
  gint8 state;
  guint len, from, to;
  
  g_return_if_fail (editor != NULL);
  len = gtk_antext_get_length (GTK_ANTEXT (editor));
  g_return_if_fail (len > 0);

  gtk_antext_freeze (GTK_ANTEXT (editor));
  from = to = 0;

  do {
     mdata = gtk_antext_get_property_data (GTK_ANTEXT (editor),
				      from);
     /* happens first time only */
     if (!mdata) {
        /* assume basic */
        mdata = new_mark_data (BASIC, NULL);
        gtk_antext_set_property_data (GTK_ANTEXT (editor),
           from, mdata);
     }
     to = from + gtk_antext_get_property_length (GTK_ANTEXT (editor),
					       from);
     if(to >= len) to = len-1;
     if(from >= to) break;

     nmdata = clone_prop_data(mdata);
     state = mdata->state;
     if (state & BASIC) {
       gtk_antext_set_property    (GTK_ANTEXT(editor),
	 from, to, editor->fonts->plain,
         &editor->colors->plain, NULL,
          nmdata);
     } else if (state & IN_COMMENT) {
       gtk_antext_set_property    (GTK_ANTEXT(editor),
	 from, to, editor->fonts->comment,
         &editor->colors->comment, NULL,
          nmdata);
     } else if (state & IN_STRING) {
       gtk_antext_set_property    (GTK_ANTEXT(editor),
	 from, to, editor->fonts->string,
         &editor->colors->string, NULL,
          nmdata);
     } else if (state & IN_KEYWORD) {
       gtk_antext_set_property    (GTK_ANTEXT(editor),
	 from, to, editor->fonts->keyword,
         &editor->colors->keywords, NULL,
          nmdata);
     } else {
        g_warning ("can not happen!");
     }
     from = to;
  } while (from < len);
  gtk_antext_thaw (GTK_ANTEXT (editor));
}

static gboolean
hilite_when_idle (GtkAnEditor *editor)
{
#ifdef EDITOR_DEBUG
  g_print ("hilite_when_idle [%u,%u]\n", 
	   editor->changed_from, editor->changed_to);
#endif

  hilite_interval (editor, editor->changed_from, editor->changed_to);
  editor->changed_from = editor->changed_to = GTK_ANTEXT (editor)->point.index;
  editor->hilite_time = 0;

  return FALSE;
}

/* --<indent stuff>------------------------------------------------------- */
/* FIXME: we should have some better api */

/* FIXME: crashes if called on the last line, if the begining of that
 * line contains a '\t'...maybe at gtktext bug... */
static void
default_one_line_ind (GtkAnEditor *editor)
{
  guint pos = gtk_editable_get_position (GTK_EDITABLE (editor));

  gtk_antext_freeze (GTK_ANTEXT (editor));
  /* FIXME: change this when they change the interface to *_move_* */
  gtk_signal_emit_by_name (GTK_OBJECT (editor), "move_to_column", 0);
  gtk_antext_set_point (GTK_ANTEXT (editor),
		      gtk_editable_get_position (GTK_EDITABLE (editor)));
  gtk_antext_insert (GTK_ANTEXT (editor), NULL, NULL, NULL, "\t", 1);
  gtk_editable_set_position (GTK_EDITABLE (editor), pos + 1); /* old pos + tab */
  gtk_antext_thaw (GTK_ANTEXT (editor));
}

/* indent-region -- calls the one line indenter repeatedly over the
 * selection.  This function should *only* be called if there *is* a
 * oneline indenter installed! */
static void
indent_region (GtkAnEditor *editor)
{
  guint from = MIN (GTK_EDITABLE (editor)->selection_start_pos,
		    GTK_EDITABLE (editor)->selection_end_pos);
  guint to = MAX (GTK_EDITABLE (editor)->selection_start_pos,
		  GTK_EDITABLE (editor)->selection_end_pos);
  guint old_pos = gtk_editable_get_position (GTK_EDITABLE (editor));
  guint pos = from;

  gtk_antext_freeze (GTK_ANTEXT (editor));
  gtk_editable_set_position (GTK_EDITABLE (editor), pos);
  while (pos < to + 1) {	/* I *mean* +1 */
    editor->one_line_ind (editor);
    to += 1;			/* the tab remember! */

    /* FIXME: change this when they change the interface to *_move_* */
    gtk_signal_emit_by_name (GTK_OBJECT (editor), "move_cursor", 0, 1);
    pos = gtk_editable_get_position (GTK_EDITABLE (editor));
    gtk_antext_set_point (GTK_ANTEXT (editor), pos);
  }
  gtk_editable_set_position (GTK_EDITABLE (editor), old_pos);
  gtk_antext_thaw (GTK_ANTEXT (editor));
}

/* --<event handles>------------------------------------------------------ */
static gint
gtk_aneditor_key_press (GtkWidget *widget, GdkEventKey *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_ANEDITOR (widget), FALSE);

  if (event->keyval == GDK_Tab) {
    if (GTK_ANEDITOR (widget)->one_line_ind) {
      gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");

      if (GTK_EDITABLE (widget)->has_selection) {
	indent_region (GTK_ANEDITOR (widget));
      } else {
	GTK_ANEDITOR (widget)->one_line_ind (GTK_ANEDITOR (widget));
      }
      return TRUE;
    }
  }

  return FALSE;
}

/* To keep track of changes for hilite_when_idle */
static void
gtk_aneditor_changed_pre (GtkAnEditor *editor)
{
  if (editor->hilite_when_idle) {
    /* the first MIN looks silly, but think about deleting the last
     * char in a text, and you will understand. */
    editor->changed_from = MIN (MIN (editor->changed_from,
				     GTK_ANTEXT (editor)->point.index),
				gtk_antext_get_length (GTK_ANTEXT (editor)) - 1);
  }
}

/* Keeps track of changes for hilite_when_idle, and installs hilite
 * function with a time delay. */
static void
gtk_aneditor_changed_post (GtkAnEditor *editor)
{
  if (editor->hilite_when_idle) {
    editor->changed_to = MIN (MAX (editor->changed_to,
				   GTK_ANTEXT (editor)->point.index),
			      gtk_antext_get_length (GTK_ANTEXT (editor)));

    /* something changed...re-highlight when idle */
    if (editor->hilite_time) gtk_timeout_remove (editor->hilite_time);
    editor->hilite_time = gtk_timeout_add (TIME_DELAY, (GtkFunction) hilite_when_idle,
					   (gpointer) editor);
  }
}
