/*
    executer.c
    Copyright (C) 2000  Kh. Naba Kumar Singh

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <gnome.h>

#include "anjuta.h"
#include "text_editor.h"
#include "messagebox.h"
#include "utilities.h"
#include "executer.h"
#include "support.h"


Executer *
executer_new (void)
{
  Executer *e = malloc (sizeof (Executer));
  if (e)
  {
    e->args_history = NULL;
    e->args = NULL;
    e->terminal = TRUE;
  }
  return e;
}

void
executer_destroy (Executer * e)
{
  gint i;
  if (e)
  {
    for (i = 0; i < g_list_length (e->args_history); i++)
      g_free (g_list_nth_data (e->args_history, i));
    g_list_free (e->args_history);
    if (e->args)
      g_free (e->args);
    g_free (e);
  }
}

void
executer_show (Executer * e)
{
  gtk_widget_show (create_executer_dialog (e));
}

GtkWidget *
create_executer_dialog (Executer * e)
{
  GtkWidget *dialog1;
  GtkWidget *dialog_vbox1;
  GtkWidget *vbox1;
  GtkWidget *frame1;
  GtkWidget *combo1;
  GtkWidget *combo_entry1;
  GtkWidget *checkbutton1;
  GtkWidget *dialog_action_area1;
  GtkWidget *button1;
  GtkWidget *button3;

  dialog1 = gnome_dialog_new (_("Execute"), NULL);
  gtk_window_set_position (GTK_WINDOW (dialog1), GTK_WIN_POS_CENTER);
  gtk_window_set_policy (GTK_WINDOW (dialog1), FALSE, FALSE, FALSE);
  gnome_dialog_set_close (GNOME_DIALOG (dialog1), TRUE);

  dialog_vbox1 = GNOME_DIALOG (dialog1)->vbox;
  gtk_widget_show (dialog_vbox1);

  vbox1 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox1);
  gtk_box_pack_start (GTK_BOX (dialog_vbox1), vbox1, TRUE, TRUE, 0);

  frame1 = gtk_frame_new (_("Command arguments (if any)"));
  gtk_widget_show (frame1);
  gtk_box_pack_start (GTK_BOX (vbox1), frame1, TRUE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (frame1), 5);

  combo1 = gtk_combo_new ();
  gtk_widget_show (combo1);
  gtk_container_add (GTK_CONTAINER (frame1), combo1);
  gtk_container_set_border_width (GTK_CONTAINER (combo1), 5);

  combo_entry1 = GTK_COMBO (combo1)->entry;
  entry_set_text_n_select (combo_entry1, e->args, TRUE);
  gtk_widget_show (combo_entry1);

  checkbutton1 = gtk_check_button_new_with_label (_("Run in Terminal"));
  gtk_widget_show (checkbutton1);
  gtk_box_pack_start (GTK_BOX (vbox1), checkbutton1, FALSE, FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (checkbutton1), 5);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (checkbutton1), TRUE);

  dialog_action_area1 = GNOME_DIALOG (dialog1)->action_area;
  gtk_widget_show (dialog_action_area1);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area1),
			     GTK_BUTTONBOX_EDGE);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (dialog_action_area1), 8);

  gnome_dialog_append_button (GNOME_DIALOG (dialog1), GNOME_STOCK_BUTTON_OK);
  button1 = g_list_last (GNOME_DIALOG (dialog1)->buttons)->data;
  gtk_widget_show (button1);
  GTK_WIDGET_SET_FLAGS (button1, GTK_CAN_DEFAULT);

  gnome_dialog_append_button (GNOME_DIALOG (dialog1),
			      GNOME_STOCK_BUTTON_CANCEL);
  button3 = g_list_last (GNOME_DIALOG (dialog1)->buttons)->data;
  gtk_widget_show (button3);
  GTK_WIDGET_SET_FLAGS (button3, GTK_CAN_DEFAULT);

  gtk_accel_group_attach (app->accel_group, GTK_OBJECT (dialog1));

  gtk_signal_connect (GTK_OBJECT (combo1), "realize",
		      GTK_SIGNAL_FUNC (on_executer_combo_realize), e);
  gtk_signal_connect (GTK_OBJECT (combo_entry1), "changed",
		      GTK_SIGNAL_FUNC (on_executer_entry_changed), e);
  gtk_signal_connect (GTK_OBJECT (checkbutton1), "toggled",
		      GTK_SIGNAL_FUNC (on_executer_checkbutton_toggled), e);
  gtk_signal_connect (GTK_OBJECT (checkbutton1), "realize",
		      GTK_SIGNAL_FUNC (on_executer_checkbutton_realize), e);

  gtk_widget_grab_focus (button1);
  gtk_widget_grab_default (button1);
  return dialog1;
}

void
on_executer_combo_realize (GtkWidget * widget, gpointer user_data)
{
  Executer *e = user_data;
  if (e->args)
    gtk_entry_set_text (GTK_ENTRY (widget), e->args);
}

void
on_executer_entry_changed (GtkEditable * editable, gpointer user_data)
{
  Executer *e = user_data;
  if (e->args)
    g_free (e->args);
  e->args = g_strdup (gtk_entry_get_text (GTK_ENTRY (editable)));
}


void
on_executer_checkbutton_toggled (GtkToggleButton * togglebutton,
				 gpointer user_data)
{
  Executer *e = user_data;
  e->terminal = gtk_toggle_button_get_active (togglebutton);
}


void
on_executer_checkbutton_realize (GtkWidget * widget, gpointer user_data)
{
  Executer *e = user_data;
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), e->terminal);
}

void
executer_execute (Executer * e)
{
  gchar *prog, *temp;
  gint s_re, e_re;
  struct stat s_stat, e_stat;
  TextEditor *te;
  pid_t pid;

  if (app->project_dbase->project_is_open == TRUE)
  {
    if (e->args)
    {
      prog =
	g_strdup_printf ("%s/%s %s", app->project_dbase->src_dir,
			 app->project_dbase->prog_name, e->args);
    }
    else
    {
      prog =
	g_strdup_printf ("%s/%s", app->project_dbase->src_dir,
			 app->project_dbase->prog_name);
    }
    chdir (app->project_dbase->src_dir);
    if (e->terminal)
    {
      if (anjuta_is_installed (app->preferences->commands.terminal, TRUE))
      {
	if (anjuta_is_installed ("anjuta_launcher", TRUE))
	{
	  gchar *command = g_strconcat ("anjuta_launcher ", prog, NULL);
	  if ((pid = fork ()) == 0)
	  {
	    execlp (app->preferences->commands.terminal,
		    app->preferences->commands.terminal, "-e", command, NULL);
	    g_error (_("Cannot execute gnome-terminal"));
	  }
	  g_free (command);
	  if (pid < 1)
	    anjuta_error (_("Cannot execute gnome-terminal"));
	  else
	    anjuta_register_child_process (pid, NULL, NULL);
	}
      }
    }
    else
    {
      if (anjuta_is_installed (app->preferences->commands.shell, TRUE))
      {

	if ((pid = fork ()) == 0)
	{
	  execlp (app->preferences->commands.shell,
		  app->preferences->commands.shell, "-c", prog, NULL);
	  g_error (_("Cannot execute command shell"));
	}
	if (pid < 0)
	  anjuta_error (_("Error: Cannot run the program"));
	else
	  anjuta_register_child_process (pid, NULL, NULL);
      }
    }
    if (prog)
      g_free (prog);
    return;
  }
  else
  {
    te = anjuta_get_current_text_editor ();
    if (te == NULL)
      return;
    if (te->full_filename)
    {
      prog = g_strdup (te->full_filename);
      temp = get_file_extension (prog);
      if (temp)
	*(--temp) = '\0';
      else
	temp = prog;
      s_re = stat (te->full_filename, &s_stat);
      e_re = stat (prog, &e_stat);
    }
    else
    {
      anjuta_warning (_
		      ("This file is not saved in proper format. Save it first"));
      return;
    }
    if ((te->status == TEXT_EDITOR_NOT_SAVED)
	|| (e_re != 0) || (s_re != 0) || (e_stat.st_mtime < s_stat.st_mtime))
    {
      anjuta_warning (_("The executable is not up to date. Build it first."));
      g_free (prog);
      return;
    }
    if (e->args)
    {
      temp = g_strdup_printf ("%s %s", prog, e->args);
    }
    else
    {
      temp = g_strdup (prog);
    }
    g_free (prog);
    prog = g_dirname (te->full_filename);
    chdir (prog);
    g_free (prog);
    if (e->terminal)
    {
      if (anjuta_is_installed (app->preferences->commands.terminal, TRUE))
      {
	if (anjuta_is_installed ("anjuta_launcher", TRUE))
	{
	  gchar *command = g_strconcat ("anjuta_launcher ", temp, NULL);
	  if ((pid = fork ()) == 0)
	  {
	    execlp (app->preferences->commands.terminal,
		    app->preferences->commands.terminal,
		    "-e", command, temp, NULL);
	    g_error (_("Cannot execute gnome-terminal"));
	  }
	  g_free (command);
	  if (pid < 0)
	    anjuta_error (_("Cannot execute gnome-terminal"));
	  else
	    anjuta_register_child_process (pid, NULL, NULL);
	}
      }
    }
    else
    {
      if (anjuta_is_installed (app->preferences->commands.shell, TRUE))
      {
	if ((pid = fork ()) == 0)
	{
	  execlp (app->preferences->commands.shell,
		  app->preferences->commands.shell, "-c", temp, NULL);
	  g_error (_("Cannot execute command shell"));
	}
	if (pid < 0)
	  anjuta_error (_("Error: Cannot run the program"));
	else
	  anjuta_register_child_process (pid, NULL, NULL);
      }
    }
    g_free (temp);
    return;
  }
}
