/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* vim: set sw=8: */

/*
 * guppi-gnumeric-manger.c:
 *
 * Copyright (C) 2000-2001 Jody Goldberg (jgoldberg@home.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>

#include "guppi-gnumeric-manager.h"
#include "guppi-gnumeric-vector.h"
#include "guppi-gnumeric-bonobo-view.h"
#include "guppi-gnumeric-interpreter.h"

#include <guppi-chart-selector.h>
#include <guppi-root-group-item.h>
#include <guppi-seq-scalar.h>
#include <guppi-seq-date.h>
#include <guppi-seq-string.h>
#include <guppi-useful.h>
#include <guppi-marker.h>

#include <gnome-xml/parser.h>
#include <gal/util/e-xml-utils.h>

#define PARENT_TYPE BONOBO_EMBEDDABLE_TYPE

#ifdef DEBUG_GUPPI_GNUMERIC
int debug_guppi_gnumeric = 10;
#else
int debug_guppi_gnumeric = 0;
#endif
#ifndef DISABLE_DEBUG
#define d(level, code)	do { if (debug_guppi_gnumeric >= level) { code; } } while (0)
#else
#define d(level, code)	
#endif

/*************************************************************************/

static void
gup_gnm_manager_clear_arrangement (GupGnmManager *manager)
{
	if (manager->data_ids != NULL) {
		guppi_free (manager->data_ids);
		manager->data_ids = NULL;
		manager->header_ids = NULL;
		manager->arrangement_len = -1;
	}
}

void
gup_gnm_manager_add_vector (GupGnmManager *manager,
			    GupGnmVector *vector, unsigned id)
{
	g_return_if_fail (IS_GUP_GNM_MANAGER(manager));

	gup_gnm_manager_clear_arrangement (manager);

	if (manager->vectors->len <= id)
		g_ptr_array_set_size (manager->vectors, id+1);
	g_ptr_array_index (manager->vectors, id) = vector;
}

GupGnmVector *
gup_gnm_manager_get_vector (GupGnmManager const *manager, unsigned id)
{
	g_return_val_if_fail (IS_GUP_GNM_MANAGER(manager), NULL);
	g_return_val_if_fail (manager->vectors != NULL, NULL);
	g_return_val_if_fail (manager->vectors->len > id, NULL);

	return g_ptr_array_index (manager->vectors, id);
}

static inline GupGnmManager *
gup_gnm_manager_from_servant (PortableServer_Servant servant)
{
	return GUP_GNM_MANAGER (bonobo_object_from_servant (servant));
}

static void
cb_configure_destroy (GtkObject *object, GupGnmManager *manager)
{
	d(0, printf ("Guppi : configure destroy %p\n", object));

	g_return_if_fail (IS_GUP_GNM_MANAGER (manager));
	manager->sample.canvas = NULL;
	manager->sample.root_item = NULL;
	manager->sample.control = NULL;
}

static Bonobo_Control
impl_configure (PortableServer_Servant servant,
		const CORBA_char *type,
		CORBA_Environment *ev)
{
	GupGnmManager *manager = gup_gnm_manager_from_servant (servant);
	BonoboControl	*control;

	d(0, printf ("Guppi : get configure %s\n", type));

	if (!strcmp (type, "Sample")) {
		if (manager->sample.control == CORBA_OBJECT_NIL) {
			GuppiRootGroupView *view = gup_gnm_manager_get_plot (manager);
			manager->sample.canvas = guppi_root_group_view_make_canvas (view,
				&manager->sample.root_item);
			gtk_signal_connect (GTK_OBJECT (manager->sample.canvas),
				"destroy",
				GTK_SIGNAL_FUNC (cb_configure_destroy), manager);
			gtk_widget_show_all (GTK_WIDGET (manager->sample.canvas));
			manager->sample.control =
				bonobo_control_new (GTK_WIDGET (manager->sample.canvas));
		}
		control = manager->sample.control;

	} else if (!strcmp (type, "Type")) {
		if (manager->chart_type_selector.control == CORBA_OBJECT_NIL) {
			GtkWidget *selector = guppi_chart_selector_widget (manager);
			gtk_widget_show_all (selector); /* Very important ! */
			manager->chart_type_selector.control =
				bonobo_control_new (selector);
		}
		control = manager->chart_type_selector.control;
	}

	return CORBA_Object_duplicate (BONOBO_OBJREF (control), ev);
}

void
gup_gnm_manager_set_plottype (GupGnmManager *manager, xmlNode *plotinfo)
{
	xmlNs	*ns;
	xmlNode *plot;

	g_return_if_fail (IS_GUP_GNM_MANAGER (manager));
	g_return_if_fail (plotinfo != NULL);

	if (manager->doc == NULL) {
		xmlNode *legend;

		manager->doc = xmlNewDoc ("1.0");
		manager->doc->xmlRootNode =
			xmlNewDocNode (manager->doc, NULL, "Graph", NULL);
		ns = xmlNewNs (manager->doc->xmlRootNode,
			"http://www.gnumeric.org/graph_v1", "graph");
		xmlSetNs (manager->doc->xmlRootNode, ns);

		legend = xmlNewChild (manager->doc->xmlRootNode, ns,
			"Legend", NULL);
		xmlNewChild (legend, ns, "Position", "east");
	} else {
		xmlNode	*p = e_xml_get_child_by_name (manager->doc->xmlRootNode,
						      "Plots");
		if (p != NULL) {
			xmlUnlinkNode (p);
			xmlFreeNode (p);
		}
		ns = manager->doc->xmlRootNode->ns;
	}
	plot = xmlNewChild (
		xmlNewChild (manager->doc->xmlRootNode, ns, "Plots", NULL),
		ns, "Plot", NULL);

	xmlAddChild (plot,
		xmlCopyNode (e_xml_get_child_by_name (plotinfo, "Type"), TRUE));
	xmlAddChild (plot,
		xmlCopyNode (e_xml_get_child_by_name (plotinfo, "Attributes"), TRUE));

	gup_gnm_manager_generate_series (manager);
	xmlDocDump (stdout, manager->doc);
}

static GUPPI_GNUMERIC_MANAGER(Buffer) *
impl_get_spec (PortableServer_Servant servant, CORBA_Environment *ev)
{
	GupGnmManager *manager = gup_gnm_manager_from_servant (servant);
	GUPPI_GNUMERIC_MANAGER(Buffer)  *spec;
	xmlChar *mem;
	int size;

	xmlDocDumpMemory (manager->doc, &mem, &size);

	spec = GUPPI_GNUMERIC_MANAGER(Buffer__alloc) ();
	spec->_length = spec->_maximum = size;
	spec->_buffer = mem;
	spec->_release = CORBA_FALSE;

	return spec;
}

static void
impl_set_spec (PortableServer_Servant servant,
	       GUPPI_GNUMERIC_MANAGER(Buffer) const *spec,
	       CORBA_Environment * ev)
{
	GupGnmManager *manager = gup_gnm_manager_from_servant (servant);
	xmlParserCtxtPtr pctxt;

	/* A limit in libxml */
	g_return_if_fail (spec->_length >= 4);

	pctxt = xmlCreatePushParserCtxt (NULL, NULL,
		spec->_buffer, spec->_length, NULL);
	xmlParseChunk (pctxt, "", 0, TRUE);

	if (manager->doc != NULL)
		xmlFreeDoc (manager->doc);
	manager->doc = pctxt->myDoc;

	d (1, {
		   puts ("Guppi : import spec");
		   xmlDocDump (stdout, manager->doc);
	});

	gup_gnm_manager_markup_spec (manager);
	xmlFreeParserCtxt (pctxt);
}

static GNOME_Gnumeric_Scalar_Vector
impl_addVector (PortableServer_Servant servant,
		const GNOME_Gnumeric_VectorSelection subscriber,
		const GNOME_Gnumeric_VectorType type,
		const GNOME_Gnumeric_VectorID id,
		CORBA_Environment *ev)
{
	GupGnmManager *manager = gup_gnm_manager_from_servant (servant);
	GupGnmVector *vector;

	d(5, printf ("addVector manager == %p, type = %d, id = %d\n", manager, type, id));

	vector = gup_gnm_vector_new (manager,
		subscriber, type, id);
	return gup_gnm_vector_servant_get (vector);
}

static void
impl_clearVectors (PortableServer_Servant servant,
		   CORBA_Environment * ev)
{
	GupGnmManager *manager = gup_gnm_manager_from_servant (servant);

	d(5, printf ("clearVectors manager == %p\n", manager));
}

static void
impl_arrangeVectors (PortableServer_Servant servant,
		     GNOME_Gnumeric_VectorIDs const * data,
		     GNOME_Gnumeric_VectorIDs const * optional_headers,
		     CORBA_Environment * ev)
{
	GupGnmManager *manager = gup_gnm_manager_from_servant (servant);
	unsigned i;

	d(5, printf ("arrangeVectors manager == %p\n", manager));

	gup_gnm_manager_clear_arrangement (manager);

	manager->arrangement_len = data->_length;
	manager->data_ids = guppi_new (int, manager->arrangement_len *2);
	manager->header_ids = manager->data_ids + manager->arrangement_len;

	for (i = 0 ; i < manager->arrangement_len ; i++) {
		manager->data_ids[i] = data->_buffer[i];
		manager->header_ids[i] = optional_headers->_buffer[i];
	}

	/* if no type has been selected yet, just store the data,
	 * we'll generate later.
	 */
	if (manager->doc != NULL)
		gup_gnm_manager_generate_series (manager);
}

static void
impl_seriesSetDimension (PortableServer_Servant servant,
			 const GNOME_Gnumeric_SeriesID seriesID,
			 const CORBA_char *dim,
			 const GNOME_Gnumeric_VectorID vectorID,
			 CORBA_Environment *ev)
{
	GupGnmManager *manager = gup_gnm_manager_from_servant (servant);

	d(5, printf ("seriesSetDimension manager == %p\n", manager));
}

static void
impl_seriesDelete (PortableServer_Servant servant,
		   const GNOME_Gnumeric_SeriesID seriesID,
		   CORBA_Environment * ev)
{
	GupGnmManager *manager = gup_gnm_manager_from_servant (servant);

	d(5, printf ("seriesDelete manager == %p\n", manager));
}

static GtkObjectClass *gup_gnm_manager_parent_class = NULL;

void
gup_gnm_manager_add_view (GupGnmManager *manager, GupGnmView *view)
{

}

void
gup_gnm_manager_remove_view (GupGnmManager *manager, GupGnmView *view)
{
}

/**
 * gup_gnm_manager_add_wrapper :
 * @manager :
 * @wrapper :
 *
 * Adds @wrapper to the collection of sequences whose lifecyle is controllled
 * by the @manager.
 */
void
gup_gnm_manager_add_wrapper (GupGnmManager *manager,  GuppiData *wrapper)
{
	if (manager->wrapper_sequences == NULL)
		manager->wrapper_sequences = g_ptr_array_new ();
	g_ptr_array_add	(manager->wrapper_sequences, wrapper);
}

/**
 * gup_gnm_manager_clear_wrappers :
 * @manager :
 *
 * Clear all the wrappers in the @manager.
 */
void
gup_gnm_manager_clear_wrappers (GupGnmManager *manager)
{
	if (manager->vectors != NULL) {
		int i;
		
		i = manager->vectors->len;
		while (i-- > 0) {
			GupGnmVector *vector = g_ptr_array_index (manager->vectors, i);
			if (vector != NULL)
				gup_gnm_vector_clear_names (vector);
		}
	}
	if (manager->wrapper_sequences != NULL) {
		int i;
		
		i = manager->wrapper_sequences->len;
		while (i-- > 0) {
			GupGnmVector *vector = g_ptr_array_index (manager->wrapper_sequences, i);
			if (vector != NULL)
				guppi_unref (GTK_OBJECT (vector));
		}
		g_ptr_array_free (manager->wrapper_sequences, TRUE);
		manager->wrapper_sequences = NULL;
	}
}

static void
gup_gnm_manager_destroy (GtkObject *obj)
{
	GupGnmManager *manager = GUP_GNM_MANAGER (obj);

	d(2, puts ("GUPPI : GupGnmManager destroyed"));

	gup_gnm_manager_clear_wrappers (manager);
	if (manager->series.names != NULL)
		guppi_unref0 (manager->series.names);
	if (manager->series.colours != NULL)
		guppi_unref0 (manager->series.colours);
	if (manager->series.markers != NULL)
		g_array_free (manager->series.markers, TRUE);

	if (manager->vectors != NULL) {
		int i;
		
		i = manager->vectors->len;
		while (i-- > 0) {
			GupGnmVector *vector = g_ptr_array_index (manager->vectors, i);
			if (vector != NULL)
				guppi_unref (GTK_OBJECT (vector));
		}
		g_ptr_array_free (manager->vectors, TRUE);
		manager->vectors = NULL;
	}

	if (manager->doc != NULL) {
		xmlFreeDoc (manager->doc);
		manager->doc = NULL;
	}

	if (gup_gnm_manager_parent_class->destroy)
		gup_gnm_manager_parent_class->destroy (obj);
}

static void
gup_gnm_manager_class_init (GupGnmManagerClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	POA_GUPPI_GNUMERIC_MANAGER(epv) *epv = &klass->epv;

	object_class->destroy = gup_gnm_manager_destroy;

	gup_gnm_manager_parent_class = gtk_type_class (PARENT_TYPE);

	epv->configure		= impl_configure;
	epv->_get_spec		= impl_get_spec;
	epv->_set_spec		= impl_set_spec;
	epv->addVector		= impl_addVector;
	epv->clearVectors	= impl_clearVectors;
	epv->arrangeVectors	= impl_arrangeVectors;
	epv->seriesSetDimension	= impl_seriesSetDimension;
	epv->seriesDelete	= impl_seriesDelete;
}

static void
gup_gnm_manager_init (GtkObject *object)
{
	GupGnmManager *manager = GUP_GNM_MANAGER (object);
	manager->plot = NULL;
	manager->views = NULL;
	manager->chart_type_selector.widget = NULL;
	manager->chart_type_selector.control = CORBA_OBJECT_NIL;
	manager->sample.canvas = NULL;
	manager->sample.root_item = NULL;
	manager->sample.control = CORBA_OBJECT_NIL;
	manager->vectors = g_ptr_array_new ();
	manager->wrapper_sequences = NULL;
	manager->series.names = guppi_seq_string_new ();
	manager->series.colours = guppi_color_palette_new ();
	manager->series.markers = g_array_new (FALSE, TRUE, sizeof (GuppiMarker));
	manager->doc = NULL;
	manager->data_ids = NULL;
	manager->header_ids = NULL;
	manager->arrangement_len = -1;
}

GtkType
gup_gnm_manager_get_type (void)
{
	static GtkType type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"GupGnmManager",
			sizeof (GupGnmManager),
			sizeof (GupGnmManagerClass),
			(GtkClassInitFunc) gup_gnm_manager_class_init,
			(GtkObjectInitFunc) gup_gnm_manager_init,
			NULL, /* reserved 1 */
			NULL, /* reserved 2 */
			(GtkClassInitFunc) NULL
		};

		type = bonobo_x_type_unique (
			PARENT_TYPE,
			POA_GUPPI_GNUMERIC_MANAGER(init), NULL,
			GTK_STRUCT_OFFSET (GupGnmManagerClass, epv),
			&info);
	}

	return type;
}

static void
gup_gnm_manager_print (GnomePrintContext * ctx, double width, double height,
		       Bonobo_PrintScissor const *scissor, gpointer closure)
{
	GuppiRootGroupView *plot =
		gup_gnm_manager_get_plot (GUP_GNM_MANAGER (closure));
	guppi_element_view_print_to_bbox (GUPPI_ELEMENT_VIEW (plot),
		ctx, 0, 0, width, height);
}

GupGnmManager *
gup_gnm_manager_new (void)
{
	GupGnmManager *manager;
	BonoboPrint *print;

	manager = guppi_type_new (gup_gnm_manager_get_type ());
	bonobo_embeddable_construct (BONOBO_EMBEDDABLE (manager),
		gup_gnm_bonobo_view_factory, NULL);

	d(2, printf ("Manager::new() == %p\n", manager));

	/* Register the Bonobo::Print interface */
	print = bonobo_print_new (gup_gnm_manager_print, manager);
	if (!print) {
		bonobo_object_unref (BONOBO_OBJECT (manager));
		return NULL;
	}

	bonobo_object_add_interface (BONOBO_OBJECT (manager),
		BONOBO_OBJECT (print));


	return manager;
}

/**
 * gup_gnm_manager_get_plot :
 * @manager :
 *
 * Get a possibly cached plot.
 */
GuppiRootGroupView *
gup_gnm_manager_get_plot (GupGnmManager *manager)
{
	if (manager->plot == NULL)
		manager->plot = guppi_gnumeric_manager_make_plot (manager);
	return manager->plot;
}
