/* This is -*- C -*- */
/* $Id: guppi-layout.c,v 1.24 2001/05/06 08:26:43 trow Exp $ */

/*
 * guppi-layout.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>

#include <gtk/gtkmain.h>

#include <math.h>
#include <guppi-useful.h>
#include "guppi-layout.h"

static GtkObjectClass *parent_class = NULL;

struct CustomLUInfo {
  double x0, y0, x1, y1;
};

static gboolean
custom_solve_fallback (GuppiMatrix *m, GuppiVector *vec, gint i, gpointer user_data)
{
  struct CustomLUInfo *info = user_data;

  switch (i % 4) {
  case 0: return info->x0;
  case 1: return info->x1;
  case 2: return info->y1;
  case 3: return info->y0;
  }

  return TRUE;
}

static GuppiVector *
custom_solve (GuppiMatrix *m, GuppiVector *vec,
	      double x0, double x1, double y0, double y1)
{
  struct CustomLUInfo info;
  double x, y, w, h;

  g_return_val_if_fail (m != NULL, NULL);
  g_return_val_if_fail (vec != NULL, NULL);

  g_return_val_if_fail (guppi_matrix_is_square (m), NULL);
  
  guppi_2sort (&x0, &x1);
  guppi_2sort (&y0, &y1);

  w = x1 - x0;
  h = y1 - y0;
  x = (x0 + x1) / 2;
  y = (y0 + y1) / 2;

  info.x0 = x - w / 3;
  info.x1 = x + w / 3;
  info.y0 = y - h / 3;
  info.y1 = y + h / 3;

  return guppi_matrix_solve_with_fallback (m, vec, custom_solve_fallback, &info);
}
 
/**************************************************************************/

enum {
  LEFT, RIGHT, TOP, BOTTOM, WIDTH, HEIGHT,
  SCREEN_X0, SCREEN_X1, SCREEN_Y0, SCREEN_Y1, CONSTANT
};

typedef struct _Term Term;

struct _Term {
  gint type;
  double c;
  GuppiGeometry *geom;
};

static Term *
term_new (gint type, double c, GuppiGeometry * geom)
{
  Term *t;

  g_return_val_if_fail (c != 0, NULL);

  t = guppi_new0 (Term, 1);
  t->type = type;
  t->c = c;
  t->geom = geom;
  guppi_ref (geom);

  return t;
}

static Term *
term_copy (Term * t)
{
  return term_new (t->type, t->c, t->geom);
}

static Term *
term_clone_converted (Term * src, GHashTable * conv_table)
{
  GuppiGeometry *dest_geom;

  g_return_val_if_fail (src != NULL, NULL);
  g_return_val_if_fail (conv_table != NULL, NULL);

  if (src->geom) {
    dest_geom = g_hash_table_lookup (conv_table, src->geom);
    g_assert (dest_geom != NULL);
  } else {
    dest_geom = NULL;
  }

  return term_new (src->type, src->c, dest_geom);
}


static void
term_free (Term * t)
{
  if (t) {
    guppi_unref0 (t->geom);
    guppi_free (t);
  }
}

/**************************************************************************/

typedef struct _Rule Rule;

struct _Rule {
  GList *terms;
};

static Rule *
rule_new (void)
{
  return guppi_new0 (Rule, 1);
}

static void
rule_add_term (Rule * r, Term * t)
{
  g_return_if_fail (r != NULL);
  g_return_if_fail (t != NULL);

  r->terms = g_list_append (r->terms, t);
}

static gboolean
rule_contains_geom (Rule * r, GuppiGeometry * geom)
{
  GList *i = r->terms;
  while (i != NULL) {
    Term *t = (Term *) i->data;
    if (t->geom == geom)
      return TRUE;
    i = g_list_next (i);
  }
  return FALSE;
}

static void
rule_replace (Rule * r, GuppiGeometry * old, GuppiGeometry * nuevo)
{
  GList *i = r->terms;
  while (i != NULL) {
    Term *t = (Term *) i->data;
    if (t->geom == old)
      t->geom = nuevo;
    i = g_list_next (i);
  }
}

static Rule *
rule_copy (Rule * r)
{
  Rule *c = rule_new ();
  GList *i = r->terms;
  while (i != NULL) {
    rule_add_term (c, term_copy ((Term *) i->data));
    i = g_list_next (i);
  }
  return c;
}

static Rule *
rule_clone_converted (Rule * src, GHashTable * conv_table)
{
  Rule *dest;
  GList *iter;

  g_return_val_if_fail (src != NULL, NULL);
  g_return_val_if_fail (conv_table != NULL, NULL);

  dest = rule_new ();

  iter = src->terms;
  while (iter != NULL) {
    dest->terms = g_list_append (dest->terms,
				 term_clone_converted ((Term *) iter->data,
						       conv_table));
    iter = g_list_next (iter);
  }

  return dest;
}

static void
rule_free (Rule * r)
{
  if (r) {
    GList *iter = r->terms;
    while (iter) {
      term_free ((Term *) iter->data);
      iter = g_list_next (iter);
    }
    g_list_free (r->terms);
    r->terms = NULL;
    guppi_free (r);
  }
}

static GList *
rule_list_copy (GList * l)
{
  GList *c = NULL;
  while (l != NULL) {
    c = g_list_append (c, rule_copy ((Rule *) l->data));
    l = g_list_next (l);
  }
  return c;
}

static void
rule_list_free (GList * l)
{
  GList *iter = l;
  while (iter != NULL) {
    rule_free ((Rule *) iter->data);
    iter = g_list_next (iter);
  }
  g_list_free (l);
}

/**************************************************************************/

typedef struct _GuppiLayoutPrivate GuppiLayoutPrivate;
struct _GuppiLayoutPrivate {
  GList *geometries;
  GList *rules;

  gint N_geom, N_rules;

  GuppiMatrix *rule_matrix;
  GuppiMatrix *coor_matrix;

  gboolean calculating_layout;
  double x0, y0, x1, y1;

  guint pending_layout;
};

#define priv(x) ((GuppiLayoutPrivate*)((GUPPI_LAYOUT(x))->opaque_internals))

static void
reset_precalculated_layout (GuppiLayout * lay)
{
  GuppiLayoutPrivate *p;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));

  p = priv (lay);

  guppi_matrix_free (p->rule_matrix);
  p->rule_matrix = NULL;

  guppi_matrix_free (p->coor_matrix);
  p->coor_matrix = NULL;
}

static GList *
geom_list_copy (GList * l)
{
  GList *copy = NULL;
  while (l != NULL) {
    copy = g_list_append (copy, GUPPI_GEOMETRY (l->data));
    guppi_ref (l->data);
    l = g_list_next (l);
  }
  return copy;
}

static void
geom_list_free (GList * l)
{
  GList *iter = l;
  while (iter != NULL) {
    guppi_unref (iter->data);
    iter = g_list_next (iter);
  }
  g_list_free (l);
}

static void
guppi_layout_finalize (GtkObject * obj)
{
  GuppiLayout *lay = GUPPI_LAYOUT (obj);
  GuppiLayoutPrivate *p = priv (lay);

  guppi_finalized (obj);

  geom_list_free (p->geometries);
  p->geometries = NULL;

  rule_list_free (p->rules);
  p->rules = NULL;

  reset_precalculated_layout (lay);

  if (p->pending_layout) {
    gtk_idle_remove (p->pending_layout);
    p->pending_layout = 0;
  }

  guppi_free (lay->opaque_internals);
  lay->opaque_internals = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
guppi_layout_class_init (GuppiLayoutClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  parent_class = gtk_type_class (GTK_TYPE_OBJECT);

  object_class->finalize = guppi_layout_finalize;
}

static void
guppi_layout_init (GuppiLayout * obj)
{
  GuppiLayoutPrivate *p;

  obj->opaque_internals = guppi_new0 (GuppiLayoutPrivate, 1);
  p = (GuppiLayoutPrivate *) obj->opaque_internals;

  p->x0 = 0;
  p->x1 = -1;
  p->y0 = 0;
  p->y1 = -1;
}

GtkType guppi_layout_get_type (void)
{
  static GtkType guppi_layout_type = 0;
  if (!guppi_layout_type) {
    static const GtkTypeInfo guppi_layout_info = {
      "GuppiLayout",
      sizeof (GuppiLayout),
      sizeof (GuppiLayoutClass),
      (GtkClassInitFunc) guppi_layout_class_init,
      (GtkObjectInitFunc) guppi_layout_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_layout_type = gtk_type_unique (GTK_TYPE_OBJECT, &guppi_layout_info);
  }
  return guppi_layout_type;
}

GuppiLayout *
guppi_layout_new (void)
{
  return GUPPI_LAYOUT (guppi_type_new (guppi_layout_get_type ()));
}

GuppiLayout *
guppi_layout_copy (GuppiLayout * lay)
{
  GuppiLayout *copy;
  GuppiLayoutPrivate *p;
  GuppiLayoutPrivate *pcopy;

  g_return_val_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay), NULL);

  copy = guppi_layout_new ();

  p = priv (lay);
  pcopy = priv (copy);

  pcopy->geometries = geom_list_copy (p->geometries);
  pcopy->rules = rule_list_copy (p->rules);
  pcopy->N_geom = p->N_geom;
  pcopy->N_rules = p->N_rules;
  pcopy->x0 = p->x0;
  pcopy->y0 = p->y0;
  pcopy->x1 = p->x1;
  pcopy->y1 = p->y1;

  return copy;
}

GuppiLayout *
guppi_layout_clone_converted (GuppiLayout * src,
			      GHashTable * conversion_table)
{
  GuppiLayout *dest;
  GuppiLayoutPrivate *p_src;
  GuppiLayoutPrivate *p_dest;
  GList *iter;
  GuppiGeometry *src_geom;
  GuppiGeometry *dest_geom;
  Rule *src_rule;
  Rule *dest_rule;

  g_return_val_if_fail (src != NULL, NULL);
  g_return_val_if_fail (conversion_table != NULL, NULL);

  dest = guppi_layout_new ();

  p_src = priv (src);
  p_dest = priv (dest);

  iter = p_src->geometries;
  while (iter) {
    src_geom = GUPPI_GEOMETRY (iter->data);
    dest_geom = g_hash_table_lookup (conversion_table, src_geom);
    g_assert (dest_geom != NULL);
    p_dest->geometries = g_list_append (p_dest->geometries, dest_geom);
    guppi_ref (dest_geom);
    iter = g_list_next (iter);
  }

  /* Need to copy rules */
  iter = p_src->rules;
  while (iter) {
    src_rule = (Rule *) iter->data;
    dest_rule = rule_clone_converted (src_rule, conversion_table);
    p_dest->rules = g_list_append (p_dest->rules, dest_rule);
    iter = g_list_next (iter);
  }

  return dest;
}

void
guppi_layout_connect_geometry (GuppiLayout * lay, GuppiGeometry * geom)
{
  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL && GUPPI_IS_GEOMETRY (geom));

  guppi_geometry_connect_to_layout (geom, lay);
}

static gboolean
registered (GuppiLayout * lay, GuppiGeometry * geom)
{
  GuppiLayoutPrivate *p;

  g_return_val_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay), FALSE);
  g_return_val_if_fail (geom != NULL && GUPPI_IS_GEOMETRY (geom), FALSE);

  p = priv (lay);

  if (!guppi_geometry_registered (geom))
    return FALSE;

  return g_list_find (p->geometries, geom) != NULL;
}

static void
register_geom (GuppiLayout * lay, GuppiGeometry * geom)
{
  GuppiLayoutPrivate *p;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL && GUPPI_IS_GEOMETRY (geom));

  p = priv (lay);

  if (registered (lay, geom))
    return;

  guppi_geometry_set_registered (geom, TRUE);

  p->geometries = g_list_append (p->geometries, geom);
  guppi_ref (geom);
  ++p->N_geom;

  reset_precalculated_layout (lay);
}

static gint
code_ext (GuppiLayout * lay, Term * t)
{
  GuppiLayoutPrivate *p;
  gint c = 0;

  g_return_val_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay), -1);
  g_return_val_if_fail (t != NULL, -1);

  p = priv (lay);

  c = g_list_index (p->geometries, t->geom);
  if (t->type == WIDTH || t->type == HEIGHT)
    c = 2 * c + 4 * p->N_geom;
  else
    c *= 4;

  switch (t->type) {
  case LEFT:
  case WIDTH:
    break;
  case RIGHT:
  case HEIGHT:
    c += 1;
    break;
  case TOP:
    c += 2;
    break;
  case BOTTOM:
    c += 3;
    break;
  default:
    g_warning ("Illegal term typecode");
    return -1;
  }

  return c;
}

static void
add_rule (GuppiLayout * lay, Rule * r)
{
  GuppiLayoutPrivate *p;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (r != NULL);
  g_return_if_fail (r->terms != NULL);

  p = priv (lay);

  p->rules = g_list_append (p->rules, r);
  ++p->N_rules;
}

#ifdef EXTRA_UNUSED_FN
static void
remove_rule (GuppiLayout * lay, Rule * r)
{
  GuppiLayoutPrivate *p;
  GList *i;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (r != NULL);

  p = priv (lay);
  i = g_list_find (p->rules, r);
  g_assert (i != NULL);
  p->rules = g_list_remove_link (p->rules, i);
  g_list_free_1 (i);
  --p->N_rules;
}
#endif

static GuppiMatrix *
build_rule_matrix (GuppiLayout * lay)
{
  GuppiLayoutPrivate *p;
  gint R, M, W;
  GuppiMatrix *m;
  GList *ri;
  gint r, c;

  p = priv (lay);

  R = p->N_rules;		/* # of rules */
  M = p->N_geom;		/* # of items */

  /*
   * Width of matrix: 6 entries per geometry
   * (edge coordinates + width & height) then 5 constants:
   * screen bounds and CONSTANT
   */
  W = 6 * M + 5;
  m = guppi_matrix_new (R, W);
  r = 0;
  for (ri = p->rules; ri != NULL; ri = g_list_next (ri)) {
    Rule *rule = (Rule *) ri->data;
    GList *ti;
    for (ti = rule->terms; ti != NULL; ti = g_list_next (ti)) {
      Term *term = (Term *) ti->data;
      if (term->type == SCREEN_X0)
	c = 6 * M;
      else if (term->type == SCREEN_X1)
	c = 6 * M + 1;
      else if (term->type == SCREEN_Y0)
	c = 6 * M + 2;
      else if (term->type == SCREEN_Y1)
	c = 6 * M + 3;
      else if (term->type == CONSTANT)
	c = 6 * M + 4;
      else
	c = code_ext (lay, term);

      guppi_matrix_entry (m, r, c) += term->c;
    }
    ++r;
  }

  return m;
}

/*
 * We use a Grahm-Schmidt-type sequence of operations to 
 * remove redundant rules from our rule matrix.
 */


static GuppiMatrix *
simplify_rule_matrix (GuppiMatrix *m)
{
  gint r, j;
  gint nonzero_rows = 0;
  g_return_val_if_fail (m != NULL, NULL);

  for (r = 0; r < guppi_matrix_rows (m); ++r) {
    for (j = 0; j < r; ++j) {
      double sc = guppi_matrix_row_dot (m, r, j);
      guppi_matrix_subtract_scaled_row_from_row (m, sc, j, r);
    }
    if (guppi_matrix_row_is_nonzero (m, r)) {
      guppi_matrix_normalize_row (m, r);
      ++nonzero_rows;
    }
  }

  if (nonzero_rows != guppi_matrix_rows (m)) {
    GuppiMatrix *collapsed = guppi_matrix_new (nonzero_rows, guppi_matrix_cols (m));
    gint i = 0;

    if (guppi_is_very_verbose ())
      g_message ("redundant rules: %d", guppi_matrix_rows (m) - nonzero_rows);

    for (r = 0; r < guppi_matrix_rows (m); ++r) {
      if (guppi_matrix_row_is_nonzero (m, r)) {
	for (j = 0; j < guppi_matrix_cols (m); ++j)
	  guppi_matrix_entry (collapsed, i, j) = guppi_matrix_entry (m, r, j);
	++i;
      }
    }
    g_assert (nonzero_rows == i);
    guppi_matrix_free (m);
    m = collapsed;
  }

  return m;
}

/* Build our matrix of unknowns coordinates */

static GuppiMatrix *
build_coordinate_matrix (GuppiLayout *lay, GuppiMatrix *rule_matrix)
{
  GuppiLayoutPrivate *p;
  GuppiMatrix *m;
  gint C, R;
  gint i, j;

  p = priv (lay);

  C = 4 * p->N_geom;
  R = guppi_matrix_rows (rule_matrix);
  if (R < C)
    g_warning ("Under-specified layout! (rules=%d, coordinates=%d)", R, C);
  if (R > C)
    g_warning ("Over-specified layout! (rules=%d, coordinates=%d)", R, C);

  m = guppi_matrix_new (C, C);

  /* Copy the piece of the rule matrix related to coordinates
     over to the coordinate matrix. */
  for (i = 0; i < R; ++i)
    for (j = 0; j < C; ++j)
      guppi_matrix_entry (m, i, j) = guppi_matrix_entry (rule_matrix, i, j);

  /*
   * If our system is under-specified, the last C-R rows of our matrix
   * are all zero. 
   * If it is over-specified, the last R-C rules are dropped.
   */

  return m;
}

static void
coordinate_matrix_diagnostics (GuppiLayout *lay, GuppiMatrix *coor_matrix)
{
  GuppiLayoutPrivate *p;
  GList *gi;
  gint i;

  p = priv (lay);
  gi = p->geometries;

  i = 0;
  for (gi = p->geometries; gi != NULL; gi = g_list_next (gi)) {
    gboolean l, r, t, b;

    l = ! guppi_matrix_column_is_nonzero (coor_matrix, i++);
    r = ! guppi_matrix_column_is_nonzero (coor_matrix, i++);
    t = ! guppi_matrix_column_is_nonzero (coor_matrix, i++);
    b = ! guppi_matrix_column_is_nonzero (coor_matrix, i++);

    if (l || r || t || b) {
      GuppiGeometry *g = (GuppiGeometry *) gi->data;
      g_print ("Unspecified geometry for %s: ",
	       gtk_type_name (GTK_OBJECT_TYPE
			      (guppi_geometry_user_data (g))));
      if (l)
	g_print ("left ");
      if (r)
	g_print ("right ");
      if (t)
	g_print ("top ");
      if (b)
	g_print ("bottom ");
      g_print ("\n");
    }
  }
}

static GuppiVector *
build_coordinate_vec (GuppiLayout *lay,
		      GuppiMatrix *rule_matrix,
		      double x0, double y0, double x1, double y1)
{
  GuppiLayoutPrivate *p;
  GuppiVector *vec = guppi_vector_new (guppi_matrix_rows (rule_matrix));
  gint i;

  p = priv (lay);

  for (i = 0; i < guppi_matrix_rows (rule_matrix); ++i) {
    GList *gi;
    double c;
    gint j;

    j = 4 * p->N_geom;

    for (gi = p->geometries; gi != NULL; gi = g_list_next (gi)) {
      GuppiGeometry *geom = (GuppiGeometry *) gi->data;
      double c;
      double sz;

      c = guppi_matrix_entry (rule_matrix, i, j);
      if (fabs (c) > rule_matrix->epsilon) {
	sz = guppi_geometry_natural_width (geom);
	guppi_vector_entry (vec, i) -= c * MAX (sz, 0);
      }
      ++j;

      c = guppi_matrix_entry (rule_matrix, i, j);
      if (fabs (c) > rule_matrix->epsilon) {
	sz = guppi_geometry_natural_height (geom);
	guppi_vector_entry (vec, i) -= c * MAX (sz, 0);
      }
      ++j;
    }

    c = guppi_matrix_entry (rule_matrix, i, j);
    if (fabs (c) > rule_matrix->epsilon) {
      guppi_vector_entry (vec, i) -= c * x0;
    }
    ++j;

    c = guppi_matrix_entry (rule_matrix, i, j);
    if (fabs (c) > rule_matrix->epsilon) {
      guppi_vector_entry (vec, i) -= c * x1;
    }
    ++j;

    c = guppi_matrix_entry (rule_matrix, i, j);
    if (fabs (c) > rule_matrix->epsilon) {
      guppi_vector_entry (vec, i) -= c * y0;
    }
    ++j;

    c = guppi_matrix_entry (rule_matrix, i, j);
    if (fabs (c) > rule_matrix->epsilon) {
      guppi_vector_entry (vec, i) -= c * y1;
    }
    ++j;

    c = guppi_matrix_entry (rule_matrix, i, j);	/* constant term */
    if (fabs (c) > rule_matrix->epsilon) {
      guppi_vector_entry (vec, i) -= c;
    }
    ++j;

    g_assert (j == guppi_matrix_cols (rule_matrix));
  }

  return vec;
}

/***************************************************************************/

void
guppi_layout_natural_width (GuppiLayout * lay, GuppiGeometry * geom)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);

  register_geom (lay, geom);

  r = rule_new ();
  rule_add_term (r, term_new (RIGHT, 1.0, geom));
  rule_add_term (r, term_new (LEFT, -1.0, geom));
  rule_add_term (r, term_new (WIDTH, -1.0, geom));

  add_rule (lay, r);
}

void
guppi_layout_natural_height (GuppiLayout * lay, GuppiGeometry * geom)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);

  register_geom (lay, geom);

  r = rule_new ();
  rule_add_term (r, term_new (BOTTOM, -1.0, geom));
  rule_add_term (r, term_new (TOP, 1.0, geom));
  rule_add_term (r, term_new (HEIGHT, -1.0, geom));

  add_rule (lay, r);
}

void
guppi_layout_left_edge_relative (GuppiLayout * lay,
				 GuppiGeometry * geom, double perc)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);
  g_return_if_fail (perc >= 0 && perc <= 1);

  register_geom (lay, geom);

  r = rule_new ();
  if (perc != 1)
    rule_add_term (r, term_new (SCREEN_X0, 1 - perc, NULL));
  if (perc != 0)
    rule_add_term (r, term_new (SCREEN_X1, perc, NULL));
  rule_add_term (r, term_new (LEFT, -1.0, geom));

  add_rule (lay, r);
}

void
guppi_layout_right_edge_relative (GuppiLayout * lay,
				  GuppiGeometry * geom, double perc)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);
  g_return_if_fail (perc >= 0 && perc <= 1);

  register_geom (lay, geom);

  r = rule_new ();
  if (perc != 1)
    rule_add_term (r, term_new (SCREEN_X0, 1 - perc, NULL));
  if (perc != 0)
    rule_add_term (r, term_new (SCREEN_X1, perc, NULL));
  rule_add_term (r, term_new (RIGHT, -1.0, geom));

  add_rule (lay, r);
}

void
guppi_layout_top_edge_relative (GuppiLayout * lay,
				GuppiGeometry * geom, double perc)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);
  g_return_if_fail (perc >= 0 && perc <= 1);

  register_geom (lay, geom);

  r = rule_new ();
  if (perc != 1)
    rule_add_term (r, term_new (SCREEN_Y0, 1 - perc, NULL));
  if (perc != 0)
    rule_add_term (r, term_new (SCREEN_Y1, perc, NULL));
  rule_add_term (r, term_new (TOP, -1.0, geom));

  add_rule (lay, r);
}

void
guppi_layout_bottom_edge_relative (GuppiLayout * lay,
				   GuppiGeometry * geom, double perc)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);
  g_return_if_fail (perc >= 0 && perc <= 1);

  register_geom (lay, geom);

  r = rule_new ();
  if (perc != 1)
    rule_add_term (r, term_new (SCREEN_Y0, 1 - perc, NULL));
  if (perc != 0)
    rule_add_term (r, term_new (SCREEN_Y1, perc, NULL));
  rule_add_term (r, term_new (BOTTOM, -1.0, geom));

  add_rule (lay, r);
}


void
guppi_layout_flush_left (GuppiLayout * lay,
			 GuppiGeometry * geom, double margin)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);

  register_geom (lay, geom);

  r = rule_new ();
  rule_add_term (r, term_new (LEFT, 1.0, geom));
  rule_add_term (r, term_new (SCREEN_X0, -1.0, NULL));
  if (margin != 0)
    rule_add_term (r, term_new (CONSTANT, -margin, NULL));

  add_rule (lay, r);
}

void
guppi_layout_flush_right (GuppiLayout * lay,
			  GuppiGeometry * geom, double margin)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);

  register_geom (lay, geom);

  r = rule_new ();
  rule_add_term (r, term_new (RIGHT, -1.0, geom));
  rule_add_term (r, term_new (SCREEN_X1, 1.0, NULL));
  if (margin != 0)
    rule_add_term (r, term_new (CONSTANT, -margin, NULL));

  add_rule (lay, r);
}

void
guppi_layout_flush_top (GuppiLayout * lay,
			GuppiGeometry * geom, double margin)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);

  register_geom (lay, geom);

  r = rule_new ();
  rule_add_term (r, term_new (TOP, 1.0, geom));
  rule_add_term (r, term_new (SCREEN_Y1, -1.0, NULL));
  if (margin != 0)
    rule_add_term (r, term_new (CONSTANT, margin, NULL));

  add_rule (lay, r);
}

void
guppi_layout_flush_bottom (GuppiLayout * lay,
			   GuppiGeometry * geom, double margin)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);

  register_geom (lay, geom);

  r = rule_new ();
  rule_add_term (r, term_new (BOTTOM, -1.0, geom));
  rule_add_term (r, term_new (SCREEN_Y0, 1.0, NULL));
  if (margin != 0)
    rule_add_term (r, term_new (CONSTANT, margin, NULL));

  add_rule (lay, r);
}

void
guppi_layout_width_relative (GuppiLayout * lay,
			     GuppiGeometry * geom, double perc)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);
  g_return_if_fail (perc > 0 && perc <= 1);

  register_geom (lay, geom);

  r = rule_new ();
  rule_add_term (r, term_new (RIGHT, 1.0, geom));
  rule_add_term (r, term_new (LEFT, -1.0, geom));
  rule_add_term (r, term_new (SCREEN_X1, -perc, NULL));
  rule_add_term (r, term_new (SCREEN_X0, perc, NULL));

  add_rule (lay, r);
}

void
guppi_layout_height_relative (GuppiLayout * lay,
			      GuppiGeometry * geom, double perc)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);
  g_return_if_fail (perc > 0 && perc <= 1);

  register_geom (lay, geom);

  r = rule_new ();
  rule_add_term (r, term_new (TOP, 1.0, geom));
  rule_add_term (r, term_new (BOTTOM, -1.0, geom));
  rule_add_term (r, term_new (SCREEN_Y1, -perc, NULL));
  rule_add_term (r, term_new (SCREEN_Y0, perc, NULL));

  add_rule (lay, r);
}

void
guppi_layout_center_horizontally (GuppiLayout * lay, GuppiGeometry * geom)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);

  register_geom (lay, geom);

  r = rule_new ();
  rule_add_term (r, term_new (LEFT, 1.0, geom));
  rule_add_term (r, term_new (RIGHT, 1.0, geom));
  rule_add_term (r, term_new (SCREEN_X0, -1.0, NULL));
  rule_add_term (r, term_new (SCREEN_X1, -1.0, NULL));

  add_rule (lay, r);
}

void
guppi_layout_center_vertically (GuppiLayout * lay, GuppiGeometry * geom)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);

  register_geom (lay, geom);

  r = rule_new ();
  rule_add_term (r, term_new (TOP, 1.0, geom));
  rule_add_term (r, term_new (BOTTOM, 1.0, geom));
  rule_add_term (r, term_new (SCREEN_Y0, -1.0, NULL));
  rule_add_term (r, term_new (SCREEN_Y1, -1.0, NULL));

  add_rule (lay, r);
}

void
guppi_layout_horizontal_fill (GuppiLayout * lay,
			      GuppiGeometry * geom,
			      double l_margin, double r_margin)
{
  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);
  guppi_layout_flush_left (lay, geom, l_margin);
  guppi_layout_flush_right (lay, geom, r_margin);
}

void
guppi_layout_vertical_fill (GuppiLayout * lay,
			    GuppiGeometry * geom,
			    double t_margin, double b_margin)
{
  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL);
  guppi_layout_flush_top (lay, geom, t_margin);
  guppi_layout_flush_bottom (lay, geom, b_margin);
}

void
guppi_layout_aligned_left_edge (GuppiLayout * lay,
				GuppiGeometry * a, GuppiGeometry * b)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (a != NULL);
  g_return_if_fail (b != NULL);

  register_geom (lay, a);
  register_geom (lay, b);

  r = rule_new ();
  rule_add_term (r, term_new (LEFT, 1.0, a));
  rule_add_term (r, term_new (LEFT, -1.0, b));

  add_rule (lay, r);
}

void
guppi_layout_aligned_right_edge (GuppiLayout * lay,
				 GuppiGeometry * a, GuppiGeometry * b)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (a != NULL);
  g_return_if_fail (b != NULL);

  register_geom (lay, a);
  register_geom (lay, b);

  r = rule_new ();
  rule_add_term (r, term_new (RIGHT, 1.0, a));
  rule_add_term (r, term_new (RIGHT, -1.0, b));

  add_rule (lay, r);
}

void
guppi_layout_aligned_top_edge (GuppiLayout * lay,
			       GuppiGeometry * a, GuppiGeometry * b)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (a != NULL);
  g_return_if_fail (b != NULL);

  register_geom (lay, a);
  register_geom (lay, b);

  r = rule_new ();
  rule_add_term (r, term_new (TOP, 1.0, a));
  rule_add_term (r, term_new (TOP, -1.0, b));

  add_rule (lay, r);
}

void
guppi_layout_aligned_bottom_edge (GuppiLayout * lay,
				  GuppiGeometry * a, GuppiGeometry * b)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (a != NULL);
  g_return_if_fail (b != NULL);

  register_geom (lay, a);
  register_geom (lay, b);

  r = rule_new ();
  rule_add_term (r, term_new (BOTTOM, 1.0, a));
  rule_add_term (r, term_new (BOTTOM, -1.0, b));

  add_rule (lay, r);
}



void
guppi_layout_horizontally_adjacent (GuppiLayout * lay,
				    GuppiGeometry * a,
				    GuppiGeometry * b, double gap)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (a != NULL);
  g_return_if_fail (b != NULL);

  register_geom (lay, a);
  register_geom (lay, b);

  r = rule_new ();
  rule_add_term (r, term_new (LEFT, 1.0, b));
  rule_add_term (r, term_new (RIGHT, -1.0, a));
  if (gap != 0)
    rule_add_term (r, term_new (CONSTANT, -gap, NULL));

  add_rule (lay, r);
}

void
guppi_layout_vertically_adjacent (GuppiLayout * lay,
				  GuppiGeometry * a,
				  GuppiGeometry * b, double gap)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (a != NULL);
  g_return_if_fail (b != NULL);

  register_geom (lay, a);
  register_geom (lay, b);

  r = rule_new ();
  rule_add_term (r, term_new (TOP, 1.0, b));
  rule_add_term (r, term_new (BOTTOM, -1.0, a));
  if (gap != 0)
    rule_add_term (r, term_new (CONSTANT, gap, NULL));

  add_rule (lay, r);
}

void
guppi_layout_hbox2 (GuppiLayout * lay,
		    GuppiGeometry * left, GuppiGeometry * right, double gap)
{
  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (GUPPI_IS_LAYOUT (lay));

  g_return_if_fail (left != NULL);
  g_return_if_fail (GUPPI_IS_GEOMETRY (left));

  g_return_if_fail (right != NULL);
  g_return_if_fail (GUPPI_IS_GEOMETRY (right));

  guppi_layout_horizontally_adjacent (lay, left, right, gap);
  guppi_layout_aligned_top_edge (lay, left, right);
  guppi_layout_aligned_bottom_edge (lay, left, right);
}

void
guppi_layout_hbox3 (GuppiLayout * lay,
		    GuppiGeometry * left,
		    GuppiGeometry * center, GuppiGeometry * right, double gap)
{
  g_return_if_fail (lay != NULL);
  g_return_if_fail (GUPPI_IS_LAYOUT (lay));

  g_return_if_fail (left != NULL);
  g_return_if_fail (GUPPI_IS_GEOMETRY (left));

  g_return_if_fail (center != NULL);
  g_return_if_fail (GUPPI_IS_GEOMETRY (center));

  g_return_if_fail (right != NULL);
  g_return_if_fail (GUPPI_IS_GEOMETRY (right));

  guppi_layout_horizontally_adjacent (lay, left, center, gap);
  guppi_layout_aligned_top_edge (lay, left, center);
  guppi_layout_aligned_bottom_edge (lay, left, center);

  guppi_layout_horizontally_adjacent (lay, center, right, gap);
  guppi_layout_aligned_top_edge (lay, center, right);
  guppi_layout_aligned_bottom_edge (lay, center, right);
}

void
guppi_layout_vbox2 (GuppiLayout * lay,
		    GuppiGeometry * upper, GuppiGeometry * lower, double gap)
{
  g_return_if_fail (lay != NULL);
  g_return_if_fail (GUPPI_IS_LAYOUT (lay));

  g_return_if_fail (upper != NULL);
  g_return_if_fail (GUPPI_IS_GEOMETRY (upper));

  g_return_if_fail (lower != NULL);
  g_return_if_fail (GUPPI_IS_GEOMETRY (lower));

  guppi_layout_vertically_adjacent (lay, upper, lower, gap);
  guppi_layout_aligned_left_edge (lay, upper, lower);
  guppi_layout_aligned_right_edge (lay, upper, lower);
}

void
guppi_layout_vbox3 (GuppiLayout * lay,
		    GuppiGeometry * upper,
		    GuppiGeometry * center, GuppiGeometry * lower, double gap)
{
  g_return_if_fail (lay != NULL);
  g_return_if_fail (GUPPI_IS_LAYOUT (lay));

  g_return_if_fail (upper != NULL);
  g_return_if_fail (GUPPI_IS_GEOMETRY (upper));

  g_return_if_fail (center != NULL);
  g_return_if_fail (GUPPI_IS_GEOMETRY (center));

  g_return_if_fail (lower != NULL);
  g_return_if_fail (GUPPI_IS_GEOMETRY (lower));

  guppi_layout_vertically_adjacent (lay, upper, center, gap);
  guppi_layout_aligned_left_edge (lay, upper, center);
  guppi_layout_aligned_right_edge (lay, upper, center);

  guppi_layout_vertically_adjacent (lay, center, lower, gap);
  guppi_layout_aligned_left_edge (lay, center, lower);
  guppi_layout_aligned_right_edge (lay, center, lower);
}

void
guppi_layout_width_ratio (GuppiLayout * lay,
			  GuppiGeometry * a, GuppiGeometry * b, double ratio)
{
  Rule *r;

  g_return_if_fail (lay != NULL);
  g_return_if_fail (a != NULL);
  g_return_if_fail (b != NULL);

  register_geom (lay, a);
  register_geom (lay, b);

  r = rule_new ();
  rule_add_term (r, term_new (RIGHT, 1.0, a));
  rule_add_term (r, term_new (LEFT, -1.0, a));
  rule_add_term (r, term_new (RIGHT, -ratio, b));
  rule_add_term (r, term_new (LEFT, ratio, b));

  add_rule (lay, r);
}

void
guppi_layout_height_ratio (GuppiLayout * lay,
			   GuppiGeometry * a, GuppiGeometry * b, double ratio)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (a != NULL);
  g_return_if_fail (b != NULL);

  register_geom (lay, a);
  register_geom (lay, b);

  r = rule_new ();
  rule_add_term (r, term_new (BOTTOM, 1.0, a));
  rule_add_term (r, term_new (TOP, -1.0, a));
  rule_add_term (r, term_new (BOTTOM, -ratio, b));
  rule_add_term (r, term_new (TOP, ratio, b));

  add_rule (lay, r);
}

void
guppi_layout_aspect_ratio (GuppiLayout * lay, GuppiGeometry * a, double ar)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (a != NULL);

  register_geom (lay, a);

  r = rule_new ();
  rule_add_term (r, term_new (RIGHT, 1.0, a));
  rule_add_term (r, term_new (LEFT, -1.0, a));
  rule_add_term (r, term_new (BOTTOM, ar, a));
  rule_add_term (r, term_new (TOP, -ar, a));

  add_rule (lay, r);
}

void
guppi_layout_width_equals_height (GuppiLayout * lay,
				  GuppiGeometry * w, GuppiGeometry * h)
{
  Rule *r;

  g_return_if_fail (lay != NULL);
  g_return_if_fail (GUPPI_IS_LAYOUT (lay));

  g_return_if_fail (w != NULL);
  g_return_if_fail (GUPPI_IS_GEOMETRY (w));

  g_return_if_fail (h != NULL);
  g_return_if_fail (GUPPI_IS_GEOMETRY (h));

  register_geom (lay, w);
  register_geom (lay, h);

  r = rule_new ();

  rule_add_term (r, term_new (RIGHT, 1.0, w));
  rule_add_term (r, term_new (LEFT, -1.0, w));
  rule_add_term (r, term_new (BOTTOM, 1.0, h));
  rule_add_term (r, term_new (TOP, -1.0, h));

  add_rule (lay, r);
}

void
guppi_layout_same_x_center (GuppiLayout * lay,
			    GuppiGeometry * a, GuppiGeometry * b)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (a != NULL);
  g_return_if_fail (b != NULL);

  register_geom (lay, a);
  register_geom (lay, b);

  r = rule_new ();
  rule_add_term (r, term_new (RIGHT, 1.0, a));
  rule_add_term (r, term_new (LEFT, 1.0, a));
  rule_add_term (r, term_new (RIGHT, -1.0, b));
  rule_add_term (r, term_new (LEFT, -1.0, b));

  add_rule (lay, r);
}

void
guppi_layout_same_y_center (GuppiLayout * lay,
			    GuppiGeometry * a, GuppiGeometry * b)
{
  Rule *r;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (a != NULL && GUPPI_IS_GEOMETRY (a));
  g_return_if_fail (b != NULL && GUPPI_IS_GEOMETRY (b));

  register_geom (lay, a);
  register_geom (lay, b);

  r = rule_new ();
  rule_add_term (r, term_new (BOTTOM, 1.0, a));
  rule_add_term (r, term_new (TOP, 1.0, a));
  rule_add_term (r, term_new (BOTTOM, -1.0, b));
  rule_add_term (r, term_new (TOP, -1.0, b));

  add_rule (lay, r);
}

/***************************************************************************/

void
guppi_layout_remove (GuppiLayout * lay, GuppiGeometry * geom)
{
  GuppiLayoutPrivate *p;
  GList *i;
  GList *node;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL && GUPPI_IS_GEOMETRY (geom));

  p = priv (lay);

  node = g_list_find (p->geometries, geom);
  g_return_if_fail (node != NULL);
  p->geometries = g_list_remove_link (p->geometries, node);
  g_list_free_1 (node);
  --p->N_geom;

  i = p->rules;
  while (i != NULL) {
    GList *next = g_list_next (i);
    Rule *r = (Rule *) i->data;
    if (rule_contains_geom (r, geom)) {
      rule_free (r);
      i->data = NULL;
      p->rules = g_list_remove_link (p->rules, i);
      g_list_free_1 (i);
      --p->N_rules;
    }
    i = next;
  }

  guppi_geometry_disconnect_from_layout (geom);
  guppi_unref (geom);
}

void
guppi_layout_replace (GuppiLayout * lay,
		      GuppiGeometry * old, GuppiGeometry * nuevo)
{
  GuppiLayoutPrivate *p;
  GList *i;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (old != NULL && GUPPI_IS_GEOMETRY (old));
  g_return_if_fail (nuevo != NULL && GUPPI_IS_GEOMETRY (nuevo));

  if (old == nuevo)
    return;

  p = priv (lay);

  i = p->geometries;
  while (i != NULL) {
    if (i->data == old) {
      guppi_geometry_set_registered (old, FALSE);
      guppi_geometry_set_registered (nuevo, FALSE);
      guppi_ref (nuevo);
      guppi_unref (old);
      i->data = nuevo;
      i = NULL;
    } else
      i = g_list_next (i);
  }

  i = p->rules;
  while (i != NULL) {
    if (i->data != NULL) {
      rule_replace ((Rule *) i->data, old, nuevo);
    }
    i = g_list_next (i);
  }

  guppi_geometry_disconnect_from_layout (old);
  guppi_layout_connect_geometry (lay, nuevo);
}


/***************************************************************************/

void
guppi_layout_bounds (GuppiLayout * lay,
		     double *x0, double *y0, double *x1, double *y1)
{
  GuppiLayoutPrivate *p;
  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  p = priv (lay);
  if (x0)
    *x0 = p->x0;
  if (y0)
    *y0 = p->y0;
  if (x1)
    *x1 = p->x1;
  if (y1)
    *y1 = p->y1;
}

void
guppi_layout_set_bounds (GuppiLayout * lay,
			 double x0, double y0, double x1, double y1)
{
  GuppiLayoutPrivate *p;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));

  p = priv (lay);

  guppi_2sort (&x0, &x1);
  guppi_2sort (&y0, &y1);

  if (p->x0 != x0 || p->x1 != x1 || p->y0 != y0 || p->y1 != y1) {
    p->x0 = x0;
    p->x1 = x1;
    p->y0 = y0;
    p->y1 = y1;
    guppi_layout_calc_delayed (lay);
  }
}

void
guppi_layout_set_bounds_from_geometry (GuppiLayout * lay,
				       GuppiGeometry * geom)
{
  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  g_return_if_fail (geom != NULL && GUPPI_IS_GEOMETRY (geom));

  guppi_layout_set_bounds (lay,
			   guppi_geometry_left (geom),
			   guppi_geometry_bottom (geom),
			   guppi_geometry_right (geom),
			   guppi_geometry_top (geom));
}

void
guppi_layout_calc (GuppiLayout * lay)
{
  GuppiLayoutPrivate *p;
  GuppiVector *v;
  GuppiVector *s;
  gint i = 0;
  GList *iter;

  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));

  p = priv (lay);

  //guppi_stop_in_debugger();

  if (p->pending_layout) {
    gtk_idle_remove (p->pending_layout);
    p->pending_layout = 0;
  }

  /* Defer all layouts until our bounds are set. */
  if (p->x0 > p->x1 || p->y0 > p->y1)
    return;

  p->calculating_layout = TRUE;

  if (p->rule_matrix == NULL) {
    GuppiMatrix *rule_matrix;
    GuppiMatrix *coor_matrix;

    rule_matrix = build_rule_matrix (lay);
    rule_matrix = simplify_rule_matrix (rule_matrix);
    coor_matrix = build_coordinate_matrix (lay, rule_matrix);
    coordinate_matrix_diagnostics (lay, coor_matrix);

    p->rule_matrix = rule_matrix;
    p->coor_matrix = coor_matrix;
  }

  v = build_coordinate_vec (lay, p->rule_matrix, p->x0, p->y0, p->x1, p->y1);
  s = custom_solve (p->coor_matrix, v, p->x0, p->y0, p->x1, p->y1);
  
  if (s != NULL) {
    iter = p->geometries;
    while (iter != NULL) {
      GuppiGeometry *geom = (GuppiGeometry *) iter->data;

      guppi_geometry_set_position (geom,
				   CLAMP (guppi_vector_entry (s, i),   p->x0, p->x1),
				   CLAMP (guppi_vector_entry (s, i+1), p->x0, p->x1),
				   CLAMP (guppi_vector_entry (s, i+2), p->y0, p->y1),
				   CLAMP (guppi_vector_entry (s, i+3), p->y0, p->y1));

#if 0
      g_print ("%s: L:%g R:%g T:%g B:%g\n",
	       gtk_type_name (GTK_OBJECT_TYPE
			      (guppi_geometry_user_data (geom))),
	       guppi_geometry_left (geom), guppi_geometry_right (geom),
	       guppi_geometry_top (geom), guppi_geometry_bottom (geom));
#endif

      i += 4;

      iter = g_list_next (iter);
    }
  }

  guppi_vector_free (v);
  guppi_vector_free (s);

  p->calculating_layout = FALSE;
}

static gint
do_idle_calc (gpointer foo)
{
  GuppiLayout *lay = GUPPI_LAYOUT (foo);
  GuppiLayoutPrivate *p = priv (lay);
  p->pending_layout = 0;
  guppi_layout_calc (lay);
  return FALSE;
}

void
guppi_layout_calc_delayed (GuppiLayout * lay)
{
  GuppiLayoutPrivate *p;
  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  p = priv (lay);
  if (p->pending_layout == 0) {
    p->pending_layout = gtk_idle_add_priority (G_PRIORITY_HIGH_IDLE,
					       do_idle_calc, lay);
  }
}

void
guppi_layout_calc_flush (GuppiLayout * lay)
{
  GuppiLayoutPrivate *p;
  g_return_if_fail (lay != NULL && GUPPI_IS_LAYOUT (lay));
  p = priv (lay);

  /* Force pending layout to occur now. */
  if (p->pending_layout)
    do_idle_calc (lay);

}

/* $Id: guppi-layout.c,v 1.24 2001/05/06 08:26:43 trow Exp $ */
