/* This is -*- C -*- */
/* $Id: guppi-geometry.c,v 1.14 2001/01/16 23:36:26 trow Exp $ */

/*
 * guppi-geometry.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>

#include <gtk/gtksignal.h>

#include <guppi-useful.h>
#include "guppi-geometry.h"
#include "guppi-layout.h"

#define EPSILON 1e-8

typedef struct _GuppiGeometryPrivate GuppiGeometryPrivate;
struct _GuppiGeometryPrivate {
  gboolean registered;

  gboolean positioned;
  double left, right, top, bottom;

  gboolean have_natural_width, have_natural_height;
  double natural_width, natural_height;

  double (*width_callback) (gpointer);
  double (*height_callback) (gpointer);
  gpointer user_data;

  GuppiLayout *layout;
  guint changed_size_sig;
  guint destroy_sig;
};

#define priv(x) ((GuppiGeometryPrivate*)((GUPPI_GEOMETRY(x))->opaque_internals))


static GtkObjectClass *parent_class = NULL;

enum {
  CHANGED_POSITION,
  CHANGED_SIZE,
  LAST_SIGNAL
};

static guint gg_signals[LAST_SIGNAL] = { 0 };

static void
guppi_geometry_finalize (GtkObject * obj)
{
  GuppiGeometry *gg = GUPPI_GEOMETRY (obj);

  guppi_finalized (obj);

  guppi_geometry_disconnect_from_layout (gg);

  guppi_free (gg->opaque_internals);
  gg->opaque_internals = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
guppi_geometry_class_init (GuppiGeometryClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  parent_class = gtk_type_class (GTK_TYPE_OBJECT);

  object_class->finalize = guppi_geometry_finalize;

  gg_signals[CHANGED_POSITION] =
    gtk_signal_new ("changed_position",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiGeometryClass, changed_position),
		    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

  gg_signals[CHANGED_SIZE] =
    gtk_signal_new ("changed_size",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiGeometryClass, changed_size),
		    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals (object_class, gg_signals, LAST_SIGNAL);

}

static void
guppi_geometry_init (GuppiGeometry * obj)
{
  obj->opaque_internals = guppi_new0 (GuppiGeometryPrivate, 1);
}

GtkType guppi_geometry_get_type (void)
{
  static GtkType guppi_geometry_type = 0;
  if (!guppi_geometry_type) {
    static const GtkTypeInfo guppi_geometry_info = {
      "GuppiGeometry",
      sizeof (GuppiGeometry),
      sizeof (GuppiGeometryClass),
      (GtkClassInitFunc) guppi_geometry_class_init,
      (GtkObjectInitFunc) guppi_geometry_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_geometry_type =
      gtk_type_unique (GTK_TYPE_OBJECT, &guppi_geometry_info);
  }
  return guppi_geometry_type;
}

GuppiGeometry *
guppi_geometry_new (void)
{
  return GUPPI_GEOMETRY (guppi_type_new (guppi_geometry_get_type ()));
}

gboolean guppi_geometry_positioned (GuppiGeometry * gg)
{
  g_return_val_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg), FALSE);
  return priv (gg)->positioned;
}

double
guppi_geometry_left (GuppiGeometry * gg)
{
  g_return_val_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg), 0);
  return priv (gg)->left;
}

double
guppi_geometry_right (GuppiGeometry * gg)
{
  g_return_val_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg), 0);
  return priv (gg)->right;
}

double
guppi_geometry_top (GuppiGeometry * gg)
{
  g_return_val_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg), 0);
  return priv (gg)->top;
}

double
guppi_geometry_bottom (GuppiGeometry * gg)
{
  g_return_val_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg), 0);
  return priv (gg)->bottom;
}

double
guppi_geometry_width (GuppiGeometry * gg)
{
  g_return_val_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg), 0);
  return priv (gg)->right - priv (gg)->left;
}

double
guppi_geometry_height (GuppiGeometry * gg)
{
  g_return_val_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg), 0);
  return priv (gg)->top - priv (gg)->bottom;
}

void
guppi_geometry_conv (GuppiGeometry * gg,
		     double x, double y, double *t_x, double *t_y)
{
  GuppiGeometryPrivate *p;
  g_return_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg));
  p = priv (gg);

  if (t_x)
    *t_x = (x - p->left) / (p->right - p->left);

  if (t_y)
    *t_y = (y - p->bottom) / (p->top - p->bottom);
}

void
guppi_geometry_unconv (GuppiGeometry * gg,
		       double t_x, double t_y, double *x, double *y)
{
  GuppiGeometryPrivate *p;
  g_return_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg));
  p = priv (gg);

  if (x)
    *x = p->left + t_x * (p->right - p->left);

  if (y)
    *y = p->bottom + t_y * (p->top - p->bottom);
}



void
guppi_geometry_set_position (GuppiGeometry * gg,
			     double l, double r, double t, double b)
{
  GuppiGeometryPrivate *p;
  g_return_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg));
  p = priv (gg);

  guppi_2sort (&l, &r);
  guppi_2sort (&b, &t);

  if (!p->positioned ||
      p->left != l || p->right != r || p->top != t || p->bottom != b) {
    p->positioned = TRUE;
    p->left = l;
    p->right = r;
    p->top = t;
    p->bottom = b;
    gtk_signal_emit (GTK_OBJECT (gg), gg_signals[CHANGED_POSITION]);
  }
}

gboolean guppi_geometry_contains (GuppiGeometry * gg, double x, double y)
{
  GuppiGeometryPrivate *p;
  g_return_val_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg), FALSE);
  p = priv (gg);

  return guppi_between (p->left, x, p->right) &&
    guppi_between (p->bottom, y, p->top);
}

void
guppi_geometry_set_width_callback (GuppiGeometry * gg, double (*f) (gpointer))
{
  g_return_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg));
  g_return_if_fail (f != NULL);

  priv (gg)->width_callback = f;
}

void
guppi_geometry_set_height_callback (GuppiGeometry * gg,
				    double (*f) (gpointer))
{
  g_return_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg));
  g_return_if_fail (f != NULL);

  priv (gg)->height_callback = f;
}

void
guppi_geometry_set_user_data (GuppiGeometry * gg, gpointer p)
{
  g_return_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg));
  priv (gg)->user_data = p;
}

gconstpointer guppi_geometry_user_data (GuppiGeometry * gg)
{
  g_return_val_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg), NULL);
  return priv (gg)->user_data;
}


static void
guppi_geometry_calc_natural_size_real (GuppiGeometry * gg,
				       gboolean do_w, gboolean do_h)
{
  GuppiGeometryPrivate *p;
  double w, h;
  gboolean changed = FALSE;

  g_return_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg));
  p = priv (gg);

  if (do_w && p->width_callback) {
    w = p->width_callback (p->user_data);
    if ((!p->have_natural_width) || fabs (p->natural_width - w) > EPSILON) {
      changed = TRUE;
    }
    p->have_natural_width = TRUE;
    p->natural_width = w;
  }

  if (do_h && p->height_callback) {
    h = p->height_callback (p->user_data);
    if ((!p->have_natural_height) || fabs (p->natural_height - h) > EPSILON) {
      changed = TRUE;
    }
    p->have_natural_height = TRUE;
    p->natural_height = h;
  }

  if (changed) {
    gtk_signal_emit (GTK_OBJECT (gg), gg_signals[CHANGED_SIZE]);
  }
}

double
guppi_geometry_natural_width (GuppiGeometry * gg)
{
  GuppiGeometryPrivate *p;
  g_return_val_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg), 0);
  p = priv (gg);

  if (!p->have_natural_width && p->width_callback)
    guppi_geometry_calc_natural_size_real (gg, TRUE, FALSE);

  return p->natural_width;
}

double
guppi_geometry_natural_height (GuppiGeometry * gg)
{
  GuppiGeometryPrivate *p;
  g_return_val_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg), 0);
  p = priv (gg);

  if (!p->have_natural_height && p->height_callback)
    guppi_geometry_calc_natural_size_real (gg, FALSE, TRUE);

  return p->natural_height;
}

void
guppi_geometry_calc_natural_size (GuppiGeometry * gg)
{
  g_return_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg));
  guppi_geometry_calc_natural_size_real (gg, TRUE, TRUE);
}

gboolean guppi_geometry_registered (GuppiGeometry * gg)
{
  g_return_val_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg), FALSE);
  return priv (gg)->registered;
}

void
guppi_geometry_set_registered (GuppiGeometry * gg, gboolean x)
{
  g_return_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg));
  priv (gg)->registered = x;
}

void
guppi_geometry_connect_to_layout (GuppiGeometry *gg, GuppiLayout *lay)
{
  GuppiGeometryPrivate *p;

  g_return_if_fail (gg && GUPPI_IS_GEOMETRY (gg));
  g_return_if_fail (lay && GUPPI_IS_LAYOUT (lay));

  p = priv (gg);

  if (p->layout == lay)
    return;

  guppi_geometry_disconnect_from_layout (gg);

  p->layout = lay;

  p->changed_size_sig =
    gtk_signal_connect_object (GTK_OBJECT (gg),
			       "changed_size",
			       GTK_SIGNAL_FUNC (guppi_layout_calc_delayed),
			       GTK_OBJECT (lay));

  p->destroy_sig =
    gtk_signal_connect_object (GTK_OBJECT (lay),
			       "destroy",
			       GTK_SIGNAL_FUNC (guppi_geometry_disconnect_from_layout),
			       GTK_OBJECT (gg));
}

void
guppi_geometry_disconnect_from_layout (GuppiGeometry * gg)
{
  GuppiGeometryPrivate *p;

  g_return_if_fail (gg != NULL && GUPPI_IS_GEOMETRY (gg));
  p = priv (gg);

  if (p->changed_size_sig)
    gtk_signal_disconnect (GTK_OBJECT (gg), p->changed_size_sig);
  if (p->destroy_sig)
    gtk_signal_disconnect (GTK_OBJECT (p->layout), p->destroy_sig);

  p->changed_size_sig = 0;
  p->destroy_sig = 0;
  p->layout = NULL;
}

/* $Id: guppi-geometry.c,v 1.14 2001/01/16 23:36:26 trow Exp $ */
