/* This is -*- C -*- */
/* $Id: guppi-seq-boolean-core-impl.c,v 1.11 2001/05/06 07:56:15 trow Exp $ */

/*
 * guppi-seq-boolean-core-impl.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
/* #include <gnome.h> */
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-string.h>
#include <guppi-convenient.h>
#include <guppi-data-impl-plug-in.h>
#include "guppi-seq-boolean-core-impl.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_boolean_core_impl_get_arg (GtkObject * obj, GtkArg * arg,
				     guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_boolean_core_impl_set_arg (GtkObject * obj, GtkArg * arg,
				     guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_boolean_core_impl_finalize (GtkObject * obj)
{
  GuppiSeqBooleanCoreImpl *impl = GUPPI_SEQ_BOOLEAN_CORE_IMPL (obj);

  guppi_unref0 (impl->garray);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/***************************************************************************/

#define ASIZE(core) (((core)->size >> 5) + (((core)->size&31)?1:0))
#define BITVEC_GET(data, i) ((data)[(i)>>5] & (1 << ((i)&31)))
#define BITVEC_SET(data, i) ((data)[(i)>>5] |= (1 << ((i)&31)))
#define BITVEC_UNSET(data, i) ((data)[(i)>>5] &= ~(1 << ((i)&31)))
#define LAST_MASK(size) ((guint32)(~0) >> (32-((size)&31)))

static gboolean
v_seq_boolean_get (const GuppiSeqBooleanImpl * impl, gint i)
{
  const GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  const guint32 *data = (const guint32 *) guppi_garray_data (core->garray);

  i -= core->index_basis;
  return BITVEC_GET (data, i);
}

static void
v_seq_boolean_set (GuppiSeqBooleanImpl * impl, gint i, gboolean x)
{
  const GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  guint32 *data = (guint32 *) guppi_garray_data (core->garray);

  i -= core->index_basis;
  if (x)
    BITVEC_SET (data, i);
  else
    BITVEC_UNSET (data, i);

  guppi_seq_impl_set_missing (GUPPI_SEQ_IMPL (impl), i + core->index_basis,
			      FALSE);
}

static void
v_seq_boolean_set_all (GuppiSeqBooleanImpl * impl, gboolean x)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  guint32 *data = (guint32 *) guppi_garray_data (core->garray);
  gint i, N;

  N = ASIZE (core);
  if (x) {
    for (i = 0; i < N - 1; ++i)
      data[i] = ~0;
  } else {
    for (i = 0; i < N - 1; ++i)
      data[i] = 0;
  }

  data[N - 1] = x ? LAST_MASK (core->size) : 0;
}

static void
v_seq_boolean_set_many (GuppiSeqBooleanImpl * impl,
			const gint * indices, gsize N, gboolean x)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  guint32 *data = (guint32 *) guppi_garray_data (core->garray);
  gint i, k;

  if (x) {


    for (i = 0; i < N; ++i) {
      k = indices[i] - core->index_basis;
      if (0 <= k && k < core->size)
	BITVEC_SET (data, k);
    }

  } else {

    for (i = 0; i < N; ++i) {
      k = indices[i] - core->index_basis;
      if (0 <= k && k < core->size)
	BITVEC_UNSET (data, k);
    }

  }

}

static void
v_seq_boolean_insert (GuppiSeqBooleanImpl * impl, gint i, gboolean x)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  guint32 *data;
  gint j, old_size, N, bottom;

  if (guppi_garray_size (core->garray) <= ASIZE (core)) {
    old_size = guppi_garray_size (core->garray);
    guppi_garray_set_size (core->garray, MAX (32, 2 * ASIZE (core)));
    data = (guint32 *) guppi_garray_data (core->garray);
    for (j = old_size; j < guppi_garray_size (core->garray); ++j)
      data[j] = 0;
  }

  data = (guint32 *) guppi_garray_data (core->garray);

  if (core->size == 0)
    core->index_basis = i;

  i -= core->index_basis;
  ++core->size;

  N = ASIZE (core);
  bottom = i >> 5;
  for (j = N - 1; j >= bottom; --j) {
    if (j + 1 < N)
      data[j + 1] |= data[j] >> 1;
    if (j != bottom || (i & 31) == 0)
      data[j] = data[j] << 1;
    else {
      guint32 bottom_mask = ((guint32) (~0)) >> (32 - (i & 31));
      data[j] = (data[j] & bottom_mask) | ((data[j] << 1) & ~bottom_mask);
    }
  }

  if (x)
    BITVEC_SET (data, i);
  else
    BITVEC_UNSET (data, i);

  guppi_seq_impl_insert_missing (GUPPI_SEQ_IMPL (impl),
				 i + core->index_basis,
				 FALSE, 1);
}

static void
v_seq_boolean_insert_many (GuppiSeqBooleanImpl * impl, gint i,
			   gboolean x, gsize step)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  guint32 *data;
  gint old_size, j, sz;
  gint step_big, step_small, N, bottom;

  /* grow the array, if necessary */
  sz = 1 + ((core->size + step) >> 5);
  if (guppi_garray_size (core->garray) <= sz) {
    old_size = guppi_garray_size (core->garray);
    guppi_garray_set_size (core->garray, MAX (32, 2 * sz));
    data = (guint32 *) guppi_garray_data (core->garray);
    for (j = old_size; j < guppi_garray_size (core->garray); ++j)
      data[j] = 0;
  }

  data = (guint32 *) guppi_garray_data (core->garray);

  if (core->size == 0)
    core->index_basis = i;

  i -= core->index_basis;
  core->size += step;

  N = ASIZE (core);
  bottom = i >> 5;
  step_big = step >> 5;
  step_small = step & 31;

  for (j = N - 1 - step_big; j >= bottom; --j) {
    if (step_small && j + step_big + 1 < N)
      data[j + step_big + 1] |= data[j] >> (32 - step_small);
    if (j == bottom && step_big == 0 && (i & 31) != 0) {
      guint32 bottom_mask = ((guint32) (~0)) >> (32 - (i & 31));
      data[j] =
	(data[j] & bottom_mask) | ((data[j] << step_small) & ~bottom_mask);
    } else
      data[j + step_big] = data[j] << step_small;
  }

  /* initialize the inserted slots */
  /* this could be optimized to operate an guint32 at a time, but
   * I'll opt for obvious correctness right now... */
  if (x) {
    for (j = 0; j < step; ++j)
      BITVEC_SET (data, i + j);
  } else {
    for (j = 0; j < step; ++j)
      BITVEC_UNSET (data, i + j);
  }

  guppi_seq_impl_insert_missing (GUPPI_SEQ_IMPL (impl),
				 i + core->index_basis,
				 FALSE, N);
}

static void
v_seq_boolean_bitwise_and (GuppiSeqBooleanImpl * impl,
			   const GuppiSeqBooleanImpl * other)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  GuppiSeqBooleanCoreImpl *other_core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (other);
  gint i;
  guint32 *data;
  const guint32 *other_data;

  if (core->index_basis == other_core->index_basis &&
      core->size == other_core->size) {
    data = (guint32 *) guppi_garray_data (core->garray);
    other_data = (const guint32 *) guppi_garray_data (other_core->garray);
    for (i = 0; i < ASIZE (core); ++i)
      data[i] &= other_data[i];
    return;
  }

  /* We'll implement the hard case later. */
  g_assert_not_reached ();
}

static void
v_seq_boolean_bitwise_or (GuppiSeqBooleanImpl * impl,
			  const GuppiSeqBooleanImpl * other)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  GuppiSeqBooleanCoreImpl *other_core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (other);
  gint i;
  guint32 *data;
  const guint32 *other_data;

  if (core->index_basis == other_core->index_basis &&
      core->size == other_core->size) {
    data = (guint32 *) guppi_garray_data (core->garray);
    other_data = (const guint32 *) guppi_garray_data (other_core->garray);
    for (i = 0; i < ASIZE (core); ++i)
      data[i] |= other_data[i];
    return;
  }

  /* We'll implement the hard case later. */
  g_assert_not_reached ();
}

static void
v_seq_boolean_bitwise_xor (GuppiSeqBooleanImpl * impl,
			   const GuppiSeqBooleanImpl * other)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  GuppiSeqBooleanCoreImpl *other_core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (other);
  gint i;
  guint32 *data;
  const guint32 *other_data;

  if (core->index_basis == other_core->index_basis &&
      core->size == other_core->size) {
    data = (guint32 *) guppi_garray_data (core->garray);
    other_data = (const guint32 *) guppi_garray_data (other_core->garray);
    for (i = 0; i < ASIZE (core); ++i)
      data[i] ^= other_data[i];
    return;
  }

  /* We'll implement the hard case later. */
  g_assert_not_reached ();
}

static void
v_seq_boolean_bitwise_not (GuppiSeqBooleanImpl * impl)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  gint i, N;
  guint32 *data;

  data = (guint32 *) guppi_garray_data (core->garray);
  N = ASIZE (core);
  for (i = 0; i < N; ++i)
    data[i] = ~data[i];

  /* Make sure surplus bits are zero */
  data[N - 1] &= LAST_MASK (core->size);
}

static gint
v_seq_boolean_next_true (const GuppiSeqBooleanImpl * impl, gint i)
{
  const GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  const guint32 *data;
  gint j, k;
  guint32 q;

  if (core->size == 0)
    return 1;

  data = (const guint32 *) guppi_garray_data (core->garray);
  i -= core->index_basis;

  if (i < 0) {
    if (data[0] & 1)
      return core->index_basis;
    i = 0;
  }

  j = i >> 5;
  k = i & 31;
  ++i;
  /* Check of our first guint32 */
  if (k != 31 && (q = (data[j] >> (k + 1)))) {
    while (!(q & 1)) {
      q = q >> 1;
      ++i;
    }
    return i + core->index_basis;
  }
  ++j;

  while (j < ASIZE (core) && data[j] == 0)
    ++j;
  if (j >= ASIZE (core))
    return core->index_basis + core->size;
  q = data[j];
  i = j << 5;
  while (!(q & 1)) {
    q = q >> 1;
    ++i;
  }
  return i + core->index_basis;
}

static gsize
v_seq_boolean_true_count (const GuppiSeqBooleanImpl * impl)
{
  const GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  guint32 *data;
  static guint8 *bitcount = NULL;
  gint q, i, count, N;
  guint32 last_mask;


  /* Initialize our bit-count dictionary, which pre-tallies the number
     of on-bits in each integer 0..255 */
  if (bitcount == NULL) {
    bitcount = guppi_new (guint8, 256);
    guppi_permanent_alloc (bitcount);
    for (i = 0; i < 256; ++i) {
      count = 0;
      q = i;
      while (q) {
	if (q & 1)
	  ++count;
	q = q >> 1;
      }
      bitcount[i] = count;
    }
  }

  data = (guint32 *) guppi_garray_data (core->garray);
  N = ASIZE (core);

  /* Be paranoid and insure that the surplus bits are zero. */
  last_mask = LAST_MASK (core->size);
  data[N - 1] &= last_mask;

  count = 0;
  for (i = 0; i < N; ++i) {
    q = data[i];
    count += bitcount[q & 0xff] + bitcount[(q >> 8) & 0xff] +
      bitcount[(q >> 16) & 0xff] + bitcount[(q >> 24) & 0xff];
  }

  return (gsize) count;
}

/* This could probably be optimized some more. */
static gint
v_seq_boolean_range_query (const GuppiSeqScalar * seq,
			   GuppiSeqBooleanImpl * impl,
			   double min, double max, gboolean do_and)
{
  gconstpointer ptr;
  gint stride;
  GuppiSeqBooleanCoreImpl *core;
  guint32 *data;
  gint i, i0, i1;
  double x;
  gint hits = 0;

  ptr = guppi_seq_scalar_raw (seq, &stride);
  if (ptr == NULL)
    return -1;

  core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  guppi_seq_indices (GUPPI_SEQ (seq), &i0, &i1);

  data = (guint32 *) guppi_garray_data (core->garray);
  i0 = MAX (i0, core->index_basis);
  i1 = MIN (i1, core->index_basis - 1 + core->size);

  /* Adjust to point at first raw data element */
  ptr = (gconstpointer) (((guint8 *) ptr) + i0 * stride);

  if (do_and) {
    /* AND behavior */
    guint32 mask = 1;
    gint offset = 0;
    guint32 data_chunk = data[offset];

    i = i0;
    while (i <= i1) {

      if (data_chunk & mask) {
	x = *(const double *) ptr;
	if (x < min || max < x) {
	  data_chunk &= ~mask;
	} else {
	  ++hits;
	}
      }

      mask = mask << 1;
      if (mask == 0) {
	data[offset] = data_chunk;
	mask = 1;
	++offset;
	while ((data_chunk = data[offset]) == 0 && i <= i1) {
	  ++offset;
	  ptr = (gconstpointer) (((guint8 *) ptr) + (stride << 5));
	  i += 32;
	}
      }
      ptr = (gconstpointer) (((guint8 *) ptr) + stride);
      ++i;
    }
    if (mask != 1)
      data[offset] = data_chunk;

  } else {
    /* overwrite behavior */
    guint32 mask = 1;
    gint offset = 0;
    guint32 data_chunk = 0;

    for (i = i0; i <= i1; ++i) {

      if (mask == 1)
	data_chunk = 0;

      x = *(const double *) ptr;
      if (min <= x && x <= max) {
	data_chunk |= mask;
	++hits;
      }
      mask = mask << 1;
      if (mask == 0) {
	mask = 1;
	data[offset] = data_chunk;
	++offset;
      }
      ptr = (gconstpointer) (((guint8 *) ptr) + stride);
    }
    if (mask != 1)
      data[offset] = data_chunk;
  }
  return hits;
}

static void
v_seq_size_hint (GuppiSeqImpl * impl, gsize n)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  if (guppi_garray_size (core->garray) < 1 + (n >> 5))
    guppi_garray_set_size (core->garray, 1 + (n >> 5));
}

static void
v_seq_get_bounds (const GuppiSeqImpl * impl, gint * min, gint * max)
{
  const GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);

  if (min)
    *min = core->index_basis;

  if (max)
    *max = core->index_basis - 1 + core->size;
}

static void
v_seq_shift_indices (GuppiSeqImpl * impl, gint delta)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  core->index_basis += delta;
}

static gboolean
v_seq_validate (const GuppiSeqImpl * impl, const gchar * str,
		gchar * error_msg, gsize errlen)
{
  return TRUE;
}

static void
v_seq_get (const GuppiSeqImpl * impl, gint i, gchar * buf, gsize len)
{
  gboolean x = v_seq_boolean_get (GUPPI_SEQ_BOOLEAN_IMPL (impl), i);

  guppi_boolean2string (x, buf, len);
}

static void
v_seq_set (GuppiSeqImpl * impl, gint i, const gchar * buf)
{
  gboolean x;
  g_return_if_fail (guppi_string_is_boolean (buf));
  x = guppi_string2boolean (buf);
  v_seq_boolean_set (GUPPI_SEQ_BOOLEAN_IMPL (impl), i, x);
}

static void
v_seq_insert (GuppiSeqImpl * impl, gint i, const gchar * buf)
{
  gboolean x;
  g_return_if_fail (guppi_string_is_boolean (buf));
  x = guppi_string2boolean (buf);
  v_seq_boolean_insert (GUPPI_SEQ_BOOLEAN_IMPL (impl), i, x);
}

static void
v_seq_insert_missing (GuppiSeqImpl *impl, gint i)
{
  v_seq_boolean_insert (GUPPI_SEQ_BOOLEAN_IMPL (impl), i, FALSE);
  guppi_seq_impl_set_missing (impl, i, TRUE);
}

static void
v_seq_delete_many (GuppiSeqImpl * impl, gint i, gsize N)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  guint32 *data = (guint32 *) guppi_garray_data (core->garray);
  gint j, step_small, step_big, top, bottom;

  i -= core->index_basis;

  N = MIN (N, core->size - i);
  if (N == 0)
    return;

  top = ASIZE (core);
  bottom = i >> 5;
  step_big = N >> 5;
  step_small = N & 31;

  if (i & 31) {
    guint32 mask = ((guint32) (~0)) >> (32 - (i & 31));
    guint32 q = data[bottom];
    q &= mask;
    if (bottom + step_big < top)
      q |= (data[bottom + step_big] >> step_small) & ~mask;
    if (bottom + step_big + 1 < top)
      q |= (data[bottom + step_big + 1] << (32 - step_small)) & ~mask;
    data[bottom] = q;
    ++bottom;
  }
  for (j = bottom; j + step_big < top; ++j) {
    data[j] = data[j + step_big] >> step_small;
    if (j + step_big + 1 < top && step_small)
      data[j] |= data[j + step_big + 1] << (32 - step_small);
  }

  core->size -= N;
}

static void
v_seq_grow_to_include (GuppiSeqImpl * impl, gint j0, gint j1)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  gint front_load, back_load;

  if (core->size == 0) {
    v_seq_boolean_insert_many (GUPPI_SEQ_BOOLEAN_IMPL (impl),
			       0, FALSE, j1 - j0 + 1);
    core->index_basis = j0;
    return;
  }

  front_load = core->index_basis - j0;
  back_load = j1 - core->index_basis - (gint) core->size + 1;

  /* Grow to the front */
  if (0 < front_load) {
    v_seq_boolean_insert_many (GUPPI_SEQ_BOOLEAN_IMPL (impl),
			       core->index_basis, FALSE, front_load);
    core->index_basis = j0;
  }
  /* Grow the back */
  if (0 < back_load) {
    v_seq_boolean_insert_many (GUPPI_SEQ_BOOLEAN_IMPL (impl),
			       core->index_basis + (gint) core->size,
			       FALSE, back_load);
  }
}

static GuppiDataImpl *
v_data_copy (GuppiDataImpl * impl)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  GuppiSeqBooleanCoreImpl *copy;
  gint i;
  guint32 *data;
  guint32 *copy_data;

  copy =
    GUPPI_SEQ_BOOLEAN_CORE_IMPL (guppi_type_new
				 (GUPPI_TYPE_SEQ_BOOLEAN_CORE_IMPL));

  copy->index_basis = core->index_basis;
  copy->size = core->size;
  copy->garray = guppi_garray_new (sizeof (guint32));
  guppi_garray_set_size (copy->garray, ASIZE (copy));

  data = (guint32 *) guppi_garray_data (core->garray);
  copy_data = (guint32 *) guppi_garray_data (copy->garray);
  for (i = 0; i < ASIZE (core); ++i)
    copy_data[i] = data[i];

  return GUPPI_DATA_IMPL (copy);

}

static gint
v_data_size_in_bytes (GuppiDataImpl * impl)
{
  GuppiSeqBooleanCoreImpl *core = GUPPI_SEQ_BOOLEAN_CORE_IMPL (impl);
  return guppi_garray_size (core->garray) * sizeof (guint32) +
    sizeof (GuppiSeqBooleanCoreImpl);
}

/***************************************************************************/

static void
guppi_seq_boolean_core_impl_class_init (GuppiSeqBooleanCoreImplClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiDataImplClass *data_class = GUPPI_DATA_IMPL_CLASS (klass);
  GuppiSeqImplClass *seq_class = GUPPI_SEQ_IMPL_CLASS (klass);
  GuppiSeqBooleanImplClass *seq_boolean_class =
    GUPPI_SEQ_BOOLEAN_IMPL_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_SEQ_BOOLEAN_IMPL);

  object_class->get_arg = guppi_seq_boolean_core_impl_get_arg;
  object_class->set_arg = guppi_seq_boolean_core_impl_set_arg;
  object_class->finalize = guppi_seq_boolean_core_impl_finalize;

  data_class->impl_name = _("Core Boolean Sequence");

  seq_boolean_class->get = v_seq_boolean_get;
  seq_boolean_class->set = v_seq_boolean_set;
  seq_boolean_class->set_all = v_seq_boolean_set_all;
  seq_boolean_class->set_many = v_seq_boolean_set_many;
  seq_boolean_class->insert = v_seq_boolean_insert;
  seq_boolean_class->insert_many = v_seq_boolean_insert_many;
  seq_boolean_class->bitwise_and = v_seq_boolean_bitwise_and;
  seq_boolean_class->bitwise_or = v_seq_boolean_bitwise_or;
  seq_boolean_class->bitwise_xor = v_seq_boolean_bitwise_xor;
  seq_boolean_class->bitwise_not = v_seq_boolean_bitwise_not;
  seq_boolean_class->next_true = v_seq_boolean_next_true;
  seq_boolean_class->true_count = v_seq_boolean_true_count;
  seq_boolean_class->range_query = v_seq_boolean_range_query;

  seq_class->size_hint = v_seq_size_hint;
  seq_class->get_bounds = v_seq_get_bounds;
  seq_class->shift_indices = v_seq_shift_indices;
  seq_class->validate = v_seq_validate;
  seq_class->get = v_seq_get;
  seq_class->set = v_seq_set;
  seq_class->insert = v_seq_insert;
  seq_class->delete_many = v_seq_delete_many;
  seq_class->grow_to_include = v_seq_grow_to_include;

  data_class->copy = v_data_copy;
  data_class->get_size_in_bytes = v_data_size_in_bytes;
}

static void
guppi_seq_boolean_core_impl_init (GuppiSeqBooleanCoreImpl * obj)
{
  obj->index_basis = 0;
  obj->size = 0;
  obj->garray = guppi_garray_new (sizeof (guint32));
}

GtkType guppi_seq_boolean_core_impl_get_type (void)
{
  static GtkType guppi_seq_boolean_core_impl_type = 0;
  if (!guppi_seq_boolean_core_impl_type) {
    static const GtkTypeInfo guppi_seq_boolean_core_impl_info = {
      "GuppiSeqBooleanCoreImpl",
      sizeof (GuppiSeqBooleanCoreImpl),
      sizeof (GuppiSeqBooleanCoreImplClass),
      (GtkClassInitFunc) guppi_seq_boolean_core_impl_class_init,
      (GtkObjectInitFunc) guppi_seq_boolean_core_impl_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_seq_boolean_core_impl_type =
      gtk_type_unique (GUPPI_TYPE_SEQ_BOOLEAN_IMPL,
		       &guppi_seq_boolean_core_impl_info);
  }
  return guppi_seq_boolean_core_impl_type;
}

/**************************************************************************/

static GuppiDataImpl *
make_impl (void)
{
  return
    GUPPI_DATA_IMPL (guppi_type_new (guppi_seq_boolean_core_impl_get_type ()));
}

GuppiPlugIn *guppi_plug_in (void);

GuppiPlugIn *
guppi_plug_in (void)
{
  GuppiPlugIn *pi;
  GuppiDataImplPlugIn *dimpi;

  pi = guppi_data_impl_plug_in_new ();
  dimpi = GUPPI_DATA_IMPL_PLUG_IN (pi);

  pi->magic_number = GUPPI_PLUG_IN_MAGIC_NUMBER;
  dimpi->impl_constructor = make_impl;

  return pi;
}


/* $Id: guppi-seq-boolean-core-impl.c,v 1.11 2001/05/06 07:56:15 trow Exp $ */
