/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-category-core-impl.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
/* #include <gnome.h> */
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-convenient.h>
#include <guppi-data-impl-plug-in.h>
#include "guppi-category-core-impl.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_category_core_impl_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_category_core_impl_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_category_core_impl_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}



static void
guppi_category_core_impl_finalize (GtkObject * obj)
{
  GuppiCategoryCoreImpl *core = GUPPI_CATEGORY_CORE_IMPL (obj);

  g_hash_table_foreach (core->name2code, guppi_free_hash_key, NULL);

  g_hash_table_destroy (core->name2code);
  g_hash_table_destroy (core->code2name);

  core->name2code = core->code2name = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/***************************************************************************/

static GuppiDataImpl *
v_copy (GuppiDataImpl * impl_to_copy)
{
  g_assert_not_reached ();
  return NULL;
}

static gint
v_get_size_in_bytes (GuppiDataImpl * impl)
{
  gint sz = 0;

  sz += sizeof (GuppiCategoryCoreImpl);

  return sz;
}

static void
v_get_size_info (GuppiDataImpl * impl, gchar * str, gsize N)
{
  GuppiCategoryCoreImpl *core = GUPPI_CATEGORY_CORE_IMPL (impl);
  g_snprintf (str, N, "%d", g_hash_table_size (core->code2name));
}

static gsize
v_size (GuppiCategoryImpl * impl)
{
  GuppiCategoryCoreImpl *core = GUPPI_CATEGORY_CORE_IMPL (impl);

  return g_hash_table_size (core->code2name);
}

static void
v_codes (GuppiCategoryImpl * impl, code_t * min, code_t * max,
	 code_t * unused)
{
  GuppiCategoryCoreImpl *core = GUPPI_CATEGORY_CORE_IMPL (impl);

  if (min)
    *min = core->min_code;

  if (max)
    *max = core->max_code;

  if (unused)
    *unused = core->max_code == GUPPI_INVALID_CODE ? 0 : core->max_code + 1;
}

static void
v_define (GuppiCategoryImpl * impl, gchar * str, code_t c)
{
  GuppiCategoryCoreImpl *core = GUPPI_CATEGORY_CORE_IMPL (impl);
  gpointer old_key;
  gpointer cptr = GUPPI_CATEGORY_CODE_TO_POINTER (c);

  /* The hashes could get badly out of sync if this function is called
     w/o sufficient checks. */

  /* Make sure we don't leak any old keys. */
  if (g_hash_table_lookup_extended (core->code2name, cptr, NULL, &old_key))
    guppi_free (old_key);

  g_hash_table_remove (core->name2code, str);
  g_hash_table_remove (core->code2name, cptr);

  g_hash_table_insert (core->name2code, str, cptr);
  g_hash_table_insert (core->code2name, cptr, str);

  if (core->min_code == GUPPI_INVALID_CODE || core->min_code > c)
    core->min_code = c;

  if (core->max_code == GUPPI_INVALID_CODE || core->max_code < c)
    core->max_code = c;
}

static const gchar *
v_code2name (GuppiCategoryImpl * impl, code_t c)
{
  GuppiCategoryCoreImpl *core = GUPPI_CATEGORY_CORE_IMPL (impl);
  gpointer cptr = GUPPI_CATEGORY_CODE_TO_POINTER (c);

  return (const gchar *) g_hash_table_lookup (core->code2name, cptr);
}

static code_t
v_name2code (GuppiCategoryImpl * impl, const gchar * str)
{
  GuppiCategoryCoreImpl *core = GUPPI_CATEGORY_CORE_IMPL (impl);
  gpointer cptr;

  if (g_hash_table_lookup_extended (core->name2code, str, NULL, &cptr))
    return GUPPI_CATEGORY_POINTER_TO_CODE (cptr);
  else
    return GUPPI_INVALID_CODE;
}

struct category_foreach {
  GuppiCategoryFn fn;
  gpointer user_data;
};

static void
foreach_handler (gpointer key, gpointer val, gpointer data)
{
  struct category_foreach *foo = (struct category_foreach *) data;

  foo->fn ((const gchar *) key,
	   GUPPI_CATEGORY_POINTER_TO_CODE (val), foo->user_data);
}

static void
v_foreach (GuppiCategoryImpl * impl, GuppiCategoryFn fn, gpointer user_data)
{
  GuppiCategoryCoreImpl *core = GUPPI_CATEGORY_CORE_IMPL (impl);
  struct category_foreach foo = { fn, user_data };

  g_hash_table_foreach (core->name2code, foreach_handler, &foo);
}

/***************************************************************************/

static void
guppi_category_core_impl_class_init (GuppiCategoryCoreImplClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiDataImplClass *data_class = GUPPI_DATA_IMPL_CLASS (klass);
  GuppiCategoryImplClass *category_class = GUPPI_CATEGORY_IMPL_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_CATEGORY_IMPL);

  object_class->get_arg = guppi_category_core_impl_get_arg;
  object_class->set_arg = guppi_category_core_impl_set_arg;
  object_class->destroy = guppi_category_core_impl_destroy;
  object_class->finalize = guppi_category_core_impl_finalize;

  data_class->impl_name = _("Core Category");

  data_class->copy = v_copy;
  data_class->get_size_in_bytes = v_get_size_in_bytes;
  data_class->get_size_info = v_get_size_info;

  category_class->size = v_size;
  category_class->codes = v_codes;
  category_class->define = v_define;
  category_class->code2name = v_code2name;
  category_class->name2code = v_name2code;
  category_class->foreach = v_foreach;

}

static void
guppi_category_core_impl_init (GuppiCategoryCoreImpl * obj)
{
  obj->min_code = GUPPI_INVALID_CODE;
  obj->max_code = GUPPI_INVALID_CODE;

  obj->name2code = g_hash_table_new (g_str_hash, g_str_equal);
  obj->code2name = g_hash_table_new (NULL, NULL);
}

GtkType guppi_category_core_impl_get_type (void)
{
  static GtkType guppi_category_core_impl_type = 0;
  if (!guppi_category_core_impl_type) {
    static const GtkTypeInfo guppi_category_core_impl_info = {
      "GuppiCategoryCoreImpl",
      sizeof (GuppiCategoryCoreImpl),
      sizeof (GuppiCategoryCoreImplClass),
      (GtkClassInitFunc) guppi_category_core_impl_class_init,
      (GtkObjectInitFunc) guppi_category_core_impl_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_category_core_impl_type =
      gtk_type_unique (GUPPI_TYPE_CATEGORY_IMPL,
		       &guppi_category_core_impl_info);
  }
  return guppi_category_core_impl_type;
}

GtkObject *
guppi_category_core_impl_new (void)
{
  return GTK_OBJECT (guppi_type_new (guppi_category_core_impl_get_type ()));
}

/***************************************************************************/

static GuppiDataImpl *
make_impl (void)
{
  return
    GUPPI_DATA_IMPL (guppi_type_new (guppi_category_core_impl_get_type ()));
}

GuppiPlugIn *guppi_plug_in (void);

GuppiPlugIn *
guppi_plug_in (void)
{
  GuppiPlugIn *pi;
  GuppiDataImplPlugIn *dimpi;

  pi = guppi_data_impl_plug_in_new ();
  dimpi = GUPPI_DATA_IMPL_PLUG_IN (pi);

  pi->magic_number = GUPPI_PLUG_IN_MAGIC_NUMBER;
  dimpi->impl_constructor = make_impl;

  return pi;
}
