/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-category.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 * Copyright (C) 2001 The Free Software Foundation
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
/* #include <gnome.h> */
#include <stdlib.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-convenient.h>
#include "guppi-category.h"
#include "guppi-category-impl.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_category_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_category_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_category_finalize (GtkObject * obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

struct OutPair {
  GuppiXMLDocument *doc;
  xmlNodePtr node;
};

static void
xml_cont_fn (const gchar *str, code_t code, gpointer data)
{
  struct OutPair *pair = (struct OutPair *) data;
  gchar buf[64];
  xmlNodePtr cat_node;

  cat_node = xmlNewNode (pair->doc->ns, "Category");

  xmlNewChild (cat_node, pair->doc->ns, "name", str);
  
  g_snprintf (buf, 64, "%d", (gint) code);
  xmlNewChild (cat_node, pair->doc->ns, "code", buf);
  
  xmlAddChild (pair->node, cat_node);
}

static xmlNodePtr
export_xml_content (GuppiData *data, GuppiXMLDocument *doc)
{
  struct OutPair pair;
  
  pair.doc = doc;
  pair.node = xmlNewNode (doc->ns, "Content");
  
  guppi_category_foreach (GUPPI_CATEGORY (data), xml_cont_fn, &pair);

  return pair.node;
}


static void
guppi_category_class_init (GuppiCategoryClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiDataClass *data_class = GUPPI_DATA_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_DATA);

  data_class->type_name = _("Category");
  data_class->impl_type = GUPPI_TYPE_CATEGORY_IMPL;
  data_class->export_xml_content = export_xml_content;

  object_class->get_arg = guppi_category_get_arg;
  object_class->set_arg = guppi_category_set_arg;
  object_class->finalize = guppi_category_finalize;

}

static void
guppi_category_init (GuppiCategory * obj)
{

}

GtkType guppi_category_get_type (void)
{
  static GtkType guppi_category_type = 0;
  if (!guppi_category_type) {
    static const GtkTypeInfo guppi_category_info = {
      "GuppiCategory",
      sizeof (GuppiCategory),
      sizeof (GuppiCategoryClass),
      (GtkClassInitFunc) guppi_category_class_init,
      (GtkObjectInitFunc) guppi_category_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_category_type =
      gtk_type_unique (GUPPI_TYPE_DATA, &guppi_category_info);
  }
  return guppi_category_type;
}

GuppiData *
guppi_category_new (void)
{
  return guppi_data_newv (GUPPI_TYPE_CATEGORY, NULL, 0, NULL);
}

/**************************************************************************/

/* Data Op Stuff */

typedef struct _GuppiDataOp_Category GuppiDataOp_Category;
struct _GuppiDataOp_Category {
  GuppiDataOp op;

  gchar *name;
  code_t code;
};

static void
op_define (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Category *cat_op = (GuppiDataOp_Category *) op;

  GuppiCategoryImpl *impl;
  GuppiCategoryImplClass *impl_class;

  impl = GUPPI_CATEGORY_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_CATEGORY_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->define);
  impl_class->define (impl, cat_op->name, cat_op->code);
}

/***************************************************************************/

gsize guppi_category_size (GuppiCategory * cat)
{
  GuppiCategoryImpl *impl;
  GuppiCategoryImplClass *impl_class;

  g_return_val_if_fail (cat != NULL && GUPPI_IS_CATEGORY (cat), 0);

  impl = GUPPI_CATEGORY_IMPL (guppi_data_impl (GUPPI_DATA (cat)));
  impl_class = GUPPI_CATEGORY_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->size);
  return impl_class->size (impl);
}

code_t guppi_category_min_code (GuppiCategory * cat)
{
  GuppiCategoryImpl *impl;
  GuppiCategoryImplClass *impl_class;
  code_t c = GUPPI_INVALID_CODE;

  g_return_val_if_fail (cat != NULL && GUPPI_IS_CATEGORY (cat), 0);

  impl = GUPPI_CATEGORY_IMPL (guppi_data_impl (GUPPI_DATA (cat)));
  impl_class = GUPPI_CATEGORY_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->codes);
  impl_class->codes (impl, &c, NULL, NULL);

  return c;
}

code_t guppi_category_max_code (GuppiCategory * cat)
{
  GuppiCategoryImpl *impl;
  GuppiCategoryImplClass *impl_class;
  code_t c = GUPPI_INVALID_CODE;

  g_return_val_if_fail (cat != NULL && GUPPI_IS_CATEGORY (cat), 0);

  impl = GUPPI_CATEGORY_IMPL (guppi_data_impl (GUPPI_DATA (cat)));
  impl_class = GUPPI_CATEGORY_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->codes);
  impl_class->codes (impl, NULL, &c, NULL);

  return c;
}

code_t guppi_category_unused_code (GuppiCategory * cat)
{
  GuppiCategoryImpl *impl;
  GuppiCategoryImplClass *impl_class;
  code_t mc = GUPPI_INVALID_CODE;
  code_t c = GUPPI_INVALID_CODE;

  g_return_val_if_fail (cat != NULL && GUPPI_IS_CATEGORY (cat), 0);

  impl = GUPPI_CATEGORY_IMPL (guppi_data_impl (GUPPI_DATA (cat)));
  impl_class = GUPPI_CATEGORY_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->codes);
  impl_class->codes (impl, NULL, &mc, &c);

  if (c == GUPPI_INVALID_CODE && mc != GUPPI_INVALID_CODE)
    c = mc + 1;

  return c;
}

gboolean
guppi_category_define (GuppiCategory * cat, const gchar * str, code_t c)
{
  GuppiCategoryImpl *impl;
  GuppiCategoryImplClass *impl_class;
  GuppiDataOp_Category op;

  g_return_val_if_fail (cat != NULL && GUPPI_IS_CATEGORY (cat), FALSE);
  g_return_val_if_fail (str != NULL, FALSE);
  g_return_val_if_fail (c != GUPPI_INVALID_CODE, FALSE);
  g_return_val_if_fail (guppi_data_can_change (GUPPI_DATA (cat)), FALSE);

  impl = GUPPI_CATEGORY_IMPL (guppi_data_impl (GUPPI_DATA (cat)));
  impl_class = GUPPI_CATEGORY_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->code2name && impl_class->name2code);

  if (impl_class->code2name (impl, c) != NULL ||
      impl_class->name2code (impl, str) != GUPPI_INVALID_CODE)
    return FALSE;

  op.op.op = op_define;
  op.name = guppi_strdup (str);
  op.code = c;

  guppi_data_add_pending_op (GUPPI_DATA (cat), (GuppiDataOp *) & op);
  guppi_data_changed (GUPPI_DATA (cat));

  return TRUE;
}

code_t guppi_category_add_by_name (GuppiCategory * cat, const gchar * str)
{
  code_t c;

  g_return_val_if_fail (cat != NULL && GUPPI_IS_CATEGORY (cat),
			GUPPI_INVALID_CODE);
  g_return_val_if_fail (str != NULL, GUPPI_INVALID_CODE);
  g_return_val_if_fail (guppi_data_can_change (GUPPI_DATA (cat)),
			GUPPI_INVALID_CODE);

  c = guppi_category_unused_code (cat);
  g_return_val_if_fail (c != GUPPI_INVALID_CODE, GUPPI_INVALID_CODE);

  if (!guppi_category_define (cat, str, c))
    c = GUPPI_INVALID_CODE;

  return c;
}

gboolean guppi_category_contains (GuppiCategory * cat, const gchar * str)
{
  GuppiCategoryImpl *impl;
  GuppiCategoryImplClass *impl_class;

  g_return_val_if_fail (cat != NULL && GUPPI_IS_CATEGORY (cat), FALSE);
  g_return_val_if_fail (str != NULL, FALSE);

  impl = GUPPI_CATEGORY_IMPL (guppi_data_impl (GUPPI_DATA (cat)));
  impl_class = GUPPI_CATEGORY_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->name2code);

  return impl_class->name2code (impl, str) != GUPPI_INVALID_CODE;
}

gboolean guppi_category_contains_code (GuppiCategory * cat, code_t c)
{
  GuppiCategoryImpl *impl;
  GuppiCategoryImplClass *impl_class;

  g_return_val_if_fail (cat != NULL && GUPPI_IS_CATEGORY (cat), FALSE);
  g_return_val_if_fail (c != GUPPI_INVALID_CODE, FALSE);

  impl = GUPPI_CATEGORY_IMPL (guppi_data_impl (GUPPI_DATA (cat)));
  impl_class = GUPPI_CATEGORY_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->code2name);

  return impl_class->code2name (impl, c) != NULL;
}

code_t guppi_category_find_by_name (GuppiCategory * cat, const gchar * str)
{
  GuppiCategoryImpl *impl;
  GuppiCategoryImplClass *impl_class;

  g_return_val_if_fail (cat != NULL && GUPPI_IS_CATEGORY (cat),
			GUPPI_INVALID_CODE);
  g_return_val_if_fail (str != NULL, GUPPI_INVALID_CODE);

  impl = GUPPI_CATEGORY_IMPL (guppi_data_impl (GUPPI_DATA (cat)));
  impl_class = GUPPI_CATEGORY_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->name2code);

  return impl_class->name2code (impl, str);
}

const gchar *
guppi_category_find_by_code (GuppiCategory * cat, code_t c)
{
  GuppiCategoryImpl *impl;
  GuppiCategoryImplClass *impl_class;

  g_return_val_if_fail (cat != NULL && GUPPI_IS_CATEGORY (cat), NULL);
  g_return_val_if_fail (c != GUPPI_INVALID_CODE, NULL);

  impl = GUPPI_CATEGORY_IMPL (guppi_data_impl (GUPPI_DATA (cat)));
  impl_class = GUPPI_CATEGORY_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->code2name);

  return impl_class->code2name (impl, c);
}

gboolean
guppi_category_rename (GuppiCategory * cat, code_t c, const gchar * str)
{
  GuppiCategoryImpl *impl;
  GuppiCategoryImplClass *impl_class;
  GuppiDataOp_Category op;
  code_t alt_code;
  const gchar *old_str;

  g_return_val_if_fail (cat && GUPPI_IS_CATEGORY (cat), FALSE);
  g_return_val_if_fail (c != GUPPI_INVALID_CODE, FALSE);
  g_return_val_if_fail (str != NULL, FALSE);
  g_return_val_if_fail (guppi_data_can_change (GUPPI_DATA (cat)), FALSE);

  impl = GUPPI_CATEGORY_IMPL (guppi_data_impl (GUPPI_DATA (cat)));
  impl_class = GUPPI_CATEGORY_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->code2name && impl_class->name2code);

  old_str = impl_class->code2name (impl, c);

  /* Bad code passed in */
  if (old_str == NULL)
    return FALSE;

  /* Old string == new string, so we don't have to do anything. */
  if (!strcmp (old_str, str))
    return TRUE;

  alt_code = impl_class->name2code (impl, str);

  /* Oops... the new name is already used. */
  if (alt_code != GUPPI_INVALID_CODE)
    return FALSE;

  op.op.op = op_define;
  op.name = guppi_strdup (str);
  op.code = c;

  guppi_data_add_pending_op (GUPPI_DATA (cat), (GuppiDataOp *) & op);
  guppi_data_changed (GUPPI_DATA (cat));

  return TRUE;
}

void
guppi_category_foreach (GuppiCategory * cat, GuppiCategoryFn fn,
			gpointer user_data)
{
  GuppiCategoryImpl *impl;
  GuppiCategoryImplClass *impl_class;

  g_return_if_fail (cat != NULL && GUPPI_IS_CATEGORY (cat));
  g_return_if_fail (fn != NULL);

  impl = GUPPI_CATEGORY_IMPL (guppi_data_impl (GUPPI_DATA (cat)));
  impl_class = GUPPI_CATEGORY_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->foreach);
  impl_class->foreach (impl, fn, user_data);
}
