/* This is -*- C -*- */
/* $Id: guppi-config-item.c,v 1.6 2001/01/16 23:36:39 trow Exp $ */

/*
 * guppi-config-item.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <glib.h>
#include <gtk/gtkframe.h>

#include "guppi-convenient.h"
#include "guppi-config-item.h"

typedef struct _GuppiConfigItemPrivate GuppiConfigItemPrivate;
struct _GuppiConfigItemPrivate {

  gchar *short_name;
  gchar *name;

  GtkCTreeNode *ctree_node;

  GtkWidget *(*widget_constructor) (gpointer);
  GtkWidget *widget;
  gpointer user_data;

  gboolean initial_expand;
  gconstpointer tag;

  GuppiConfigItem *parent;
  GuppiConfigItem *prev;
  GuppiConfigItem *next;
  GuppiConfigItem *child;
};

#define priv(x) ((GuppiConfigItemPrivate*)((x)->opaque_internals))

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_config_item_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_config_item_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_config_item_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_config_item_finalize (GtkObject * obj)
{
  GuppiConfigItem *item = GUPPI_CONFIG_ITEM (obj);
  GuppiConfigItemPrivate *p = priv (item);

  guppi_finalized (obj);

  guppi_free (p->name);
  guppi_free (p->short_name);

  guppi_unref0 (p->parent);
  guppi_unref0 (p->prev);

  guppi_free (item->opaque_internals);
  item->opaque_internals = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
guppi_config_item_class_init (GuppiConfigItemClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  parent_class = gtk_type_class (GTK_TYPE_OBJECT);

  object_class->get_arg = guppi_config_item_get_arg;
  object_class->set_arg = guppi_config_item_set_arg;
  object_class->destroy = guppi_config_item_destroy;
  object_class->finalize = guppi_config_item_finalize;

}

static void
guppi_config_item_init (GuppiConfigItem * obj)
{
  obj->opaque_internals = guppi_new0 (GuppiConfigItemPrivate, 1);
}

GtkType
guppi_config_item_get_type (void)
{
  static GtkType guppi_config_item_type = 0;
  if (!guppi_config_item_type) {
    static const GtkTypeInfo guppi_config_item_info = {
      "GuppiConfigItem",
      sizeof (GuppiConfigItem),
      sizeof (GuppiConfigItemClass),
      (GtkClassInitFunc) guppi_config_item_class_init,
      (GtkObjectInitFunc) guppi_config_item_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_config_item_type =
      gtk_type_unique (GTK_TYPE_OBJECT, &guppi_config_item_info);
  }
  return guppi_config_item_type;
}

static GtkWidget *
fallback (gpointer foo)
{
  gchar *str;
  GtkWidget *w;

  str = guppi_strdup_printf ("Widget constructor for %s is NULL", (gchar *) foo);
  w = gtk_label_new (str);
  guppi_free (str);
  return w;
}

GuppiConfigItem *
guppi_config_item_new (const gchar * name,
		       const gchar * short_name,
		       GtkWidget * (*fn) (gpointer), gpointer user_data)
{
  GuppiConfigItem *item;
  GuppiConfigItemPrivate *p;

  g_return_val_if_fail (name != NULL, NULL);
  g_return_val_if_fail (short_name != NULL, NULL);

  item = GUPPI_CONFIG_ITEM (guppi_type_new (guppi_config_item_get_type ()));
  p = priv (item);

  p->name = guppi_strdup (name);
  p->short_name = guppi_strdup (short_name);
  if (fn == NULL)
    g_warning ("Widget constructor for %s is NULL", name);
  p->widget_constructor = fn ? fn : fallback;
  p->user_data = fn ? user_data : p->name;

  return item;
}

const gchar *
guppi_config_item_name (GuppiConfigItem * item)
{
  GuppiConfigItemPrivate *p;
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), NULL);
  p = priv (item);
  return p->name;
}

const gchar *
guppi_config_item_short_name (GuppiConfigItem * item)
{
  GuppiConfigItemPrivate *p;
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), NULL);
  p = priv (item);
  return p->short_name;
}

gboolean
guppi_config_item_has_widget (GuppiConfigItem * item)
{
  GuppiConfigItemPrivate *p;
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), FALSE);
  p = priv (item);

  return p->widget != NULL;
}

GtkWidget *
guppi_config_item_widget (GuppiConfigItem * item)
{
  GuppiConfigItemPrivate *p;
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), NULL);
  p = priv (item);

  if (p->widget == NULL) {
    GtkWidget *w;
    GtkWidget *fr;

    fr = gtk_frame_new (p->name);
    w = p->widget_constructor (p->user_data);
    gtk_container_set_border_width (GTK_CONTAINER (fr), 3);
    gtk_container_add (GTK_CONTAINER (fr), w);

    gtk_widget_show_all (fr);

    p->widget = fr;
  }

  return p->widget;
}

gboolean
guppi_config_item_expand_initially (GuppiConfigItem * item)
{
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), FALSE);
  return priv (item)->initial_expand;
}

void
guppi_config_item_set_initial_expansion (GuppiConfigItem * item, gboolean x)
{
  g_return_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item));
  priv (item)->initial_expand = x;
}

gconstpointer
guppi_config_item_tag (GuppiConfigItem * item)
{
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), NULL);
  return priv (item)->tag;
}

void
guppi_config_item_set_tag (GuppiConfigItem * item, gconstpointer tag)
{
  g_return_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item));
  priv (item)->tag = tag;
}

GuppiConfigItem *
guppi_config_item_prev (GuppiConfigItem * item)
{
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), NULL);
  return priv (item)->prev;
}

GuppiConfigItem *
guppi_config_item_next (GuppiConfigItem * item)
{
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), NULL);
  return priv (item)->next;
}

GuppiConfigItem *
guppi_config_item_parent (GuppiConfigItem * item)
{
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), NULL);
  return priv (item)->parent;
}

GuppiConfigItem *
guppi_config_item_first_sibling (GuppiConfigItem * item)
{
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), NULL);

  while (priv (item)->prev)
    item = priv (item)->prev;

  return item;
}

GuppiConfigItem *
guppi_config_item_last_sibling (GuppiConfigItem * item)
{
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), NULL);

  while (priv (item)->next)
    item = priv (item)->next;

  return item;
}

GuppiConfigItem *
guppi_config_item_first_child (GuppiConfigItem * item)
{
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), NULL);
  return priv (item)->child;
}

GuppiConfigItem *
guppi_config_item_last_child (GuppiConfigItem * item)
{
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), NULL);

  item = guppi_config_item_first_child (item);
  if (item)
    item = guppi_config_item_last_sibling (item);

  return item;
}

void
guppi_config_item_add_before (GuppiConfigItem * item, GuppiConfigItem * add)
{
  GuppiConfigItem *first_sib;
  GuppiConfigItem *last_sib;
  GuppiConfigItemPrivate *p;
  GuppiConfigItemPrivate *f;
  GuppiConfigItemPrivate *l;

  GuppiConfigItem *iter;

  g_return_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item));
  g_return_if_fail (add == NULL || GUPPI_IS_CONFIG_ITEM (add));
  g_return_if_fail (add == NULL || GTK_OBJECT_FLOATING (add));

  if (add == NULL)
    return;

  first_sib = guppi_config_item_first_sibling (add);
  last_sib = guppi_config_item_last_sibling (add);

  p = priv (item);
  f = priv (first_sib);
  l = priv (last_sib);

  for (iter = first_sib; iter != NULL; iter = guppi_config_item_next (iter)) {
    guppi_refcounting_assign (priv (iter)->parent, p->parent);
  }

  guppi_refcounting_assign (l->next, item);

  guppi_refcounting_assign (f->prev, p->prev);

  if (p->prev) {
    guppi_refcounting_assign (priv (p->prev)->next, first_sib);
  }

  guppi_refcounting_assign (p->prev, last_sib);

  if (p->parent && priv (p->parent)->child == item) {
    guppi_refcounting_assign (priv (p->parent)->child, first_sib)
  }
}

void
guppi_config_item_add_after (GuppiConfigItem * item, GuppiConfigItem * add)
{
  GuppiConfigItem *first_sib;
  GuppiConfigItem *last_sib;
  GuppiConfigItemPrivate *p;
  GuppiConfigItemPrivate *f;
  GuppiConfigItemPrivate *l;

  GuppiConfigItem *iter;

  g_return_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item));
  g_return_if_fail (add == NULL || GUPPI_IS_CONFIG_ITEM (add));
  g_return_if_fail (add == NULL || GTK_OBJECT_FLOATING (add));

  if (add == NULL)
    return;

  first_sib = guppi_config_item_first_sibling (add);
  last_sib = guppi_config_item_last_sibling (add);

  p = priv (item);
  f = priv (first_sib);
  l = priv (last_sib);

  for (iter = first_sib; iter != NULL; iter = guppi_config_item_next (iter)) {
    guppi_refcounting_assign (priv (iter)->parent, p->parent);
  }

  guppi_refcounting_assign (f->prev, item);

  guppi_refcounting_assign (l->next, p->next);

  if (p->next) {
    guppi_refcounting_assign (priv (p->next)->prev, last_sib);
  }

  guppi_refcounting_assign (p->next, first_sib);
}

void
guppi_config_item_add_below (GuppiConfigItem * item, GuppiConfigItem * add)
{
  GuppiConfigItem *first_sib;
  GuppiConfigItemPrivate *p;
  GuppiConfigItemPrivate *a;
  GuppiConfigItemPrivate *f;

  GuppiConfigItem *iter;

  g_return_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item));
  g_return_if_fail (add == NULL || GUPPI_IS_CONFIG_ITEM (add));
  g_return_if_fail (add == NULL || GTK_OBJECT_FLOATING (add));

  if (add == NULL)
    return;

  first_sib = guppi_config_item_first_sibling (add);

  p = priv (item);
  a = priv (add);
  f = priv (first_sib);

  if (p->child) {

    guppi_config_item_add_before (p->child, first_sib);

  } else {

    for (iter = first_sib; iter; iter = guppi_config_item_next (iter)) {
      guppi_refcounting_assign (priv (iter)->parent, item);
    }

    guppi_refcounting_assign (p->child, first_sib);
  }
}

void
guppi_config_item_append (GuppiConfigItem * item, GuppiConfigItem * add)
{
  g_return_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item));
  g_return_if_fail (add == NULL || GUPPI_IS_CONFIG_ITEM (add));

  if (add == NULL)
    return;

  item = guppi_config_item_last_sibling (item);
  guppi_config_item_add_after (item, add);
}

void
guppi_config_item_append_below (GuppiConfigItem * item, GuppiConfigItem * add)
{
  GuppiConfigItem *child;

  g_return_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item));
  g_return_if_fail (add == NULL || GUPPI_IS_CONFIG_ITEM (add));

  if (add == NULL)
    return;

  child = guppi_config_item_last_child (item);

  if (child)
    guppi_config_item_append (child, add);
  else
    guppi_config_item_add_below (item, add);
}

void
guppi_config_item_destroy_tree (GuppiConfigItem * item)
{
  GuppiConfigItemPrivate *p;

  g_return_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item));

  p = priv (item);

  if (p->next) 
    guppi_config_item_destroy_tree (p->next);
  p->next = NULL;

  if (p->child) 
    guppi_config_item_destroy_tree (p->child);
  p->child = NULL;

  guppi_unref0 (item);
}

void
guppi_config_item_foreach (GuppiConfigItem * item,
			   void (*fn) (GuppiConfigItem *, gpointer),
			   gpointer user_data)
{
  GuppiConfigItemPrivate *p;

  g_return_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item));
  g_return_if_fail (fn != NULL);

  p = priv (item);

  fn (item, user_data);

  if (p->next)
    guppi_config_item_foreach (p->next, fn, user_data);
  if (p->child)
    guppi_config_item_foreach (p->child, fn, user_data);
}

GuppiConfigItem *
guppi_config_item_find_similar (GuppiConfigItem * root,
				GuppiConfigItem * item)
{
  GuppiConfigItemPrivate *pr;
  GuppiConfigItemPrivate *p;

  g_return_val_if_fail (root != NULL && GUPPI_IS_CONFIG_ITEM (root), NULL);
  g_return_val_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item), NULL);

  pr = priv (root);
  p = priv (item);

  if (p->widget_constructor == pr->widget_constructor &&
      p->user_data == pr->user_data) return root;

  if (pr->next)
    guppi_config_item_find_similar (pr->next, item);

  if (pr->child)
    guppi_config_item_find_similar (pr->child, item);

  return NULL;
}

GuppiConfigItem *
guppi_config_item_find_by_tag (GuppiConfigItem * root, gconstpointer tag)
{
  GuppiConfigItemPrivate *p;
  GuppiConfigItem *item;

  g_return_val_if_fail (root != NULL && GUPPI_IS_CONFIG_ITEM (root), NULL);

  p = priv (root);
  if (p->tag == tag)
    return root;

  if (p->next) {
    item = guppi_config_item_find_by_tag (p->next, tag);
    if (item)
      return item;
  }

  if (p->child) {
    item = guppi_config_item_find_by_tag (p->child, tag);
    if (item)
      return item;
  }

  return NULL;
}

/* This is very inefficient */

static void
xfer_iter_fn (GuppiConfigItem * item, gpointer ptr)
{
  GuppiConfigItem *root = GUPPI_CONFIG_ITEM (ptr);
  GuppiConfigItem *match;
  GuppiConfigItemPrivate *p = priv (item);

  match = guppi_config_item_find_similar (root, item);

  if (match) {
    p->widget = priv (match)->widget;
    priv (match)->widget = NULL;
  }
}

void
guppi_config_item_widget_transfer (GuppiConfigItem * dest,
				   GuppiConfigItem * src)
{
  g_return_if_fail (dest != NULL && GUPPI_IS_CONFIG_ITEM (dest));
  g_return_if_fail (src == NULL || GUPPI_IS_CONFIG_ITEM (src));

  if (src == NULL)
    return;

  guppi_config_item_foreach (dest, xfer_iter_fn, src);
}

static void
ctree_iter (GuppiConfigItem * item, GtkCTree * ctree,
	    GtkCTreeNode * parent, GtkCTreeNode * sib)
{
  GuppiConfigItemPrivate *p = priv (item);
  GtkCTreeNode *node;
  GuppiConfigItem *recurse;
  gchar *txt[1];
  txt[0] = (gchar *) guppi_config_item_short_name (item);

  /* Sanity check our node structure */

  if (p->next) {
    g_assert (priv (p->next)->prev == item);
    g_assert (priv (p->next)->parent == p->parent);
  }
  if (p->prev) {
    g_assert (priv (p->prev)->next == item);
    g_assert (priv (p->prev)->parent == p->parent);
  }
  if (p->child) {
    GuppiConfigItem *i = p->child;
    while (i) {
      g_assert (priv (i)->parent == item);
      i = guppi_config_item_next (i);
    }
  }


  node = gtk_ctree_insert_node (ctree, parent, sib,
				txt, 5,
				NULL, NULL, NULL, NULL,
				guppi_config_item_first_child (item) == NULL,
				guppi_config_item_expand_initially (item));

  gtk_ctree_node_set_row_data (ctree, node, item);

  recurse = guppi_config_item_last_child (item);
  if (recurse)
    ctree_iter (recurse, ctree, node, NULL);

  recurse = guppi_config_item_prev (item);
  if (recurse)
    ctree_iter (recurse, ctree, parent, node);
}

void
guppi_config_item_build_ctree (GuppiConfigItem * item, GtkCTree * ctree)
{
  g_return_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item));
  g_return_if_fail (ctree != NULL && GTK_IS_CTREE (ctree));

  ctree_iter (guppi_config_item_last_sibling (item), ctree, NULL, NULL);
}


/* $Id: guppi-config-item.c,v 1.6 2001/01/16 23:36:39 trow Exp $ */
