/* This is -*- C -*- */
/* vim: set sw=2: */
/* $Id: guppi-unique-id.c,v 1.4 2000/12/18 06:04:34 trow Exp $ */

/*
 * guppi-unique-id.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <fcntl.h>
#include <sys/types.h>
#include "guppi-convenient.h"
#include "guppi-unique-id.h"

static void
init_unique_id ()
{
  gint fd;
  unsigned seed = 0;

  /* If we have /dev/random, read some random bytes to use as a seed. */
  fd = open ("/dev/random", O_RDONLY);
  if (fd >= 0) {
    if (read (fd, &seed, sizeof (seed)) != sizeof (seed))
      seed = 0;
    close (fd);
  }

  /* Otherwise, try to generate a quasirandom number from some system info. */
  if (seed == 0)
    seed = (unsigned)time (NULL) * (unsigned)getpid () + (unsigned)getppid ();

  srandom (seed);
}

guppi_uniq_t
guppi_unique_id (void)
{
  gboolean initialized = FALSE;
  guint32 time_part, rand_part;
  guppi_uniq_t uid = 0;
  static guint count_part = 0x0bee;

  if (!initialized) {
    init_unique_id ();
    initialized = TRUE;
  }

  
  time_part = (guint32) time (NULL);
  rand_part = (guint32) random ();

  uid = (((guppi_uniq_t)rand_part) << 32)
    | (guppi_uniq_t)(time_part & 0xffff0000)
    | (guppi_uniq_t)(count_part & 0x0000ffff);

  ++count_part;
  
  return uid;
}

void
guppi_uniq2str (guppi_uniq_t x, gchar *str)
{
  g_snprintf (str, 17, "%x-%x", (guint32)(x >> 32), (guint32)(x & 0xffffffff));
}

guppi_uniq_t
guppi_str2uniq (const gchar *str)
{
  guint32 a, b;
  if (sscanf (str, "%x-%x", &a, &b) != 2)
    return 0;

  return (((guppi_uniq_t)a) << 32) | ((guppi_uniq_t)b);
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

static guint
uniq_hash (gconstpointer key)
{
  guppi_uniq_t x = *(guppi_uniq_t *)key;

  return ((guint)(x & 0xffffffff)) ^ (guint)(x >> 32);
}

static gint
uniq_cmp (gconstpointer a, gconstpointer b)
{
  guppi_uniq_t x = *(guppi_uniq_t *)a;
  guppi_uniq_t y = *(guppi_uniq_t *)b;

  return guppi_uniq_cmp (x, y);
}

GHashTable *
guppi_uniq_table_new (void)
{
  return g_hash_table_new (uniq_hash, uniq_cmp);
}

void
guppi_uniq_table_destroy (GHashTable *hash)
{
  g_return_if_fail (hash);
  g_hash_table_foreach (hash, guppi_free_hash_key, NULL);
  g_hash_table_destroy (hash);
}

void
guppi_uniq_table_add (GHashTable *hash, guppi_uniq_t x)
{
  g_return_if_fail (hash);
  if (!guppi_uniq_table_contains (hash, x)) {
    guppi_uniq_t *p = guppi_new0 (guppi_uniq_t, 1);
    *p = x;
    g_hash_table_insert (hash, p, p);
  }
}

gboolean
guppi_uniq_table_contains (GHashTable *hash, guppi_uniq_t x)
{
  g_return_val_if_fail (hash, FALSE);
  return g_hash_table_lookup (hash, &x) != NULL;
}

/* $Id: guppi-unique-id.c,v 1.4 2000/12/18 06:04:34 trow Exp $ */
