/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-context-xml.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <ctype.h>
#include <string.h>
#include <gtk/gtkmain.h>

#ifdef GNOME_XML_HEADER_VERSION_1
#include <gnome-xml/parserInternals.h>
#elif defined(GNOME_XML_HEADER_VERSION_2)
#include <libxml/parserInternals.h>
#else
#error Neither xml header location.  Something is wrong.
#endif

#include <guppi-convenient.h>
#include <guppi-debug.h>
#include "guppi-context-xml.h"

static GtkObjectClass * parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_context_xml_get_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_context_xml_set_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_context_xml_finalize (GtkObject *obj)
{
  GuppiContextXML *ctx = GUPPI_CONTEXT_XML (obj);

  /* This properly deallocates any user data */
  guppi_context_xml_set_user_data (ctx, NULL);

  if (ctx->element_begin_fns) {
    g_hash_table_foreach (ctx->element_begin_fns, guppi_free_hash_key, NULL);
    g_hash_table_destroy (ctx->element_begin_fns);
    ctx->element_begin_fns = NULL;
  }

  if (ctx->element_end_fns) {
    g_hash_table_foreach (ctx->element_end_fns, guppi_free_hash_key, NULL);
    g_hash_table_destroy (ctx->element_end_fns);
    ctx->element_end_fns = NULL;
  }

  if (ctx->element_contexts) {
    g_hash_table_foreach (ctx->element_contexts, guppi_free_hash_key, NULL);
    g_hash_table_foreach (ctx->element_contexts, guppi_unref_hash_val, NULL);
    g_hash_table_destroy (ctx->element_contexts);
    ctx->element_contexts = NULL;
  }

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
guppi_context_xml_class_init (GuppiContextXMLClass *klass)
{
  GtkObjectClass* object_class = (GtkObjectClass *)klass;

  parent_class = gtk_type_class (GTK_TYPE_OBJECT);

  object_class->get_arg = guppi_context_xml_get_arg;
  object_class->set_arg = guppi_context_xml_set_arg;
  object_class->finalize = guppi_context_xml_finalize;
}

static void
guppi_context_xml_init (GuppiContextXML *obj)
{
  obj->strip_characters = TRUE;
}

GtkType
guppi_context_xml_get_type (void)
{
  static GtkType guppi_context_xml_type = 0;
  if (!guppi_context_xml_type) {
    static const GtkTypeInfo guppi_context_xml_info = {
      "GuppiContextXML",
      sizeof(GuppiContextXML),
      sizeof(GuppiContextXMLClass),
      (GtkClassInitFunc)guppi_context_xml_class_init,
      (GtkObjectInitFunc)guppi_context_xml_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_context_xml_type = gtk_type_unique (GTK_TYPE_OBJECT,
					      &guppi_context_xml_info);
  }
  return guppi_context_xml_type;
}

GuppiContextXML *
guppi_context_xml_new (void)
{
  return GUPPI_CONTEXT_XML (guppi_type_new (guppi_context_xml_get_type ()));
}

/*****************************************************************************/

void
guppi_context_xml_set_user_data (GuppiContextXML *ctx, gpointer ud)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));

  if (ctx->please_free_data) {
    g_assert (!(ctx->please_unref_data || ctx->please_call_this_destroy_fn));
    guppi_free (ctx->user_data);
  } else if (ctx->please_unref_data) {
    g_assert (!ctx->please_call_this_destroy_fn);
    guppi_unref (ctx->user_data);
  } else if (ctx->please_call_this_destroy_fn) {
    ctx->please_call_this_destroy_fn (ctx->user_data);
  }

  ctx->user_data = ud;
  ctx->please_free_data = FALSE;
  ctx->please_unref_data = FALSE;
  ctx->please_call_this_destroy_fn = NULL;
}

void
guppi_context_xml_set_user_data_free (GuppiContextXML *ctx, gpointer ud)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));

  guppi_context_xml_set_user_data (ctx, ud);
  ctx->please_free_data = TRUE;
}

void
guppi_context_xml_set_user_data_unref (GuppiContextXML *ctx, gpointer ud)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));

  guppi_context_xml_set_user_data (ctx, ud);
  ctx->please_unref_data = TRUE;
}

void
guppi_context_xml_set_user_data_destroy_fn (GuppiContextXML *ctx,
					    gpointer ud,
					    void (*fn) (gpointer))
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));
  g_return_if_fail (fn);

  guppi_context_xml_set_user_data (ctx, ud);
  ctx->please_call_this_destroy_fn = fn;
}

void
guppi_context_xml_set_begin_context_fn (GuppiContextXML *ctx,
					xml_begin_context_fn fn)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));

  ctx->begin_context = fn;
}

void
guppi_context_xml_set_end_context_fn (GuppiContextXML *ctx,
				      xml_end_context_fn fn)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));

  ctx->end_context = fn;
}

void
guppi_context_xml_set_characters_fn (GuppiContextXML *ctx,
				     xml_char_fn fn)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));

  ctx->characters = fn;
}

void
guppi_context_xml_set_begin_element_fn (GuppiContextXML *ctx,
					xml_begin_element_fn fn)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));

  ctx->begin_element = fn;
}

void
guppi_context_xml_set_end_element_fn (GuppiContextXML *ctx,
				      xml_end_element_fn fn)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));

  ctx->end_element = fn;
}

void
guppi_context_xml_add_begin_element_fn (GuppiContextXML *ctx,
					const CHAR *name,
					xml_begin_element_fn fn)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));

  if (ctx->element_begin_fns == NULL)
    ctx->element_begin_fns = g_hash_table_new (g_str_hash, g_str_equal);

  g_hash_table_insert (ctx->element_begin_fns, guppi_strdup (name), fn);
}

void
guppi_context_xml_add_end_element_fn (GuppiContextXML *ctx,
				      const CHAR *name,
				      xml_end_element_fn fn)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));

  if (ctx->element_end_fns == NULL)
    ctx->element_end_fns = g_hash_table_new (g_str_hash, g_str_equal);

  g_hash_table_insert (ctx->element_end_fns, guppi_strdup (name), fn);
}

void
guppi_context_xml_add_element_fns (GuppiContextXML *ctx,
				   const CHAR *name,
				   xml_begin_element_fn beg_fn,
				   xml_end_element_fn end_fn)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));

  guppi_context_xml_add_begin_element_fn (ctx, name, beg_fn);
  guppi_context_xml_add_end_element_fn (ctx, name, end_fn);
}

void
guppi_context_xml_add_element_context (GuppiContextXML *ctx,
				       const CHAR *name,
				       GuppiContextXML *new_ctx)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));
  g_return_if_fail (new_ctx && GUPPI_IS_CONTEXT_XML (new_ctx));

  if (ctx->element_contexts == NULL)
    ctx->element_contexts = g_hash_table_new (g_str_hash, g_str_equal);

  g_hash_table_insert (ctx->element_contexts, guppi_strdup (name), new_ctx);
  guppi_ref (new_ctx);
}

/*****************************************************************************/

static void
gtk_iter_hack (void)
{
  static gint i = 0;

  /*
    It looks like this causes some flaming death... 
    Apparently some code somewhere isn't properly re-enterant.
    I think the problem is with the GnomeCanvas, but I don't want
    to point fingers...
  */

#if 0
  if (i == 0) {
    while (gtk_events_pending ())
      gtk_main_iteration ();
  }

  ++i;
  if (i == 8)
    i = 0;
#endif
}

void
guppi_context_xml_process_begin_context (GuppiContextXML *ctx,
					 GuppiContextXML *outer_ctx)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));
  g_return_if_fail (outer_ctx == NULL || GUPPI_IS_CONTEXT_XML (outer_ctx));

  if (ctx->begin_context)
    ctx->begin_context (ctx->user_data,
			outer_ctx ? outer_ctx->user_data : NULL);
}

void
guppi_context_xml_process_end_context (GuppiContextXML *ctx,
				       GuppiContextXML *outer_ctx)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));
  g_return_if_fail (outer_ctx == NULL || GUPPI_IS_CONTEXT_XML (outer_ctx));

  if (ctx->end_context)
    ctx->end_context (ctx->user_data,
		      outer_ctx ? outer_ctx->user_data : NULL);
}

void
guppi_context_xml_process_characters (GuppiContextXML *ctx,
				      const CHAR *ch, gint len)
{
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));
  g_return_if_fail (ch);

  if (ctx->characters) {

    if (ctx->strip_characters) {

      /* remove leading white-space */
      while (len > 0 && isspace ((guchar)*ch)) {
	++ch;
	--len;
      }

      /* remove trailing white-space */
      while (len > 0 && isspace ((guchar)ch[len-1])) {
	--len;
      }

    }

    if (len > 0)
      ctx->characters (ch, len, ctx->user_data);

  }

  gtk_iter_hack ();
}

GuppiContextXML *
guppi_context_xml_process_begin_element (GuppiContextXML *ctx,
					 const CHAR *name,
					 const CHAR **attrs)
{
  gpointer n = (gpointer)name;
  gpointer ptr;

  g_return_val_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx), NULL);

  gtk_iter_hack ();

  if (ctx->element_contexts && 
      (ptr = g_hash_table_lookup (ctx->element_contexts, n))) {

    g_assert (GUPPI_IS_CONTEXT_XML (ptr));

    return GUPPI_CONTEXT_XML (ptr);

  } else if (ctx->element_begin_fns &&
	     (ptr = g_hash_table_lookup (ctx->element_begin_fns, n))) {

    xml_begin_element_fn fn = (xml_begin_element_fn)ptr;

    return fn (name, attrs, ctx->user_data);

  } else if (ctx->begin_element) {

    return ctx->begin_element (name, attrs, ctx->user_data);

  }

  return NULL;
}

void
guppi_context_xml_process_end_element (GuppiContextXML *ctx, const CHAR *name)
{
  gpointer n = (gpointer)name;
  gpointer ptr;

  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));

  if (ctx->element_end_fns && 
      (ptr = g_hash_table_lookup (ctx->element_end_fns, n))) {

    xml_end_element_fn fn = (xml_end_element_fn)ptr;

    fn (name, ctx->user_data);

  } else if (ctx->end_element) {
    
    ctx->end_element (name, ctx->user_data);
    
  }
}

/*****************************************************************************/

void
guppi_context_xml_warning (GuppiContextXML *xml, const gchar *format_str, ...)
{
  va_list args;

  g_return_if_fail (xml && GUPPI_IS_CONTEXT_XML (xml));
  g_return_if_fail (format_str);

  va_start (args, format_str);
  fputs ("*** XML Warning: ", stderr);
  vfprintf (stderr, format_str, args);
  va_end (args);
}

void
guppi_context_xml_error (GuppiContextXML *xml, const gchar *format_str, ...)
{
  va_list args;

  g_return_if_fail (xml && GUPPI_IS_CONTEXT_XML (xml));
  g_return_if_fail (format_str);

  va_start (args, format_str);
  fputs ("*** XML Error: ", stderr);
  vfprintf (stderr, format_str, args);
  va_end (args);

}

/*****************************************************************************/

typedef struct _ContextFrame ContextFrame;
struct _ContextFrame {
  gint depth;
  GuppiContextXML *context;
};

typedef struct _ParseState ParseState;
struct _ParseState {
  gint depth;
  GList *frames;
};

#define current_frame(s) ((ContextFrame *)(s)->frames->data)

static GuppiContextXML *
current_context (ParseState *state)
{
  if (! state->frames)
    return NULL;
  if (! state->frames->data)
    return NULL;
  return ((ContextFrame *)state->frames->data)->context;
}


static void
push_context (ParseState *state, GuppiContextXML *ctx)
{
  ContextFrame *frame;
  GuppiContextXML *current_ctx;

  g_return_if_fail (state);
  g_return_if_fail (ctx && GUPPI_IS_CONTEXT_XML (ctx));

  current_ctx = current_context (state);

  frame = guppi_new0 (ContextFrame, 1);

  frame->depth = state->depth;
  frame->context = ctx;
  guppi_ref (ctx);
  guppi_sink (ctx);

  guppi_context_xml_process_begin_context (ctx, current_ctx);

  state->frames = g_list_prepend (state->frames, frame);
}

static void
pop_context (ParseState *state)
{
  ContextFrame *frame = current_frame (state);
  GuppiContextXML *ctx;
  GuppiContextXML *outer_ctx;
  GList *old_link;

  g_return_if_fail (state);

  ctx = frame->context;

  /* Pop off the context. */
  old_link = state->frames;
  state->frames = g_list_remove_link (state->frames, state->frames);
  g_list_free_1 (old_link);

  outer_ctx = current_context (state);

  guppi_context_xml_process_end_context (frame->context,
					 outer_ctx);
  guppi_unref (frame->context);
  guppi_free (frame);
}

static void
xml_characters (gpointer user_data,
		const CHAR *ch,
		gint len)
{
  ParseState *state = (ParseState *)user_data;
  GuppiContextXML *ctx = current_context (state);

  guppi_context_xml_process_characters (ctx, ch, len);
}

static void
xml_start_element (gpointer user_data,
		   const CHAR *name,
		   const CHAR **attrs)
{
  ParseState *state = (ParseState *)user_data;
  GuppiContextXML *ctx = current_context (state);
  GuppiContextXML *new_ctx;

  ++state->depth;

  new_ctx = guppi_context_xml_process_begin_element (ctx, name, attrs);
  if (new_ctx && new_ctx != ctx) {
    push_context (state, new_ctx);
  }
}

static void
xml_end_element (gpointer user_data,
		 const CHAR *name)
{
  ParseState *state = (ParseState *)user_data;
  GuppiContextXML *ctx = current_context (state);
  ContextFrame *frame = current_frame (state);

  if (frame->depth == state->depth) {

    pop_context (state);

  } else {

    guppi_context_xml_process_end_element (ctx, name);

  }

  --state->depth;
}

gint
guppi_context_xml_parse (GuppiContextXML *root_ctx,
			 const gchar *filename)
{
  gint ret = 0;
  xmlSAXHandler sax;
  xmlParserCtxtPtr xml_ctxt;
  ParseState state;

  g_return_val_if_fail (root_ctx && GUPPI_IS_CONTEXT_XML (root_ctx), -1);
  g_return_val_if_fail (filename, -1);

  /* Initialize our SAX thingie. */
  memset (&sax, 0, sizeof (sax));
  sax.characters = xml_characters;
  sax.startElement = xml_start_element;
  sax.endElement = xml_end_element;

  /* Initialize our parse state. */
  state.depth = 0;
  state.frames = NULL;

  push_context (&state, root_ctx);

  xml_ctxt = xmlCreateFileParserCtxt (filename);
  if (xml_ctxt == NULL)
    return -1;

  xml_ctxt->sax = &sax;
  xml_ctxt->userData = &state;

  xmlParseDocument (xml_ctxt);

  if (xml_ctxt->wellFormed)
    ret = 0;
  else
    ret = -1;

  xml_ctxt->sax = NULL;
  xmlFreeParserCtxt (xml_ctxt);

  pop_context (&state);

  return ret;
}

/* Code dup */
gint
guppi_context_xml_parse_memory (GuppiContextXML *root_ctx,
				gpointer ptr, gint N)
{
  gint ret = 0;
  xmlSAXHandler sax;
  xmlParserCtxtPtr xml_ctxt;
  ParseState state;

  g_return_val_if_fail (root_ctx && GUPPI_IS_CONTEXT_XML (root_ctx), -1);
  g_return_val_if_fail (ptr != NULL, -1);
  g_return_val_if_fail (N >= 0, -1);

  if (N == 0)
    return 0;

  memset (&sax, 0, sizeof (sax));
  sax.characters = xml_characters;
  sax.startElement = xml_start_element;
  sax.endElement = xml_end_element;

  state.depth = 0;
  state.frames = NULL;

  push_context (&state, root_ctx);

  xml_ctxt = xmlCreateMemoryParserCtxt ((gchar *) ptr, N);

  if (xml_ctxt == NULL)
    return -1;

  xml_ctxt->sax = &sax;
  xml_ctxt->userData = &state;

  xmlParseDocument (xml_ctxt);

  if (xml_ctxt->wellFormed)
    ret = 0;
  else
    ret = -1;

  xml_ctxt->sax = NULL;
  xmlFreeParserCtxt (xml_ctxt);

  pop_context (&state);

  return ret;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

static GuppiContextXML *
debug_begin_element (const CHAR *name, const CHAR **attrs, gpointer foo)
{
  gint i=0;

  g_print ("begin %s: ", name);

  while (attrs && attrs[i] && attrs[i+1]) {
    g_print ("%s='%s' ", attrs[i], attrs[i+1]);
    i += 2;
  }
  g_print ("\n");

  return NULL;
}

static void
debug_characters (const CHAR *txt, gint len, gpointer foo)
{
  g_print ("chars: ");
  fwrite (txt, len, 1, stdout);
  g_print ("\n");
}

static void
debug_end_element (const CHAR *name, gpointer foo)
{
  g_print ("end %s\n", name);
}

GuppiContextXML *
guppi_context_xml_debug (void)
{
  GuppiContextXML *ctx;

  ctx = guppi_context_xml_new ();
  guppi_context_xml_set_begin_element_fn (ctx, debug_begin_element);
  guppi_context_xml_set_characters_fn (ctx, debug_characters);
  guppi_context_xml_set_end_element_fn (ctx, debug_end_element);

  return ctx;
}
