/* This is -*- C -*- */
/* $Id: guppi-style.c,v 1.16 2001/01/16 23:36:36 trow Exp $ */

/*
 * guppi-style.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>

#include <gtk/gtkframe.h>
#include <gtk/gtklabel.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <libgnomeui/gnome-color-picker.h>

#include <guppi-convenient.h>
#include <guppi-metrics.h>
#include <guppi-rgb.h>
#include "guppi-style.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0,
  ARG_MARKER,
  ARG_MARKER_SIZE1,
  ARG_MARKER_SIZE2,
  ARG_COLOR,
  ARG_COLOR_DIM,
  ARG_COLOR_BRIGHT,
  ARG_LINE_WIDTH
};

enum {
  CHANGED,
  LAST_SIGNAL
};

guint sty_signals[LAST_SIGNAL] = { 0 };

static void
guppi_style_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiStyle *sty = GUPPI_STYLE (obj);

  switch (arg_id) {

  case ARG_MARKER:
    GTK_VALUE_INT (*arg) = (gint) guppi_style_marker (sty);
    break;

  case ARG_MARKER_SIZE1:
    GTK_VALUE_DOUBLE (*arg) = guppi_style_marker_size1 (sty);
    break;

  case ARG_MARKER_SIZE2:
    GTK_VALUE_DOUBLE (*arg) = guppi_style_marker_size2 (sty);
    break;

  case ARG_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_style_color (sty);
    break;

  case ARG_COLOR_DIM:
    GTK_VALUE_UINT (*arg) = guppi_style_color_dim (sty);
    break;

  case ARG_COLOR_BRIGHT:
    GTK_VALUE_UINT (*arg) = guppi_style_color_bright (sty);
    break;

  case ARG_LINE_WIDTH:
    GTK_VALUE_DOUBLE (*arg) = guppi_style_line_width (sty);
    break;

  default:
    break;
  }
}


static void
guppi_style_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiStyle *sty = GUPPI_STYLE (obj);

  switch (arg_id) {

  case ARG_MARKER:
    guppi_style_set_marker (sty, (GuppiMarker) GTK_VALUE_INT (*arg));
    break;

  case ARG_MARKER_SIZE1:
    guppi_style_set_marker_size1 (sty, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_MARKER_SIZE2:
    guppi_style_set_marker_size2 (sty, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_COLOR:
    guppi_style_set_color (sty, GTK_VALUE_UINT (*arg));
    break;

  case ARG_COLOR_DIM:
    guppi_style_set_color_dim (sty, GTK_VALUE_UINT (*arg));
    break;

  case ARG_COLOR_BRIGHT:
    guppi_style_set_color_bright (sty, GTK_VALUE_UINT (*arg));
    break;

  case ARG_LINE_WIDTH:
    guppi_style_set_line_width (sty, GTK_VALUE_DOUBLE (*arg));
    break;

  default:
    break;
  }
}


static void
guppi_style_finalize (GtkObject * obj)
{
  GuppiStyle *sty = GUPPI_STYLE (obj);

  guppi_finalized (obj);

  guppi_free (sty->label);
  sty->label = NULL;

  if (sty->pending_delayed_change) {
    gtk_idle_remove (sty->pending_delayed_change);
    sty->pending_delayed_change = 0;
  }

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/*****************************************************************************/

typedef struct _StyleWidget StyleWidget;
struct _StyleWidget {

  GuppiStyle *style;

  GtkOptionMenu *marker_menu;

  GuppiMarker marker_in_spinners;

  GtkLabel *spin1_label;
  GtkSpinButton *spin1;

  GtkLabel *spin2_label;
  GtkSpinButton *spin2;

  GnomeColorPicker *color_picker;
};

static void
set_marker_cb (GtkWidget * menu_item, GuppiStyle * style)
{
  gpointer p;
  GuppiMarker m;

  g_return_if_fail (menu_item != NULL);
  g_return_if_fail (style != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (style));

  p = gtk_object_get_user_data (GTK_OBJECT (menu_item));
  m = (GuppiMarker) GPOINTER_TO_INT (p);

  guppi_style_set_marker (style, m);
}

static void
set_size1_cb (GtkSpinButton * sb, GuppiStyle * style)
{
  double x;

  g_return_if_fail (sb != NULL);
  g_return_if_fail (GTK_IS_SPIN_BUTTON (sb));
  g_return_if_fail (style != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (style));

  x = gtk_spin_button_get_value_as_float (sb);
  guppi_style_set_marker_size1 (style, x);
}

static void
set_size2_cb (GtkSpinButton * sb, GuppiStyle * style)
{
  double x;

  g_return_if_fail (sb != NULL);
  g_return_if_fail (GTK_IS_SPIN_BUTTON (sb));
  g_return_if_fail (style != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (style));

  x = gtk_spin_button_get_value_as_float (sb);

  guppi_style_set_marker_size2 (style, x);
}

static void
set_color_cb (GtkWidget * cp, guint r, guint g, guint b, guint a,
	      GuppiStyle * style)
{
  g_return_if_fail (style != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (style));

  guppi_style_set_color (style,
			 RGBA_TO_UINT (r >> 8, g >> 8, b >> 8, a >> 8));
}


static void
build_marker_menu (GuppiStyle * style, StyleWidget * sw)
{
  GtkMenu *menu;
  gint i;

  g_return_if_fail (style != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (style));
  g_return_if_fail (sw != NULL);

  sw->marker_menu = GTK_OPTION_MENU (gtk_option_menu_new ());

  menu = GTK_MENU (gtk_menu_new ());

  for (i = 0; i < (gint) GUPPI_MARKER_LAST; ++i) {
    const GuppiMarkerInfo *info = guppi_marker_info ((GuppiMarker) i);
    if (info && info->name) {
      GtkWidget *menu_item = gtk_menu_item_new_with_label (info->name);
      gtk_object_set_user_data (GTK_OBJECT (menu_item), GINT_TO_POINTER (i));
      gtk_signal_connect (GTK_OBJECT (menu_item),
			  "activate", GTK_SIGNAL_FUNC (set_marker_cb), style);
      gtk_widget_show (menu_item);
      gtk_menu_append (menu, menu_item);
    }
  }

  gtk_option_menu_set_menu (sw->marker_menu, GTK_WIDGET (menu));
}

static void
set_marker_menu (StyleWidget * sw, GuppiMarker m)
{
  GList *item;
  gint pick = 0;

  item =
    GTK_MENU_SHELL (gtk_option_menu_get_menu (sw->marker_menu))->children;
  while (item != NULL) {
    gpointer p;
    GuppiMarker this_m;

    p = gtk_object_get_user_data (GTK_OBJECT (item->data));
    this_m = (GuppiMarker) GPOINTER_TO_INT (p);

    if (this_m == m) {
      gtk_option_menu_set_history (sw->marker_menu, pick);
      return;
    }

    item = g_list_next (item);
    ++pick;
  }

  g_assert_not_reached ();
}

static void
push_style_state_to_widget (GuppiStyle * style, StyleWidget * sw)
{
  GuppiMarker m;
  GtkAdjustment *adj;
  guint r, g, b, a;

  g_return_if_fail (style != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (style));
  g_return_if_fail (sw != NULL);

  m = guppi_style_marker (style);

  set_marker_menu (sw, m);

  if (m != sw->marker_in_spinners) {
    const GuppiMarkerInfo *info = guppi_marker_info (m);


    gtk_label_set_text (sw->spin1_label,
			info->size1_desc ? info->size1_desc : _("N/A"));

    if (info->size1_desc) {
      adj = gtk_spin_button_get_adjustment (sw->spin1);
      adj->lower = info->size1_min;
      adj->upper = info->size1_max;
      adj->step_increment = 1;
      gtk_adjustment_changed (adj);
    }

    gtk_label_set_text (sw->spin2_label,
			info->size2_desc ? info->size2_desc : _("N/A"));

    if (info->size2_desc) {
      adj = gtk_spin_button_get_adjustment (sw->spin2);
      adj->lower = info->size2_min;
      adj->upper = info->size2_max;
      adj->step_increment = 1;
      gtk_adjustment_changed (adj);
    }


    gtk_widget_set_sensitive (GTK_WIDGET (sw->spin1_label),
			      info->size1_desc != NULL);

    gtk_widget_set_sensitive (GTK_WIDGET (sw->spin1),
			      info->size1_desc != NULL);

    gtk_widget_set_sensitive (GTK_WIDGET (sw->spin2_label),
			      info->size2_desc != NULL);

    gtk_widget_set_sensitive (GTK_WIDGET (sw->spin2),
			      info->size2_desc != NULL);

    sw->marker_in_spinners = m;
  }

  adj = gtk_spin_button_get_adjustment (sw->spin1);
  adj->value = guppi_style_marker_size1 (style);
  gtk_adjustment_value_changed (adj);

  adj = gtk_spin_button_get_adjustment (sw->spin2);
  adj->value = guppi_style_marker_size2 (style);
  gtk_adjustment_value_changed (adj);


  UINT_TO_RGBA (guppi_style_color (style), &r, &g, &b, &a);
  gnome_color_picker_set_i8 (sw->color_picker, r, g, b, a);
}

static void
widget_destroy (GtkWidget * w, StyleWidget * sw)
{
  g_return_if_fail (sw != NULL);

  gtk_signal_disconnect_by_func (GTK_OBJECT (sw->style),
				 GTK_SIGNAL_FUNC (push_style_state_to_widget),
				 sw);
  guppi_unref0 (sw->style);
}


static GtkWidget *
edit_widget (GuppiStyle * style)
{
  GtkWidget *frame;
  GtkTable *table;
  StyleWidget *sw;

  g_return_val_if_fail (style != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_STYLE (style), NULL);

  sw = guppi_new0 (StyleWidget, 1);
  sw->marker_in_spinners = GUPPI_MARKER_LAST;

  sw->style = style;
  guppi_ref (sw->style);

  build_marker_menu (style, sw);

  sw->spin1_label = GTK_LABEL (gtk_label_new ("?"));
  sw->spin1 = GTK_SPIN_BUTTON (gtk_spin_button_new (NULL, 1, 1));
  gtk_spin_button_set_numeric (sw->spin1, TRUE);
  gtk_signal_connect (GTK_OBJECT (sw->spin1),
		      "changed", GTK_SIGNAL_FUNC (set_size1_cb), style);

  sw->spin2_label = GTK_LABEL (gtk_label_new ("?"));
  sw->spin2 = GTK_SPIN_BUTTON (gtk_spin_button_new (NULL, 1, 1));
  gtk_spin_button_set_numeric (sw->spin2, TRUE);
  gtk_signal_connect (GTK_OBJECT (sw->spin2),
		      "changed", GTK_SIGNAL_FUNC (set_size2_cb), style);


  sw->color_picker = GNOME_COLOR_PICKER (gnome_color_picker_new ());
  gnome_color_picker_set_use_alpha (sw->color_picker, TRUE);
  gtk_signal_connect (GTK_OBJECT (sw->color_picker),
		      "color_set", GTK_SIGNAL_FUNC (set_color_cb), style);


  gtk_signal_connect (GTK_OBJECT (style),
		      "changed",
		      GTK_SIGNAL_FUNC (push_style_state_to_widget), sw);

  push_style_state_to_widget (style, sw);

  /* Build the rest of our widget */

  frame = gtk_frame_new (_("Style"));
  table = GTK_TABLE (gtk_table_new (4, 2, FALSE));
  gtk_container_add (GTK_CONTAINER (frame), GTK_WIDGET (table));

  gtk_table_attach (table, GTK_WIDGET (sw->marker_menu),
		    0, 2, 0, 1, 0, 0, 0, 0);

  gtk_table_attach (table, GTK_WIDGET (sw->spin1_label),
		    0, 1, 1, 2, 0, 0, 0, 0);

  gtk_table_attach (table, GTK_WIDGET (sw->spin1), 1, 2, 1, 2, 0, 0, 0, 0);

  gtk_table_attach (table, GTK_WIDGET (sw->spin2_label),
		    0, 1, 2, 3, 0, 0, 0, 0);

  gtk_table_attach (table, GTK_WIDGET (sw->spin2), 1, 2, 2, 3, 0, 0, 0, 0);

  gtk_table_attach (table, gtk_label_new (_("Color")),
		    0, 1, 3, 4, 0, 0, 0, 0);

  gtk_table_attach (table, GTK_WIDGET (sw->color_picker),
		    1, 2, 3, 4, 0, 0, 0, 0);

  gtk_signal_connect (GTK_OBJECT (frame),
		      "destroy", GTK_SIGNAL_FUNC (widget_destroy), sw);

  gtk_widget_show_all (frame);

  return frame;
}

/*****************************************************************************/

#define add_arg(str, t, symb) \
gtk_object_add_arg_type("GuppiStyle::" str, t, GTK_ARG_READWRITE, symb)

static void
guppi_style_class_init (GuppiStyleClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiStyleClass *style_class = GUPPI_STYLE_CLASS (klass);

  parent_class = gtk_type_class (GTK_TYPE_OBJECT);

  object_class->get_arg = guppi_style_get_arg;
  object_class->set_arg = guppi_style_set_arg;
  object_class->finalize = guppi_style_finalize;
  style_class->edit_widget = edit_widget;

  sty_signals[CHANGED] =
    gtk_signal_new ("changed",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiStyleClass, changed),
		    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals (object_class, sty_signals, LAST_SIGNAL);

  add_arg ("marker", GTK_TYPE_INT, ARG_MARKER);
  add_arg ("marker_size1", GTK_TYPE_DOUBLE, ARG_MARKER_SIZE1);
  add_arg ("marker_size2", GTK_TYPE_DOUBLE, ARG_MARKER_SIZE2);
  add_arg ("color", GTK_TYPE_UINT, ARG_COLOR);
  add_arg ("color_dim", GTK_TYPE_UINT, ARG_COLOR_DIM);
  add_arg ("color_bright", GTK_TYPE_UINT, ARG_COLOR_BRIGHT);
  add_arg ("line_width", GTK_TYPE_DOUBLE, ARG_LINE_WIDTH);
}

static void
guppi_style_init (GuppiStyle * obj)
{
  const GuppiMarkerInfo *info;

  obj->marker = GUPPI_MARKER_CIRCLE;

  info = guppi_marker_info (obj->marker);
  obj->marker_sz1 = info->size1_default;
  obj->marker_sz2 = info->size2_default;

  obj->color = RGBA_BLACK;
}

GtkType guppi_style_get_type (void)
{
  static GtkType guppi_style_type = 0;
  if (!guppi_style_type) {
    static const GtkTypeInfo guppi_style_info = {
      "GuppiStyle",
      sizeof (GuppiStyle),
      sizeof (GuppiStyleClass),
      (GtkClassInitFunc) guppi_style_class_init,
      (GtkObjectInitFunc) guppi_style_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_style_type = gtk_type_unique (GTK_TYPE_OBJECT, &guppi_style_info);
  }
  return guppi_style_type;
}

GuppiStyle *
guppi_style_new (void)
{
  return GUPPI_STYLE (guppi_type_new (guppi_style_get_type ()));
}

/****************************************************************************/

void
guppi_style_set_marker (GuppiStyle * sty, GuppiMarker m)
{
  g_return_if_fail (sty != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (sty));

  if (sty->marker != m) {
    sty->marker = m;
    guppi_style_changed (sty);
  }
}

void
guppi_style_set_marker_size1 (GuppiStyle * sty, double sz)
{
  g_return_if_fail (sty != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (sty));
  g_return_if_fail (sz >= 0);

  if (sty->marker_sz1 != sz) {
    sty->marker_sz1 = sz;
    guppi_style_changed (sty);
  }
}

void
guppi_style_set_marker_size2 (GuppiStyle * sty, double sz)
{
  g_return_if_fail (sty != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (sty));
  g_return_if_fail (sz >= 0);

  if (sty->marker_sz2 != sz) {
    sty->marker_sz2 = sz;
    guppi_style_changed (sty);
  }
}

void
guppi_style_set_color (GuppiStyle * sty, guint32 c)
{
  g_return_if_fail (sty != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (sty));

  if (sty->color != c) {
    sty->color = c;
    guppi_style_changed (sty);
  }
}

void
guppi_style_set_color_dim (GuppiStyle * sty, guint32 c)
{
  g_return_if_fail (sty != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (sty));

  if (sty->color_dim != c) {
    sty->color_dim = c;
    guppi_style_changed (sty);
  }
}

void
guppi_style_set_color_bright (GuppiStyle * sty, guint32 c)
{
  g_return_if_fail (sty != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (sty));

  if (sty->color_bright != c) {
    sty->color_bright = c;
    guppi_style_changed (sty);
  }
}

void
guppi_style_set_line_width (GuppiStyle * sty, double w)
{
  g_return_if_fail (sty != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (sty));
  g_return_if_fail (w >= 0);

  if (sty->line_width != w) {
    sty->line_width = w;
    guppi_style_changed (sty);
  }
}

/*****************************************************************************/

void
guppi_style_copy (GuppiStyle * dest, const GuppiStyle * src)
{
  g_return_if_fail (dest != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (dest));
  g_return_if_fail (src != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (src));

  guppi_style_freeze (dest);

  guppi_style_set_marker (dest, guppi_style_marker (src));
  guppi_style_set_marker_size1 (dest, guppi_style_marker_size1 (src));
  guppi_style_set_marker_size2 (dest, guppi_style_marker_size2 (src));
  guppi_style_set_color (dest, guppi_style_color (src));
  guppi_style_set_color_dim (dest, guppi_style_color_dim (src));
  guppi_style_set_color_bright (dest, guppi_style_color_bright (src));
  guppi_style_set_line_width (dest, guppi_style_line_width (src));

  guppi_style_thaw (dest);
}


/*****************************************************************************/

void
guppi_style_changed (GuppiStyle * sty)
{
  g_return_if_fail (sty != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (sty));

  if (sty->pending_delayed_change) {
    gtk_idle_remove (sty->pending_delayed_change);
    sty->pending_delayed_change = 0;
  }

  if (sty->freeze_count > 0) {
    sty->pending_change = TRUE;
  } else {
    gtk_signal_emit (GTK_OBJECT (sty), sty_signals[CHANGED]);
    sty->pending_change = FALSE;
  }
}

static gint
delayed_changer (gpointer foo)
{
  GuppiStyle *sty = GUPPI_STYLE (foo);
  sty->pending_delayed_change = 0;
  guppi_style_changed (sty);
  return FALSE;
}

void
guppi_style_changed_delayed (GuppiStyle * sty)
{
  g_return_if_fail (sty != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (sty));

  if (sty->freeze_count > 0)
    sty->pending_change = TRUE;
  else if (sty->pending_delayed_change == 0)
    sty->pending_delayed_change = gtk_idle_add_priority (G_PRIORITY_HIGH_IDLE,
							 delayed_changer,
							 sty);
}

void
guppi_style_flush_changes (GuppiStyle * sty)
{
  g_return_if_fail (sty != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (sty));

  if (sty->pending_delayed_change != 0)
    guppi_style_changed (sty);
}

void
guppi_style_freeze (GuppiStyle * sty)
{
  g_return_if_fail (sty != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (sty));
  ++sty->freeze_count;
}

void
guppi_style_thaw (GuppiStyle * sty)
{
  g_return_if_fail (sty != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (sty));
  g_return_if_fail (sty->freeze_count > 0);
  --sty->freeze_count;
  if (sty->freeze_count == 0 && sty->pending_change)
    guppi_style_changed (sty);
}

/*****************************************************************************/

static const guint32 alien_style_colors[] = {
  0x9c9cffff, 0x9c3163ff, 0xffffceff, 0xceffffff,
  0x630063ff, 0xff8484ff, 0x0063ceff, 0xceceffff,
  0x000084ff, 0xff00ffff, 0xffff00ff, 0x00ffffff,
  0x840084ff, 0x840000ff, 0x008484ff, 0x0000ffff,
  0x00ceffff, 0xceffffff, 0xceffceff, 0xffff9cff,
  0x9cceffff, 0xff9cceff, 0xce9cffff, 0xffce9cff,
  0x3163ffff, 0x31ceceff, 0x9cce00ff, 0xffce00ff,
  0xff9c00ff, 0xff6300ff, 0x63639cff, 0x949494ff,
  0x003163ff, 0x319c63ff, 0x003100ff, 0x313100ff,
  0x9c3100ff, 0x9c3163ff, 0x31319cff, 0x313131ff,
  0xffffffff, 0xff0000ff, 0x00ff00ff, 0x0000ffff,
  0xffff00ff, 0xff00ffff, 0x00ffffff, 0x840000ff,
  0x008400ff, 0x000084ff, 0x848400ff, 0x840084ff,
  0x008484ff, 0xc6c6c6ff, 0x848484ff
};

static gint const alien_style_count =
  sizeof (alien_style_colors) / sizeof (guint32);

static const guint32 stock_style_colors[] = {
  0xff3000ff, 0x80ff00ff, 0x00ffcfff, 0x2000ffff,
  0xff008fff, 0xffbf00ff, 0x00ff10ff, 0x009fffff,
  0xaf00ffff, 0xff0000ff, 0xafff00ff, 0x00ff9fff,
  0x0010ffff, 0xff00bfff, 0xff8f00ff, 0x20ff00ff,
  0x00cfffff, 0x8000ffff, 0xff0030ff, 0xdfff00ff,
  0x00ff70ff, 0x0040ffff, 0xff00efff, 0xff6000ff,
  0x50ff00ff, 0x00ffffff, 0x5000ffff, 0xff0060ff,
  0xffef00ff, 0x00ff40ff, 0x0070ffff, 0xdf00ffff
};

static gint const stock_style_count =
  sizeof (stock_style_colors) / sizeof (guint32);


GuppiStyle *
guppi_style_stock (gint i, int which_collection)
{
  static GuppiStyle **stock_styles = NULL;

  g_return_val_if_fail (i >= 0, NULL);

  if (which_collection == 2)
    return guppi_style_stock_alien (i);

  if (i >= stock_style_count)
    return NULL;

  if (stock_styles == NULL) {
    stock_styles = guppi_new0 (GuppiStyle *, stock_style_count);
    guppi_permanent_alloc (stock_styles);
  }

  if (stock_styles[i] == NULL) {
    guint32 c;
    GuppiStyle *sty;

    sty = stock_styles[i] = guppi_style_new ();
    guppi_permanent_alloc (sty);
    guppi_ref (sty);

    sty->label = guppi_strdup_printf (_("Stock Style #%d"), i + 1);
    guppi_permanent_alloc (sty->label);

    sty->marker = GUPPI_MARKER_CIRCLE;
    sty->marker_sz1 = guppi_in2pt (1.0 / 32);
    sty->marker_sz2 = guppi_in2pt (1.0 / 96);
    sty->line_width = guppi_in2pt (1.0 / 64);

    c = stock_style_colors[i];
    sty->color = sty->color_bright = c;
    sty->color_dim = UINT_RGBA_CHANGE_A (c, UINT_RGBA_A (c) / 2);
  }

  return stock_styles[i];
}

GuppiStyle *
guppi_style_stock_alien (gint i)
{
  static GuppiStyle **stock_styles = NULL;

  g_return_val_if_fail (i >= 0, NULL);

  if (i >= alien_style_count)
    return NULL;

  if (stock_styles == NULL) {
    stock_styles = guppi_new0 (GuppiStyle *, alien_style_count);
    guppi_permanent_alloc (stock_styles);
  }

  if (stock_styles[i] == NULL) {
    guint32 c;
    GuppiStyle *sty;

    sty = stock_styles[i] = guppi_style_new ();
    guppi_permanent_alloc (sty);
    guppi_ref (sty);

    sty->label = guppi_strdup_printf (_("Alien Style #%d"), i + 1);
    guppi_permanent_alloc (sty->label);

    sty->marker = GUPPI_MARKER_CIRCLE;
    sty->marker_sz1 = guppi_in2pt (1.0 / 32);
    sty->marker_sz2 = guppi_in2pt (1.0 / 96);
    sty->line_width = guppi_in2pt (1.0 / 64);

    c = alien_style_colors[i];
    sty->color = sty->color_bright = c;
    sty->color_dim = UINT_RGBA_CHANGE_A (c, UINT_RGBA_A (c) / 2);
  }

  return stock_styles[i];
}

GtkWidget *
guppi_style_edit_widget (GuppiStyle * style)
{
  GuppiStyleClass *sc;

  g_return_val_if_fail (style != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_STYLE (style), NULL);

  sc = GUPPI_STYLE_CLASS (GTK_OBJECT (style)->klass);
  return sc->edit_widget ? sc->edit_widget (style) : NULL;
}

void
guppi_style_edit_window (GuppiStyle * style)
{
  GtkWidget *w = guppi_style_edit_widget (style);
  if (w) {
    GtkWidget *win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_container_add (GTK_CONTAINER (win), w);
    gtk_widget_show_all (win);
  }
}

/* $Id: guppi-style.c,v 1.16 2001/01/16 23:36:36 trow Exp $ */
