/* This is -*- C -*- */
/* vim: set sw=2: */
/* $Id: guppi-gsml.c,v 1.1 2001/02/04 05:37:54 trow Exp $ */

/*
 * guppi-gsml.c
 *
 * GSML == Guppi's Stupid Markup Language
 *
 * Copyright (C) 2001 The Free Software Foundation
 *
 * Developed by Jon Trowbridge <trow@gnu.org>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <guppi-memory.h>
#include "guppi-text-tokens.h"
#include "guppi-gsml.h"

static void
push_and_scale (GuppiTextBlock *block, double factor)
{
  guppi_text_block_add (block, guppi_text_token_new_push ());
  if (factor > 0)
    guppi_text_block_add (block, guppi_text_token_new_resize_font (RESIZE_FONT_SCALED, factor));
}

static void
push_and_shift_and_scale (GuppiTextBlock *block, double shift, double factor)
{
  guppi_text_block_add (block, guppi_text_token_new_push ());

  if (shift != 0)
    guppi_text_block_add (block, guppi_text_token_new_raise_lower (shift));

  if (factor > 0)
    guppi_text_block_add (block, guppi_text_token_new_resize_font (RESIZE_FONT_SCALED, factor));
}

typedef struct _GSMLParse GSMLParse;
struct _GSMLParse {
  GuppiTextBlock *block;
  gboolean inside_singleton;
};

static void
xml_char_cb (const CHAR *in_str, gint len, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;
  gchar *str = guppi_strndup (in_str, len);
  gchar **strv = g_strsplit (str, " ", 0);
  gint i=0;

  if (gsml->inside_singleton)
    return;

  while (strv[i]) {
    g_strstrip (strv[i]);
    if (i != 0)
      guppi_text_block_add (gsml->block,
			    guppi_text_token_new_space (1.0));
    guppi_text_block_add (gsml->block,
			  guppi_text_token_new_word (strv[i]));
    ++i;
  }

  guppi_free (str);
  guppi_free (strv);
}

static void
xml_pop_end_cb (const CHAR *tag, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;

  guppi_text_block_add (gsml->block, guppi_text_token_new_pop ());
}

static void
xml_singleton_end_cb (const CHAR *tag, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;

  gsml->inside_singleton = FALSE;
}

static GuppiContextXML *
xml_GSML_cb (const CHAR *tag, const CHAR **env, gpointer user_data)
{
  /* Do nothing */
  return NULL;
}

static void
xml_GSML_end_cb (const CHAR *tag, gpointer user_data)
{
  /* Do nothing */
}

static GuppiContextXML *
xml_SP_cb (const CHAR *tag, const CHAR **env, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;
  double sz = 1.0;

  if (gsml->inside_singleton)
    return NULL;

  if (env && env[0] && !g_strcasecmp (env[0], "size"))
    sz = atof (env[1]);

  guppi_text_block_add (gsml->block,
			guppi_text_token_new_space (sz));

  gsml->inside_singleton = TRUE;

  return NULL;
}

static GuppiContextXML *
xml_NOBR_cb (const CHAR *tag, const CHAR **env, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;

  if (gsml->inside_singleton)
    return NULL;

  guppi_text_block_add (gsml->block, guppi_text_token_new_push ());
  guppi_text_block_add (gsml->block, guppi_text_token_new_nobreak ());

  return NULL;
}

static GuppiContextXML *
xml_SCALE_cb (const CHAR *tag, const CHAR **env, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;
  double factor = 1.0;

  if (gsml->inside_singleton)
    return NULL;

  if (env && env[0] && !g_strcasecmp (env[0], "factor"))
    factor = atof (env[1]);

  push_and_scale (gsml->block, factor);

  return NULL;
}

static GuppiContextXML *
xml_BR_cb (const CHAR *tag, const CHAR **env, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;

  if (gsml->inside_singleton)
    return NULL;
  
  guppi_text_block_add (gsml->block, 
			guppi_text_token_new_hard_break ());

  gsml->inside_singleton = TRUE;

  return NULL;
}

static GuppiContextXML *
xml_CENTER_cb (const CHAR *tag, const CHAR **env, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;

  if (gsml->inside_singleton)
    return NULL;

  guppi_text_block_add (gsml->block, 
			      guppi_text_token_new_push ());

  guppi_text_block_add (gsml->block, 
			      guppi_text_token_new_justify (GTK_JUSTIFY_CENTER));

  return NULL;
}

static GuppiContextXML *
xml_LEFT_cb (const CHAR *tag, const CHAR **env, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;

  if (gsml->inside_singleton)
    return NULL;

  guppi_text_block_add (gsml->block, guppi_text_token_new_push ());

  guppi_text_block_add (gsml->block, 
			guppi_text_token_new_justify (GTK_JUSTIFY_LEFT));

  return NULL;
}

static GuppiContextXML *
xml_RIGHT_cb (const CHAR *tag, const CHAR **env, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;

  if (gsml->inside_singleton)
    return NULL;

  guppi_text_block_add (gsml->block, guppi_text_token_new_push ());

  guppi_text_block_add (gsml->block, 
			guppi_text_token_new_justify (GTK_JUSTIFY_RIGHT));

  return NULL;
}

static GuppiContextXML *
xml_FILL_cb (const CHAR *tag, const CHAR **env, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;

  if (gsml->inside_singleton)
    return NULL;

  guppi_text_block_add (gsml->block, guppi_text_token_new_push ());

  guppi_text_block_add (gsml->block, 
			guppi_text_token_new_justify (GTK_JUSTIFY_FILL));

  return NULL;
}

#define SUP_RAISE 0.6
#define SUP_SCALE 0.6

static GuppiContextXML *
xml_SUP_cb (const CHAR *tag, const CHAR **env, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;

  if (gsml->inside_singleton)
    return NULL;

  push_and_shift_and_scale (gsml->block, SUP_RAISE, SUP_SCALE);

  return NULL;
}

#define SUB_LOWER -0.3
#define SUB_SCALE SUP_SCALE

static GuppiContextXML *
xml_SUB_cb (const CHAR *tag, const CHAR **env, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;

  if (gsml->inside_singleton)
    return NULL;

  push_and_shift_and_scale (gsml->block, SUB_LOWER, SUB_SCALE);

  return NULL;
}

static GuppiContextXML *
xml_RAT_cb (const CHAR *tag, const CHAR **env, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;
  const gchar *ip = NULL, *num = NULL, *den = NULL;
  gint i;
  const double size = 0.6;
  const double num_raise = 0.4;
  const double den_lower = -0.1;

  if (gsml->inside_singleton)
    return NULL;
  
  if (env == NULL) {
    g_warning ("Numerator and denominator not specified in <frac/>");
    return NULL;
  }

  for (i=0; env[i] && env[i+1]; i += 2) {
    if (!g_strcasecmp (env[i], "ip"))
      ip = env[i+1];
    else if (!g_strcasecmp (env[i], "num"))
      num = env[i+1];
    else if (!g_strcasecmp (env[i], "den"))
      den = env[i+1];
  }

  if (ip) {
    guppi_text_block_add (gsml->block, 
			  guppi_text_token_new_word (ip));
    if (num || den)
      guppi_text_block_add (gsml->block,
			    guppi_text_token_new_space (0.3));
  }

  if (num) {

    push_and_shift_and_scale (gsml->block, num_raise, size);

    guppi_text_block_add (gsml->block, guppi_text_token_new_word (num));

    guppi_text_block_add (gsml->block, guppi_text_token_new_pop ());
  }

  if (num || den)
    guppi_text_block_add (gsml->block, guppi_text_token_new_word ("/"));

  if (den) {

    push_and_shift_and_scale (gsml->block, den_lower, size);

    guppi_text_block_add (gsml->block, guppi_text_token_new_word (den));

    guppi_text_block_add (gsml->block, guppi_text_token_new_pop ());
  }

  gsml->inside_singleton = TRUE;

  return NULL;
}

static GuppiContextXML *
xml_SCI_cb (const CHAR *tag, const CHAR **env, gpointer user_data)
{
  GSMLParse *gsml = (GSMLParse *) user_data;
  const gchar *man = "1.0", *exp = "0", *base = "10";
  gint significant_digits = -1;
  gint i;

  if (gsml->inside_singleton)
    return NULL;

  if (env == NULL)
    return NULL;

  for (i=0; env[i]; i += 2) {
    if (!g_strcasecmp (env[i], "man"))
      man = env[i+1];
    else if (!g_strcasecmp (env[i], "exp"))
      exp = env[i+1];
    else if (!g_strcasecmp (env[i], "base"))
      base = env[i+1];
    else if (!g_strcasecmp (env[i], "sig"))
      significant_digits = atoi (env[i]);
  }

  guppi_text_block_add (gsml->block, guppi_text_token_new_push ());
  guppi_text_block_add (gsml->block, guppi_text_token_new_nobreak ());

  guppi_text_block_add (gsml->block, guppi_text_token_new_word (man));

  /* Slightly shrink & raise our mult sign */

  guppi_text_block_add (gsml->block, guppi_text_token_new_space (0.25));

  push_and_shift_and_scale (gsml->block, 0.1, 0.75);
  guppi_text_block_add (gsml->block, guppi_text_token_new_word ("x"));
  guppi_text_block_add (gsml->block, guppi_text_token_new_pop ());

  guppi_text_block_add (gsml->block, guppi_text_token_new_space (0.15));

  guppi_text_block_add (gsml->block, guppi_text_token_new_word (base));

  /* Raise our exponent */

  push_and_shift_and_scale (gsml->block, SUP_RAISE, SUP_SCALE);
  guppi_text_block_add (gsml->block, guppi_text_token_new_word (exp));
  guppi_text_block_add (gsml->block, guppi_text_token_new_pop ());

  guppi_text_block_add (gsml->block, guppi_text_token_new_pop ());

  gsml->inside_singleton = TRUE;

  return NULL;
}

void
guppi_text_block_parse_gsml (GuppiTextBlock *block, const gchar *gsml_str)
{
  GuppiContextXML *root_ctx = NULL;
  gchar *str;
  GSMLParse *parse;

  g_return_unless_is_guppi_text_block (block);

  if (gsml_str == NULL)
    return;

  parse = guppi_new0 (GSMLParse, 1);
  parse->block = block;
  parse->inside_singleton = FALSE;

  root_ctx = guppi_context_xml_new ();

  guppi_context_xml_set_characters_fn (root_ctx, xml_char_cb);

  guppi_context_xml_add_element_fns (root_ctx, "gsml",
				     xml_GSML_cb, xml_GSML_end_cb);

  guppi_context_xml_add_element_fns (root_ctx, "sp",
				     xml_SP_cb, xml_singleton_end_cb);

  guppi_context_xml_add_element_fns (root_ctx, "br",
				     xml_BR_cb, xml_singleton_end_cb);

  guppi_context_xml_add_element_fns (root_ctx, "nobr",
				     xml_NOBR_cb, xml_pop_end_cb);

  guppi_context_xml_add_element_fns (root_ctx, "scale",
				     xml_SCALE_cb, xml_pop_end_cb);

  guppi_context_xml_add_element_fns (root_ctx, "center",
				     xml_CENTER_cb, xml_pop_end_cb);

  guppi_context_xml_add_element_fns (root_ctx, "left",
				     xml_LEFT_cb, xml_pop_end_cb);

  guppi_context_xml_add_element_fns (root_ctx, "right",
				     xml_RIGHT_cb, xml_pop_end_cb);

  guppi_context_xml_add_element_fns (root_ctx, "fill",
				     xml_FILL_cb, xml_pop_end_cb);

  guppi_context_xml_add_element_fns (root_ctx, "sup",
				     xml_SUP_cb, xml_pop_end_cb);

  guppi_context_xml_add_element_fns (root_ctx, "sub", 
				     xml_SUB_cb, xml_pop_end_cb);

  guppi_context_xml_add_element_fns (root_ctx, "rat",
				     xml_RAT_cb, xml_singleton_end_cb);

  guppi_context_xml_add_element_fns (root_ctx, "sci",
				     xml_SCI_cb, xml_singleton_end_cb);

  guppi_context_xml_set_user_data_free (root_ctx, parse);


  str = guppi_strdup_printf ("<gsml>%s</gsml>", gsml_str);

  guppi_text_block_freeze (block);
  guppi_text_block_clear (block);
  guppi_context_xml_parse_memory (root_ctx, (gpointer) str, strlen (str));
  guppi_text_block_thaw (block);

  guppi_free (str);
}


/* $Id: guppi-gsml.c,v 1.1 2001/02/04 05:37:54 trow Exp $ */
