/* This is -*- C -*- */
/* $Id: guppi-group-view.c,v 1.9 2001/02/04 05:37:54 trow Exp $ */

/*
 * guppi-group-view.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <guppi-useful.h>
#include "guppi-canvas-item.h"
#include "guppi-canvas-group.h"
#include "guppi-group-view.h"
#include "guppi-layout.h"

typedef struct _GuppiGroupViewPrivate GuppiGroupViewPrivate;
struct _GuppiGroupViewPrivate {
  GList *elements;
  GuppiLayout *layout;

  GList *new_elements;
  GuppiLayout *new_layout;

  guint lay_geom_sig1;
  guint lay_geom_sig2;

  gboolean touched;
};

#define priv(x) ((GuppiGroupViewPrivate*)((GUPPI_GROUP_VIEW(x))->opaque_internals))

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_group_view_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_group_view_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_group_view_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_group_view_finalize (GtkObject * obj)
{
  GuppiGroupView *grp = GUPPI_GROUP_VIEW (obj);
  GuppiGroupViewPrivate *p = priv (grp);
  GuppiGeometry *geom;

  guppi_unref0 (p->layout);
  guppi_unref0 (p->new_layout);

  g_list_foreach (p->elements, guppi_unref2, NULL);
  g_list_free (p->elements);
  g_list_foreach (p->new_elements, guppi_unref2, NULL);
  g_list_free (p->new_elements);

  geom = guppi_element_view_geometry (GUPPI_ELEMENT_VIEW (grp));

  if (p->lay_geom_sig1)
    gtk_signal_disconnect (GTK_OBJECT (geom), p->lay_geom_sig1);
  if (p->lay_geom_sig2)
    gtk_signal_disconnect (GTK_OBJECT (geom), p->lay_geom_sig2);

  guppi_free (grp->opaque_internals);
  grp->opaque_internals = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/****************************************************************************/

static void
make_item_iter_fn (GuppiElementView * view, gpointer user_data)
{
  GnomeCanvasGroup *group = GNOME_CANVAS_GROUP (user_data);
  GnomeCanvas *canvas = GNOME_CANVAS_ITEM (group)->canvas;

  guppi_element_view_make_canvas_item (view, canvas, group);
}

static GuppiCanvasItem *
make_canvas_item (GuppiElementView * view,
		  GnomeCanvas * canvas, GnomeCanvasGroup * canvas_group)
{
  GuppiElementViewClass *klass;
  GnomeCanvasItem *gnoitem;

  klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);
  g_assert (klass->canvas_item_type);
  gnoitem = gnome_canvas_item_new (canvas_group,
				   klass->canvas_item_type, NULL);

  guppi_group_view_foreach (GUPPI_GROUP_VIEW (view),
			    make_item_iter_fn, gnoitem);

  return GUPPI_CANVAS_ITEM (gnoitem);
}

static void
print (GuppiElementView * view, GnomePrintContext * pc)
{
  GuppiGroupViewPrivate *p = priv (GUPPI_GROUP_VIEW (view));

  /* Chain our call */
  if (GUPPI_ELEMENT_VIEW_CLASS (parent_class)->print)
    GUPPI_ELEMENT_VIEW_CLASS (parent_class)->print (view, pc);

  if (p->layout)
    guppi_layout_calc_flush (p->layout);

  guppi_group_view_foreach (GUPPI_GROUP_VIEW (view),
			    (void (*)(GuppiElementView *, gpointer))
			    guppi_element_view_print, pc);
}

static GtkWidget *
group_top_widget (gpointer ptr)
{
  return gtk_label_new ("Group Configure Placeholder");
}

static void
config_iter_fn (GuppiElementView * view, gpointer ptr)
{
  GuppiConfigItem *tree = guppi_element_view_config_tree (view);

  if (ptr && tree) {
    guppi_config_item_append_below (GUPPI_CONFIG_ITEM (ptr), tree);
    guppi_unref (tree);
  }
}

static GuppiConfigItem *
config_tree (GuppiElementView * view)
{
  const gchar *label;
  GuppiConfigItem *top;

  label = guppi_element_state_label (guppi_element_view_state (view));

  top = guppi_config_item_new (label, label, group_top_widget, view);

  guppi_config_item_set_initial_expansion (top, TRUE);

  guppi_group_view_foreach (GUPPI_GROUP_VIEW (view), config_iter_fn, top);

  if (guppi_config_item_first_child (top) == NULL) {
    guppi_unref (top);
    top = NULL;
  }

  return top;
}

/****************************************************************************/

static void
guppi_group_view_class_init (GuppiGroupViewClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiElementViewClass *view_class = GUPPI_ELEMENT_VIEW_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_ELEMENT_VIEW);

  object_class->get_arg = guppi_group_view_get_arg;
  object_class->set_arg = guppi_group_view_set_arg;
  object_class->destroy = guppi_group_view_destroy;
  object_class->finalize = guppi_group_view_finalize;

  view_class->canvas_item_type = GUPPI_TYPE_CANVAS_GROUP;
  view_class->make_canvas_item = make_canvas_item;
  view_class->print = print;
  view_class->config_tree = config_tree;
}

static void
guppi_group_view_init (GuppiGroupView * obj)
{
  GuppiGroupViewPrivate *p;
  p = guppi_new0 (GuppiGroupViewPrivate, 1);
  obj->opaque_internals = p;
  p->new_layout = guppi_layout_new ();
}

GtkType guppi_group_view_get_type (void)
{
  static GtkType guppi_group_view_type = 0;
  if (!guppi_group_view_type) {
    static const GtkTypeInfo guppi_group_view_info = {
      "GuppiGroupView",
      sizeof (GuppiGroupView),
      sizeof (GuppiGroupViewClass),
      (GtkClassInitFunc) guppi_group_view_class_init,
      (GtkObjectInitFunc) guppi_group_view_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_group_view_type =
      gtk_type_unique (GUPPI_TYPE_ELEMENT_VIEW, &guppi_group_view_info);
  }
  return guppi_group_view_type;
}

/****************************************************************************/

static void
touch (GuppiGroupView * grp)
{
  priv (grp)->touched = TRUE;
}

gboolean 
guppi_group_view_has (GuppiGroupView * grp, GuppiElementView * view)
{
  g_return_val_if_fail (grp != NULL && GUPPI_IS_GROUP_VIEW (grp), FALSE);
  g_return_val_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view), FALSE);

  return g_list_find (priv (grp)->new_elements, view) != NULL;
}

void
guppi_group_view_add (GuppiGroupView * grp, GuppiElementView * view)
{
  GuppiGroupViewPrivate *p;

  g_return_if_fail (grp != NULL && GUPPI_IS_GROUP_VIEW (grp));
  g_return_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view));

  p = priv (grp);

  if (!guppi_group_view_has (grp, view)) {
    GuppiGroupViewClass *klass;
    klass = GUPPI_GROUP_VIEW_CLASS (GTK_OBJECT (grp)->klass);

    p->new_elements = g_list_append (p->new_elements, view);

    if (klass->add_hook)
      klass->add_hook (grp, view);

    gtk_signal_connect_object (GTK_OBJECT (view),
			       "changed_structure",
			       GTK_SIGNAL_FUNC
			       (guppi_element_view_changed_structure),
			       GTK_OBJECT (grp));

    guppi_ref (view);
    guppi_sink (view);
    touch (grp);
  }
}

void
guppi_group_view_remove (GuppiGroupView * grp, GuppiElementView * view)
{
  GuppiGroupViewPrivate *p;
  GuppiGeometry *geom;
  GList *node;

  g_return_if_fail (grp != NULL && GUPPI_IS_GROUP_VIEW (grp));
  g_return_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view));

  p = priv (grp);

  node = g_list_find (p->new_elements, view);
  g_return_if_fail (node != NULL);

  GUPPI_ENTER;

  geom = guppi_element_view_geometry (view);

  guppi_layout_remove (p->new_layout, geom);

  gtk_signal_disconnect_by_func (GTK_OBJECT (view),
				 GTK_SIGNAL_FUNC
				 (guppi_element_view_changed_structure), grp);

  guppi_unref (view);
  p->new_elements = g_list_remove_link (p->new_elements, node);
  g_list_free_1 (node);

  touch (grp);

  GUPPI_EXIT;
}

void
guppi_group_view_replace (GuppiGroupView * grp,
			  GuppiElementView * old, GuppiElementView * nuevo)
{
  GuppiGroupViewPrivate *p;
  GuppiGeometry *old_geom;
  GuppiGeometry *nuevo_geom;
  GList *old_node;

  g_return_if_fail (grp != NULL && GUPPI_IS_GROUP_VIEW (grp));
  g_return_if_fail (old != NULL && GUPPI_IS_ELEMENT_VIEW (old));
  g_return_if_fail (nuevo != NULL && GUPPI_IS_ELEMENT_VIEW (nuevo));

  p = priv (grp);

  old_node = g_list_find (p->new_elements, old);
  g_return_if_fail (old_node != NULL);
  g_return_if_fail (g_list_find (p->new_elements, nuevo) == NULL);

  GUPPI_ENTER;

  old_geom = guppi_element_view_geometry (old);
  nuevo_geom = guppi_element_view_geometry (nuevo);

  guppi_layout_replace (p->new_layout, old_geom, nuevo_geom);

  guppi_unref (old);
  old_node->data = nuevo;
  guppi_ref (nuevo);

  touch (grp);

  GUPPI_EXIT;
}

void
guppi_group_view_foreach (GuppiGroupView * grp,
			  void (*fn) (GuppiElementView *, gpointer),
			  gpointer user_data)
{
  GuppiGroupViewPrivate *p;
  GList *iter;

  g_return_if_fail (grp != NULL && GUPPI_IS_GROUP_VIEW (grp));
  g_return_if_fail (fn != NULL);

  p = priv (grp);

  for (iter = p->elements; iter != NULL; iter = g_list_next (iter))
    fn (GUPPI_ELEMENT_VIEW (iter->data), user_data);
}

void
guppi_group_view_foreach_pending (GuppiGroupView * grp,
				  void (*fn) (GuppiElementView *, gpointer),
				  gpointer user_data)
{
  GuppiGroupViewPrivate *p;
  GList *iter;

  g_return_if_fail (grp != NULL && GUPPI_IS_GROUP_VIEW (grp));
  g_return_if_fail (fn != NULL);

  p = priv (grp);

  for (iter = p->new_elements; iter != NULL; iter = g_list_next (iter))
    fn (GUPPI_ELEMENT_VIEW (iter->data), user_data);
}

static void
connect_geom_iter_fn (GuppiElementView * view, gpointer ptr)
{
  GuppiGroupView *grp = GUPPI_GROUP_VIEW (ptr);
  GuppiGroupViewPrivate *p;
  GuppiGeometry *geom =
    guppi_element_view_geometry (GUPPI_ELEMENT_VIEW (view));

  p = priv (grp);

  guppi_layout_connect_geometry (p->layout, geom);
}

static void
layout_refresh (GuppiGroupView * grp)
{
  GuppiGroupViewPrivate *p;
  double x0, y0, x1, y1;

  p = priv (grp);
  guppi_element_view_get_bbox_pt (GUPPI_ELEMENT_VIEW (grp), &x0, &y0, &x1,
				  &y1);
  if (p->layout)
    guppi_layout_set_bounds (p->layout, x0, y0, x1, y1);
}

void
guppi_group_view_commit_changes (GuppiGroupView * grp)
{
  GuppiGroupViewPrivate *p;
  GuppiGeometry *geom;
  GList *iter;

  g_return_if_fail (grp != NULL && GUPPI_IS_GROUP_VIEW (grp));

  p = priv (grp);
  if (!p->touched)
    return;

  geom = guppi_element_view_geometry (GUPPI_ELEMENT_VIEW (grp));

  /* Carefully copy our list of elements. */

  /* We are doing the refs here... */
  for (iter = p->new_elements; iter != NULL; iter = g_list_next (iter))
    guppi_ref (iter->data);

  for (iter = p->elements; iter != NULL; iter = g_list_next (iter))
    guppi_unref (iter->data);

  g_list_free (p->elements);
  p->elements = NULL;

  /* ...rather than here, just to be ultra-paranoid. */
  for (iter = p->new_elements; iter != NULL; iter = g_list_next (iter))
    p->elements = g_list_append (p->elements, iter->data);


  /* Copying our layout is easier. */

  guppi_unref (p->layout);
  p->layout = guppi_layout_copy (p->new_layout);

  /* Now make sure our objects send signals to that layout object,
     not the old one. */
  guppi_group_view_foreach (grp, connect_geom_iter_fn, grp);

  guppi_layout_set_bounds_from_geometry (p->layout, geom);

  /* If our group's bounds change, we have to re-calculate the layout
     of all of the objects inside. */

  if (p->lay_geom_sig1)
    gtk_signal_disconnect (GTK_OBJECT (geom), p->lay_geom_sig1);
  if (p->lay_geom_sig2)
    gtk_signal_disconnect (GTK_OBJECT (geom), p->lay_geom_sig2);

  p->lay_geom_sig1 =
    gtk_signal_connect_object (GTK_OBJECT (geom),
			       "changed_position",
			       GTK_SIGNAL_FUNC (layout_refresh),
			       GTK_OBJECT (grp));

  p->lay_geom_sig2 =
    gtk_signal_connect_object (GTK_OBJECT (geom),
			       "changed_size",
			       GTK_SIGNAL_FUNC (layout_refresh),
			       GTK_OBJECT (grp));

  p->touched = FALSE;

  guppi_element_view_changed_delayed (GUPPI_ELEMENT_VIEW (grp));
}

static void
commit_iter (GuppiElementView * view, gpointer foo)
{
  if (GUPPI_IS_GROUP_VIEW (view))
    guppi_group_view_commit_all_changes (GUPPI_GROUP_VIEW (view));
}

void
guppi_group_view_commit_all_changes (GuppiGroupView * grp)
{
  g_return_if_fail (grp != NULL && GUPPI_IS_GROUP_VIEW (grp));

  guppi_group_view_foreach_pending (grp, commit_iter, NULL);

  guppi_group_view_commit_changes (grp);
}

/*****************************************************************************/

void
guppi_group_view_layout_natural_width (GuppiGroupView * grp,
				       GuppiElementView * v)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  guppi_group_view_add (grp, v);
  guppi_layout_natural_width (priv (grp)->new_layout,
			      guppi_element_view_geometry (v));
  touch (grp);
}

void
guppi_group_view_layout_natural_height (GuppiGroupView * grp,
					GuppiElementView * v)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  guppi_group_view_add (grp, v);
  guppi_layout_natural_height (priv (grp)->new_layout,
			       guppi_element_view_geometry (v));
  touch (grp);
}

void
guppi_group_view_layout_left_edge_relative (GuppiGroupView * grp,
					    GuppiElementView * v, double perc)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  g_return_if_fail (0 <= perc && perc <= 1);

  guppi_group_view_add (grp, v);
  guppi_layout_left_edge_relative (priv (grp)->new_layout,
				   guppi_element_view_geometry (v), perc);
  touch (grp);
}

void
guppi_group_view_layout_right_edge_relative (GuppiGroupView * grp,
					     GuppiElementView * v,
					     double perc)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  g_return_if_fail (0 <= perc && perc <= 1);

  guppi_group_view_add (grp, v);
  guppi_layout_right_edge_relative (priv (grp)->new_layout,
				    guppi_element_view_geometry (v), perc);
  touch (grp);
}

void
guppi_group_view_layout_top_edge_relative (GuppiGroupView * grp,
					   GuppiElementView * v, double perc)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  g_return_if_fail (0 <= perc && perc <= 1);

  guppi_group_view_add (grp, v);
  guppi_layout_top_edge_relative (priv (grp)->new_layout,
				  guppi_element_view_geometry (v), perc);
  touch (grp);
}

void
guppi_group_view_layout_bottom_edge_relative (GuppiGroupView * grp,
					      GuppiElementView * v,
					      double perc)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  g_return_if_fail (0 <= perc && perc <= 1);

  guppi_group_view_add (grp, v);
  guppi_layout_bottom_edge_relative (priv (grp)->new_layout,
				     guppi_element_view_geometry (v), perc);
  touch (grp);
}

void
guppi_group_view_layout_flush_left (GuppiGroupView * grp,
				    GuppiElementView * v, double margin)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  guppi_group_view_add (grp, v);
  guppi_layout_flush_left (priv (grp)->new_layout,
			   guppi_element_view_geometry (v), margin);
  touch (grp);
}

void
guppi_group_view_layout_flush_right (GuppiGroupView * grp,
				     GuppiElementView * v, double margin)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  guppi_group_view_add (grp, v);
  guppi_layout_flush_right (priv (grp)->new_layout,
			    guppi_element_view_geometry (v), margin);
  touch (grp);
}

void
guppi_group_view_layout_flush_top (GuppiGroupView * grp,
				   GuppiElementView * v, double margin)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  guppi_group_view_add (grp, v);
  guppi_layout_flush_top (priv (grp)->new_layout,
			  guppi_element_view_geometry (v), margin);
  touch (grp);
}

void
guppi_group_view_layout_flush_bottom (GuppiGroupView * grp,
				      GuppiElementView * v, double margin)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  guppi_group_view_add (grp, v);
  guppi_layout_flush_bottom (priv (grp)->new_layout,
			     guppi_element_view_geometry (v), margin);
  touch (grp);
}

void
guppi_group_view_layout_width_relative (GuppiGroupView * grp,
					GuppiElementView * v, double perc)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  g_return_if_fail (0 <= perc && perc <= 1);

  guppi_group_view_add (grp, v);
  guppi_layout_width_relative (priv (grp)->new_layout,
			       guppi_element_view_geometry (v), perc);
  touch (grp);
}

void
guppi_group_view_layout_height_relative (GuppiGroupView * grp,
					 GuppiElementView * v, double perc)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  g_return_if_fail (0 <= perc && perc <= 1);

  guppi_group_view_add (grp, v);
  guppi_layout_height_relative (priv (grp)->new_layout,
				guppi_element_view_geometry (v), perc);
  touch (grp);
}

void
guppi_group_view_layout_center_horizontally (GuppiGroupView * grp,
					     GuppiElementView * v)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  guppi_group_view_add (grp, v);
  guppi_layout_center_horizontally (priv (grp)->new_layout,
				    guppi_element_view_geometry (v));
  touch (grp);
}

void
guppi_group_view_layout_center_vertically (GuppiGroupView * grp,
					   GuppiElementView * v)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  guppi_group_view_add (grp, v);
  guppi_layout_center_vertically (priv (grp)->new_layout,
				  guppi_element_view_geometry (v));
  touch (grp);
}

void
guppi_group_view_layout_horizontal_fill (GuppiGroupView * grp,
					 GuppiElementView * v,
					 double lm, double rm)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  guppi_group_view_add (grp, v);
  guppi_layout_horizontal_fill (priv (grp)->new_layout,
				guppi_element_view_geometry (v), lm, rm);
  touch (grp);
}

void
guppi_group_view_layout_vertical_fill (GuppiGroupView * grp,
				       GuppiElementView * v,
				       double lm, double rm)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  guppi_group_view_add (grp, v);
  guppi_layout_vertical_fill (priv (grp)->new_layout,
			      guppi_element_view_geometry (v), lm, rm);
  touch (grp);
}

void
guppi_group_view_layout_aligned_left_edge (GuppiGroupView * grp,
					   GuppiElementView * v1,
					   GuppiElementView * v2)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_aligned_left_edge (priv (grp)->new_layout,
				  guppi_element_view_geometry (v1),
				  guppi_element_view_geometry (v2));
  touch (grp);
}

void
guppi_group_view_layout_aligned_right_edge (GuppiGroupView * grp,
					    GuppiElementView * v1,
					    GuppiElementView * v2)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_aligned_right_edge (priv (grp)->new_layout,
				   guppi_element_view_geometry (v1),
				   guppi_element_view_geometry (v2));
  touch (grp);
}

void
guppi_group_view_layout_aligned_top_edge (GuppiGroupView * grp,
					  GuppiElementView * v1,
					  GuppiElementView * v2)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_aligned_top_edge (priv (grp)->new_layout,
				 guppi_element_view_geometry (v1),
				 guppi_element_view_geometry (v2));
  touch (grp);
}

void
guppi_group_view_layout_aligned_bottom_edge (GuppiGroupView * grp,
					     GuppiElementView * v1,
					     GuppiElementView * v2)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_aligned_bottom_edge (priv (grp)->new_layout,
				    guppi_element_view_geometry (v1),
				    guppi_element_view_geometry (v2));
  touch (grp);
}

void
guppi_group_view_layout_horizontally_adjacent (GuppiGroupView * grp,
					       GuppiElementView * v1,
					       GuppiElementView * v2,
					       double gap)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_horizontally_adjacent (priv (grp)->new_layout,
				      guppi_element_view_geometry (v1),
				      guppi_element_view_geometry (v2), gap);
  touch (grp);
}

void
guppi_group_view_layout_vertically_adjacent (GuppiGroupView * grp,
					     GuppiElementView * v1,
					     GuppiElementView * v2,
					     double gap)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_vertically_adjacent (priv (grp)->new_layout,
				    guppi_element_view_geometry (v1),
				    guppi_element_view_geometry (v2), gap);
  touch (grp);
}

void
guppi_group_view_layout_hbox2 (GuppiGroupView * grp,
			       GuppiElementView * v1,
			       GuppiElementView * v2, double gap)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_hbox2 (priv (grp)->new_layout,
		      guppi_element_view_geometry (v1),
		      guppi_element_view_geometry (v2), gap);
  touch (grp);
}

void
guppi_group_view_layout_hbox3 (GuppiGroupView * grp,
			       GuppiElementView * v1,
			       GuppiElementView * v2,
			       GuppiElementView * v3, double gap)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  g_return_if_fail (v3 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v3));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_group_view_add (grp, v3);
  guppi_layout_hbox3 (priv (grp)->new_layout,
		      guppi_element_view_geometry (v1),
		      guppi_element_view_geometry (v2),
		      guppi_element_view_geometry (v3), gap);
  touch (grp);
}

void
guppi_group_view_layout_vbox2 (GuppiGroupView * grp,
			       GuppiElementView * v1,
			       GuppiElementView * v2, double gap)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_vbox2 (priv (grp)->new_layout,
		      guppi_element_view_geometry (v1),
		      guppi_element_view_geometry (v2), gap);
  touch (grp);
}

void
guppi_group_view_layout_vbox3 (GuppiGroupView * grp,
			       GuppiElementView * v1,
			       GuppiElementView * v2,
			       GuppiElementView * v3, double gap)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  g_return_if_fail (v3 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v3));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_group_view_add (grp, v3);
  guppi_layout_vbox3 (priv (grp)->new_layout,
		      guppi_element_view_geometry (v1),
		      guppi_element_view_geometry (v2),
		      guppi_element_view_geometry (v3), gap);
  touch (grp);
}

void
guppi_group_view_layout_width_ratio (GuppiGroupView * grp,
				     GuppiElementView * v1,
				     GuppiElementView * v2, double x)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  g_return_if_fail (x > 0);

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_width_ratio (priv (grp)->new_layout,
			    guppi_element_view_geometry (v1),
			    guppi_element_view_geometry (v2), x);
  touch (grp);
}

void
guppi_group_view_layout_height_ratio (GuppiGroupView * grp,
				      GuppiElementView * v1,
				      GuppiElementView * v2, double x)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  g_return_if_fail (x > 0);

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_height_ratio (priv (grp)->new_layout,
			     guppi_element_view_geometry (v1),
			     guppi_element_view_geometry (v2), x);
  touch (grp);
}

void
guppi_group_view_layout_aspect_ratio (GuppiGroupView * grp,
				      GuppiElementView * v, double x)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v));

  g_return_if_fail (x > 0);

  guppi_group_view_add (grp, v);
  guppi_layout_aspect_ratio (priv (grp)->new_layout,
			     guppi_element_view_geometry (v), x);
  touch (grp);
}

void
guppi_group_view_layout_same_width (GuppiGroupView * grp,
				    GuppiElementView * v1,
				    GuppiElementView * v2)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_same_width (priv (grp)->new_layout,
			   guppi_element_view_geometry (v1),
			   guppi_element_view_geometry (v2));
  touch (grp);
}

void
guppi_group_view_layout_same_height (GuppiGroupView * grp,
				     GuppiElementView * v1,
				     GuppiElementView * v2)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_same_height (priv (grp)->new_layout,
			    guppi_element_view_geometry (v1),
			    guppi_element_view_geometry (v2));
  touch (grp);
}

void
guppi_group_view_layout_width_equals_height (GuppiGroupView * grp,
					     GuppiElementView * v1,
					     GuppiElementView * v2)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_width_equals_height (priv (grp)->new_layout,
				    guppi_element_view_geometry (v1),
				    guppi_element_view_geometry (v2));
  touch (grp);
}

void
guppi_group_view_layout_same_size (GuppiGroupView * grp,
				   GuppiElementView * v1,
				   GuppiElementView * v2)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_same_size (priv (grp)->new_layout,
			  guppi_element_view_geometry (v1),
			  guppi_element_view_geometry (v2));
  touch (grp);
}

void
guppi_group_view_layout_same_place (GuppiGroupView * grp,
				    GuppiElementView * v1,
				    GuppiElementView * v2)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_same_place (priv (grp)->new_layout,
			   guppi_element_view_geometry (v1),
			   guppi_element_view_geometry (v2));
  touch (grp);
}

void
guppi_group_view_layout_same_x_center (GuppiGroupView * grp,
				       GuppiElementView * v1,
				       GuppiElementView * v2)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_same_x_center (priv (grp)->new_layout,
			      guppi_element_view_geometry (v1),
			      guppi_element_view_geometry (v2));
  touch (grp);
}

void
guppi_group_view_layout_same_y_center (GuppiGroupView * grp,
				       GuppiElementView * v1,
				       GuppiElementView * v2)
{
  g_return_if_fail (grp != NULL);
  g_return_if_fail (GUPPI_IS_GROUP_VIEW (grp));

  g_return_if_fail (v1 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v1));

  g_return_if_fail (v2 != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (v2));

  guppi_group_view_add (grp, v1);
  guppi_group_view_add (grp, v2);
  guppi_layout_same_y_center (priv (grp)->new_layout,
			      guppi_element_view_geometry (v1),
			      guppi_element_view_geometry (v2));
  touch (grp);
}


/* $Id: guppi-group-view.c,v 1.9 2001/02/04 05:37:54 trow Exp $ */
