/* This is -*- C -*- */
/* $Id: guppi-seq-scalar.c,v 1.27 2001/01/16 23:36:21 trow Exp $ */

/*
 * guppi-seq-scalar.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <math.h>
#include <stdlib.h>

#include <gtk/gtklabel.h>
#include <gtk/gtksignal.h>
#include <gtk/gtktable.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-convenient.h>
#include <guppi-string.h>
#include "guppi-seq-scalar.h"
#include "guppi-seq-scalar-impl.h"
#include "guppi-seq-boolean.h"
/* #include <gnome.h> */

typedef struct _GuppiSeqScalarPrivate GuppiSeqScalarPrivate;
struct _GuppiSeqScalarPrivate {
  gboolean have_minmax, have_sum, have_sum_abs, have_var, have_quartiles;
  double min, max, sum, sum_abs, var, q1, median, q3;

  gboolean save_minmax, save_sum, save_sum_abs;

  gpointer sorted_copy;
};

#define priv(x) ((GuppiSeqScalarPrivate*)((x)->opaque_internals))

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_scalar_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_scalar_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_scalar_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_seq_scalar_finalize (GtkObject * obj)
{
  GuppiSeqScalar *seq = GUPPI_SEQ_SCALAR (obj);
  GuppiSeqScalarPrivate *p = priv (seq);

  guppi_free0 (p->sorted_copy);
  guppi_free0 (seq->opaque_internals);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
changed (GuppiData * data)
{
  GuppiSeqScalar *ss = GUPPI_SEQ_SCALAR (data);
  GuppiSeqScalarPrivate *p = priv (ss);

  if (GUPPI_DATA_CLASS (parent_class)->changed)
    GUPPI_DATA_CLASS (parent_class)->changed (data);

  p->have_minmax = p->save_minmax;
  p->have_sum = p->save_sum;
  p->have_sum_abs = p->save_sum_abs;
  p->have_var = FALSE;
  p->have_quartiles = FALSE;

  p->save_minmax = FALSE;
  p->save_sum = FALSE;

  guppi_free (p->sorted_copy);
  p->sorted_copy = NULL;
}

static void
write_xml_element (GuppiSeq *seq, gint i, GuppiOutputXML *out)
{
  double x;

  x = guppi_seq_scalar_get (GUPPI_SEQ_SCALAR (seq), i);
  
  guppi_output_xml_elementf (out, "scalar", "%g", x);
}

static GuppiData *
create_xml_object (GuppiAttributesXML *ax)
{
  return guppi_seq_scalar_new ();
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

typedef struct _ScalarContext ScalarContext;
struct _ScalarContext {
  gboolean in_scalar;
  GuppiSeqScalar *seq;
};


static GuppiContextXML *
begin_element (const CHAR *name, const CHAR **attr, gpointer ud)
{
  ScalarContext *sc = (ScalarContext *)ud;

  g_return_val_if_fail (sc, NULL);

  if (!strcmp (name, "gpi:scalar")) {
    if (sc->in_scalar) {
      g_warning ("Nested scalar elements not allowed.");
      return NULL;
    }
    sc->in_scalar = TRUE;
  } else if (! strcmp (name, "gpi:missing")) {
    if (sc->in_scalar) {
      g_warning ("missing tag not allowed inside of scalar.");
      return NULL;
    }
    guppi_seq_append_missing (GUPPI_SEQ (sc->seq));
  } else {
    g_warning ("Unknown tag: %s", name);
    return NULL;
  }

  return NULL;
}

static void
end_element (const CHAR *name, gpointer ud)
{
  ScalarContext *sc = (ScalarContext *)ud;

  g_return_if_fail (sc);

  if (!strcmp (name, "gpi:scalar")) {
    sc->in_scalar = FALSE;
  }
}

static void
characters (const CHAR *s, gint len, gpointer ud)
{
  ScalarContext *sc = (ScalarContext *)ud;

  g_return_if_fail (sc);

  if (sc->in_scalar) {
    gchar *str = guppi_strndup (s, len);

    if (guppi_string_is_number (str)) {
      double x = atof (str);
      guppi_seq_scalar_append (sc->seq, x);
    } else {
      g_warning ("Illegal characters inside scalar: %s", str);
    }
    guppi_free (str);
  } else {
    g_warning ("Illegal characters.");
  }
}

static GuppiContextXML *
content_xml_context (GuppiData *d, GuppiAttributesXML *ax)
{
  GuppiContextXML *ctx;
  ScalarContext *sc;

  /* I don't do any reference counting w/ our data object.  This could
     come back to haunt me. */
  sc = guppi_new0 (ScalarContext, 1);
  sc->in_scalar = FALSE;
  sc->seq = GUPPI_SEQ_SCALAR (d);
  
  ctx = guppi_context_xml_new ();
  guppi_context_xml_set_begin_element_fn (ctx, begin_element);
  guppi_context_xml_set_end_element_fn (ctx, end_element);
  guppi_context_xml_set_characters_fn (ctx, characters);
  guppi_context_xml_set_user_data_free (ctx, sc);

  return ctx;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

static void
guppi_seq_scalar_class_init (GuppiSeqScalarClass * klass)
{
  static GtkWidget *info_display (GuppiData *);

  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiDataClass *data_class = GUPPI_DATA_CLASS (klass);
  GuppiSeqClass *seq_class = GUPPI_SEQ_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_SEQ);

  seq_class->write_xml_element = write_xml_element;

  data_class->type_name = _("Scalar Sequence");
  data_class->impl_type = GUPPI_TYPE_SEQ_SCALAR_IMPL;
  data_class->changed = changed;
  data_class->create_xml_object = create_xml_object;
  data_class->content_xml_context = content_xml_context;
  data_class->info_display = info_display;

  object_class->get_arg = guppi_seq_scalar_get_arg;
  object_class->set_arg = guppi_seq_scalar_set_arg;
  object_class->destroy = guppi_seq_scalar_destroy;
  object_class->finalize = guppi_seq_scalar_finalize;

}

static void
guppi_seq_scalar_init (GuppiSeqScalar * obj)
{
  obj->opaque_internals = guppi_new0 (GuppiSeqScalarPrivate, 1);
}

GtkType guppi_seq_scalar_get_type (void)
{
  static GtkType guppi_seq_scalar_type = 0;
  if (!guppi_seq_scalar_type) {
    static const GtkTypeInfo guppi_seq_scalar_info = {
      "GuppiSeqScalar",
      sizeof (GuppiSeqScalar),
      sizeof (GuppiSeqScalarClass),
      (GtkClassInitFunc) guppi_seq_scalar_class_init,
      (GtkObjectInitFunc) guppi_seq_scalar_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_seq_scalar_type =
      gtk_type_unique (GUPPI_TYPE_SEQ, &guppi_seq_scalar_info);
  }
  return guppi_seq_scalar_type;
}

GuppiData *
guppi_seq_scalar_new (void)
{
  return guppi_data_newv (GUPPI_TYPE_SEQ_SCALAR, NULL, 0, NULL);
}

/*****************************************************************************/

/* Sequence Operation Stuff */

typedef struct _GuppiDataOp_Scalar GuppiDataOp_Scalar;
struct _GuppiDataOp_Scalar {
  GuppiDataOp op;

  gint i;
  gsize N;

  double x;

  gconstpointer in_ptr;
  gint in_stride;

  gpointer out_ptr;
  gint out_stride;
};

static void
op_set (GuppiData *d, GuppiDataOp *op)
{
  GuppiSeqScalar *ss = GUPPI_SEQ_SCALAR (d);
  GuppiSeqScalarPrivate *p = priv (ss);
  GuppiDataOp_Scalar *scalar_op = (GuppiDataOp_Scalar *) op;

  gboolean was_missing;
  double old_value;
  double x = scalar_op->x;

  GuppiSeqScalarImpl *impl;
  GuppiSeqScalarImplClass *impl_class;

  was_missing = guppi_seq_missing (GUPPI_SEQ (d), scalar_op->i);
  old_value = was_missing ? 0 : guppi_seq_scalar_get (ss, scalar_op->i);

  impl = GUPPI_SEQ_SCALAR_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->set);
  impl_class->set (impl, scalar_op->i, x);

  /*
   * Try to incrementally update some of our stats, so that we can
   * possibly avoid future recalcs.
   */

  if (p->have_sum) {
    p->sum += x - old_value;
    p->save_sum = TRUE;
  }
  if (p->have_sum_abs) {
    p->sum_abs += fabs(x) - fabs(old_value);
    p->save_sum_abs = TRUE;
  }

  if (!was_missing && p->min == p->max) {

    /* We could probably do something for this case */

  } else if (!was_missing && old_value == p->min) {
    if (x < old_value) {
      p->min = x;
      p->save_minmax = TRUE;
    }
  } else if (!was_missing && old_value == p->max) {
    if (x > old_value) {
      p->max = x;
      p->save_minmax = TRUE;
    }
  } else if (x < p->min) {
    p->min = x;
    p->save_minmax = TRUE;
  } else if (x > p->max) {
    p->max = x;
    p->save_minmax = TRUE;
  } else if (p->min < x && x < p->max) {
    p->save_minmax = TRUE;
  }
}

static void
op_set_many (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Scalar *scalar_op = (GuppiDataOp_Scalar *) op;

  GuppiSeqScalarImpl *impl;
  GuppiSeqScalarImplClass *impl_class;

  impl = GUPPI_SEQ_SCALAR_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->set_many) {
    impl_class->set_many (impl,
			  scalar_op->i,
			  scalar_op->in_ptr,
			  scalar_op->in_stride, scalar_op->N);
  } else {
    /* Do our insert one piece at a time --- ugh */
    const guchar *byte_ptr = (const guchar *) scalar_op->in_ptr;
    gint count = scalar_op->N;
    gint j = scalar_op->i;

    g_assert (impl_class->set);
    for (; count-- > 0; ++j) {
      impl_class->set (impl, j, *(const double *) byte_ptr);
      byte_ptr += scalar_op->in_stride;
    }
  }
}
static void
op_insert (GuppiData *d, GuppiDataOp *op)
{
  GuppiSeqScalar *ss = GUPPI_SEQ_SCALAR (d);
  GuppiSeqScalarPrivate *p = priv (ss);
  GuppiDataOp_Scalar *scalar_op = (GuppiDataOp_Scalar *) op;
  double x = scalar_op->x;

  GuppiSeqScalarImpl *impl;
  GuppiSeqScalarImplClass *impl_class;

  impl = GUPPI_SEQ_SCALAR_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->insert);

  impl_class->insert (impl, scalar_op->i, x);

  if (p->have_sum) {
    p->sum += x;
    p->save_sum = TRUE;
  }
  if (p->have_sum_abs) {
    p->sum += fabs(x);
    p->save_sum_abs = TRUE;
  }

  if (p->have_minmax) {
    p->min = MIN (p->min, x);
    p->max = MAX (p->max, x);
    p->save_minmax = TRUE;
  }
}

static void
op_insert_many (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Scalar *scalar_op = (GuppiDataOp_Scalar *) op;

  GuppiSeqScalarImpl *impl;
  GuppiSeqScalarImplClass *impl_class;

  impl = GUPPI_SEQ_SCALAR_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->insert_many) {
    impl_class->insert_many (impl,
			     scalar_op->i,
			     scalar_op->in_ptr,
			     scalar_op->in_stride, scalar_op->N);
  } else {
    /* Do our insert one piece at a time --- ugh */
    const guchar *byte_ptr = (const guchar *) scalar_op->in_ptr;
    gint j;

    g_assert (impl_class->insert);
    for (j = 0; j < scalar_op->N; ++j) {
      impl_class->insert (impl, scalar_op->i + j, *(const double *) byte_ptr);
      byte_ptr += scalar_op->in_stride;
    }
  }
}

/*****************************************************************************/

/* Stuff for the sorted copy */

typedef struct _SortPair SortPair;
struct _SortPair {
  double x;
  gint i;
};

static int
sorted_pair_compare (const void *a, const void *b)
{
  double x = ((const SortPair *) a)->x;
  double y = ((const SortPair *) b)->x;

  return (x > y) - (x < y);
}

static void
make_sorted_copy (GuppiSeqScalar * seq)
{
  gint i, i0, i1, j, stride;
  gsize N;
  SortPair *sc;
  gboolean has_missing;
  gconstpointer ptr;

  if (priv (seq)->sorted_copy)
    return;

  has_missing = guppi_seq_has_missing (GUPPI_SEQ (seq));
  N = guppi_seq_count (GUPPI_SEQ (seq));
  ptr = guppi_seq_scalar_raw (seq, &stride);

  sc = guppi_new (SortPair, N);

  j = 0;
  guppi_seq_bounds (GUPPI_SEQ (seq), &i0, &i1);
  for (i = i0; i <= i1; ++i) {
    if (!has_missing || guppi_seq_available (GUPPI_SEQ (seq), i)) {
      double x;
      if (ptr)
	x = guppi_seq_scalar_raw_get (ptr, stride, i);
      else
	x = guppi_seq_scalar_get (seq, i);

      sc[j].x = x;
      sc[j].i = i;
      ++j;
    }
  }

  qsort (sc, N, sizeof (SortPair), sorted_pair_compare);

  priv (seq)->sorted_copy = sc;
}

static SortPair *
get_sorted_copy (GuppiSeqScalar * seq)
{
  GuppiSeqScalarPrivate *p;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq), NULL);

  p = priv (seq);

  if (p->sorted_copy == NULL)
    make_sorted_copy (seq);

  return (SortPair *) p->sorted_copy;
}


/*****************************************************************************/

double
guppi_seq_scalar_get (const GuppiSeqScalar * seq, gint i)
{
  const GuppiSeqScalarImpl *impl;
  GuppiSeqScalarImplClass *impl_class;

  g_return_val_if_fail (seq != NULL, 0);
  g_return_val_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (seq), i), 0);

  impl = GUPPI_SEQ_SCALAR_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->get);
  return (impl_class->get) (impl, i);
}

void
guppi_seq_scalar_set (GuppiSeqScalar * seq, gint i, double x)
{
  GuppiDataOp_Scalar op;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));
  g_return_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (seq), i));

  if (guppi_seq_missing (GUPPI_SEQ (seq), i) ||
      guppi_seq_scalar_get (seq, i) != x) {

    op.op.op = op_set;
    op.i = i;
    op.x = x;

    guppi_seq_changed_set (GUPPI_SEQ (seq), i, i, (GuppiDataOp *) & op);
  }
}

void
guppi_seq_scalar_set_many (GuppiSeqScalar * seq, gint start,
			   const double *vals, gint stride, gsize N)
{
  GuppiDataOp_Scalar op;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  if (N == 0)
    return;

  g_return_if_fail (vals != NULL);

  op.op.op = op_set_many;
  op.i = start;
  op.in_ptr = vals;
  op.in_stride = stride;
  op.N = N;

  guppi_seq_size_hint (GUPPI_SEQ (seq),
		       MAX (guppi_seq_size (GUPPI_SEQ (seq)), start + N));
  guppi_seq_changed_set (GUPPI_SEQ (seq), start, start + N,
			 (GuppiDataOp *) & op);
}

void
guppi_seq_scalar_prepend (GuppiSeqScalar * seq, double x)
{
  gint first;

  g_return_if_fail (seq != NULL);
  g_return_if_fail (GUPPI_IS_SEQ (seq));

  first = guppi_seq_min_index (GUPPI_SEQ (seq));
  guppi_seq_scalar_insert (seq, first, x);
}

void
guppi_seq_scalar_prepend_many (GuppiSeqScalar * seq,
			       gconstpointer ptr, gint stride, gsize N)
{
  gint first;

  g_return_if_fail (seq != NULL);
  g_return_if_fail (GUPPI_IS_SEQ (seq));

  first = guppi_seq_min_index (GUPPI_SEQ (seq));
  guppi_seq_scalar_insert_many (seq, first, ptr, stride, N);
}

void
guppi_seq_scalar_prepend_repeating (GuppiSeqScalar * seq, double x, gsize N)
{
  guppi_seq_scalar_prepend_many (seq, &x, 0, N);
}

void
guppi_seq_scalar_append (GuppiSeqScalar * seq, double x)
{
  gint last;
  last = guppi_seq_max_index (GUPPI_SEQ (seq));
  guppi_seq_scalar_insert (seq, last + 1, x);
}

void
guppi_seq_scalar_append_many (GuppiSeqScalar * seq,
			      gconstpointer ptr, gint stride, gsize N)
{
  gint last;

  g_return_if_fail (seq != NULL);
  g_return_if_fail (GUPPI_IS_SEQ (seq));

  last = guppi_seq_max_index (GUPPI_SEQ (seq));
  guppi_seq_scalar_insert_many (seq, last + 1, ptr, stride, N);
}

void
guppi_seq_scalar_append_repeating (GuppiSeqScalar * seq, double x, gsize N)
{
  guppi_seq_scalar_append_many (seq, &x, 0, N);
}

void
guppi_seq_scalar_insert (GuppiSeqScalar * seq, gint i, double x)
{
  GuppiDataOp_Scalar op;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  op.op.op = op_insert;
  op.i = i;
  op.x = x;

  guppi_seq_changed_insert (GUPPI_SEQ (seq), i, 1, (GuppiDataOp *) & op);
}

void
guppi_seq_scalar_insert_many (GuppiSeqScalar * seq, gint i,
			      gconstpointer ptr, gint stride, gsize N)
{
  GuppiDataOp_Scalar op;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  if (N == 0)
    return;

  g_return_if_fail (ptr != NULL);

  op.op.op = op_insert_many;
  op.i = i;
  op.in_ptr = ptr;
  op.in_stride = stride;
  op.N = N;

  guppi_seq_size_hint (GUPPI_SEQ (seq), guppi_seq_size (GUPPI_SEQ (seq)) + N);
  guppi_seq_changed_insert (GUPPI_SEQ (seq), i, N, (GuppiDataOp *) & op);
}

void
guppi_seq_scalar_insert_repeating (GuppiSeqScalar * seq, gint i,
				   double x, gsize N)
{
  guppi_seq_scalar_insert_many (seq, i, &x, 0, N);
}

static gint
do_range_query (GuppiSeqScalar * seq,
		GuppiSeqBoolean * bseq,
		double min, double max, gboolean do_and)
{
  const GuppiSeqScalarImpl *impl;
  const GuppiSeqScalarImplClass *impl_class;
#if 0
  GuppiSeqBooleanImpl *b_impl;
  GuppiSeqBooleanImplClass *b_impl_class;
#endif

  gint hits;
  SortPair *sorted_copy;

  double seq_min, seq_max;
  gint count = 0;

  g_return_val_if_fail (seq != NULL, 0);
  g_return_val_if_fail (bseq != NULL, 0);

  guppi_2sort (&min, &max);
  seq_min = guppi_seq_scalar_min (seq);
  seq_max = guppi_seq_scalar_max (seq);

  if (min <= seq_max && seq_max <= max) {
    if (do_and) {
      /* do nothing */
    } else {
      guppi_seq_boolean_set_all (bseq, TRUE);
    }
    return guppi_seq_size (GUPPI_SEQ (seq));
  }

  impl = GUPPI_SEQ_SCALAR_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->range_query) {
    hits = (impl_class->range_query) (impl, bseq, min, max, do_and);
    if (hits >= 0)
      return hits;
  }
#if 0
  b_impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (GUPPI_DATA (bseq)));
  b_impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (b_impl)->klass);

  if (b_impl_class->range_query) {
    hits = (b_impl_class->range_query) (seq, b_impl, min, max, do_and);
    if (hits >= 0)
      return hits;
  }
#endif

  sorted_copy = get_sorted_copy (seq);

  if (sorted_copy) {
    gint k0, k1, k, a, b;
    gint set_buffer[500];
    gint sb_N = 500, sb_i;
    gint seq_count, N;
    GuppiSeqBoolean *op_bool;

    N = guppi_seq_size (GUPPI_SEQ (seq));
    seq_count = guppi_seq_count (GUPPI_SEQ (seq));

    if (do_and) {

      op_bool = GUPPI_SEQ_BOOLEAN (guppi_seq_boolean_new ());
      guppi_seq_size_hint (GUPPI_SEQ (op_bool), N);
      guppi_seq_boolean_append_many (op_bool, FALSE, N);
      guppi_seq_set_min_index (GUPPI_SEQ (op_bool),
			       guppi_seq_min_index (GUPPI_SEQ (seq)));

    } else {
      guppi_seq_boolean_clear (bseq);
      op_bool = bseq;
    }

    if (min <= sorted_copy[0].x)
      k0 = 0;
    else if (min > sorted_copy[seq_count - 1].x)
      k0 = seq_count;
    else {

      a = 0;
      b = seq_count - 1;
      k0 = (a + b) / 2;

      while (!(sorted_copy[k0 - 1].x < min && min <= sorted_copy[k0].x)) {
	gint k0_old = k0;
	if (min <= sorted_copy[k0 - 1].x)
	  b = k0 + 1;
	else			/* if (min > sorted_copy[m].x) */
	  a = k0 - 1;
	k0 = (a + b) / 2;
	if (k0 == k0_old)
	  ++k0;
      }

    }

    if (sorted_copy[seq_count - 1].x <= max)
      k1 = seq_count - 1;
    else if (max < sorted_copy[0].x)
      k1 = -1;
    else {
      a = k0;
      b = seq_count - 1;
      k1 = (a + b) / 2;

      while (!(sorted_copy[k1].x <= max && max < sorted_copy[k1 + 1].x)) {
	gint k1_old = k1;
	if (sorted_copy[k1].x > max)
	  b = k1 - 1;
	else			/* if (max >= sorted_copy[k1+1].x) */
	  a = k1 + 1;
	k1 = (a + b) / 2;
	if (k1 == k1_old)
	  --k1;
      }
    }

    if (k0 == 0 && k1 == seq_count - 1) {

      guppi_seq_boolean_set_all (op_bool, TRUE);

    } else {

      sb_i = 0;
      for (k = k0; k <= k1; ++k) {
	set_buffer[sb_i] = sorted_copy[k].i;
	++sb_i;
	if (sb_i == sb_N) {
	  guppi_seq_boolean_set_many (op_bool, set_buffer, sb_N, TRUE);
	  sb_i = 0;
	}
      }

      /* flush the set buffer */
      if (sb_i > 0)
	guppi_seq_boolean_set_many (op_bool, set_buffer, sb_i, TRUE);

    }

    if (do_and) {
      guppi_seq_boolean_bitwise_and (bseq, op_bool);
      guppi_unref0 (op_bool);
    }

    return guppi_seq_boolean_true_count (bseq);

  } else {
    /* no sorted copy available */
    g_assert_not_reached ();
  }

  return count;
}

GuppiSeqBoolean *
guppi_seq_scalar_range_query (GuppiSeqScalar * seq, double min, double max)
{
  GuppiSeqBoolean *bseq;

  guppi_2sort (&min, &max);

  g_return_val_if_fail (seq != NULL, NULL);

  bseq = GUPPI_SEQ_BOOLEAN (guppi_seq_boolean_new_aligned (GUPPI_SEQ (seq)));
  do_range_query (seq, bseq, min, max, FALSE);

  return bseq;
}

gint
guppi_seq_scalar_in_place_range_query (GuppiSeqScalar * seq,
				       GuppiSeqBoolean * bseq,
				       double min, double max)
{
  return do_range_query (seq, bseq, min, max, FALSE);
}

gint
guppi_seq_scalar_bitwise_and_range_query (GuppiSeqScalar * seq,
					  GuppiSeqBoolean * bseq,
					  double min, double max)
{
  return do_range_query (seq, bseq, min, max, TRUE);
}

static void
calc_minmax (GuppiSeqScalar * seq)
{
  GuppiSeqScalarPrivate *p;
  GuppiSeqScalarImpl *impl;
  GuppiSeqScalarImplClass *impl_class;

  p = priv (seq);
  impl = GUPPI_SEQ_SCALAR_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->range) {

    (impl_class->range) (impl, &p->min, &p->max);

  } else {
    gint i0 = guppi_seq_min_index (GUPPI_SEQ (seq));
    gint i1 = guppi_seq_max_index (GUPPI_SEQ (seq));
    gint i;
    double x0 = 0, x1 = 0, y;
    gboolean has_missing = guppi_seq_has_missing (GUPPI_SEQ (seq));
    gconstpointer ptr;
    gint stride;

    ptr = guppi_seq_scalar_raw (seq, &stride);

    while (has_missing && guppi_seq_missing (GUPPI_SEQ (seq), i0) && i0 <= i1)
      ++i0;

    if (i0 <= i1) {
      if (ptr)
	x0 = x1 = guppi_seq_scalar_raw_get (ptr, stride, i0);
      else
	x0 = x1 = guppi_seq_scalar_get (seq, i0);
    }

    for (i = i0 + 1; i <= i1; ++i) {
      if (!(has_missing && guppi_seq_missing (GUPPI_SEQ (seq), i))) {
	if (ptr)
	  y = guppi_seq_scalar_raw_get (ptr, stride, i);
	else
	  y = guppi_seq_scalar_get (seq, i);
	if (y < x0)
	  x0 = y;
	if (y > x1)
	  x1 = y;
      }
    }
    p->min = x0;
    p->max = x1;
  }

  p->have_minmax = TRUE;
}

double
guppi_seq_scalar_min (GuppiSeqScalar * seq)
{
  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq), 0);
  g_return_val_if_fail (guppi_seq_nonempty (GUPPI_SEQ (seq)), 0);

  if (!priv (seq)->have_minmax)
    calc_minmax (seq);

  return priv (seq)->min;
}

double
guppi_seq_scalar_max (GuppiSeqScalar * seq)
{
  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq), 0);
  g_return_val_if_fail (guppi_seq_nonempty (GUPPI_SEQ (seq)), 0);

  if (!priv (seq)->have_minmax)
    calc_minmax (seq);

  return priv (seq)->max;
}

double
guppi_seq_scalar_sum (GuppiSeqScalar * seq)
{
  GuppiSeqScalarPrivate *p;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq), 0);

  if (guppi_seq_empty (GUPPI_SEQ (seq)))
    return 0;

  p = priv (seq);

  if (!p->have_sum) {

    GuppiSeqScalarImpl *impl;
    GuppiSeqScalarImplClass *impl_class;

    impl = GUPPI_SEQ_SCALAR_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
    impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS (GTK_OBJECT (impl)->klass);

    if (impl_class->sum) {

      p->sum = (impl_class->sum) (impl);

    } else {
      gint i0 = guppi_seq_min_index (GUPPI_SEQ (seq));
      gint i1 = guppi_seq_max_index (GUPPI_SEQ (seq));
      gint i;
      double y = 0;
      gboolean has_missing = guppi_seq_has_missing (GUPPI_SEQ (seq));

      for (i = i0; i <= i1; ++i)
	if (!(has_missing && guppi_seq_missing (GUPPI_SEQ (seq), i)))
	  y += guppi_seq_scalar_get (seq, i);

      p->sum = y;
    }

    p->have_sum = TRUE;
  }

  return p->sum;
}

double
guppi_seq_scalar_sum_abs (GuppiSeqScalar * seq)
{
  GuppiSeqScalarPrivate *p;
  
  g_return_val_if_fail (seq && GUPPI_IS_SEQ_SCALAR (seq), 0);

  if (guppi_seq_empty (GUPPI_SEQ (seq)))
    return 0;

  p = priv (seq);

  if (!p->have_sum_abs) {
    gint i0 = guppi_seq_min_index (GUPPI_SEQ (seq));
    gint i1 = guppi_seq_max_index (GUPPI_SEQ (seq));
    gint i;
    double y = 0;
    gboolean has_missing = guppi_seq_has_missing (GUPPI_SEQ (seq));

    for (i = i0; i <= i1; ++i)
      if (!(has_missing && guppi_seq_missing (GUPPI_SEQ (seq), i)))
	y += fabs (guppi_seq_scalar_get (seq, i));

    p->sum_abs = y;
    p->have_sum_abs = TRUE;
  }

  return p->sum_abs;
}

double
guppi_seq_scalar_mean (GuppiSeqScalar * seq)
{
  double sum;
  gsize n;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq), 0);

  n = guppi_seq_count (GUPPI_SEQ (seq));
  g_return_val_if_fail (n > 0, 0);
  sum = guppi_seq_scalar_sum (seq);

  return sum / n;
}

double
guppi_seq_scalar_var (GuppiSeqScalar * seq)
{
  GuppiSeqScalarPrivate *p;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq), 0);

  p = priv (seq);

  if (!p->have_var) {

    const GuppiSeqScalarImpl *impl;
    GuppiSeqScalarImplClass *impl_class;

    impl = GUPPI_SEQ_SCALAR_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
    impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS (GTK_OBJECT (impl)->klass);

    if (impl_class->var) {

      p->var = (impl_class->var) (impl);

    } else {

      double x, om, mean = 0, sumsq = 0;
      gint i, i0, i1, count = 0, stride;
      gconstpointer ptr;
      gboolean has_missing;

      guppi_seq_indices (GUPPI_SEQ (seq), &i0, &i1);
      has_missing = guppi_seq_has_missing (GUPPI_SEQ (seq));

      ptr = guppi_seq_scalar_raw (seq, &stride);

      for (i = i0; i <= i1; ++i) {

	if (!(has_missing && guppi_seq_missing (GUPPI_SEQ (seq), i))) {

	  if (ptr)
	    x = guppi_seq_scalar_raw_get (ptr, stride, i);
	  else
	    x = guppi_seq_scalar_get (seq, i);

	  ++count;
	  om = mean;
	  mean += (x - mean) / count;
	  if (count > 1)
	    sumsq += (x - mean) * (x - om);
	}
      }

      p->var = sumsq / count;
    }

    p->have_var = TRUE;
  }

  return p->var;
}

double
guppi_seq_scalar_vars (GuppiSeqScalar * seq)
{
  double v;
  gsize n;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq), 0);

  n = guppi_seq_count (GUPPI_SEQ (seq));
  g_return_val_if_fail (n > 1, 0);

  v = guppi_seq_scalar_var (seq);
  return (n * (n - 1)) * v;
}

double
guppi_seq_scalar_sdev (GuppiSeqScalar * seq)
{
  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq), 0);
  return sqrt (guppi_seq_scalar_var (seq));
}

double
guppi_seq_scalar_sdevs (GuppiSeqScalar * seq)
{
  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq), 0);
  return sqrt (guppi_seq_scalar_vars (seq));
}

static void
calc_quartiles (GuppiSeqScalar * seq)
{
  GuppiSeqScalarPrivate *p;
  const GuppiSeqScalarImpl *impl;
  GuppiSeqScalarImplClass *impl_class;

  p = priv (seq);

  if (p->have_quartiles)
    return;

  impl = GUPPI_SEQ_SCALAR_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  /* Try to use an implementation-specific calculation. */
  p->have_quartiles = impl_class->quartiles &&
    (impl_class->quartiles) (impl, &p->q1, &p->median, &p->q3);

  /* If that doesn't work, use our fallback calculation. */
  if (!p->have_quartiles) {
    gsize N = guppi_seq_count (GUPPI_SEQ (seq));

    if (N > 0) {
      SortPair *sc = get_sorted_copy (seq);
      gint i;
      double t;

      g_assert (sc != NULL);

      t = 0.25 * (N - 1);
      i = (gint) t;
      p->q1 = (i + 1 - t) * sc[i].x + (t - i) * sc[i + 1].x;

      t = 0.5 * (N - 1);
      i = (gint) t;
      p->median = (i + 1 - t) * sc[i].x + (t - i) * sc[i + 1].x;

      t = 0.75 * (N - 1);
      i = (gint) t;
      p->q3 = (i + 1 - t) * sc[i].x + (t - i) * sc[i + 1].x;

    } else {

      p->q1 = p->median = p->q3 = 0;

    }
  }

  p->have_quartiles = TRUE;
}

double
guppi_seq_scalar_q1 (GuppiSeqScalar * seq)
{
  GuppiSeqScalarPrivate *p;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq), 0);
  g_return_val_if_fail (guppi_seq_nonempty (GUPPI_SEQ (seq)), 0);

  p = priv (seq);

  if (!p->have_quartiles)
    calc_quartiles (seq);

  return p->q1;
}

double
guppi_seq_scalar_median (GuppiSeqScalar * seq)
{
  GuppiSeqScalarPrivate *p;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq), 0);
  g_return_val_if_fail (guppi_seq_nonempty (GUPPI_SEQ (seq)), 0);

  p = priv (seq);

  if (!p->have_quartiles)
    calc_quartiles (seq);

  return p->median;
}

double
guppi_seq_scalar_q3 (GuppiSeqScalar * seq)
{
  GuppiSeqScalarPrivate *p;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq), 0);
  g_return_val_if_fail (guppi_seq_nonempty (GUPPI_SEQ (seq)), 0);

  p = priv (seq);

  if (!p->have_quartiles)
    calc_quartiles (seq);

  return p->q3;
}

double
guppi_seq_scalar_percentile (GuppiSeqScalar * seq, double p)
{
  const GuppiSeqScalarImpl *impl;
  GuppiSeqScalarImplClass *impl_class;
  double x;
  gint N;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_SCALAR (seq), 0);
  g_return_val_if_fail (guppi_seq_nonempty (GUPPI_SEQ (seq)), 0);
  g_return_val_if_fail (0 <= p && p <= 1, 0);

  impl = GUPPI_SEQ_SCALAR_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->percentile && impl_class->percentile (impl, p, &x))
    return x;

  N = guppi_seq_count (GUPPI_SEQ (seq));
  if (N > 0) {
    SortPair *sc = get_sorted_copy (seq);
    gint i;
    double t;

    t = p * (N - 1);
    i = (gint) t;
    return (i + 1 - t) * sc[i].x + (t - i) * sc[i + 1].x;

  }

  g_assert_not_reached ();
  return 0;
}

gconstpointer guppi_seq_scalar_raw (const GuppiSeqScalar * seq, gint * stride)
{
  const GuppiSeqScalarImpl *impl;
  GuppiSeqScalarImplClass *impl_class;

  g_return_val_if_fail (seq != NULL, NULL);
  g_return_val_if_fail (stride != NULL, NULL);

  impl = GUPPI_SEQ_SCALAR_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->raw_access == NULL)
    return NULL;

  return (impl_class->raw_access) (impl, stride);
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

enum {
  MIN_LABEL,
  MAX_LABEL,
  MEAN_LABEL,
  SDEV_LABEL,
  MEDIAN_LABEL,
  Q1_LABEL,
  Q3_LABEL,
  IQR_LABEL,
  LAST_LABEL
};

static const gchar *label_names [LAST_LABEL] = {
  N_("Min"), N_("Max"), N_("Mean"), N_("Std Dev"),
  N_("Median"), N_("Q1"), N_("Q3"), N_("IRQ")
};

typedef struct _info info;
struct _info {
  GuppiData *data;
  GtkLabel *labels[LAST_LABEL];
};

static void
push_state (GuppiData *data, gpointer user_data)
{
  GuppiSeqScalar *seq = GUPPI_SEQ_SCALAR (data);
  info *x = (info *)user_data;
  gchar buf[64];
  double q1, q3;

  g_snprintf (buf, 64, "%g", guppi_seq_scalar_min (seq));
  gtk_label_set_text (x->labels[MIN_LABEL], buf);

  g_snprintf (buf, 64, "%g", guppi_seq_scalar_max (seq));
  gtk_label_set_text (x->labels[MAX_LABEL], buf);

  g_snprintf (buf, 64, "%g", guppi_seq_scalar_mean (seq));
  gtk_label_set_text (x->labels[MEAN_LABEL], buf);

  g_snprintf (buf, 64, "%g", guppi_seq_scalar_sdev (seq));
  gtk_label_set_text (x->labels[SDEV_LABEL], buf);

  g_snprintf (buf, 64, "%g", guppi_seq_scalar_median (seq));
  gtk_label_set_text (x->labels[MEDIAN_LABEL], buf);

  g_snprintf (buf, 64, "%g", q1 = guppi_seq_scalar_q1 (seq));
  gtk_label_set_text (x->labels[Q1_LABEL], buf);

  g_snprintf (buf, 64, "%g", q3 = guppi_seq_scalar_q3 (seq));
  gtk_label_set_text (x->labels[Q3_LABEL], buf);

  g_snprintf (buf, 64, "%g", q3 - q1);
  gtk_label_set_text (x->labels[IQR_LABEL], buf);
}

static void
destroy_cb (GtkWidget *w, gpointer user_data)
{
  info *x = (info *)user_data;

  gtk_signal_disconnect_by_data (GTK_OBJECT (x->data), x);
  guppi_unref0 (x->data);
}

static GtkWidget *
info_display (GuppiData *data)
{
  info *x = guppi_new0 (info, 1);
  GtkTable *table = GTK_TABLE (gtk_table_new (LAST_LABEL, 2, FALSE));
  GtkWidget *w;
  gint i;

  x->data = data;
  guppi_ref (data);

  for (i=0; i<LAST_LABEL; ++i) {
    w = gtk_label_new (_(label_names[i]));
    gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
    gtk_table_attach (table, w, 0, 1, i, i+1, GTK_EXPAND | GTK_FILL, 0, 6, 1);
    gtk_widget_show (w);
    x->labels[i] = GTK_LABEL (gtk_label_new (""));
    gtk_table_attach_defaults (table, w = GTK_WIDGET (x->labels[i]),
			       1, 2, i, i+1);
    gtk_widget_show (w);
  }

  push_state (data, x);

  gtk_signal_connect (GTK_OBJECT (data), "changed",
		      GTK_SIGNAL_FUNC (push_state), x);
  gtk_signal_connect (GTK_OBJECT (table), "destroy",
		      GTK_SIGNAL_FUNC (destroy_cb), x);

  return GTK_WIDGET (table);
}

/* $Id: guppi-seq-scalar.c,v 1.27 2001/01/16 23:36:21 trow Exp $ */
