/* This is -*- C -*- */
/* $Id: guppi-legend-state.c,v 1.13 2000/12/14 20:22:57 trow Exp $ */

/*
 * guppi-legend-state.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <guppi-useful.h>
#include <guppi-seq-string.h>
#include <guppi-seq-categorical.h>
#include "guppi-legend-state.h"
#include "guppi-legend-view.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0,
  ARG_LABELS,
  ARG_STYLES,
  ARG_BOX_WIDTH,
  ARG_BOX_HEIGHT,
  ARG_BOX_EDGE_THICKNESS,
  ARG_BOX_EDGE_COLOR,
  ARG_EDGE_MARGIN,
  ARG_LABEL_OFFSET,
  ARG_LABEL_FONT,
  ARG_LABEL_COLOR
};

static void
guppi_legend_state_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiLegendState *state = GUPPI_LEGEND_STATE (obj);

  switch (arg_id) {

  case ARG_LABELS:
    GTK_VALUE_POINTER (*arg) = guppi_legend_state_labels (state);
    break;

  case ARG_STYLES:
    GTK_VALUE_POINTER (*arg) = guppi_legend_state_styles (state);
    break;

  case ARG_BOX_WIDTH:
    GTK_VALUE_DOUBLE (*arg) = guppi_legend_state_box_width (state);
    break;

  case ARG_BOX_HEIGHT:
    GTK_VALUE_DOUBLE (*arg) = guppi_legend_state_box_height (state);
    break;

  case ARG_BOX_EDGE_THICKNESS:
    GTK_VALUE_DOUBLE (*arg) = guppi_legend_state_box_edge_thickness (state);
    break;

  case ARG_BOX_EDGE_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_legend_state_box_edge_color (state);
    break;

  case ARG_EDGE_MARGIN:
    GTK_VALUE_DOUBLE (*arg) = guppi_legend_state_edge_margin (state);
    break;

  case ARG_LABEL_OFFSET:
    GTK_VALUE_DOUBLE (*arg) = guppi_legend_state_label_offset (state);
    break;

  case ARG_LABEL_FONT:
    GTK_VALUE_POINTER (*arg) = guppi_legend_state_label_font (state);
    break;

  case ARG_LABEL_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_legend_state_label_color (state);
    break;

  default:
    break;
  };
}

static void
guppi_legend_state_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiLegendState *state = GUPPI_LEGEND_STATE (obj);

  switch (arg_id) {

  case ARG_LABELS:
    guppi_legend_state_set_labels (state,
				   GUPPI_DATA0 (GTK_VALUE_POINTER (*arg)));
    break;

  case ARG_STYLES:
    guppi_legend_state_set_styles (state,
				   GUPPI_SEQ_STYLE0 (GTK_VALUE_POINTER
						     (*arg)));
    break;

  case ARG_BOX_WIDTH:
    guppi_legend_state_set_box_width (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_BOX_HEIGHT:
    guppi_legend_state_set_box_height (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_BOX_EDGE_THICKNESS:
    guppi_legend_state_set_box_edge_thickness (state,
					       GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_BOX_EDGE_COLOR:
    guppi_legend_state_set_box_edge_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_LABEL_OFFSET:
    guppi_legend_state_set_label_offset (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_LABEL_FONT:
    guppi_legend_state_set_label_font (state,
				       GNOME_FONT (GTK_VALUE_POINTER (*arg)));
    break;

  case ARG_LABEL_COLOR:
    guppi_legend_state_set_label_color (state, GTK_VALUE_UINT (*arg));
    break;

  default:
    break;
  };
}

static void
guppi_legend_state_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_legend_state_finalize (GtkObject * obj)
{
  GuppiLegendState *state = GUPPI_LEGEND_STATE (obj);

  guppi_unref0 (state->label_font);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/*****************************************************************************/

#define add_arg(str, t, symb) \
gtk_object_add_arg_type("GuppiLegendState::" str, t, GTK_ARG_READWRITE, symb)

#define add_arg_w(str, t, symb) \
gtk_object_add_arg_type("GuppiLegendState::" str, t, GTK_ARG_WRITABLE, symb)

static void
guppi_legend_state_class_init (GuppiLegendStateClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiElementStateClass *state_class = GUPPI_ELEMENT_STATE_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_ELEMENT_STATE);

  object_class->get_arg = guppi_legend_state_get_arg;
  object_class->set_arg = guppi_legend_state_set_arg;
  object_class->destroy = guppi_legend_state_destroy;
  object_class->finalize = guppi_legend_state_finalize;

  state_class->view_type = GUPPI_TYPE_LEGEND_VIEW;

  add_arg ("labels", GTK_TYPE_POINTER, ARG_LABELS);
  add_arg ("styles", GTK_TYPE_POINTER, ARG_STYLES);
  add_arg ("box_width", GTK_TYPE_DOUBLE, ARG_BOX_WIDTH);
  add_arg ("box_height", GTK_TYPE_DOUBLE, ARG_BOX_HEIGHT);
  add_arg ("box_edge_thickness", GTK_TYPE_DOUBLE, ARG_BOX_EDGE_THICKNESS);
  add_arg ("box_edge_color", GTK_TYPE_UINT, ARG_BOX_EDGE_COLOR);
  add_arg ("edge_margin", GTK_TYPE_DOUBLE, ARG_EDGE_MARGIN);
  add_arg ("label_offset", GTK_TYPE_DOUBLE, ARG_LABEL_OFFSET);
  add_arg ("label_font", GTK_TYPE_POINTER, ARG_LABEL_FONT);
  add_arg ("label_color", GTK_TYPE_UINT, ARG_LABEL_COLOR);

}

static void
guppi_legend_state_init (GuppiLegendState * obj)
{
  guppi_element_state_add_shared (GUPPI_ELEMENT_STATE (obj),
				  SHARED_LABEL_DATA, guppi_shared_data ());

  guppi_element_state_add_shared (GUPPI_ELEMENT_STATE (obj),
				  SHARED_STYLE_DATA, guppi_shared_data ());

  obj->box_width = guppi_in2pt (1 / 4.0);
  obj->box_height = guppi_in2pt (1 / 4.0);
  obj->box_edge_thickness = guppi_in2pt (1 / 48.0);
  obj->box_edge_color = RGBA_BLACK;
  obj->edge_margin = guppi_in2pt (1 / 8.0);
  obj->label_offset = guppi_in2pt (1 / 16.0);

  obj->label_font = guppi_default_font ();
  guppi_ref (obj->label_font);

  obj->label_color = RGBA_BLACK;
}

GtkType guppi_legend_state_get_type (void)
{
  static GtkType guppi_legend_state_type = 0;
  if (!guppi_legend_state_type) {
    static const GtkTypeInfo guppi_legend_state_info = {
      "GuppiLegendState",
      sizeof (GuppiLegendState),
      sizeof (GuppiLegendStateClass),
      (GtkClassInitFunc) guppi_legend_state_class_init,
      (GtkObjectInitFunc) guppi_legend_state_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_legend_state_type =
      gtk_type_unique (GUPPI_TYPE_ELEMENT_STATE, &guppi_legend_state_info);
  }
  return guppi_legend_state_type;
}

GuppiElementState *
guppi_legend_state_new (void)
{
  return GUPPI_ELEMENT_STATE (guppi_type_new (guppi_legend_state_get_type ()));
}

/*****************************************************************************/

GuppiData *
guppi_legend_state_labels (GuppiLegendState * state)
{
  GtkObject *obj;

  g_return_val_if_fail (state != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_LEGEND_STATE (state), NULL);

  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (state),
					SHARED_LABEL_DATA);
  return GUPPI_DATA0 (obj);
}

GuppiSeqStyle *
guppi_legend_state_styles (GuppiLegendState * state)
{
  GtkObject *obj;

  g_return_val_if_fail (state != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_LEGEND_STATE (state), NULL);

  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (state),
					SHARED_STYLE_DATA);
  return GUPPI_SEQ_STYLE0 (obj);
}

void
guppi_legend_state_set_labels (GuppiLegendState * state, GuppiData * data)
{
  g_return_if_fail (state && GUPPI_IS_LEGEND_STATE (state));
  g_return_if_fail (data == NULL || GUPPI_IS_DATA (data));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (state),
				  SHARED_LABEL_DATA, data);
}

void
guppi_legend_state_set_styles (GuppiLegendState * state, GuppiSeqStyle * sd)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_LEGEND_STATE (state));
  g_return_if_fail (sd == NULL || GUPPI_IS_SEQ_STYLE (sd));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (state),
				  SHARED_STYLE_DATA, sd);
}


void
guppi_legend_state_set_box_width (GuppiLegendState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_LEGEND_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->box_width != x) {
    state->box_width = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_legend_state_set_box_height (GuppiLegendState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_LEGEND_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->box_height != x) {
    state->box_height = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_legend_state_set_box_edge_thickness (GuppiLegendState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_LEGEND_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->box_edge_thickness != x) {
    state->box_edge_thickness = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_legend_state_set_box_edge_color (GuppiLegendState * state, guint32 x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_LEGEND_STATE (state));

  if (state->box_edge_color != x) {
    state->box_edge_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_legend_state_set_edge_margin (GuppiLegendState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_LEGEND_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->edge_margin != x) {
    state->edge_margin = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_legend_state_set_label_offset (GuppiLegendState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_LEGEND_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->label_offset != x) {
    state->label_offset = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_legend_state_set_label_font (GuppiLegendState * state, GnomeFont * f)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_LEGEND_STATE (state));
  g_return_if_fail (f != NULL);
  g_return_if_fail (GNOME_IS_FONT (f));

  if (state->label_font != f) {

    guppi_refcounting_assign (state->label_font, f);

    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_legend_state_set_label_color (GuppiLegendState * state, guint32 x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_LEGEND_STATE (state));

  if (state->label_color != x) {
    state->label_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}


/*****************************************************************************/

void
guppi_legend_state_entry_bounds (GuppiLegendState * state,
				 gint * i0, gint * i1)
{
  GuppiData *labels;
  GuppiSeqStyle *ssty;

  g_return_if_fail (state && GUPPI_IS_LEGEND_STATE (state));

  if (i0) *i0 = 0;
  if (i1) *i1 = -1;

  ssty = guppi_legend_state_styles (state);
  labels = guppi_legend_state_labels (state);

  if (ssty == NULL || labels == NULL)
    return;

  if (GUPPI_IS_SEQ_STRING (labels)) {
    guppi_seq_common_bounds (GUPPI_SEQ (ssty), GUPPI_SEQ (labels), i0, i1);
  } else if (GUPPI_IS_SEQ_CATEGORICAL (labels)) {
    GuppiCategory *cat;
    cat = guppi_seq_categorical_category (GUPPI_SEQ_CATEGORICAL (labels));
    if (i0) *i0 = 0;
    if (i1) *i1 = (gint)guppi_category_size (cat) - 1;
  } else {
    g_warning ("Unknown data type.");
  }
}

guint32
guppi_legend_state_entry_color (GuppiLegendState * state, gint i)
{
  GuppiSeqStyle *ss;
  GuppiStyle *sty;
  gint i0, i1;

  g_return_val_if_fail (state && GUPPI_IS_LEGEND_STATE (state), 0);

  guppi_legend_state_entry_bounds (state, &i0, &i1);
  g_return_val_if_fail (i0 <= i && i <= i1, 0);

  ss = guppi_legend_state_styles (state);

  if (ss != NULL) {
    sty = guppi_seq_style_get (ss, i);
    return guppi_style_color (sty);
  }

  return 0;
}

const gchar *
guppi_legend_state_entry_text (GuppiLegendState * state, gint i)
{
  GuppiData *labels;
  gint i0, i1;

  g_return_val_if_fail (state != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_LEGEND_STATE (state), NULL);

  guppi_legend_state_entry_bounds (state, &i0, &i1);
  g_return_val_if_fail (i0 <= i && i <= i1, NULL);

  labels = guppi_legend_state_labels (state);
  
  if (GUPPI_IS_SEQ_STRING (labels)) 
    return guppi_seq_string_get (GUPPI_SEQ_STRING (labels), i);
  else if (GUPPI_IS_SEQ_CATEGORICAL (labels)) {
    GuppiCategory *cat;
    cat = guppi_seq_categorical_category (GUPPI_SEQ_CATEGORICAL (labels));
    return guppi_category_find_by_code (cat, i);
  }

  g_warning ("Unknown data type.");
  return NULL;
}

double
guppi_legend_state_max_label_width (GuppiLegendState * state)
{
  GnomeFont *font;
  gint i, i0, i1;
  double max = 0;

  g_return_val_if_fail (state != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_LEGEND_STATE (state), 0);

  font = guppi_legend_state_label_font (state);
  guppi_legend_state_entry_bounds (state, &i0, &i1);

  for (i = i0; i <= i1; ++i) {
    const gchar *txt = guppi_legend_state_entry_text (state, i);
    double len = gnome_font_get_width_string (font, txt);
    max = MAX (len, max);
  }

  return max;
}

double
guppi_legend_state_natural_width (GuppiLegendState * state)
{
  double nat_w = 0;

  g_return_val_if_fail (state != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_LEGEND_STATE (state), 0);

  nat_w += 2 * guppi_legend_state_edge_margin (state);
  nat_w += guppi_legend_state_label_offset (state);
  nat_w += guppi_legend_state_max_label_width (state);
  nat_w += guppi_legend_state_box_width (state);

  return nat_w;
}

double
guppi_legend_state_natural_height (GuppiLegendState * state)
{
  GnomeFont *font;
  double nat_h = 0;
  double natural_skip;
  gint i0, i1, N;

  g_return_val_if_fail (state != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_LEGEND_STATE (state), 0);

  font = guppi_legend_state_label_font (state);

  natural_skip = MAX (0.5 * gnome_font_get_ascender (font),
		      guppi_legend_state_box_edge_thickness (state));

  guppi_legend_state_entry_bounds (state, &i0, &i1);
  N = i1 - i0 + 1;

  nat_h += 2 * guppi_legend_state_edge_margin (state);
  nat_h += (N - 1) * natural_skip;
  nat_h += N * MAX (guppi_legend_state_box_height (state),
		    gnome_font_get_ascender (font));

  return nat_h;
}



/* $Id: guppi-legend-state.c,v 1.13 2000/12/14 20:22:57 trow Exp $ */
