/* This is -*- C -*- */
/* $Id: guppi-seq-integer-core-impl.c,v 1.6 2001/01/16 23:35:52 trow Exp $ */

/*
 * guppi-seq-integer-core-impl.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
/* #include <gnome.h> */
#include <stdlib.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-convenient.h>
#include <guppi-data-impl-plug-in.h>
#include "guppi-seq-integer-core-impl.h"


static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_integer_core_impl_get_arg (GtkObject * obj, GtkArg * arg,
				     guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_integer_core_impl_set_arg (GtkObject * obj, GtkArg * arg,
				     guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_integer_core_impl_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_seq_integer_core_impl_finalize (GtkObject * obj)
{
  GuppiSeqIntegerCoreImpl *impl = GUPPI_SEQ_INTEGER_CORE_IMPL (obj);

  guppi_unref0 (impl->garray);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/*****************************************************************************/

static void
v_seq_integer_range (const GuppiSeqIntegerImpl * impl, gint * min, gint * max)
{
  const GuppiSeqIntegerCoreImpl *core = GUPPI_SEQ_INTEGER_CORE_IMPL (impl);
  gint x0 = 0, x1 = 0, i, a;
  const gint *ptr = (const gint *) guppi_garray_data (core->garray);

  if (!core->cached_minmax) {

    if (core->size) {
      x0 = x1 = ptr[0];
      for (i = 1; i < core->size; ++i) {
	a = ptr[i];
	if (a < x0)
	  x0 = a;
	if (a > x1)
	  x1 = a;
      }
    }

    /* Casting out const is not nice. */
    ((GuppiSeqIntegerCoreImpl *) core)->min = x0;
    ((GuppiSeqIntegerCoreImpl *) core)->max = x1;
    ((GuppiSeqIntegerCoreImpl *) core)->cached_minmax = TRUE;
  }

  if (min)
    *min = core->min;

  if (max)
    *max = core->max;
}

static gint
v_seq_integer_frequency (const GuppiSeqIntegerImpl * impl, gint k)
{
  const GuppiSeqIntegerCoreImpl *core = GUPPI_SEQ_INTEGER_CORE_IMPL (impl);
  gint count = 0, i;
  const gint *ptr = (const gint *) guppi_garray_data (core->garray);

  for (i = 0; i < core->size; ++i)
    if (k == ptr[i])
      ++count;

  return count;
}

static gint
v_seq_integer_get (const GuppiSeqIntegerImpl * impl, gint i)
{
  const GuppiSeqIntegerCoreImpl *core = GUPPI_SEQ_INTEGER_CORE_IMPL (impl);
  const gint *ptr = (const gint *) guppi_garray_data (core->garray);

  return ptr[i - core->index_basis];
}

static void
v_seq_integer_set (GuppiSeqIntegerImpl * impl, gint i, gint val)
{
  GuppiSeqIntegerCoreImpl *core = GUPPI_SEQ_INTEGER_CORE_IMPL (impl);
  gint *ptr = (gint *) guppi_garray_data (core->garray);
  gint old_val;

  i -= core->index_basis;

  old_val = ptr[i];

  if (core->cached_minmax) {

    if (val < core->min)
      core->min = val;

    if (val > core->max)
      core->max = val;

    if ((old_val == core->min && val != core->min) ||
	(old_val == core->max && val != core->max))
      core->cached_minmax = FALSE;
  }

  ptr[i] = val;
}

static void
v_seq_integer_insert (GuppiSeqIntegerImpl * impl, gint i,
		      const gint * ptr, gsize N)
{
  GuppiSeqIntegerCoreImpl *core = GUPPI_SEQ_INTEGER_CORE_IMPL (impl);
  gint *data;
  gint a, j;

  i -= core->index_basis;

  if (guppi_garray_size (core->garray) <= core->size)
    guppi_garray_set_size (core->garray,
			   MAX (MAX (20, 2 * core->size),
				core->size + 2 * N));

  if (core->cached_minmax) {
    for (j = 0; j < N; ++j) {
      a = ptr[j];
      if (a < core->min)
	core->min = a;
      if (a > core->max)
	core->max = a;
    }
  }

  data = (gint *) guppi_garray_data (core->garray);
  for (j = core->size - N; i <= j; --j)
    data[j + N] = data[j];
  for (j = 0; j < N; ++j)
    data[i + j] = ptr[j];
  core->size += N;
}

static void
v_seq_size_hint (GuppiSeqImpl * impl, gsize N)
{
  GuppiSeqIntegerCoreImpl *core = GUPPI_SEQ_INTEGER_CORE_IMPL (impl);

  if (guppi_garray_size (core->garray) < N)
    guppi_garray_set_size (core->garray, N);
}

static void
v_seq_get_bounds (const GuppiSeqImpl * impl, gint * min, gint * max)
{
  const GuppiSeqIntegerCoreImpl *core = GUPPI_SEQ_INTEGER_CORE_IMPL (impl);

  if (min)
    *min = core->index_basis;

  if (max)
    *max = core->index_basis - 1 + core->size;
}

static void
v_seq_shift_indices (GuppiSeqImpl * impl, gint delta)
{
  GuppiSeqIntegerCoreImpl *core = GUPPI_SEQ_INTEGER_CORE_IMPL (impl);
  core->index_basis += delta;
}

static gboolean
v_seq_validate (const GuppiSeqImpl * impl, const gchar * str,
		gchar * error_msg, gsize errlen)
{
  return TRUE;
}

static void
v_seq_get (const GuppiSeqImpl * impl, gint i, gchar * buf, gsize len)
{
  gint val;
  val = v_seq_integer_get (GUPPI_SEQ_INTEGER_IMPL (impl), i);
  g_snprintf (buf, len, "%d", val);
}

static void
v_seq_set (GuppiSeqImpl * impl, gint i, const gchar * buf)
{
  v_seq_integer_set (GUPPI_SEQ_INTEGER_IMPL (impl), i, atoi (buf));
}

static void
v_seq_insert (GuppiSeqImpl * impl, gint i, const gchar * buf)
{
  gint foo = atoi (buf);
  v_seq_integer_insert (GUPPI_SEQ_INTEGER_IMPL (impl), i, &foo, 1);
}

static void
v_seq_delete_many (GuppiSeqImpl * impl, gint i, gsize N)
{
  GuppiSeqIntegerCoreImpl *core = GUPPI_SEQ_INTEGER_CORE_IMPL (impl);
  gint *data;
  gint j;

  /* This could be optimized. */
  core->cached_minmax = FALSE;

  data = (gint *) guppi_garray_data (core->garray);
  i -= core->index_basis;

  for (j = i; j + N < core->size; ++j)
    data[j] = data[j + N];

  core->size -= N;
}

static GuppiDataImpl *
v_data_copy (GuppiDataImpl * impl)
{
  GuppiSeqIntegerCoreImpl *core = GUPPI_SEQ_INTEGER_CORE_IMPL (impl);
  GuppiSeqIntegerCoreImpl *copy;
  gint i;
  gint *data;
  gint *copy_data;

  copy =
    GUPPI_SEQ_INTEGER_CORE_IMPL (guppi_type_new
				 (GUPPI_TYPE_SEQ_INTEGER_CORE_IMPL));

  copy->index_basis = core->index_basis;
  copy->size = core->size;
  guppi_garray_set_size (copy->garray, core->size);

  data = (gint *) guppi_garray_data (core->garray);
  copy_data = (gint *) guppi_garray_data (copy->garray);

  for (i = 0; i < core->size; ++i)
    copy_data[i] = data[i];

  return GUPPI_DATA_IMPL (copy);
}

static gint
v_data_size_in_bytes (GuppiDataImpl * impl)
{
  GuppiSeqIntegerCoreImpl *core = GUPPI_SEQ_INTEGER_CORE_IMPL (impl);

  return guppi_garray_size (core->garray) * sizeof (double) +
    sizeof (GuppiSeqIntegerCoreImpl);
}

/*****************************************************************************/

static void
guppi_seq_integer_core_impl_class_init (GuppiSeqIntegerCoreImplClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiDataImplClass *data_class = GUPPI_DATA_IMPL_CLASS (klass);
  GuppiSeqImplClass *seq_class = GUPPI_SEQ_IMPL_CLASS (klass);
  GuppiSeqIntegerImplClass *seq_integer_class =
    GUPPI_SEQ_INTEGER_IMPL_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_SEQ_INTEGER_IMPL);

  object_class->get_arg = guppi_seq_integer_core_impl_get_arg;
  object_class->set_arg = guppi_seq_integer_core_impl_set_arg;
  object_class->destroy = guppi_seq_integer_core_impl_destroy;
  object_class->finalize = guppi_seq_integer_core_impl_finalize;

  data_class->impl_name = _("Core Integer Sequence");

  seq_integer_class->range = v_seq_integer_range;
  seq_integer_class->frequency = v_seq_integer_frequency;
  seq_integer_class->get = v_seq_integer_get;
  seq_integer_class->set = v_seq_integer_set;
  seq_integer_class->insert = v_seq_integer_insert;

  seq_class->size_hint = v_seq_size_hint;
  seq_class->get_bounds = v_seq_get_bounds;
  seq_class->shift_indices = v_seq_shift_indices;

  seq_class->validate = v_seq_validate;
  seq_class->get = v_seq_get;
  seq_class->set = v_seq_set;
  seq_class->insert = v_seq_insert;
  seq_class->delete_many = v_seq_delete_many;

  data_class->copy = v_data_copy;
  data_class->get_size_in_bytes = v_data_size_in_bytes;

}

static void
guppi_seq_integer_core_impl_init (GuppiSeqIntegerCoreImpl * obj)
{
  obj->index_basis = 0;
  obj->size = 0;
  obj->garray = guppi_garray_new (sizeof (gint));
}

GtkType guppi_seq_integer_core_impl_get_type (void)
{
  static GtkType guppi_seq_integer_core_impl_type = 0;
  if (!guppi_seq_integer_core_impl_type) {
    static const GtkTypeInfo guppi_seq_integer_core_impl_info = {
      "GuppiSeqIntegerCoreImpl",
      sizeof (GuppiSeqIntegerCoreImpl),
      sizeof (GuppiSeqIntegerCoreImplClass),
      (GtkClassInitFunc) guppi_seq_integer_core_impl_class_init,
      (GtkObjectInitFunc) guppi_seq_integer_core_impl_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_seq_integer_core_impl_type =
      gtk_type_unique (GUPPI_TYPE_SEQ_INTEGER_IMPL,
		       &guppi_seq_integer_core_impl_info);
  }
  return guppi_seq_integer_core_impl_type;
}

/****************************************************************************/

static GuppiDataImpl *
make_impl (void)
{
  return
    GUPPI_DATA_IMPL (guppi_type_new (guppi_seq_integer_core_impl_get_type ()));
}

GuppiPlugIn *guppi_plug_in (void);

GuppiPlugIn *
guppi_plug_in (void)
{
  GuppiPlugIn *pi;
  GuppiDataImplPlugIn *dimpi;

  pi = guppi_data_impl_plug_in_new ();
  dimpi = GUPPI_DATA_IMPL_PLUG_IN (pi);

  pi->magic_number = GUPPI_PLUG_IN_MAGIC_NUMBER;
  dimpi->impl_constructor = make_impl;

  return pi;
}


/* $Id: guppi-seq-integer-core-impl.c,v 1.6 2001/01/16 23:35:52 trow Exp $ */
