/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-output-xml.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "guppi-memory.h"
#include "guppi-output-xml.h"

#define GUPPI_XML_PREFIX "gpi"
#define INDENT_SIZE 2

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_output_xml_get_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_output_xml_set_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_output_xml_destroy (GtkObject *obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_output_xml_finalize (GtkObject *obj)
{
  GuppiOutputXML *out = GUPPI_OUTPUT_XML (obj);

  guppi_finalized (obj);

  guppi_output_xml_close (out);
  
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
guppi_output_xml_class_init (GuppiOutputXMLClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  parent_class = gtk_type_class (GTK_TYPE_OBJECT);

  object_class->get_arg = guppi_output_xml_get_arg;
  object_class->set_arg = guppi_output_xml_set_arg;
  object_class->destroy = guppi_output_xml_destroy;
  object_class->finalize = guppi_output_xml_finalize;

}

static void
guppi_output_xml_init (GuppiOutputXML *obj)
{
  obj->fd = -1;
}

GtkType guppi_output_xml_get_type (void)
{
  static GtkType guppi_output_xml_type = 0;
  if (!guppi_output_xml_type) {
    static const GtkTypeInfo guppi_output_xml_info = {
      "GuppiOutputXML",
      sizeof (GuppiOutputXML),
      sizeof (GuppiOutputXMLClass),
      (GtkClassInitFunc) guppi_output_xml_class_init,
      (GtkObjectInitFunc) guppi_output_xml_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_output_xml_type =
      gtk_type_unique (GTK_TYPE_OBJECT, &guppi_output_xml_info);
  }
  return guppi_output_xml_type;
}

GuppiOutputXML *
guppi_output_xml_new (void)
{
  GtkObject *obj = guppi_type_new (guppi_output_xml_get_type ());
  return GUPPI_OUTPUT_XML (obj);
}

void
guppi_output_xml_open_file (GuppiOutputXML *out, const gchar *filename)
{
  gint fd;

  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (filename);

  fd = creat (filename, 00644);
  g_return_if_fail (fd >= 0);
  guppi_output_xml_open_fd (out, fd);
  out->own_fd = TRUE;
}

void
guppi_output_xml_open_fd (GuppiOutputXML *out, gint fd)
{
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (fd >= 0);

  if (out->fd >= 0)
    guppi_output_xml_close (out);

  out->fd = fd;
  out->zfd = gzdopen (fd, "w");

  out->own_fd = FALSE;
}

void
guppi_output_xml_close (GuppiOutputXML *out)
{
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));

  if (out->zfd)
    gzclose (out->zfd);
  else if (out->fd >= 0 && out->own_fd)
    close (out->fd);
  out->fd = -1;
  out->zfd = NULL;
}

/****************************************************************************/

static void
push_tag (GuppiOutputXML *out, const gchar *str)
{
  out->tag_stack = g_list_prepend (out->tag_stack, guppi_strdup (str));
  out->indent += INDENT_SIZE;
}

static gchar *
pop_tag (GuppiOutputXML *out)
{
  gchar *tag;

  g_return_val_if_fail (out->tag_stack, NULL);
  tag = (gchar *) out->tag_stack->data;
  out->tag_stack = g_list_remove (out->tag_stack, tag);
  out->indent -= INDENT_SIZE;
  return tag;
}

static void
transform_quotes (gchar *p)
{
  g_return_if_fail (p);

  while (*p) {
    if (*p == '\'')
      *p = '"';
    ++p;
  }
}

/****************************************************************************/

void
guppi_output_xml_print (GuppiOutputXML *out, const gchar *str)
{
  gint i;
  gint len;

  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (str);

  if (!out->topmatter) {
    out->topmatter = TRUE;
    guppi_output_xml_print (out, "<?xml version=\"1.0\"?>");
  }

  if (out->zfd) {

    for (i = 0; i < out->indent; ++i)
      gzwrite (out->zfd, " ", 1);

    len = strlen (str);
    gzwrite (out->zfd, (gpointer)str, len);

    gzwrite (out->zfd, "\n", 1);

  } else if (out->fd >= 0) {

    for (i = 0; i < out->indent; ++i)
      write (out->fd, " ", 1);

    len = strlen(str);
    while (len) {
      gint n = write (out->fd, str, len);
      if (n == -1) {
	g_warning ("GuppiOutputXML write error.");
	return;
      }
      len -=n;
      str += n;
    }
    
    write (out->fd, "\n", 1); /* add a trailing newline */
  } else {

    for (i = 0; i < out->indent; ++i)
      putc (' ', stdout);

    puts (str);

  }
}

void
guppi_output_xml_printf (GuppiOutputXML *out, const gchar *format_str, ...)
{
  va_list args;

  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (format_str);

  va_start (args, format_str);
  guppi_output_xml_vprintf (out, format_str, args);
  va_end (args);
}

void
guppi_output_xml_vprintf (GuppiOutputXML *out, const gchar *format_str,
			  va_list args)
{
  gchar *buffer;

  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (format_str);

  buffer = guppi_strdup_vprintf (format_str, args);

  guppi_output_xml_print (out, buffer);
  guppi_free (buffer);
}

void
guppi_output_xml_element (GuppiOutputXML *out, const gchar *tag,
			  const gchar *body)
{
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (tag);
  g_return_if_fail (body);

  guppi_output_xml_printf (out, "<%s:%s>%s</%s:%s>",
			   GUPPI_XML_PREFIX, tag,
			   body,
			   GUPPI_XML_PREFIX, tag);
}

void
guppi_output_xml_elementf (GuppiOutputXML *out, const gchar *tag,
			   const gchar *body_format, ...)
{
  va_list args;
  gchar *body_buffer;
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (tag);
  g_return_if_fail (body_format);
  va_start (args, body_format);
  body_buffer = guppi_strdup_vprintf (body_format, args);
  guppi_output_xml_element (out, tag, body_buffer);
  guppi_free (body_buffer);
  va_end (args);
}

void
guppi_output_xml_tag (GuppiOutputXML *out, const gchar *tag)
{
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (tag);
  guppi_output_xml_printf (out, "<%s:%s>", GUPPI_XML_PREFIX, tag);
  push_tag (out, tag);
}

void
guppi_output_xml_tagf (GuppiOutputXML *out,
		       const gchar *tag, const gchar *attr_format, ...)
{
  va_list args;
  gchar *attr_buffer;
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (tag);
  g_return_if_fail (attr_format);
  va_start (args, attr_format);
  attr_buffer = guppi_strdup_vprintf (attr_format, args);
  transform_quotes (attr_buffer);
  guppi_output_xml_printf (out, "<%s:%s %s>",
			   GUPPI_XML_PREFIX, tag, attr_buffer);
  push_tag (out, tag);
  guppi_free (attr_buffer);
  va_end (args);
}

void
guppi_output_xml_end_tag (GuppiOutputXML *out)
{
  gchar *tag;
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  tag = pop_tag (out);
  g_return_if_fail (tag);
  guppi_output_xml_printf (out, "</%s:%s>", GUPPI_XML_PREFIX, tag);
  guppi_free (tag);
}

void
guppi_output_xml_end_tag_check (GuppiOutputXML *out,
				const gchar *expected_tag)
{
  /* Code dup from previous function --- I should know better by this
     point in my life. */
  gchar *tag;
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (expected_tag);
  tag = pop_tag (out);
  g_return_if_fail (tag);
  if (strcmp (tag, expected_tag))
    g_error ("XML tags out of balance: expected \"%s\", got \"%s\"",
	     expected_tag, tag);
  guppi_output_xml_printf (out, "</%s:%s>", GUPPI_XML_PREFIX, tag);
  guppi_free (tag);
}

void
guppi_output_xml_empty_tag (GuppiOutputXML *out, const gchar *tag)
{
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (tag);
  guppi_output_xml_printf (out, "<%s:%s/>", GUPPI_XML_PREFIX, tag);
}

void
guppi_output_xml_empty_tagf (GuppiOutputXML *out,
			     const gchar *tag,
			     const gchar *attr_format, ...)
{
  va_list args;
  gchar *attr_buffer;
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (attr_format);
  va_start (args, attr_format);
  attr_buffer = guppi_strdup_vprintf (attr_format, args);
  transform_quotes (attr_buffer);
  guppi_output_xml_printf (out, "<%s:%s %s/>",
			   GUPPI_XML_PREFIX, tag, attr_buffer);
  guppi_free (attr_buffer);
  va_end (args);
}

void
guppi_output_xml_comment (GuppiOutputXML *out, const gchar *comment_text)
{
  gchar *buffer;
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (comment_text != NULL);
  buffer = guppi_strdup_printf ("<!-- %s -->", comment_text);
  guppi_output_xml_print (out, buffer);
  guppi_free (buffer);
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

void
guppi_output_xml_attribute_start (GuppiOutputXML *out, const gchar *name)
{
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (name);

  guppi_output_xml_tag (out, "Attribute");
  guppi_output_xml_element (out, "name", name);
  guppi_output_xml_tag (out, "value");
}

void
guppi_output_xml_attribute_end (GuppiOutputXML *out)
{
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  guppi_output_xml_end_tag_check (out, "value");
  guppi_output_xml_end_tag_check (out, "Attribute");
}

void
guppi_output_xml_attribute (GuppiOutputXML *out,
			    const gchar *name,
			    const gchar *value)
{
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (name && value);
  
  guppi_output_xml_tag (out, "Attribute");
  guppi_output_xml_element (out, "name", name);
  guppi_output_xml_element (out, "value", value);
  guppi_output_xml_end_tag_check (out, "Attribute");

}

void
guppi_output_xml_attributef (GuppiOutputXML *out,
			     const gchar *name,
			     const gchar *value_format,
			     ...)
{
  va_list args;
  gchar *value_buf;

  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));
  g_return_if_fail (name && value_format);

  va_start (args, value_format);
  value_buf = guppi_strdup_vprintf (value_format, args);

  guppi_output_xml_attribute (out, name, value_buf);

  guppi_free (value_buf);
  va_end (args);
}

