/* $Id: guppi-scatter-item.c,v 1.27 2000/12/16 03:42:46 trow Exp $ */

/*
 * guppi-scatter-item.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <guppi-useful.h>
#include "guppi-scatter-tools.h"
#include "guppi-scatter-item.h"

static GtkObjectClass *parent_class = NULL;

static void
guppi_scatter_item_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_scatter_item_finalize (GtkObject * obj)
{
  GuppiScatterItem *item = GUPPI_SCATTER_ITEM (obj);
  gint i;

  guppi_unref0 (item->local_x_visibility_mask);
  guppi_unref0 (item->local_y_visibility_mask);

  for (i = 0; i < GUPPI_SCATTER_STATE_STYLE_COUNT; ++i)
    guppi_unref0 (item->style_view[i]);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}


/*****************************************************************************/

static void
state_changed (GuppiCanvasItem * gci)
{
  guppi_canvas_item_request_update (gci);
}

static void
view_changed (GuppiCanvasItem * gci)
{
  GuppiScatterItem *item = GUPPI_SCATTER_ITEM (gci);

  /* Request speed rendering, if we aren't doing it already... */
  if (item->speed_render == 0)
    item->speed_render = 1;

  guppi_canvas_item_request_update (gci);
}

static gint
query_slack (GuppiCanvasItem * gci)
{
  GuppiScatterState *state =
    GUPPI_SCATTER_STATE (guppi_canvas_item_state (gci));
  double max_size = 0;
  gint i;

  /* A hack to estimate the maximum size of our markers */
  for (i = 0; i < GUPPI_SCATTER_STATE_STYLE_COUNT; ++i) {
    GuppiScatterStyle *gss = guppi_scatter_state_get_style (state, i);
    if (gss) {
      double q = 0;
      q += guppi_scatter_style_size1_gradient_max (gss);
      q += guppi_scatter_style_size2_gradient_max (gss);
      q *= 2;
      max_size = MAX (max_size, q);
    }
  }

  max_size = guppi_pt2px (max_size) * guppi_canvas_item_scale (gci);

  return 3 + (gint) ceil (max_size);
}

static void
update_style (GuppiCanvasItem * gci, gint i)
{
  GuppiScatterState *state =
    GUPPI_SCATTER_STATE (guppi_canvas_item_state (gci));
  GuppiScatterStyleView *sv;

  g_return_if_fail (i >= 0 && i < GUPPI_SCATTER_STATE_STYLE_COUNT);

  sv = GUPPI_SCATTER_ITEM (gci)->style_view[i];
  guppi_scatter_style_view_set_scale (sv, guppi_canvas_item_scale (gci));
  guppi_scatter_style_view_set_style (sv, state->style[i]);

}

static void
choose_speed_rendering_level (GuppiCanvasItem * gci, gint vis_points)
{
  GuppiScatterItem *item = GUPPI_SCATTER_ITEM (gci);
  GuppiScatterState *state =
    GUPPI_SCATTER_STATE (guppi_canvas_item_state (gci));
  gint stride;

  item->speed_render = 0;

#if ALLOW_SPEED_RENDERING
  if (vis_points > 250) {

    item->speed_render = 0;

    if (guppi_seq_scalar_raw (guppi_scatter_state_x_data (state), &stride) ==
	NULL) ++item->speed_render;

    if (guppi_seq_scalar_raw (guppi_scatter_state_y_data (state), &stride) ==
	NULL) ++item->speed_render;

    item->speed_render += vis_points / 1500;
  }
#endif
}

static void
update (GuppiCanvasItem * gci, double aff[6], ArtSVP * svp, gint flags)
{
  GuppiScatterState *state =
    GUPPI_SCATTER_STATE (guppi_canvas_item_state (gci));
  GuppiScatterItem *item = GUPPI_SCATTER_ITEM (gci);
  GuppiSeqScalar *x_data;
  GuppiSeqScalar *y_data;
  gint i, i0, i1, j0, j1 = 0;
  double x0, y0, x1, y1;

  guppi_canvas_item_get_bbox_vp (gci, &x0, &y0, &x1, &y1);

  x_data = GUPPI_SEQ_SCALAR0 (guppi_scatter_state_x_data (state));
  y_data = GUPPI_SEQ_SCALAR0 (guppi_scatter_state_y_data (state));

  if (x_data == NULL || y_data == NULL) {
    return;
  }

  /* Update our styles */

  for (i = 0; i < GUPPI_SCATTER_STATE_STYLE_COUNT; ++i)
    update_style (gci, i);

  /* Build our visibility masks */

  guppi_seq_common_bounds (GUPPI_SEQ (x_data), GUPPI_SEQ (y_data), &i0, &i1);

  if (item->local_x_visibility_mask) {
    guppi_seq_indices (GUPPI_SEQ (item->local_x_visibility_mask), &j0, &j1);

    if (i0 != j0 || i1 != j1) {
      guppi_unref0 (item->local_x_visibility_mask);
    }
  }

  if (item->local_y_visibility_mask) {
    guppi_seq_indices (GUPPI_SEQ (item->local_y_visibility_mask), &j0, &j1);

    if (i0 != j0 || i1 != j1) {
      guppi_unref0 (item->local_y_visibility_mask);
    }
  }

  if (item->local_x_visibility_mask == NULL) {
    item->local_x_visibility_mask =
      GUPPI_SEQ_BOOLEAN (guppi_seq_boolean_new ());
    guppi_seq_grow_to_include_range (GUPPI_SEQ
				     (item->local_x_visibility_mask), i0, i1);
  }

  if (item->local_y_visibility_mask == NULL) {
    item->local_y_visibility_mask =
      GUPPI_SEQ_BOOLEAN (guppi_seq_boolean_new ());
    guppi_seq_grow_to_include_range (GUPPI_SEQ
				     (item->local_y_visibility_mask), i0, i1);
  }

  if (item->speed_render == 1) {
    GuppiSeqScalar *x_data = guppi_scatter_state_x_data (state);
    GuppiSeqScalar *y_data = guppi_scatter_state_y_data (state);
    double vx0, vy0, vx1, vy1;
    gint vis_pts;

    /* Let's figure out how many points are on the screen! */

    guppi_element_view_get_bbox_vp (guppi_canvas_item_view (gci),
				    &vx0, &vy0, &vx1, &vy1);


    guppi_seq_scalar_in_place_range_query (x_data,
					   item->local_x_visibility_mask,
					   vx0, vx1);

    guppi_seq_scalar_in_place_range_query (y_data,
					   item->local_y_visibility_mask,
					   vy0, vy1);
    guppi_seq_boolean_bitwise_and (item->local_x_visibility_mask,
				   item->local_y_visibility_mask);

    vis_pts = guppi_seq_boolean_true_count (item->local_x_visibility_mask);


    /* Choose a rendering level. */

    choose_speed_rendering_level (gci, vis_pts);
  }

}

static gint
full_render_cb (gpointer user_data)
{
  GuppiCanvasItem *gci = GUPPI_CANVAS_ITEM (user_data);
  GuppiScatterItem *item = GUPPI_SCATTER_ITEM (gci);
  guppi_canvas_item_request_total_redraw (gci);
  item->speed_render = 0;
  item->pending_full_render = 0;

  return FALSE;
}

static void
render (GuppiCanvasItem * gci, GnomeCanvasBuf * buf)
{
  GuppiScatterItem *item;
  GuppiScatterState *state;
  GuppiElementView *gev;

  GuppiSeqScalar *x_data;
  GuppiSeqScalar *y_data;
  GuppiSeqInteger *style_data = NULL;

  gconstpointer x_data_raw, y_data_raw = NULL;
  gint x_data_stride = 0, y_data_stride = 0;

  GuppiSeqScalar *color_grad_data = NULL;
  gint cgrad_i0 = 0, cgrad_i1 = -1;
  double cgrad_min = 0, cgrad_max = 1;
  gconstpointer color_grad_data_raw = NULL;
  gint color_grad_data_stride = 0;

  GuppiSeqScalar *size1_grad_data = NULL;
  gint sz1grad_i0 = 0, sz1grad_i1 = -1;
  double sz1grad_min = 0, sz1grad_max = 1;
  gconstpointer size1_grad_data_raw = NULL;
  gint size1_grad_data_stride = 0;

  GuppiSeqScalar *size2_grad_data = NULL;
  gint sz2grad_i0 = 0, sz2grad_i1 = -1;
  double sz2grad_min = 0, sz2grad_max = 1;
  gconstpointer size2_grad_data_raw = NULL;
  gint size2_grad_data_stride = 0;

  gint i, imax, sty_i0 = 0, sty_i1 = -1;
  GuppiScatterStyleView *sview;
  gint style_num, last_style_num = -1;
  gboolean mutable_marker = FALSE;
  guint32 last_color = 0;

  double vx0, vy0, vx1, vy1;
  gint slack, w, h, N;

  GuppiAlphaTemplate *atemp = NULL;
  guint rnow = 0, gnow = 0, bnow = 0, anow = 0;

  w = buf->rect.x1 - buf->rect.x0;
  h = buf->rect.y1 - buf->rect.y0;

  item = GUPPI_SCATTER_ITEM (gci);
  state = GUPPI_SCATTER_STATE (guppi_canvas_item_state (gci));
  gev = guppi_canvas_item_view (gci);

  x_data = guppi_scatter_state_x_data (state);
  y_data = guppi_scatter_state_y_data (state);

  if (x_data == NULL || y_data == NULL)
    return;

  x_data_raw = guppi_seq_scalar_raw (x_data, &x_data_stride);
  y_data_raw = guppi_seq_scalar_raw (y_data, &y_data_stride);

  /*
     Find the bounding box for the area we are viewing, in plot/view
     coordinates.  We allow some slack that is based on our estimate of
     how big our largest marker can be.
   */

  slack = query_slack (gci);
  guppi_canvas_item_c2vp (gci, buf->rect.x0 - slack, buf->rect.y0 - slack,
			  &vx0, &vy0);
  guppi_canvas_item_c2vp (gci, buf->rect.x1 + slack, buf->rect.y1 + slack,
			  &vx1, &vy1);

  /* Next we do a window query for our viewed region. */

  N = guppi_seq_scalar_in_place_range_query (x_data,
					     item->local_x_visibility_mask,
					     vx0, vx1);
  if (N == 0)
    return;


  N = guppi_seq_scalar_in_place_range_query (y_data,
					     item->local_y_visibility_mask,
					     vy0, vy1);

  if (N == 0)
    return;

  style_data = guppi_scatter_state_style_data (state);
  if (style_data)
    guppi_seq_indices (GUPPI_SEQ (style_data), &sty_i0, &sty_i1);


  /* Prepare our gradient data sets. */

  color_grad_data = guppi_scatter_state_color_gradient_data (state);
  if (color_grad_data) {
    color_grad_data_raw = guppi_seq_scalar_raw (color_grad_data,
						&color_grad_data_stride);
    guppi_seq_indices (GUPPI_SEQ (color_grad_data), &cgrad_i0, &cgrad_i1);
    cgrad_min = guppi_seq_scalar_min (color_grad_data);
    cgrad_max = guppi_seq_scalar_max (color_grad_data);
    /* If our color data is constant, this will ensure that every point
       gets converted to 0.5 */
    if (cgrad_min == cgrad_max) {
      cgrad_min -= 1;
      cgrad_max += 1;
    }
  }

  size1_grad_data = guppi_scatter_state_size1_gradient_data (state);
  if (size1_grad_data) {
    size1_grad_data_raw = guppi_seq_scalar_raw (size1_grad_data,
						&size1_grad_data_stride);
    guppi_seq_indices (GUPPI_SEQ (size1_grad_data), &sz1grad_i0, &sz1grad_i1);
    sz1grad_min = guppi_seq_scalar_min (size1_grad_data);
    sz1grad_max = guppi_seq_scalar_max (size1_grad_data);
    if (sz1grad_min == sz1grad_max) {
      sz1grad_min -= 1;
      sz1grad_min += 1;
    }
    mutable_marker = TRUE;
  }

  size2_grad_data = guppi_scatter_state_size2_gradient_data (state);
  if (size2_grad_data) {
    size2_grad_data_raw = guppi_seq_scalar_raw (size2_grad_data,
						&size2_grad_data_stride);
    guppi_seq_indices (GUPPI_SEQ (size2_grad_data), &sz2grad_i0, &sz2grad_i1);
    sz2grad_min = guppi_seq_scalar_min (size2_grad_data);
    sz2grad_max = guppi_seq_scalar_max (size2_grad_data);
    if (sz2grad_min == sz2grad_max) {
      sz2grad_min -= 1;
      sz2grad_min += 1;
    }
    mutable_marker = TRUE;
  }

  /* Combine our two visibility masks and find their first common element. */

  guppi_seq_boolean_bitwise_and (item->local_x_visibility_mask,
				 item->local_y_visibility_mask);

  i = guppi_seq_boolean_first_true (item->local_x_visibility_mask);

  guppi_seq_common_bounds (GUPPI_SEQ (x_data), GUPPI_SEQ (y_data), NULL,
			   &imax);


  /* If this is going to be a partial rendering, de-schedule any full
     renderings. */

  if (item->pending_full_render != 0) {
    gtk_timeout_remove (item->pending_full_render);
    item->pending_full_render = 0;
  }

  /* Our main loop.  We want to keep slow operations outside of here... */

  while (i <= imax) {
    double vx, vy;
    gint x, y;
    gint speed_step;

    if (x_data_raw)
      vx = guppi_seq_scalar_raw_get (x_data_raw, x_data_stride, i);
    else
      vx = guppi_seq_scalar_get (x_data, i);

    if (y_data_raw)
      vy = guppi_seq_scalar_raw_get (y_data_raw, y_data_stride, i);
    else
      vy = guppi_seq_scalar_get (y_data, i);

    if (guppi_element_view_valid_vp (gev, vx, vy)) {

      guppi_canvas_item_vp2c (gci, vx, vy, &x, &y);

      if (style_data != NULL && sty_i0 <= i && i <= sty_i1)
	style_num = guppi_seq_integer_get (style_data, i);
      else
	style_num = 0;

      sview = item->style_view[style_num];


      /* Handle the marker template */
      if (style_num != last_style_num || mutable_marker) {
	double t_sz1 = -1, t_sz2 = -1;

	if (size1_grad_data && sz1grad_i0 <= i && i <= sz1grad_i1) {
	  if (size1_grad_data_raw)
	    t_sz1 = guppi_seq_scalar_raw_get (size1_grad_data_raw,
					      size1_grad_data_stride, i);
	  else
	    t_sz1 = guppi_seq_scalar_get (size1_grad_data, i);
	  t_sz1 = (t_sz1 - sz1grad_min) / (sz1grad_max - sz1grad_min);
	}

	if (size2_grad_data && sz2grad_i0 <= i && i <= sz2grad_i1) {
	  if (size2_grad_data_raw)
	    t_sz2 = guppi_seq_scalar_raw_get (size2_grad_data_raw,
					      size2_grad_data_stride, i);
	  else
	    t_sz2 = guppi_seq_scalar_get (size2_grad_data, i);
	  t_sz2 = (t_sz2 - sz2grad_min) / (sz2grad_max - sz2grad_min);
	}

	atemp =
	  guppi_scatter_style_view_marker_template (sview, t_sz1, t_sz2);
      }

      /* Handle the marker color */
      {
	guint32 color = last_color;

	if (color_grad_data && cgrad_i0 <= i && i <= cgrad_i1) {
	  double cg;
	  if (color_grad_data_raw)
	    cg = guppi_seq_scalar_raw_get (color_grad_data_raw,
					   color_grad_data_stride, i);
	  else
	    cg = guppi_seq_scalar_get (color_grad_data, i);
	  cg = (cg - cgrad_min) / (cgrad_max - cgrad_min);
	  color = guppi_scatter_style_view_marker_color (sview, cg);

	} else if (style_num != last_style_num) {
	  color = guppi_scatter_style_view_marker_color (sview, -1);
	}

	if (color != last_color) {
	  UINT_TO_RGBA (color, &rnow, &gnow, &bnow, &anow);
	  last_color = color;
	}
      }


      last_style_num = style_num;

      /****** Inlined alpha template printer *****/
      if (atemp != NULL) {
	x -= buf->rect.x0;
	y -= buf->rect.y0;

	x += atemp->x_offset;
	y += atemp->y_offset;

	/* This is a hack: it should be removed after the [xy]_offsets get
	   properly implemented for our various templates. */
	x -= atemp->width / 2;
	y -= atemp->height / 2;

	if (x + atemp->width >= 0 && x < w && y + atemp->height >= 0 && y < h) {

	  gint atpr_x0, atpr_y0, atpr_x1, atpr_y1;
	  gint atpr_i, atpr_j;
	  guchar *atpr_ap;
	  guchar *atpr_arun;
	  guchar *atpr_rgbp;
	  gchar *atpr_rgbrun;

	  /* Clip to our viewing area */
	  atpr_x0 = MAX (x, 0);
	  atpr_y0 = MAX (y, 0);
	  atpr_x1 = MIN (x + atemp->width - 1, w - 1);
	  atpr_y1 = MIN (y + atemp->height - 1, h - 1);

	  atpr_ap =
	    atemp->data + atemp->width * (atpr_y0 - y) + (atpr_x0 - x);
	  atpr_rgbp = buf->buf + buf->buf_rowstride * atpr_y0 + 3 * atpr_x0;
	  for (atpr_j = atpr_y0; atpr_j <= atpr_y1; ++atpr_j) {
	    atpr_arun = atpr_ap;
	    atpr_rgbrun = atpr_rgbp;
	    for (atpr_i = atpr_x0; atpr_i <= atpr_x1; ++atpr_i) {
	      if (*atpr_arun > 0) {
		PIXEL_RGBA (atpr_rgbrun, rnow, gnow, bnow,
			    ((1 + anow) * (1 + *atpr_arun)) >> 8);
	      }
	      ++atpr_arun;
	      atpr_rgbrun += 3;
	    }
	    atpr_ap += atemp->width;
	    atpr_rgbp += buf->buf_rowstride;
	  }
	}
      }
      /****** End of inlined alpha template printer *****/
    }

    i = guppi_seq_boolean_next_true (item->local_x_visibility_mask, i);
    speed_step = item->speed_render;
    while (speed_step > 1 && i <= imax) {
      i = guppi_seq_boolean_next_true (item->local_x_visibility_mask, i);
      --speed_step;
    }
  }

  /* If that just was a partial rendering, schedule a full rendering
     for real soon now. */
  if (item->speed_render > 1) {
    item->pending_full_render = gtk_timeout_add (30, full_render_cb, item);
  }

}

static void
foreach_class_toolkit (GuppiCanvasItem * item,
		       void (*fn) (GuppiPlotToolkit *, gpointer),
		       gpointer user_data)
{
  GuppiPlotToolkit *tk;

  tk = guppi_scatter_toolkit_drag ();
  fn (tk, user_data);
  guppi_unref (tk);

  tk = guppi_scatter_toolkit_brush ();
  fn (tk, user_data);
  guppi_unref (tk);
}

/*****************************************************************************/

static void
guppi_scatter_item_class_init (GuppiScatterItemClass * klass)
{
  GtkObjectClass *object_class;
  GuppiCanvasItemClass *gci_class;
  GnomeCanvasItemClass *item_class;

  object_class = (GtkObjectClass *) klass;
  gci_class = GUPPI_CANVAS_ITEM_CLASS (klass);
  item_class = GNOME_CANVAS_ITEM_CLASS (klass);
  parent_class = gtk_type_class (GUPPI_TYPE_CANVAS_ITEM);

  object_class->destroy = guppi_scatter_item_destroy;
  object_class->finalize = guppi_scatter_item_finalize;

  gci_class->state_changed = state_changed;
  gci_class->view_changed = view_changed;
  gci_class->guppi_update = update;
  gci_class->guppi_render = render;
  gci_class->uses_vp_coordinates = TRUE;
  gci_class->foreach_class_toolkit = foreach_class_toolkit;

  guppi_canvas_item_class_set_item_class_toolkit (gci_class,
						  guppi_scatter_toolkit_default ());
}

static void
guppi_scatter_item_init (GuppiScatterItem * si)
{
  gint i;

  for (i = 0; i < GUPPI_SCATTER_STATE_STYLE_COUNT; ++i)
    si->style_view[i] = guppi_scatter_style_view_new ();
}

GtkType guppi_scatter_item_get_type (void)
{
  static GtkType scatter_item_type = 0;

  if (!scatter_item_type) {
    static const GtkTypeInfo scatter_item_info = {
      "GuppiScatterItem",
      sizeof (GuppiScatterItem),
      sizeof (GuppiScatterItemClass),
      (GtkClassInitFunc) guppi_scatter_item_class_init,
      (GtkObjectInitFunc) guppi_scatter_item_init,
      NULL, NULL,
      (GtkClassInitFunc) NULL
    };
    scatter_item_type = gtk_type_unique (GUPPI_TYPE_CANVAS_ITEM,
					 &scatter_item_info);
  }

  return scatter_item_type;
}




/* $Id: guppi-scatter-item.c,v 1.27 2000/12/16 03:42:46 trow Exp $ */
