/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-pie-register.c
 *
 * Copyright (C) 2000 Helix Code, Inc.
 *
 * Developed by Jody Goldberg <jgoldberg@home.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include <config.h>
#include <guppi-root-group-state.h>
#include <guppi-root-group-view.h>
#include "guppi-chart-type.h"
#include "guppi-pie-register.h"
#include "guppi-XLplots-common.h"
#include "pie.xpm"

#include <guppi-seq-style.h>
#include <guppi-useful.h>

typedef struct
{
  gboolean const is_split;
  gboolean const is_3d;
} PieChartDescriptor;

static GuppiRootGroupView *
guppi_piechart_factory (GuppiMinorChartType const *t,
			GuppiSeqData   *series,
			GuppiSeqData   *labels,
			GuppiSeqString *series_names)
{
  PieChartDescriptor const *desc = t->user_data;
  double const separation = desc->is_split ? 30. : 0.;

  GuppiElementState* root_state;
  GuppiGroupView* root_view;

  GuppiData  *first_series;

  /* Data Objects */
  GuppiSeqStyle  *styles;

  /* State Objects */
  GuppiElementState* pie_state;
  GuppiElementState* legend_state;

  /* View Objects */
  GuppiElementView* pie_view;
  GuppiElementView* legend_view;

  root_state = guppi_root_group_state_new();
  root_view = GUPPI_GROUP_VIEW(guppi_element_state_make_view(root_state));

  /* Only plot the first sequence */
  first_series = guppi_seq_data_get(GUPPI_SEQ_DATA (series), 0);

  /* If the first series are labels, use them */
  if (guppi_seq_size (GUPPI_SEQ (labels)) >= 1) {
    GuppiData *tmp = guppi_seq_data_get(GUPPI_SEQ_DATA (labels), 0);
    series_names = GUPPI_SEQ_STRING (tmp);
  }

  if (series_names == NULL)
    series_names = xlplots_stock_labels (guppi_seq_size (GUPPI_SEQ (first_series)));

  /* Use default styles */
  styles = xlplots_stock_styles (guppi_seq_size (GUPPI_SEQ (first_series)));

  /* Build our state objects */
  pie_state = guppi_element_state_new("pie",
				      "data",                   first_series,
				      "data_labels",		series_names,
				      "data_styles",		styles,
				      "show_percentages",	FALSE,
				      "radius_maximize",	TRUE,
				      "radius_lock",		FALSE,
				      "base_offset",		separation,
				      "base_angle",		M_PI / -2.0,
				      "use_stock_colors",	0,
				      NULL);

  legend_state = xlplots_stock_legend ();
  guppi_element_state_changed_delayed(GUPPI_ELEMENT_STATE(pie_state));
  guppi_element_state_connect(pie_state, SHARED_LABEL_DATA,
			      legend_state, SHARED_LABEL_DATA);
  guppi_element_state_connect(pie_state, SHARED_STYLE_DATA,
			      legend_state, SHARED_STYLE_DATA);

  /* Build our views */
  pie_view = guppi_element_state_make_view(pie_state);
  legend_view = guppi_element_state_make_view(legend_state);

  /* Do our layout */
  guppi_group_view_layout_flush_top(root_view, pie_view, 7.2);
  guppi_group_view_layout_flush_left(root_view, pie_view, 7.2);
  guppi_group_view_layout_flush_bottom(root_view, pie_view, 7.2);
  guppi_group_view_layout_horizontally_adjacent(root_view, pie_view, legend_view, 7.2);
  guppi_group_view_layout_natural_width(root_view, legend_view);
  guppi_group_view_layout_natural_height(root_view, legend_view);
  guppi_group_view_layout_flush_right(root_view, legend_view, 7.2);
  guppi_group_view_layout_same_y_center(root_view, pie_view, legend_view);

  guppi_root_group_view_set_size(GUPPI_ROOT_GROUP_VIEW(root_view), 6*72, 6*72);

  guppi_group_view_commit_all_changes(root_view);

  return GUPPI_ROOT_GROUP_VIEW(root_view);
}

void
guppi_piechart_register ()
{
  GuppiMajorChartType *pie_types =
      guppi_major_chart_type_register ( _("Pie"), pie_icon, NULL);

  {
    static PieChartDescriptor const desc = { FALSE };
    guppi_minor_chart_type_register (pie_types,
      _("Pie"), "chart_pie_1_1.png",
      _("Percentage of each contributor."),
      1, 1, &guppi_piechart_factory, &desc);
  }

  {
    static PieChartDescriptor const desc = { FALSE };
    guppi_minor_chart_type_register (pie_types,
      _("3D Pie"), "chart_pie_1_2.png",
      _("Percentage of each contributor in 3D pie."),
      1, 2, &guppi_piechart_factory, &desc);
  }

  {
    static PieChartDescriptor const desc = { FALSE };
    guppi_minor_chart_type_register (pie_types,
      _("Multi-pie"), "chart_pie_1_3.png",
      _("Major totals as percentages with each wedge subdivided into"
	"secondary pies."),
      1, 3, &guppi_piechart_factory, &desc);
  }

  {
    static PieChartDescriptor const desc = { TRUE };
    guppi_minor_chart_type_register (pie_types,
      _("Split Pie"), "chart_pie_2_1.png",
      _("Percentage of each contributor with wedges split apart."),
      2, 1, &guppi_piechart_factory, &desc);
  }

  {
    static PieChartDescriptor const desc = { TRUE };
    guppi_minor_chart_type_register (pie_types,
      _("3D Split Pie"), "chart_pie_2_2.png",
      _("Percentage of each contributor with 3D wedges split apart."),
      2, 2, &guppi_piechart_factory, &desc);
  }

  {
    static PieChartDescriptor const desc = { FALSE };
    guppi_minor_chart_type_register (pie_types,
      _("Multi-pie-bars"), "chart_pie_2_3.png",
      _("Major totals as percentages with each wedge subdivided into"
	"secondary stacked bars."),
      2, 3, &guppi_piechart_factory, &desc);
  }
}
