/* This is -*- C -*- */
/* $Id: guppi-axis-item.c,v 1.25 2000/12/17 05:24:54 trow Exp $ */

/*
 * guppi-axis-item.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <guppi-useful.h>
#include <guppi-alpha-template.h>
#include <guppi-seq-scalar.h>

#include "guppi-axis-item.h"
#include "guppi-axis-view.h"
#include "guppi-axis-state.h"

static GtkObjectClass *parent_class = NULL;

static void
guppi_axis_item_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_axis_item_finalize (GtkObject * obj)
{
  GuppiAxisItem *item = GUPPI_AXIS_ITEM (obj);
  GList *iter;

  if (item->label_list) {
    iter = item->label_list;
    while (iter != NULL) {
      guppi_unref (iter->data);
      iter = g_list_next (iter);
    }
  }

  guppi_unref0 (item->legend_text);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/****************************************************************************/

static void
prepare_legend (GuppiCanvasItem * gci)
{
  GuppiAxisState *state;
  GuppiAxisItem *item;
  double rot;
  const gchar *txt;
  double sc;

  g_return_if_fail (gci != NULL);

  state = GUPPI_AXIS_STATE (guppi_canvas_item_state (gci));
  item = GUPPI_AXIS_ITEM (gci);

  sc = guppi_canvas_item_scale (gci);

  txt = guppi_axis_state_displayed_legend (state);
  if (txt == NULL) {
    guppi_unref0 (item->legend_text);
    return;
  }

  if (item->legend_text == NULL)
    item->legend_text = guppi_raster_text_new ();

  if (guppi_axis_state_position (state) == GUPPI_WEST)
    rot = -90;
  else if (guppi_axis_state_position (state) == GUPPI_EAST)
    rot = +90;
  else
    rot = 0;

  guppi_raster_text_rasterize_full (item->legend_text,
				    txt,
				    guppi_axis_state_legend_font (state),
				    sc, rot, TRUE, 1);

}

static void
prepare_labels (GuppiCanvasItem * gci)
{
  GuppiAxisMarkers *marks;
  GuppiAxisItem *item;
  GuppiAxisState *state;
  GuppiElementView *view;
  GList *iter;
  GList *new_list;
  gint i, N;
  double sc;
  double size, shrink_factor = 1, span, rot = 0;

  g_return_if_fail (gci != NULL);

  item = GUPPI_AXIS_ITEM (gci);
  state = GUPPI_AXIS_STATE (guppi_canvas_item_state (gci));
  view = guppi_canvas_item_view (gci);
  sc = guppi_canvas_item_scale (gci);

  if (guppi_axis_state_horizontal (state)) {

    marks = guppi_element_view_x_axis_markers (view);
    if (guppi_axis_state_rotate_labels (state))
      rot = -90;
    span = guppi_geometry_height (guppi_element_view_geometry (view));

  } else {


    marks = guppi_element_view_y_axis_markers (view);
    if (guppi_axis_state_rotate_labels (state))
      rot = -90;
    span = guppi_geometry_width (guppi_element_view_geometry (view));

  }

  if (marks == NULL)
    return;

  if (guppi_axis_state_shrink_labels_to_fit (state)) {
    shrink_factor = guppi_axis_state_label_shrink_to_fit_factor (state,
								 marks, span);
  }

  N = guppi_axis_markers_size (marks);

  new_list = NULL;

  for (i = 0; i < N; ++i) {
    const GuppiTick *tick;
    GnomeFont *font = NULL;
    gboolean shown = FALSE;

    tick = guppi_axis_markers_get (marks, i);

    if (tick != NULL)
      guppi_axis_state_tick_properties (state, tick,
					NULL, NULL, NULL, NULL,
					&shown, NULL, NULL, &font);

    if (shown) {

      GuppiRasterText *raster_text = NULL;

      /* We attempt to re-use our previous text rasterizations. */
      /* (This crappy linear search is ugly, but mostly harmless) */
      iter = item->label_list;
      while (iter != NULL && raster_text == NULL) {
	if (iter->data) {
	  GuppiRasterText *rt = GUPPI_RASTER_TEXT (iter->data);
	  if (!strcmp (tick->label, rt->text)) {
	    raster_text = rt;
	    iter->data = NULL;
	  }
	}
	iter = g_list_next (iter);
      }

      /* If we didn't find anything to re-use, create a clean slate. */
      if (raster_text == NULL)
	raster_text = guppi_raster_text_new ();

      /* Re-rasterize. */

      size = shrink_factor * sc;

      if (font != NULL && size > 0) {
	
	guppi_raster_text_rasterize_full (raster_text,
					  guppi_tick_label (tick),
					  font, size, rot, TRUE, 1);

	raster_text->user_data = (gpointer) tick;

	new_list = g_list_append (new_list, raster_text);

      } else {

	guppi_unref (raster_text);

      }


    }
  }

  /* Clean up the old list and replace it with the new one. */
  iter = item->label_list;
  while (iter != NULL) {
    guppi_unref (iter->data);
    iter = g_list_next (iter);
  }
  g_list_free (item->label_list);

  item->label_list = new_list;

}

static void
update (GuppiCanvasItem * gci, double aff[6], ArtSVP * svp, gint flags)
{
  prepare_legend (gci);
  prepare_labels (gci);
}

static void
render (GuppiCanvasItem * gci, GnomeCanvasBuf * buf)
{
  GuppiAxisItem *axi = GUPPI_AXIS_ITEM (gci);
  GuppiElementView *view = guppi_canvas_item_view (gci);
  GuppiAxisState *state = GUPPI_AXIS_STATE (guppi_canvas_item_state (gci));
  double sc = guppi_canvas_item_scale (gci);

  gint cx0, cy0, cx1, cy1;
  double x0 = 0, y0 = 0, x1 = 0, y1 = 0;

  guppi_canvas_item_get_bbox_c (gci, &cx0, &cy0, &cx1, &cy1);

  /* First, paint the edge (if necessary) */

  if (guppi_axis_state_show_edge (state) &&
      guppi_axis_state_edge_thickness (state) > 0) {

    double t;

    t = guppi_axis_state_edge_thickness (state) * sc;
    if (guppi_axis_state_horizontal (state)) {
      t = guppi_y_pt2px (t);
    } else {
      t = guppi_x_pt2px (t);
    }

    switch (guppi_axis_state_position (state)) {
    case GUPPI_NORTH:
      x0 = cx0;
      x1 = cx1;
      y0 = cy1 - t;
      y1 = cy1;
      break;

    case GUPPI_SOUTH:
      x0 = cx0;
      x1 = cx1;
      y0 = cy0;
      y1 = cy0 + t;
      break;

    case GUPPI_WEST:
      x0 = cx1;
      x1 = cx1 - t;
      y0 = cy0;
      y1 = cy1;
      break;

    case GUPPI_EAST:
      x0 = cx0;
      x1 = cx0 + t;
      y0 = cy0;
      y1 = cy1;
      break;

    default:
      g_assert_not_reached ();
    }

    guppi_paint_soft_box (buf, x0, y0, x1, y1,
			  guppi_axis_state_edge_color (state));
  }

  /* Next, paint the legend (if necessary) */

  if (axi->legend_text != NULL) {
    double lx = 0, ly = 0;
    double w, h;
    guint r, g, b, a;

    w = guppi_raster_text_template (axi->legend_text)->width;
    h = guppi_raster_text_template (axi->legend_text)->height;

    switch (guppi_axis_state_position (state)) {

    case GUPPI_NORTH:
      lx = (cx0 + cx1) / 2 - w / 2;
      ly = cy0;
      break;

    case GUPPI_SOUTH:
      lx = (cx0 + cx1) / 2 - w / 2;
      ly = cy1 - h;
      break;

    case GUPPI_WEST:
      lx = cx0;
      ly = (cy0 + cy1) / 2 - h / 2;
      break;

    case GUPPI_EAST:
      lx = cx1 - w;
      ly = (cy0 + cy1) / 2 - h / 2;
      break;

    default:
      g_assert_not_reached ();

    }

    UINT_TO_RGBA (guppi_axis_state_legend_color (state), &r, &g, &b, &a);

    guppi_alpha_template_print (guppi_raster_text_template (axi->legend_text),
				lx, ly, r, g, b, a, buf);

  }




  /* Next, paint the markers (again, if necessary) */

  {
    GuppiAxisMarkers *marks = NULL;

    if (guppi_axis_state_horizontal (state))
      marks = guppi_element_view_x_axis_markers (view);
    else
      marks = guppi_element_view_y_axis_markers (view);

    if (marks) {
      gint i, N;
      GList *label_iter;

      N = guppi_axis_markers_size (marks);
      label_iter = axi->label_list;

      for (i = 0; i < N; ++i) {
	const GuppiTick *tick = guppi_axis_markers_get (marks, i);
	double pos = tick->position, pos_c = 0;
	gboolean show = FALSE, show_label = FALSE;
	double length = 0, thick = 0, label_offset = 0;
	guint32 tick_color = 0, label_color = 0;
	GnomeFont *font = NULL;

	guppi_axis_state_tick_properties (state, tick,
					  &show, &tick_color, &thick, &length,
					  &show_label, &label_offset,
					  &label_color, &font);

	if (show || show_label) {
	  if (guppi_axis_state_horizontal (state))
	    guppi_canvas_item_vp2c_d (gci, pos, 0, &pos_c, NULL);
	  else
	    guppi_canvas_item_vp2c_d (gci, 0, pos, NULL, &pos_c);
	}

	/* Paint tick */
	if (show && length > 0 && thick > 0) {

	  label_offset += length;

	  if (guppi_axis_state_horizontal (state)) {
	    length = guppi_y_pt2px (length);
	    thick = guppi_x_pt2px (thick);
	  } else {
	    length = guppi_x_pt2px (length);
	    thick = guppi_y_pt2px (thick);
	  }

	  length *= sc;
	  thick *= sc;

	  switch (guppi_axis_state_position (state)) {

	  case GUPPI_NORTH:
	    x0 = pos_c - thick / 2;
	    x1 = pos_c + thick / 2;
	    y0 = cy1 - length;
	    y1 = cy1;
	    break;

	  case GUPPI_SOUTH:
	    x0 = pos_c - thick / 2;
	    x1 = pos_c + thick / 2;
	    y0 = cy0;
	    y1 = cy0 + length;
	    break;

	  case GUPPI_WEST:
	    x0 = cx1 - length;
	    x1 = cx1;
	    y0 = pos_c - thick / 2;
	    y1 = pos_c + thick / 2;
	    break;

	  case GUPPI_EAST:
	    x0 = cx0;
	    x1 = cx0 + length;
	    y0 = pos_c - thick / 2;
	    y1 = pos_c + thick / 2;
	    break;

	  default:
	    g_assert_not_reached ();
	  }

	  guppi_paint_soft_box (buf, x0, y0, x1, y1, tick_color);
	}

	/* Paint label */
	if (show_label && font != NULL) {
	  GuppiRasterText *raster_text = NULL;
	  gint xl = 0, yl = 0;
	  guint r, g, b, a;

	  label_offset *= sc;

	  while (label_iter != NULL && raster_text == NULL) {
	    GuppiRasterText *rt = GUPPI_RASTER_TEXT (label_iter->data);
	    if (rt->user_data == (gpointer) tick)
	      raster_text = rt;
	    label_iter = g_list_next (label_iter);
	  }

	  if (raster_text) {

	    gboolean print_template = TRUE;
	    GuppiAlphaTemplate *template;

	    template = guppi_raster_text_template (raster_text);

	    /* template can be NULL if we try to rasterize the empty string */
	    if (template) {


	      switch (guppi_axis_state_position (state)) {

	      case GUPPI_NORTH:
		xl = pos_c - template->width / 2;
		yl = cy1 - guppi_y_pt2px (label_offset) - template->height;
		break;

	      case GUPPI_SOUTH:
		xl = pos_c - template->width / 2;
		yl = cy0 + guppi_y_pt2px (label_offset);
		break;

	      case GUPPI_WEST:
		xl = cx1 - guppi_x_pt2px (label_offset) - template->width;
		yl = pos_c - template->height / 2;
		break;

	      case GUPPI_EAST:
		xl = cx0 + guppi_x_pt2px (label_offset);
		yl = pos_c - template->height / 2;
		break;

	      default:
		g_assert_not_reached ();
	      }

	      UINT_TO_RGBA (label_color, &r, &g, &b, &a);

	      /* Fine-tune label placement */
	      if (guppi_axis_state_horizontal (state)) {
		if (xl < cx0) {
		  if (xl + 0.50 * template->width < cx0)
		    print_template = FALSE;
		  else
		    xl = cx0;
		} else if (xl + template->width >= cx1) {
		  if (xl + 0.50 * template->width >= cx1)
		    print_template = FALSE;
		  else
		    xl = cx1 - template->width - 1;
		}
	      } else {
		if (yl < cy0) {
		  if (yl + 0.50 * template->height < cy0)
		    print_template = FALSE;
		  else
		    yl = cy0;
		} else if (yl + template->height >= cy1) {
		  if (yl + 0.50 * template->height >= cy1)
		    print_template = FALSE;
		  else
		    yl = cy1 - template->height - 1;
		}
	      }

	      if (print_template)
		guppi_alpha_template_print (template, xl, yl,
					    r, g, b, a, buf);
	    }
	  }
	}

      }
    }
  }
}

static void
state_changed (GuppiCanvasItem * gci)
{
  guppi_canvas_item_request_update (gci);
}

static void
view_changed (GuppiCanvasItem * gci)
{
  GuppiElementView *view = guppi_canvas_item_view (gci);
  guppi_canvas_item_request_update (gci);
  guppi_geometry_calc_natural_size (guppi_element_view_geometry (view));
}

static gboolean
data_drop (GuppiCanvasItem * gci, GuppiData * data)
{
  GuppiAxisState *state;

  state = GUPPI_AXIS_STATE (guppi_canvas_item_state (gci));

  if (data == NULL) {
    guppi_axis_state_set_data (state, NULL);
    return TRUE;
  }

  if (GUPPI_IS_SEQ_SCALAR (data)) {
    guppi_axis_state_set_data (state, data);
    return TRUE;
  }

  return FALSE;
}

/****************************************************************************/

static void
guppi_axis_item_class_init (GuppiAxisItemClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiCanvasItemClass *gci_class = GUPPI_CANVAS_ITEM_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_CANVAS_ITEM);

  object_class->destroy = guppi_axis_item_destroy;
  object_class->finalize = guppi_axis_item_finalize;

  gci_class->state_changed = state_changed;
  gci_class->view_changed = view_changed;
  gci_class->guppi_update = update;
  gci_class->guppi_render = render;
  gci_class->data_drop = data_drop;

  gci_class->uses_vp_coordinates = TRUE;

}

static void
guppi_axis_item_init (GuppiAxisItem * obj)
{

}

GtkType guppi_axis_item_get_type (void)
{
  static GtkType guppi_axis_item_type = 0;
  if (!guppi_axis_item_type) {
    static const GtkTypeInfo guppi_axis_item_info = {
      "GuppiAxisItem",
      sizeof (GuppiAxisItem),
      sizeof (GuppiAxisItemClass),
      (GtkClassInitFunc) guppi_axis_item_class_init,
      (GtkObjectInitFunc) guppi_axis_item_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_axis_item_type = gtk_type_unique (GUPPI_TYPE_CANVAS_ITEM,
					    &guppi_axis_item_info);
  }
  return guppi_axis_item_type;
}

/* $Id: guppi-axis-item.c,v 1.25 2000/12/17 05:24:54 trow Exp $ */
