/* $Id: guppi-splash.c,v 1.3 2000/04/21 23:22:45 trow Exp $ */

/*
 * guppi-splash.c
 *
 * Copyright (C) 1999, 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 * and Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <unistd.h>
#include <gnome.h>
#include "guppi-paths.h"
#include "guppi-splash.h"

static gboolean please_destroy_splash = FALSE;
static GtkWidget* splash = NULL;
static GtkWidget* splash_label = NULL;

static time_t splash_time;
static const gint min_splash_time = 15;

static void
gtk_catchup()
{
  /* Flush all events */
  while (gtk_events_pending())
    gtk_main_iteration();

  /* Sleep a little bit, so that gtk's idle handlers have time to catch up
     and do any rendering that needs to be done. */
  usleep(150);

  /* Flush all events again */
  while (gtk_events_pending())
    gtk_main_iteration();
}

static gint
destroy_timeout(gpointer user_data)
{
  time_t t;

  if (!please_destroy_splash)
    return TRUE;

  if (splash == NULL)
    return FALSE;

  time(&t);
  if (t-splash_time < min_splash_time) 
    return TRUE;

  if (splash)
    gtk_widget_destroy(splash);
  gtk_catchup();
  splash = NULL;
  please_destroy_splash = FALSE;

  return FALSE;
}

static void
set_style(GtkWidget* w, gpointer foo)
{
  GtkStyle* style;
  GdkColor bg = { 0xaedaee, 0xae00, 0xda00, 0xee00};
  GdkColor fg = { 0x0000ff, 0x0000, 0x0000, 0xffff};

  style = gtk_style_copy(gtk_widget_get_style(w));
  style->bg[0] = bg;
  style->fg[0] = fg;
  gtk_widget_set_style(w, style);
  gtk_widget_set_style(GTK_WIDGET(foo), style);
}

void
guppi_splash_create(void)
{
  GtkWidget* vbox;
  GtkWidget* pixmap;

  gchar* buffer;

  g_return_if_fail(splash == NULL);

  splash = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_position(GTK_WINDOW(splash), GTK_WIN_POS_CENTER);
  gtk_window_set_policy(GTK_WINDOW(splash), FALSE, FALSE, FALSE);


  buffer = g_strdup_printf(_("Welcome to Guppi %s"), VERSION);
  gtk_window_set_title(GTK_WINDOW(splash), buffer);
  g_free(buffer);

  vbox = gtk_vbox_new(FALSE, 4);

  pixmap = gnome_pixmap_new_from_file(guppi_logo_graphic_path());

  splash_label = gtk_label_new("");
  gtk_label_set_line_wrap(GTK_LABEL(splash_label), TRUE);
  gtk_label_set_justify(GTK_LABEL(splash_label), GTK_JUSTIFY_CENTER);

  /* We change the style and adjust the color scheme upon
     realization... */
  gtk_signal_connect(GTK_OBJECT(splash),
		     "realize",
		     GTK_SIGNAL_FUNC(set_style),
		     splash_label);

  gtk_box_pack_start(GTK_BOX(vbox), pixmap, TRUE, TRUE, 1);
  gtk_box_pack_end(GTK_BOX(vbox), splash_label, TRUE, TRUE, 1);
  gtk_container_add(GTK_CONTAINER(splash), vbox);

  gtk_widget_show_all(splash);
  gtk_catchup();
  time(&splash_time);

  gtk_timeout_add(1000*min_splash_time/5, destroy_timeout, NULL);
}

void
guppi_splash_destroy(void)
{
  gchar* buffer;

  g_return_if_fail(splash != NULL);

  buffer = g_strdup_printf(_("Visit us at %s"), "http://www.gnome.org/guppi");

  guppi_splash_message(buffer);

  g_free(buffer);

  please_destroy_splash = TRUE;
}

void
guppi_splash_message(const gchar* text)
{
  g_return_if_fail(text != NULL);

  if (splash) {
    gtk_catchup();
    gtk_label_set_text(GTK_LABEL(splash_label), (gchar*)text);

  }
  g_message(text);

  if (splash) {
    gtk_catchup();
  }
}


/* $Id: guppi-splash.c,v 1.3 2000/04/21 23:22:45 trow Exp $ */
