/* This is -*- C -*- */
/* $Id: guppi-plot-control.c,v 1.2 2000/04/13 19:45:20 trow Exp $ */

/*
 * guppi-plot-control.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-plot-control.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_plot_control_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_plot_control_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_plot_control_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);

  gtk_object_unref(GTK_OBJECT(GUPPI_PLOT_CONTROL(obj)->state));
  GUPPI_PLOT_CONTROL(obj)->state = NULL;
}

static void
guppi_plot_control_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_plot_control_class_init(GuppiPlotControlClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_VBOX);

  object_class->get_arg = guppi_plot_control_get_arg;
  object_class->set_arg = guppi_plot_control_set_arg;
  object_class->destroy = guppi_plot_control_destroy;
  object_class->finalize = guppi_plot_control_finalize;

}

static void
guppi_plot_control_init(GuppiPlotControl* obj)
{
  obj->state = NULL;
}

GtkType
guppi_plot_control_get_type(void)
{
  static GtkType guppi_plot_control_type = 0;
  if (!guppi_plot_control_type) {
    static const GtkTypeInfo guppi_plot_control_info = {
      "GuppiPlotControl",
      sizeof(GuppiPlotControl),
      sizeof(GuppiPlotControlClass),
      (GtkClassInitFunc)guppi_plot_control_class_init,
      (GtkObjectInitFunc)guppi_plot_control_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_plot_control_type = gtk_type_unique(GTK_TYPE_VBOX, &guppi_plot_control_info);
  }
  return guppi_plot_control_type;
}

static gint
clist_selected_row(GtkCList* clist)
{
  GList* clist_iter;
  gint i=0;
  g_return_val_if_fail(clist != NULL, -1);

  clist_iter = clist->row_list;
  while (clist_iter != NULL) {
    if (GTK_CLIST_ROW(clist_iter)->state == GTK_STATE_SELECTED)
      return i;
    ++i;
    clist_iter = g_list_next(clist_iter);
  }
  return -1;
}

static GtkWidget*
my_stock_button(const gchar* type, const gchar* text)
{
  GtkWidget* w = gnome_stock_pixmap_widget_new(NULL, type);
  GtkWidget* b = gtk_button_new();
  GtkTooltips* tip = gtk_tooltips_new();

  gtk_container_add(GTK_CONTAINER(b), w);
  gtk_tooltips_set_tip(tip, b, text, NULL);
  
  return b;
}

static void
clist_mark_visibility(GtkCList* clist, gint r, gboolean v)
{
  static GdkPixmap* vis_pixmap = NULL;
  static GdkBitmap* vis_mask = NULL;

  if (vis_pixmap == NULL) {
    gnome_stock_pixmap_gdk(GNOME_STOCK_BUTTON_APPLY, NULL,
			   &vis_pixmap, &vis_mask);
  }

  if (v) {
    if (vis_pixmap)
      gtk_clist_set_pixmap(clist,r,0,vis_pixmap,vis_mask);
    else
      gtk_clist_set_text(clist,r,0,"X");
  } else {
    gtk_clist_set_text(clist,r,0,"");
  }
}

static void
guppi_plot_control_populate_clist(GuppiPlotControl* gpc)
{
  gint i, N;
  GuppiPlotElement* e;
  gchar* row[3];

  g_return_if_fail(gpc != NULL);
  g_return_if_fail(gpc->state != NULL);
  g_return_if_fail(gpc->clist != NULL);

  gtk_clist_freeze(gpc->clist);
  gtk_clist_clear(gpc->clist);

  row[0] = "";

  N = guppi_plot_state_element_count(gpc->state); 
  for (i=0; i<N; ++i) {
    e = guppi_plot_state_get_element(gpc->state, i);

    row[1] = (gchar*)guppi_plot_element_type_name(e);
    row[2] = (gchar*)guppi_plot_element_label(e);
    gtk_clist_append(gpc->clist, row);

    clist_mark_visibility(gpc->clist, i, e->visible);
  }

  gtk_clist_set_column_width(gpc->clist, 0, 24);
  gtk_clist_set_column_resizeable(gpc->clist, 0, FALSE);
  gtk_clist_set_column_auto_resize(gpc->clist, 1, TRUE);
  gtk_clist_set_column_auto_resize(gpc->clist, 2, TRUE);

  gtk_clist_thaw(gpc->clist);
}

static void
button_new_cb(GtkButton* b, gpointer data)
{

}

static void
button_raisetop_cb(GtkButton* b, gpointer data)
{
  GuppiPlotControl* gpc;
  GuppiPlotState* state;
  gint r;
  GuppiPlotElement* e;

  gpc = GUPPI_PLOT_CONTROL(data);
  g_return_if_fail(gpc != NULL);
  state = gpc->state;
  g_return_if_fail(state != NULL);

  r = clist_selected_row(gpc->clist);
  if (r == -1)
    return;

  if (r != 0) {
    e = guppi_plot_state_get_element(state, r);
    guppi_plot_element_raise_to_top(e);
    guppi_plot_control_populate_clist(gpc);
    gtk_clist_select_row(gpc->clist, 0, 0);
  }
}

static void
button_raise_cb(GtkButton* b, gpointer data)
{
  GuppiPlotControl* gpc;
  GuppiPlotState* state;
  gint r;
  GuppiPlotElement* e;

  gpc = GUPPI_PLOT_CONTROL(data);
  g_return_if_fail(gpc != NULL);
  state = gpc->state;
  g_return_if_fail(state != NULL);

  r = clist_selected_row(gpc->clist);
  if (r == -1)
    return;

  if (r != 0) {
    e = guppi_plot_state_get_element(state, r);
    guppi_plot_element_raise(e, 1);
    guppi_plot_control_populate_clist(gpc);
    gtk_clist_select_row(gpc->clist, r-1, 0);
  }
}

static void
button_visible_cb(GtkButton* b, gpointer data)
{
  GuppiPlotControl* gpc;
  GuppiPlotState* state;
  gint r;
  GuppiPlotElement* e;

  gpc = GUPPI_PLOT_CONTROL(data);
  g_return_if_fail(gpc != NULL);
  state = gpc->state;
  g_return_if_fail(state != NULL);

  r = clist_selected_row(gpc->clist);
  if (r == -1)
    return;

  e = guppi_plot_state_get_element(state, r);
  guppi_plot_element_reverse_visibility(e);
  clist_mark_visibility(gpc->clist, r, e->visible);
  
}

static void
button_lower_cb(GtkButton* b, gpointer data)
{
  GuppiPlotControl* gpc;
  GuppiPlotState* state;
  gint r, N;
  GuppiPlotElement* e;

  gpc = GUPPI_PLOT_CONTROL(data);
  g_return_if_fail(gpc != NULL);
  state = gpc->state;
  g_return_if_fail(state != NULL);

  r = clist_selected_row(gpc->clist);
  if (r == -1)
    return;

  N = guppi_plot_state_element_count(state);
  if (r != N-1) {
    e = guppi_plot_state_get_element(state, r);
    guppi_plot_element_lower(e, 1);
    guppi_plot_control_populate_clist(gpc);
    gtk_clist_select_row(gpc->clist, r+1, 0);
  }
}

static void
button_lowerbot_cb(GtkButton* b, gpointer data)
{
  GuppiPlotControl* gpc;
  GuppiPlotState* state;
  gint r, N;
  GuppiPlotElement* e;

  gpc = GUPPI_PLOT_CONTROL(data);
  g_return_if_fail(gpc != NULL);
  state = gpc->state;
  g_return_if_fail(state != NULL);

  r = clist_selected_row(gpc->clist);
  if (r == -1)
    return;

  N = guppi_plot_state_element_count(state);
  if (r != N-1) {
    e = guppi_plot_state_get_element(state, r);
    guppi_plot_element_lower_to_bottom(e);
    guppi_plot_control_populate_clist(gpc);
    gtk_clist_select_row(gpc->clist, N-1, 0);
  }

}

static void
button_delete_cb(GtkButton* b, gpointer data)
{
  GuppiPlotControl* gpc;
  GuppiPlotState* state;
  gint r;

  gpc = GUPPI_PLOT_CONTROL(data);
  g_return_if_fail(gpc != NULL);
  state = gpc->state;
  g_return_if_fail(state != NULL);

  r = clist_selected_row(gpc->clist);
  if (r == -1)
    return;

  guppi_plot_state_remove_element(state,
				  guppi_plot_state_get_element(state, r));
}

/* We handle our add and remove signals in a brain-damaged way right now. */

static void
state_add_cb(GuppiPlotState* s, GuppiPlotElement* e, gpointer data)
{
  guppi_plot_control_populate_clist(GUPPI_PLOT_CONTROL(data));
}

static void
state_remove_cb(GuppiPlotState* s, GuppiPlotElement* e, gpointer data)
{
  guppi_plot_control_populate_clist(GUPPI_PLOT_CONTROL(data));
}

void
guppi_plot_control_construct(GuppiPlotControl* gpc, GuppiPlotState* state)
{
  GtkBox* box;
  GtkWidget* hbox_button;
  GtkWidget* b;
  GtkWidget* swin;

  g_return_if_fail(gpc != NULL);
  g_return_if_fail(state != NULL);
  g_return_if_fail(gpc->state == NULL);

  /* ***** Set up our state object ***** */

  gpc->state = state;
  gtk_object_ref(GTK_OBJECT(state));

  gtk_signal_connect(GTK_OBJECT(state), 
		     "add", 
		     GTK_SIGNAL_FUNC(state_add_cb),
		     gpc);

  gtk_signal_connect(GTK_OBJECT(state),
		     "remove", 
		     GTK_SIGNAL_FUNC(state_remove_cb),
		     gpc);

  /* ***** Set up our GUI ***** */

  box = GTK_BOX(gpc);

  /* build our clist */
  gpc->clist = GTK_CLIST(gtk_clist_new(3));
  gtk_clist_set_selection_mode(gpc->clist, GTK_SELECTION_SINGLE);
  guppi_plot_control_populate_clist(gpc);
  swin = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swin),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swin),
					GTK_WIDGET(gpc->clist));
  gtk_widget_show_all(swin);
  gtk_box_pack_start(box, swin, TRUE, TRUE, GNOME_PAD_SMALL);

  /* build the row of buttons along the bottom */
  hbox_button = gtk_hbox_new(TRUE, GNOME_PAD_SMALL);

  gtk_box_pack_start(GTK_BOX(hbox_button), 
		     b = my_stock_button(GNOME_STOCK_PIXMAP_NEW,
					 _("New")),
		     FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(b), "clicked",
		     GTK_SIGNAL_FUNC(button_new_cb), gpc);

  gtk_box_pack_start(GTK_BOX(hbox_button), 
		     b = my_stock_button(GNOME_STOCK_PIXMAP_TOP,
					 _("Raise to Top")),
		     FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(b), "clicked",
		     GTK_SIGNAL_FUNC(button_raisetop_cb), gpc);

  gtk_box_pack_start(GTK_BOX(hbox_button), 
		     b = my_stock_button(GNOME_STOCK_PIXMAP_UP,
					 _("Raise")),
		     FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(b), "clicked",
		     GTK_SIGNAL_FUNC(button_raise_cb), gpc);

  gtk_box_pack_start(GTK_BOX(hbox_button), 
		     b = my_stock_button(GNOME_STOCK_PIXMAP_REFRESH,
					 _("Toggle Visibility")),
		     FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(b), "clicked",
		     GTK_SIGNAL_FUNC(button_visible_cb), gpc);


  gtk_box_pack_start(GTK_BOX(hbox_button), 
		     b = my_stock_button(GNOME_STOCK_PIXMAP_DOWN,
					 _("Lower")),
		     FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(b), "clicked",
		     GTK_SIGNAL_FUNC(button_lower_cb), gpc);


  gtk_box_pack_start(GTK_BOX(hbox_button), 
		     b = my_stock_button(GNOME_STOCK_PIXMAP_BOTTOM,
					 _("Lower to Bottom")),
		     FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(b), "clicked",
		     GTK_SIGNAL_FUNC(button_lowerbot_cb), gpc);

  gtk_box_pack_start(GTK_BOX(hbox_button), 
		     b = my_stock_button(GNOME_STOCK_PIXMAP_TRASH,
					 _("Delete")),
		     FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(b), "clicked",
		     GTK_SIGNAL_FUNC(button_delete_cb), gpc);


  gtk_box_pack_start(box, hbox_button,
		     FALSE, FALSE, GNOME_PAD_SMALL);
}

GtkWidget*
guppi_plot_control_new(GuppiPlotState* state)
{
  GtkWidget* w;
  
  g_return_val_if_fail(state != NULL, NULL);

  w = GTK_WIDGET(gtk_type_new(guppi_plot_control_get_type()));
  guppi_plot_control_construct(GUPPI_PLOT_CONTROL(w), state);
  return w;
}



/* $Id: guppi-plot-control.c,v 1.2 2000/04/13 19:45:20 trow Exp $ */
