/* $Id: guppi-nice-values.c,v 1.1 2000/04/21 23:22:44 trow Exp $ */

/*
 * guppi-nice-values.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>.

 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <math.h>
#include "guppi-nice-values.h"

static const double base4_divisors[] = { 4, 2, 1, -1 };
static const double base8_divisors[] = { 8, 4, 2, 1, -1 };
static const double base10_divisors[] = { 10, 5, 4, 2, 1, -1 };
static const double base16_divisors[] = { 16, 8, 4, 2, 1, -1 };
static const double base32_divisors[] = { 32, 16, 8, 4, 2, 1, -1 };
static const double base64_divisors[] = { 64, 32, 16, 8, 4, 2, 1, -1 };

double*
guppi_nice_values(double pos_min, double pos_max, gint goal, gint base,
		  gsize* N)
{
  double width, mag, div, step, start, count, t;
  double delta_best=1e+8, step_best=0, start_best=0;
  gint i, count_best=0;
  const double* divisors = NULL;
  double* values;

  g_return_val_if_fail(goal > 1, NULL);

  if (pos_min > pos_max) {
    t = pos_min;
    pos_min = pos_max;
    pos_max = t;
  }

  width = fabs(pos_max - pos_min);
  mag = ceil(log(width / goal) / log(base));
  
  switch (base) {
  case 4:
    divisors = base4_divisors;
    break;
  case 8:
    divisors = base8_divisors;
    break;
  case 10:
    divisors = base10_divisors;
    break;
  case 16:
    divisors = base16_divisors;
    break;
  case 32:
    divisors = base32_divisors;
    break;
  case 64:
    divisors = base64_divisors;
    break;
  default:
    g_warning("Illegal radix (base=%d)", base);
    return NULL;
  }

  g_assert(divisors != NULL);
  for(i=0; divisors[i] > 0; ++i) {
    div = divisors[i];
    step = pow(base, mag)/div;
    start = ceil(pos_min/step)*step;
    count = floor(width/step);
    if (pos_min <= start && start <= pos_max)
      ++count;
    
    if (fabs(count-goal) < delta_best) {
      delta_best = fabs(count-goal);
      step_best = step;
      start_best = start;
      count_best = (gint)count;
    }
  }

  if (step_best <= 0) {
    g_message("Search for nice axis points failed.  This shouldn't happen.");
    return NULL;
  }

  /* Create the label list */

  if (N)
    *N = count_best;

  values = g_new(double, count_best);
  for (i=0; i < count_best; ++i) {
    t = start_best + i*step_best;
    if (fabs(t / step_best) < 1e-12)
      t = 0;
    values[i] = t;
  }

  return values;
}


/* $Id: guppi-nice-values.c,v 1.1 2000/04/21 23:22:44 trow Exp $ */
