/* This is -*- C -*- */
/* $Id: guppi-layout-grid.c,v 1.4 2000/05/03 17:06:51 trow Exp $ */

/*
 * guppi-layout-grid.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <math.h>

#include "guppi-layout-grid.h"
#include "guppi-layout-grid-allocate.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_layout_grid_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_layout_grid_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_layout_grid_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_layout_grid_finalize(GtkObject* obj)
{
  GuppiLayoutGrid* grid = GUPPI_LAYOUT_GRID(obj);
  gint i;

  if (parent_class->finalize)
    parent_class->finalize(obj);

  for (i=0; i<grid->rows * grid->cols; ++i) {
    gtk_object_unref(GTK_OBJECT(grid->cells[i]));
    grid->cells[i] = NULL;
  }
  
  g_free(grid->cells);
  grid->cells = NULL;

  grid->rows = grid->cols = 0;
}

static void
guppi_layout_grid_class_init(GuppiLayoutGridClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_OBJECT);

  object_class->get_arg = guppi_layout_grid_get_arg;
  object_class->set_arg = guppi_layout_grid_set_arg;
  object_class->destroy = guppi_layout_grid_destroy;
  object_class->finalize = guppi_layout_grid_finalize;

}

static void
guppi_layout_grid_init(GuppiLayoutGrid* obj)
{

}

GtkType
guppi_layout_grid_get_type(void)
{
  static GtkType guppi_layout_grid_type = 0;
  if (!guppi_layout_grid_type) {
    static const GtkTypeInfo guppi_layout_grid_info = {
      "GuppiLayoutGrid",
      sizeof(GuppiLayoutGrid),
      sizeof(GuppiLayoutGridClass),
      (GtkClassInitFunc)guppi_layout_grid_class_init,
      (GtkObjectInitFunc)guppi_layout_grid_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_layout_grid_type = gtk_type_unique(GTK_TYPE_OBJECT, &guppi_layout_grid_info);
  }
  return guppi_layout_grid_type;
}

GuppiLayoutGrid*
guppi_layout_grid_new(void)
{
  return GUPPI_LAYOUT_GRID(gtk_type_new(guppi_layout_grid_get_type()));
}

static gint
do_allocation_cb(gpointer data)
{
  GuppiLayoutGrid* grid = GUPPI_LAYOUT_GRID(data);

  guppi_layout_grid_allocate(grid,
			     grid->last_x0, grid->last_y0,
			     grid->last_x1, grid->last_y1);

  /* Reset idle id */
  grid->idle_id = 0;

  return FALSE;
}

void
guppi_layout_grid_schedule_allocation(GuppiLayoutGrid* grid)
{
  g_return_if_fail(grid != NULL);
  if (grid->idle_id == 0) {
    grid->idle_id = gtk_idle_add(do_allocation_cb, grid);
  }
}

static void
guppi_layout_grid_cancel_allocation(GuppiLayoutGrid* grid)
{
  g_return_if_fail(grid != NULL);
  if (grid->idle_id == 0)
    return;
  gtk_idle_remove(grid->idle_id);
  grid->idle_id = 0;
}

static void
schedule_alloc_cb(gpointer foo, GuppiLayoutGrid* grid)
{
  guppi_layout_grid_schedule_allocation(grid);
}

void
guppi_layout_grid_set_size(GuppiLayoutGrid* grid, gint cols, gint rows)
{
  gint i, N;

  g_return_if_fail(grid != NULL);
  g_return_if_fail(cols > 0);
  g_return_if_fail(rows > 0);

  /* We need to handle the resizing case.  I'll do it later. */
  g_return_if_fail(grid->cells == NULL);

  grid->cols = cols;
  grid->rows = rows;

  N = cols * rows;
  grid->cells = g_new(GuppiLayoutGridCell*, N);

  for (i=0; i<N; ++i) {
    grid->cells[i] = guppi_layout_grid_cell_new();
    gtk_signal_connect(GTK_OBJECT(grid->cells[i]),
		       "changed",
		       GTK_SIGNAL_FUNC(schedule_alloc_cb),
		       grid);
  }
}

GuppiLayoutGridCell*
guppi_layout_grid_get_cell(GuppiLayoutGrid* grid, gint c, gint r)
{
  g_return_val_if_fail(grid != NULL, NULL);
  g_return_val_if_fail(grid->cells != NULL, NULL);
  g_return_val_if_fail(c >= 0, NULL);
  g_return_val_if_fail(c < grid->cols, NULL);
  g_return_val_if_fail(r >= 0, NULL);
  g_return_val_if_fail(r < grid->rows, NULL);

  return grid->cells[r * grid->cols + c];
}

void
guppi_layout_grid_allocate(GuppiLayoutGrid* grid,
			   double x0, double y0,
			   double x1, double y1)
{
  GuppiLayoutGridAllocation* alw;
  GuppiLayoutGridAllocation* alh;
  gint i, j;
  double x, y, w, h;
  GuppiLayoutGridCell* cell;
  gboolean any_changed = FALSE;

  g_return_if_fail(grid != NULL);

  /*** First, initialize all cells ***/
  for (i=0; i<grid->cols; ++i)
    for (j=0; j<grid->rows; ++j) {
      cell = guppi_layout_grid_get_cell(grid, i, j);
      g_assert(cell != NULL);
      guppi_layout_grid_cell_calc_properties(cell);
      if (cell->changed_geometry)
	any_changed = TRUE;
    }

  /* If nothing has changed since our last allocation, we don't need
     to do any more work. */
  if (!any_changed && 
      x0 == grid->last_x0 &&
      x1 == grid->last_x1 &&
      y0 == grid->last_y0 &&
      y1 == grid->last_y1) {
    return;
  }

  /**** Handle the width allocations ****/

  alw = g_new0(GuppiLayoutGridAllocation, grid->cols);

  /* Reset the allocations */
  for (i=0; i<grid->cols; ++i) {
    alw[i].request = 0;
    alw[i].min_size = 0;
    alw[i].max_size = 0;
    alw[i].hunger = 0;
    alw[i].allocation = 0;
  }

  /* Find the proper allocation request for each column */
  for (i=0; i<grid->cols; ++i) {
    for (j=0; j<grid->rows; ++j) {

      cell = guppi_layout_grid_get_cell(grid, i, j);
      g_assert(cell != NULL);

      cell->changed_geometry = FALSE; /* reset changed flag */

      alw[i].request = MAX(alw[i].request, cell->req_width);
      alw[i].min_size = MAX(alw[i].min_size, cell->min_width);
      if (alw[i].max_size > 0 && cell->max_width > 0)
	alw[i].max_size = MIN(alw[i].max_size, cell->max_width);
      else
	alw[i].max_size = MAX(alw[i].max_size, cell->max_width);
      alw[i].hunger = MAX(alw[i].hunger, cell->hunger_width);
    }

    if (alw[i].min_size > alw[i].max_size && alw[i].max_size > 0) {
      g_warning("Incoherent min/max width in col %d: ignoring max", i);
      alw[i].max_size = 0;
    }
  }

  /* Do the allocation */
  guppi_layout_grid_calculate_allocation(fabs(x1 - x0), alw, grid->cols);


  /**** Handle the height allocations ****/

  alh = g_new0(GuppiLayoutGridAllocation, grid->rows);

  /* Reset the allocations */
  for (i=0; i<grid->rows; ++i) {
    alh[i].request = 0;
    alh[i].min_size = 0;
    alh[i].max_size = 0;
    alh[i].hunger = 0;
    alh[i].allocation = 0;
  }

  /* Find the proper allocation request for each row */
  for (i=0; i<grid->rows; ++i) {
    for (j=0; j<grid->cols; ++j) {

      cell = guppi_layout_grid_get_cell(grid, j, i);
      g_assert(cell != NULL);

      alh[i].request = MAX(alh[i].request, cell->req_height);
      alh[i].min_size = MAX(alh[i].min_size, cell->min_height);
      if (alh[i].max_size > 0 && cell->max_height > 0)
	alh[i].max_size = MIN(alh[i].max_size, cell->max_height);
      else
	alh[i].max_size = MAX(alh[i].max_size, cell->max_height);
      alh[i].hunger = MAX(alh[i].hunger, cell->hunger_height);
    }

    if (alh[i].min_size > alh[i].max_size && alh[i].max_size > 0) {
      g_warning("Incoherent min/max height in row %d: ignoring max", i);
      alh[i].max_size = 0;
    }
  }

  /* Do the allocation */
  guppi_layout_grid_calculate_allocation(fabs(y1 - y0), alh, grid->rows);



  /* Set each cell's allocation */
  y = MIN(y0, y1);
  for (j=0; j<grid->rows; ++j) {
    x = MIN(x0, x1);
    h = alh[j].allocation;
    for (i=0; i<grid->cols; ++i) {
      cell = guppi_layout_grid_get_cell(grid, i, j);
      w = alw[i].allocation;
      guppi_layout_grid_cell_set_allocation(cell, x, y, w, h);
      x += w;
    }
    y += h;
  }

  grid->last_x0 = x0;
  grid->last_x1 = x1;
  grid->last_y0 = y0;
  grid->last_y1 = y1;

  /* Clean up and go home... */
  g_free(alw);
  g_free(alh);
}

static void
resize_cb(GtkWidget* w, GtkAllocation* al, GuppiLayoutGrid* grid)
{
  double x0, y0, x1, y1;

  /* Get the bounding world coordinates of the entire visible piece of
     the canvas. */
  gnome_canvas_window_to_world(grid->connected_canvas, 0, 0, &x0, &y0);
  gnome_canvas_window_to_world(grid->connected_canvas,
			       al->width, al->height, &x1, &y1);

  guppi_layout_grid_allocate(grid, x0, y0, x1, y1);
}


void
guppi_layout_grid_connect_to_canvas(GuppiLayoutGrid* grid, GnomeCanvas* canvas)
{
  g_return_if_fail(grid != NULL);
  g_return_if_fail(canvas != NULL);
  g_return_if_fail(grid->connected_canvas == NULL);

  grid->connected_canvas = canvas;

  gtk_signal_connect(GTK_OBJECT(canvas),
		     "size_allocate",
		     GTK_SIGNAL_FUNC(resize_cb),
		     grid);
}

void
guppi_layout_grid_connect_items_gridwise(GuppiLayoutGrid* grid)
{
  GList* iter;
  GuppiLayoutGridCell* cell;
  GuppiCanvasItem* curr;
  GuppiCanvasItem* prev;
  gint r, c;

  g_return_if_fail(grid != NULL);

  /* Connect intra-cell */
  for (r=0; r<grid->rows; ++r) {
    for (c=0; c<grid->cols; ++c) {
      cell = guppi_layout_grid_get_cell(grid, c, r);
      iter = cell->items;
      prev = NULL;
      while (iter) {
	curr = GUPPI_CANVAS_ITEM(iter->data);
	if (prev) {
	  guppi_canvas_item_connect_scales(curr, prev);
	  guppi_item_state_connect_x_data(curr->state, prev->state);
	  guppi_item_state_connect_y_data(curr->state, prev->state);
	}
	prev = curr;
	iter = g_list_next(iter);
      }
    }
  }

  /* Connect by row */
  for (r=0; r<grid->rows; ++r) {
    prev = NULL;
    for (c=0; c<grid->cols; ++c) {
      cell = guppi_layout_grid_get_cell(grid, c, r);
      if (cell->items) {
	curr = GUPPI_CANVAS_ITEM(cell->items->data);
	if (prev) {
	  guppi_canvas_item_connect_y_scales(curr, prev);
	  guppi_item_state_connect_y_data(curr->state, prev->state);
	}
	prev = curr;
      }
    }
  }

  /* Connect by column */
  for (c=0; c<grid->cols; ++c) {
    prev = NULL;
    for (r=0; r<grid->rows; ++r) {
      cell = guppi_layout_grid_get_cell(grid, c, r);
      if (cell->items) {
	curr = GUPPI_CANVAS_ITEM(cell->items->data);
	if (prev) {
	  guppi_canvas_item_connect_x_scales(curr, prev);
	  guppi_item_state_connect_x_data(curr->state, prev->state);
	}
	prev = curr;
      }
    }
  }
}


/* $Id: guppi-layout-grid.c,v 1.4 2000/05/03 17:06:51 trow Exp $ */
