/* This is -*- C -*- */
/* $Id: guppi-slinreg-item.c,v 1.5 2000/04/21 23:22:41 trow Exp $ */

/*
 * guppi-slinreg-item.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <math.h>
#include <libart_lgpl/art_svp.h>
#include <guppi-rgb.h>
#include <guppi-useful.h>
#include "guppi-slinreg-item.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0,
  ARG_STATE
};

static void
guppi_slinreg_item_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  }
}

static void
guppi_slinreg_item_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {
  case ARG_STATE:
    guppi_canvas_item_set_state(GUPPI_CANVAS_ITEM(obj),
				GUPPI_ITEM_STATE(GTK_VALUE_POINTER(*arg)));
    break;

  default:
    break;
  }
}

static void
guppi_slinreg_item_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_slinreg_item_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}


static void
guppi_slinreg_item_class_init(GuppiSlinregItemClass* klass)
{
  static void guppi_slinreg_item_update(GnomeCanvasItem*, double affine[6],
					ArtSVP*, gint);
  static void guppi_slinreg_item_render(GnomeCanvasItem*, GnomeCanvasBuf*);
  static void state_set_hook(GuppiCanvasItem*, GuppiItemState*);

  GtkObjectClass* object_class = (GtkObjectClass*)klass;
  GnomeCanvasItemClass* item_class = GNOME_CANVAS_ITEM_CLASS(klass);
  GuppiCanvasItemClass* gci_class = GUPPI_CANVAS_ITEM_CLASS(klass);
 
  parent_class = gtk_type_class(GUPPI_TYPE_CANVAS_ITEM);

  gtk_object_add_arg_type("GuppiSlinregItem::state",
			  GTK_TYPE_POINTER,
			  GTK_ARG_WRITABLE,
			  ARG_STATE);

  object_class->get_arg = guppi_slinreg_item_get_arg;
  object_class->set_arg = guppi_slinreg_item_set_arg;
  object_class->destroy = guppi_slinreg_item_destroy;
  object_class->finalize = guppi_slinreg_item_finalize;

  item_class->update = guppi_slinreg_item_update;
  item_class->render = guppi_slinreg_item_render;

  gci_class->state_set_hook = state_set_hook;
}

static void
guppi_slinreg_item_init(GuppiSlinregItem* obj)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(obj);

  gci->bb_passive = TRUE;
  gci->bb_x0 = gci->bb_x1 = gci->bb_y0 = gci->bb_y1 = 0;
}

GtkType
guppi_slinreg_item_get_type(void)
{
  static GtkType guppi_slinreg_item_type = 0;
  if (!guppi_slinreg_item_type) {
    static const GtkTypeInfo guppi_slinreg_item_info = {
      "GuppiSlinregItem",
      sizeof(GuppiSlinregItem),
      sizeof(GuppiSlinregItemClass),
      (GtkClassInitFunc)guppi_slinreg_item_class_init,
      (GtkObjectInitFunc)guppi_slinreg_item_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_slinreg_item_type = gtk_type_unique(GUPPI_TYPE_CANVAS_ITEM, &guppi_slinreg_item_info);
  }
  return guppi_slinreg_item_type;
}

/*****************************************************************************/

static void
guppi_slinreg_item_update(GnomeCanvasItem* item, double affine[6],
			  ArtSVP* clip_path, gint flags)
{
  GuppiCanvasItem* gci;
  GuppiSlinregItem* si;
  GuppiSlinregState* state =
    GUPPI_SLINREG_STATE(guppi_canvas_item_state(GUPPI_CANVAS_ITEM(item)));
  ArtVpath linepath[3];
  ArtSVP* svp;
  double m, b;
  double x0, x1, y0, y1;
  double wx0, wx1, wy0, wy1, dx, dy;
  double xspan, scale, line_width, font_size, theta;
  gchar label_text[256];
  GnomeFont* label_font;

  /* Chain our calls to update */
  if (parent_class && GNOME_CANVAS_ITEM_CLASS(parent_class)->update)
    GNOME_CANVAS_ITEM_CLASS(parent_class)->update(item,affine,clip_path,flags);

  gci = GUPPI_CANVAS_ITEM(item);
  si = GUPPI_SLINREG_ITEM(item);

  if (state && state->slr && state->slr->valid) {

    m = guppi_simple_linreg_slope(state->slr);
    b = guppi_simple_linreg_intercept(state->slr);

    scale = guppi_canvas_item_magnification(gci);
    line_width = MAX(0.5, scale * state->line_width);

    xspan = fabs(gci->x1 - gci->x0);
    /* Put the endpoints of the line we draw just outside of our view. */
    x0 = gci->x0 - 0.1*xspan;
    x1 = gci->x1 + 0.1*xspan;
    y0 = m * x0 + b;
    y1 = m * x1 + b;

    guppi_canvas_item_plot2world(gci, x0, y0, &wx0, &wy0);
    guppi_canvas_item_plot2world(gci, x1, y1, &wx1, &wy1);

    linepath[0].code = ART_MOVETO;
    linepath[0].x = wx0;
    linepath[0].y = wy0;

    linepath[1].code = ART_LINETO;
    linepath[1].x = wx1;
    linepath[1].y = wy1;

    linepath[2].code = ART_END;
    linepath[2].x = 0;
    linepath[2].y = 0;

    svp = art_svp_vpath_stroke(linepath,
			       ART_PATH_STROKE_JOIN_ROUND,
			       ART_PATH_STROKE_CAP_ROUND,
			       line_width,
			       4, 0.25);

    /* Notice that I'm ignoring the affine.
       That is probably not a good idea. */
  
    gnome_canvas_item_update_svp_clip (item, &si->line_svp, svp, clip_path);

    if (state->show_label) {

      font_size = scale > 1 ? sqrt(scale) : 1.0;
      font_size *= state->label_size;
      
      g_snprintf(label_text, 256, "y=%.3gx+%.3g, R^2=%.3g",
		 m, b, guppi_simple_linreg_R_squared(state->slr));

      theta = atan2(wy1-wy0, wx1-wx0);

      label_font = state->label_font;
      if (label_font == NULL) 
	label_font = guppi_default_font();

      /* Re-render the label iff the label has changed since the previous
	 update. */
      if (si->line_label == NULL || si->label_text == NULL ||
	  si->label_size != font_size || si->label_rot != theta ||
	  strcmp(si->label_text, label_text)) {

	g_free(si->label_text);
	si->label_text = g_strdup(label_text);
	si->label_size = font_size;
	si->label_rot = theta;
	
	if (si->line_label)
	  gtk_object_unref(GTK_OBJECT(si->line_label));
	
	si->line_label = NULL;
	if (label_font)
	  si->line_label = guppi_alpha_template_text_general(label_font,
							     font_size,
							     label_text,
							     180*theta/M_PI,
							     TRUE,
							     0);
      }

      /* Calculate the location of the center of the visible part of
	 the regression line.  This is "mostly right". */
      if (y0 < gci->y0)
	x0 = (gci->y0 - b)/m;
      else if (y0 > gci->y1)
	x0 = (gci->y1 - b)/m;
      if (y1 < gci->y0)
	x1 = (gci->y0 - b)/m;
      else if (y1 > gci->y1)
	x1 = (gci->y1 - b)/m;
      x0 = (x0+x1)/2;
      y0 = m * x0 + b;

      /* Adjust slightly for the width of the line. */
      guppi_canvas_item_plot2world(gci, x0, y0, &wx0, &wy0);
      dx = -(line_width/2 + font_size/2 + 5)*sin(theta);
      dy = (line_width/2 + font_size/2 + 5)*cos(theta);
      wx0 += dx;
      wy0 += dy;

      gnome_canvas_w2c(item->canvas, wx0, wy0, 
		       &(si->label_pos_x), &(si->label_pos_y));
      
      si->label_pos_x -= si->line_label->width/2;
      si->label_pos_y -= si->line_label->height/2;
    }
            
  }

  guppi_canvas_item_request_total_redraw(GUPPI_CANVAS_ITEM(item));  
}

static void
guppi_slinreg_item_render(GnomeCanvasItem* item,
			  GnomeCanvasBuf* buf)
{
  GuppiSlinregItem* si;
  GuppiSlinregState* state;
  guchar r,g,b,a;

  /* Chain our calls to render */
  if (parent_class && GNOME_CANVAS_ITEM_CLASS(parent_class)->render)
    GNOME_CANVAS_ITEM_CLASS(parent_class)->render(item, buf);

  si = GUPPI_SLINREG_ITEM(item);
  state = GUPPI_SLINREG_STATE(guppi_canvas_item_state(GUPPI_CANVAS_ITEM(si)));

  /* This is nice and easy. */
  if (si->line_svp != NULL) {// && si->state != NULL)

    buf = guppi_canvas_item_clip_buf(GUPPI_CANVAS_ITEM(item), buf);

    gnome_canvas_render_svp(buf,
			    si->line_svp,
			    state->line_color);
  
    if (state->show_label && si->line_label) {
      UINT_TO_RGBA(state->label_color, &r, &g, &b, &a);
      guppi_alpha_template_print(si->line_label,
				 si->label_pos_x,
				 si->label_pos_y,
				 r, g, b, a,
				 buf->buf, buf->buf_rowstride,
				 buf->rect.x1 - buf->rect.x0,
				 buf->rect.y1 - buf->rect.y0,
				 buf->rect.x0, buf->rect.y0);

    }

    g_free(buf);
  }
}

static void
state_set_hook(GuppiCanvasItem* item, GuppiItemState* state)
{
  GuppiSlinregItem* si = GUPPI_SLINREG_ITEM(item);
  GuppiSlinregState* ss = GUPPI_SLINREG_STATE(state);

  g_return_if_fail(si != NULL);
  g_return_if_fail(ss != NULL);


  gtk_signal_connect_object(GTK_OBJECT(ss),
			    "reset_x_data",
			    GTK_SIGNAL_FUNC(guppi_canvas_item_request_update),
			    GTK_OBJECT(si));

  gtk_signal_connect_object(GTK_OBJECT(ss),
			    "reset_y_data",
			    GTK_SIGNAL_FUNC(guppi_canvas_item_request_update),
			    GTK_OBJECT(si));

  gtk_signal_connect_object(GTK_OBJECT(ss),
			    "changed_x_data",
			    GTK_SIGNAL_FUNC(guppi_canvas_item_request_update),
			    GTK_OBJECT(si));

  gtk_signal_connect_object(GTK_OBJECT(ss),
			    "changed_y_data",
			    GTK_SIGNAL_FUNC(guppi_canvas_item_request_update),
			    GTK_OBJECT(si));

}


/* $Id: guppi-slinreg-item.c,v 1.5 2000/04/21 23:22:41 trow Exp $ */
