/* This is -*- C -*- */
/* $Id: guppi-scatter-style-control.c,v 1.3 2000/04/25 13:57:14 trow Exp $ */

/*
 * guppi-scatter-style-control.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <guppi-useful.h>
#include <guppi-seq-scalar.h>
#include <guppi-data-picker.h>
#include "guppi-scatter-marker-menu.h"
#include "guppi-scatter-style-control.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_scatter_style_control_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_scatter_style_control_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_scatter_style_control_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);

  gtk_object_unref(GTK_OBJECT(GUPPI_SCATTER_STYLE_CONTROL(obj)->style));
  GUPPI_SCATTER_STYLE_CONTROL(obj)->style = NULL;
}

static void
guppi_scatter_style_control_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_scatter_style_control_class_init(GuppiScatterStyleControlClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_HBOX);

  object_class->get_arg = guppi_scatter_style_control_get_arg;
  object_class->set_arg = guppi_scatter_style_control_set_arg;
  object_class->destroy = guppi_scatter_style_control_destroy;
  object_class->finalize = guppi_scatter_style_control_finalize;
}

static void
guppi_scatter_style_control_init(GuppiScatterStyleControl* obj)
{
  obj->style = NULL;
}

GtkType
guppi_scatter_style_control_get_type(void)
{
  static GtkType guppi_scatter_style_control_type = 0;
  if (!guppi_scatter_style_control_type) {
    static const GtkTypeInfo guppi_scatter_style_control_info = {
      "GuppiScatterStyleControl",
      sizeof(GuppiScatterStyleControl),
      sizeof(GuppiScatterStyleControlClass),
      (GtkClassInitFunc)guppi_scatter_style_control_class_init,
      (GtkObjectInitFunc)guppi_scatter_style_control_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_scatter_style_control_type = gtk_type_unique(GTK_TYPE_HBOX, &guppi_scatter_style_control_info);
  }
  return guppi_scatter_style_control_type;
}

/*****************************************************************************/

static void
guppi_scatter_style_control_apply_state(GuppiScatterStyleControl* gssc)
{
  /* Some shorthand */
  GuppiScatterStyle* style = gssc->style;
  GladeXML* xml = gssc->glade_xml;
  
  GtkWidget* marker_menu;

  GtkWidget* marker_size_frame;
  GtkWidget* size_gradient_frame;
  GtkWidget* color_style_frame;
  GtkWidget* color_frame;
  GtkWidget* color_gradient_frame;

  GtkWidget* size1_spinner_label;
  GtkWidget* size2_spinner_label;
  GtkWidget* size1_spinner;
  GtkWidget* size2_spinner;
  GtkAdjustment* size1_spadj;
  GtkAdjustment* size2_spadj;

  GtkWidget* color_picker;
  GtkWidget* size1_picker;
  GtkWidget* size2_picker;

  GtkWidget* color_reverse;
  GtkWidget* size1_reverse;
  GtkWidget* size2_reverse;

  GtkWidget* size1_picker_label;
  GtkWidget* size2_picker_label;

  GtkWidget* color_box_high;
  GtkWidget* color_box_low;

  GtkWidget* color_from_label;
  GtkWidget* color_to_label;

  GtkWidget* color_style_menu;

  GuppiScatterMarker marker;
  const GuppiScatterMarkerInfo* info;
  gboolean sensitive, have_size1, have_size2;
  guint r, g, b, a;

  marker_menu = glade_xml_get_widget(xml, "marker_menu");
  marker_size_frame = glade_xml_get_widget(xml, "marker_size_frame");
  size_gradient_frame = glade_xml_get_widget(xml, "size_gradient_frame");
  color_style_frame = glade_xml_get_widget(xml, "color_style_frame");
  color_frame = glade_xml_get_widget(xml, "color_frame");
  color_gradient_frame = glade_xml_get_widget(xml, "color_gradient_frame");

  size1_spinner_label = glade_xml_get_widget(xml, "marker_size_1_label_A");
  size2_spinner_label = glade_xml_get_widget(xml, "marker_size_2_label_A");
  size1_spinner = glade_xml_get_widget(xml, "marker_size_1_spinner");
  size2_spinner = glade_xml_get_widget(xml, "marker_size_2_spinner");

  color_picker = glade_xml_get_widget(xml, "color_picker");
  size1_picker = glade_xml_get_widget(xml, "marker_size_1_picker");
  size2_picker = glade_xml_get_widget(xml, "marker_size_2_picker");

  color_reverse = glade_xml_get_widget(xml, "color_reverse_gradient");
  size1_reverse = glade_xml_get_widget(xml, "marker_size_1_reverse_gradient");
  size2_reverse = glade_xml_get_widget(xml, "marker_size_2_reverse_gradient");

  size1_picker_label = glade_xml_get_widget(xml, "marker_size_1_label_B");
  size2_picker_label = glade_xml_get_widget(xml, "marker_size_2_label_B");

  color_box_high = glade_xml_get_widget(xml, "colorpicker_upper");
  color_box_low = glade_xml_get_widget(xml, "colorpicker_lower");

  color_from_label = glade_xml_get_widget(xml, "color_from_label");
  color_to_label = glade_xml_get_widget(xml, "color_to_label");

  color_style_menu = glade_xml_get_widget(xml, "color_style_menu");

  marker = guppi_scatter_style_marker(style);
  info = &guppi_scatter_marker_info[marker];

  /* Set our marker type */
  guppi_scatter_marker_menu_set_selection(GUPPI_SCATTER_MARKER_MENU(marker_menu), marker);

  /* Turn off parts of the UI based on the marker type */
  sensitive = marker != SCATTER_MARKER_NONE;
  gtk_widget_set_sensitive(marker_size_frame, sensitive);
  gtk_widget_set_sensitive(size_gradient_frame, sensitive);
  gtk_widget_set_sensitive(color_style_frame, sensitive);
  gtk_widget_set_sensitive(color_frame, sensitive);
  gtk_widget_set_sensitive(color_gradient_frame, sensitive);

  /* Set the labels on our spinners & pickers*/
  have_size1 = info->size1_desc != NULL;
  have_size2 = info->size2_desc != NULL;
  gtk_widget_set_sensitive(size1_spinner_label, have_size1);
  gtk_widget_set_sensitive(size1_picker_label, have_size1);
  gtk_widget_set_sensitive(size1_picker, have_size1);
  gtk_widget_set_sensitive(size2_spinner_label, have_size2);
  gtk_widget_set_sensitive(size2_picker_label, have_size2);
  gtk_widget_set_sensitive(size2_picker, have_size2);

  gtk_label_set_text(GTK_LABEL(size1_spinner_label),
		     have_size1 ? info->size1_desc : _("n/a"));
  gtk_label_set_text(GTK_LABEL(size1_picker_label),
		     have_size1 ? info->size1_desc : _("n/a"));
  gtk_label_set_text(GTK_LABEL(size2_spinner_label),
		     have_size2 ? info->size2_desc : _("n/a"));
  gtk_label_set_text(GTK_LABEL(size2_picker_label),
		     have_size2 ? info->size2_desc : _("n/a"));

  /* Set our spinner characteristics */
  gtk_widget_set_sensitive(size1_spinner, have_size1);
  gtk_widget_set_sensitive(size2_spinner, have_size2);

  size1_spadj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(size1_spinner));
  size2_spadj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(size2_spinner));

  size1_spadj->value = style->size2;
  size1_spadj->upper = info->size1_max;
  size1_spadj->lower = info->size1_min;
  size1_spadj->step_increment = 0.5;

  size2_spadj->value = style->size2;
  size2_spadj->upper = info->size2_max;
  size2_spadj->lower = info->size2_min;
  size2_spadj->step_increment = 0.5;

  gtk_adjustment_changed(size1_spadj);
  gtk_adjustment_changed(size2_spadj);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(size1_spinner), style->size1);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(size2_spinner), style->size2);

  /* Set up our data pickers */
  guppi_data_picker_set_selected_data(GUPPI_DATA_PICKER(color_picker),
				      (GuppiData*)style->gradient_data);
  guppi_data_picker_set_selected_data(GUPPI_DATA_PICKER(size1_picker),
				      (GuppiData*)style->size1_data);
  guppi_data_picker_set_selected_data(GUPPI_DATA_PICKER(size2_picker),
				      (GuppiData*)style->size2_data);

  /* Set up our reversal toggle buttons */
  gtk_widget_set_sensitive(color_reverse, style->gradient_data != NULL);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(color_reverse),
			       style->reverse_color_gradient);

  gtk_widget_set_sensitive(size1_reverse, style->size1_data != NULL);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(size1_reverse),
			       style->reverse_size1_gradient);

  gtk_widget_set_sensitive(size2_reverse, style->size2_data != NULL);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(size2_reverse),
			       style->reverse_size2_gradient);

  /* Set up our color selectors */
  UINT_TO_RGBA(style->color_low, &r, &g, &b, &a);
  gnome_color_picker_set_i8(GNOME_COLOR_PICKER(color_box_low),r,g,b,a);

  UINT_TO_RGBA(style->color, &r, &g, &b, &a);
  gnome_color_picker_set_i8(GNOME_COLOR_PICKER(color_box_high),r,g,b,a);

  /* Activate some stuff based on if we have a color gradient */
  gtk_widget_set_sensitive(color_from_label, style->gradient_data != NULL);
  gtk_widget_set_sensitive(color_to_label, style->gradient_data != NULL);
  gtk_widget_set_sensitive(color_box_low, style->gradient_data != NULL);
  /*
  if (style->gradient_data != NULL) {
    gtk_widget_show(color_from_label);
    gtk_widget_show(color_to_label);
    gtk_widget_show(color_box_low);
  } else {
    gtk_widget_hide(color_from_label);
    gtk_widget_hide(color_to_label);
    gtk_widget_hide(color_box_low);
  }
  */

  /* Deactivate unused pieces of the interface */
  gtk_widget_set_sensitive(color_style_menu, FALSE);

}

static void
state_change_cb(GuppiScatterStyle* style, gpointer data)
{
  g_return_if_fail(data != NULL);
  guppi_scatter_style_control_apply_state(GUPPI_SCATTER_STYLE_CONTROL(data));
}

/*****************************************************************************/

static void
marker_style_cb(GtkWidget* w, GuppiScatterMarker marker, gpointer data)
{
  GuppiScatterStyle* sty = GUPPI_SCATTER_STYLE(data);

  g_return_if_fail(sty != NULL);
  guppi_scatter_style_set_marker(sty, marker);
}

static void
pick_color_grad_cb(GuppiDataPicker* gdp, GuppiData* data, gpointer x)
{
  GuppiScatterStyle* sty = GUPPI_SCATTER_STYLE(x);
  guppi_scatter_style_set_gradient_data(sty, (GuppiSeqScalar*)data);
}

static void
reverse_color_grad_cb(GtkToggleButton* b, gpointer data)
{
  GuppiScatterStyle* sty = GUPPI_SCATTER_STYLE(data);
  gboolean state;

  g_return_if_fail(sty != NULL);
  state = gtk_toggle_button_get_active(b);
  guppi_scatter_style_set_reverse_color_gradient(sty, state);

}

static void
size1_spin_cb(GtkSpinButton* spin, gpointer data)
{
  GuppiScatterStyle* style = GUPPI_SCATTER_STYLE(data);
  double x = gtk_spin_button_get_value_as_float(spin);
  guppi_scatter_style_set_marker_size1(style, x);
}

static void
size2_spin_cb(GtkSpinButton* spin, gpointer data)
{
  GuppiScatterStyle* style = GUPPI_SCATTER_STYLE(data);
  double x = gtk_spin_button_get_value_as_float(spin);
  guppi_scatter_style_set_marker_size2(style, x);
}

static void
pick_size1_grad_cb(GuppiDataPicker* gdp, GuppiData* data, gpointer x)
{
  GuppiScatterStyle* sty = GUPPI_SCATTER_STYLE(x);
  guppi_scatter_style_set_size1_data(sty, (GuppiSeqScalar*)data);
}


static void
reverse_size1_grad_cb(GtkToggleButton* b, gpointer data)
{
  GuppiScatterStyle* sty = GUPPI_SCATTER_STYLE(data);
  gboolean state;

  g_return_if_fail(sty != NULL);
  state = gtk_toggle_button_get_active(b);
  guppi_scatter_style_set_reverse_size1_gradient(sty, state);
}

static void
pick_size2_grad_cb(GuppiDataPicker* gdp, GuppiData* data, gpointer x)
{
  GuppiScatterStyle* sty = GUPPI_SCATTER_STYLE(x);
  guppi_scatter_style_set_size2_data(sty, (GuppiSeqScalar*)data);
}

static void
reverse_size2_grad_cb(GtkToggleButton* b, gpointer data)
{
  GuppiScatterStyle* sty = GUPPI_SCATTER_STYLE(data);
  gboolean state;

  g_return_if_fail(sty != NULL);
  state = gtk_toggle_button_get_active(b);
  guppi_scatter_style_set_reverse_size2_gradient(sty, state);
}

static void
color_box_low_cb(GnomeColorPicker* w, guint r, guint g, guint b, guint a,
		 gpointer data)
{
  GuppiScatterStyle* sty = GUPPI_SCATTER_STYLE(data);
  guint rgba = RGBA_TO_UINT(r >> 8, g >> 8, b >> 8, a >> 8);
  guppi_scatter_style_set_color_low(sty, rgba);
}

static void
color_box_high_cb(GnomeColorPicker* w, guint r, guint g, guint b, guint a,
		 gpointer data)
{
  GuppiScatterStyle* sty = GUPPI_SCATTER_STYLE(data);
  guint rgba = RGBA_TO_UINT(r >> 8, g >> 8, b >> 8, a >> 8);
  guppi_scatter_style_set_color(sty, rgba);
}

static void
guppi_scatter_style_control_connect_signals(GuppiScatterStyleControl* gssc)
{
  GtkWidget* w;
  GladeXML* xml;
  GuppiScatterStyle* style;

  g_return_if_fail(gssc != NULL);

  /* shorthand */
  xml = gssc->glade_xml;
  style = gssc->style;

  g_return_if_fail(xml != NULL);
  g_return_if_fail(style != NULL);

  gtk_signal_connect(GTK_OBJECT(style),
		     "changed",
		     GTK_SIGNAL_FUNC(state_change_cb),
		     gssc);

  w = glade_xml_get_widget(xml, "marker_menu");
  gtk_signal_connect(GTK_OBJECT(w),
		     "selected_marker",
		     GTK_SIGNAL_FUNC(marker_style_cb),
		     style);

  w = glade_xml_get_widget(xml, "marker_size_1_spinner");
  gtk_signal_connect(GTK_OBJECT(w),
		     "changed",
		     GTK_SIGNAL_FUNC(size1_spin_cb),
		     style);

  w = glade_xml_get_widget(xml, "marker_size_2_spinner");
  gtk_signal_connect(GTK_OBJECT(w),
		     "changed",
		     GTK_SIGNAL_FUNC(size2_spin_cb),
		     style);

  w = glade_xml_get_widget(xml, "color_reverse_gradient");
  gtk_signal_connect(GTK_OBJECT(w),
		     "toggled",
		     GTK_SIGNAL_FUNC(reverse_color_grad_cb),
		     style);

  w = glade_xml_get_widget(xml, "marker_size_1_reverse_gradient");
  gtk_signal_connect(GTK_OBJECT(w),
		     "toggled",
		     GTK_SIGNAL_FUNC(reverse_size1_grad_cb),
		     style);


  w = glade_xml_get_widget(xml, "marker_size_2_reverse_gradient");
  gtk_signal_connect(GTK_OBJECT(w),
		     "toggled",
		     GTK_SIGNAL_FUNC(reverse_size2_grad_cb),
		     style);

  w = glade_xml_get_widget(xml, "color_picker");
  gtk_signal_connect(GTK_OBJECT(w),
		     "selected_data",
		     GTK_SIGNAL_FUNC(pick_color_grad_cb),
		     style);

  w = glade_xml_get_widget(xml, "marker_size_1_picker");
  gtk_signal_connect(GTK_OBJECT(w),
		     "selected_data",
		     GTK_SIGNAL_FUNC(pick_size1_grad_cb),
		     style);

  w = glade_xml_get_widget(xml, "marker_size_2_picker");
  gtk_signal_connect(GTK_OBJECT(w),
		     "selected_data",
		     GTK_SIGNAL_FUNC(pick_size2_grad_cb),
		     style);

  w = glade_xml_get_widget(xml, "colorpicker_lower");
  gtk_signal_connect(GTK_OBJECT(w),
		     "color_set",
		     GTK_SIGNAL_FUNC(color_box_low_cb),
		     style);

  w = glade_xml_get_widget(xml, "colorpicker_upper");
  gtk_signal_connect(GTK_OBJECT(w),
		     "color_set",
		     GTK_SIGNAL_FUNC(color_box_high_cb),
		     style);

}

/*****************************************************************************/

void
guppi_scatter_style_control_construct(GuppiScatterStyleControl* gssc,
				      GuppiScatterStyle* sty)
{
  const gchar* path;

  g_return_if_fail(gssc != NULL);
  g_return_if_fail(sty != NULL);
  g_return_if_fail(gssc->style == NULL);

  gssc->style = sty;
  gtk_object_ref(GTK_OBJECT(gssc->style));

  path = guppi_glade_path("guppi-scatter-style-control.glade");
  gssc->glade_xml = path ? glade_xml_new(path, "root_table") : NULL;

  if (gssc->glade_xml == NULL)
    g_error("glade_xml_new() failed to load \"guppi-scatter-style-control.glade\"");
  gssc->widget = glade_xml_get_widget(gssc->glade_xml, "root_table");

  guppi_scatter_style_control_apply_state(gssc);
  guppi_scatter_style_control_connect_signals(gssc);

  gtk_widget_show(gssc->widget);
  gtk_container_add(GTK_CONTAINER(gssc), gssc->widget);
}

GtkWidget*
guppi_scatter_style_control_new(GuppiScatterStyle* sty)
{
  GuppiScatterStyleControl* gssc = 
    GUPPI_SCATTER_STYLE_CONTROL(gtk_type_new(guppi_scatter_style_control_get_type()));
  guppi_scatter_style_control_construct(gssc, sty);
  return GTK_WIDGET(gssc);
}



/* $Id: guppi-scatter-style-control.c,v 1.3 2000/04/25 13:57:14 trow Exp $ */
