/* This is -*- C -*- */
/* $Id: guppi-boxplot-item.c,v 1.2 2000/05/03 17:06:48 trow Exp $ */

/*
 * guppi-boxplot-item.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-boxplot-item.h"
#include "guppi-boxplot-state.h"

#include <libart_lgpl/art_svp_vpath.h>

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_boxplot_item_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_boxplot_item_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_boxplot_item_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_boxplot_item_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_boxplot_item_calc_geometry(GuppiCanvasItem* gci)
{
  GuppiBoxplotState* state = GUPPI_BOXPLOT_STATE(guppi_canvas_item_state(gci));

  if (state->data == NULL) {
    guppi_canvas_item_set_geometry(gci, 0, 0, 0, 0, 0, 0, 0, 0);
  } else if (state->horizontal) {
    guppi_canvas_item_set_geometry(gci, 0, 0, 0, 0, 10, 15, 20, 0.1);
  } else {
    guppi_canvas_item_set_geometry(gci, 10, 15, 20, 0.1, 0, 0, 0, 0);
  }
  
}

static void
guppi_boxplot_item_class_init(GuppiBoxplotItemClass* klass)
{
  static void guppi_boxplot_item_update(GnomeCanvasItem*, double affine[6],
					ArtSVP*, gint);
  static void guppi_boxplot_item_render(GnomeCanvasItem*, GnomeCanvasBuf*);
  static void state_set_hook(GuppiCanvasItem*, GuppiItemState*);

  GtkObjectClass* object_class = (GtkObjectClass*)klass;
  GnomeCanvasItemClass* item_class = GNOME_CANVAS_ITEM_CLASS(klass);
  GuppiCanvasItemClass* gci_class = GUPPI_CANVAS_ITEM_CLASS(klass);
 
  parent_class = gtk_type_class(GUPPI_TYPE_CANVAS_ITEM);

  object_class->get_arg = guppi_boxplot_item_get_arg;
  object_class->set_arg = guppi_boxplot_item_set_arg;
  object_class->destroy = guppi_boxplot_item_destroy;
  object_class->finalize = guppi_boxplot_item_finalize;

  item_class->update = guppi_boxplot_item_update;
  item_class->render = guppi_boxplot_item_render;

  gci_class->state_set_hook = state_set_hook;
  gci_class->calc_geometry = guppi_boxplot_item_calc_geometry;

}

static void
guppi_boxplot_item_init(GuppiBoxplotItem* obj)
{
  GUPPI_CANVAS_ITEM(obj)->width_hunger = 0.25;
  GUPPI_CANVAS_ITEM(obj)->height_hunger = 0.25;
}

GtkType
guppi_boxplot_item_get_type(void)
{
  static GtkType guppi_boxplot_item_type = 0;
  if (!guppi_boxplot_item_type) {
    static const GtkTypeInfo guppi_boxplot_item_info = {
      "GuppiBoxplotItem",
      sizeof(GuppiBoxplotItem),
      sizeof(GuppiBoxplotItemClass),
      (GtkClassInitFunc)guppi_boxplot_item_class_init,
      (GtkObjectInitFunc)guppi_boxplot_item_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_boxplot_item_type = gtk_type_unique(GUPPI_TYPE_CANVAS_ITEM, &guppi_boxplot_item_info);
  }
  return guppi_boxplot_item_type;
}

/*****************************************************************************/

static void
guppi_boxplot_item_update(GnomeCanvasItem* item, double affine[6],
			  ArtSVP* clip_path, gint flags)
{
  GuppiCanvasItem* gci;
  GuppiBoxplotItem* bpi;
  GuppiBoxplotState* state;
  ArtVpath path[16];
  ArtSVP* svp;
  double t0, ta, tm, tb, t1;
  double md, q1, q3, iqr;
  gint i;

  gci = GUPPI_CANVAS_ITEM(item);
  bpi = GUPPI_BOXPLOT_ITEM(item);
  state = GUPPI_BOXPLOT_STATE(guppi_canvas_item_state(gci));

  /* Chain our calls to update */
  if (parent_class && GNOME_CANVAS_ITEM_CLASS(parent_class)->update)
    GNOME_CANVAS_ITEM_CLASS(parent_class)->update(item,affine,clip_path,flags);

  if (state->data) {

    state->md = guppi_seq_scalar_median(state->data);
    state->q1 = guppi_seq_scalar_q1(state->data);
    state->q3 = guppi_seq_scalar_q3(state->data);
    state->iqr = state->q3 - state->q1;
    state->iqr_factor = 0.2;

    if (state->horizontal) {
      guppi_canvas_item_plot2world(gci, state->md, 0, &md, NULL);
      guppi_canvas_item_plot2world(gci, state->q1, 0, &q1, NULL);
      guppi_canvas_item_plot2world(gci, state->q3, 0, &q3, NULL);
    } else {
      guppi_canvas_item_plot2world(gci, 0, state->md, NULL, &md);
      guppi_canvas_item_plot2world(gci, 0, state->q1, NULL, &q1);
      guppi_canvas_item_plot2world(gci, 0, state->q3, NULL, &q3);
    }
    iqr = q3 - q1;
    
    if (state->horizontal) {
      guppi_canvas_item_plot2world(gci, 0, gci->full_y0, NULL, &t1);
      guppi_canvas_item_plot2world(gci, 0, gci->full_y1, NULL, &t0);
    } else {
      guppi_canvas_item_plot2world(gci, gci->full_x0, 0, &t0, NULL);
      guppi_canvas_item_plot2world(gci, gci->full_x1, 0, &t1, NULL);

    }

    tm = fabs(t1-t0);
    t0 += tm*0.15;
    t1 -= tm*0.15;
    tm = (t0+t1)/2;
    ta = (3*t0+t1)/4;
    tb = (t0+3*t1)/4;

    i=0;

    path[i].code = ART_MOVETO_OPEN;
    path[i].x = q1 - iqr * state->iqr_factor;
    path[i].y = ta;
    ++i;
    
    path[i].code = ART_LINETO;
    path[i].x = path[i-1].x;
    path[i].y = tb;
    ++i;

    path[i].code = ART_MOVETO_OPEN;
    path[i].x = path[i-2].x;
    path[i].y = tm;
    ++i;

    path[i].code = ART_LINETO;
    path[i].x = q1;
    path[i].y = tm;
    ++i;

    path[i].code = ART_MOVETO;
    path[i].x = q1;
    path[i].y = t1;
    ++i;

    path[i].code = ART_LINETO;
    path[i].x = q1;
    path[i].y = t0;
    ++i;

    path[i].code = ART_LINETO;
    path[i].x = q3;
    path[i].y = t0;
    ++i;

    path[i].code = ART_LINETO;
    path[i].x = q3;
    path[i].y = t1;
    ++i;

    path[i].code = ART_LINETO;
    path[i].x = path[i-4].x;
    path[i].y = path[i-4].y;
    ++i;

    path[i].code = ART_MOVETO_OPEN;
    path[i].x = md;
    path[i].y = t0;
    ++i;

    path[i].code = ART_LINETO;
    path[i].x = md;
    path[i].y = t1;
    ++i;

    path[i].code = ART_MOVETO_OPEN;
    path[i].x = q3 + iqr * state->iqr_factor;
    path[i].y = ta;
    ++i;

    path[i].code = ART_LINETO;
    path[i].x = path[i-1].x;
    path[i].y = tb;
    ++i;

    path[i].code = ART_MOVETO_OPEN;
    path[i].x = path[i-1].x;
    path[i].y = tm;
    ++i;

    path[i].code = ART_LINETO;
    path[i].x = q3;
    path[i].y = tm;
    ++i;

    path[i].code = ART_END;
    path[i].x = path[i].y = 0;

    /* Transpose path coordinates if not horizontal */
    if (!state->horizontal) {
      for (i=0; path[i].code != ART_END; ++i) {
	tm = path[i].x;
	path[i].x = path[i].y;
	path[i].y = tm;
      }
    }

    {
      /* This is *so* abusive; but libart_lgpl spews those damn
       colinear warning messages everywhere... */
      static FILE* devnull = NULL;
      FILE* stdout_copy = stdout;
      if (devnull == NULL) 
	devnull = fopen("/dev/null", "w");
      stdout = devnull;

      svp = art_svp_vpath_stroke(path,
				 ART_PATH_STROKE_JOIN_ROUND,
				 ART_PATH_STROKE_CAP_ROUND,
				 MAX(state->width, 0.25),
				 4, 0.25);

      stdout = stdout_copy;
    }

    gnome_canvas_item_update_svp_clip(item, &bpi->box_svp, svp, clip_path);
    
  } else {
    /* We don't have any data */
    if (bpi->box_svp)
      art_svp_free(bpi->box_svp);
    bpi->box_svp = NULL;
  }
  
  guppi_canvas_item_request_total_redraw(gci);
}

static void
guppi_boxplot_item_render(GnomeCanvasItem* item,
			  GnomeCanvasBuf* buf)
{
  GuppiCanvasItem* gci;
  GuppiBoxplotItem* bpi;
  GuppiBoxplotState* state;

  gci = GUPPI_CANVAS_ITEM(item);
  bpi = GUPPI_BOXPLOT_ITEM(item);
  state = GUPPI_BOXPLOT_STATE(guppi_canvas_item_state(gci));

  /* Chain our calls to render */
  if (parent_class && GNOME_CANVAS_ITEM_CLASS(parent_class)->render)
    GNOME_CANVAS_ITEM_CLASS(parent_class)->render(item, buf);

  if (bpi->box_svp) {
    buf = guppi_canvas_item_clip_buf(gci, buf);
    gnome_canvas_render_svp(buf, bpi->box_svp, state->color);
    g_free(buf);
  }
}

static void
changed_data(GuppiItemState* state, GuppiCanvasItem* gci)
{
  gci->need_geometry_recalc = TRUE;
  guppi_canvas_item_calc_geometry(gci);
  guppi_canvas_item_request_update(gci);
}

static void
state_set_hook(GuppiCanvasItem* item, GuppiItemState* state)
{
  GuppiBoxplotItem* si = GUPPI_BOXPLOT_ITEM(item);
  GuppiBoxplotState* ss = GUPPI_BOXPLOT_STATE(state);

  g_return_if_fail(si != NULL);
  g_return_if_fail(ss != NULL);


  gtk_signal_connect(GTK_OBJECT(ss),
		     "reset_x_data",
		     GTK_SIGNAL_FUNC(changed_data),
		     si);

  gtk_signal_connect(GTK_OBJECT(ss),
		     "reset_y_data",
		     GTK_SIGNAL_FUNC(changed_data),
		     si);

  gtk_signal_connect_object(GTK_OBJECT(ss),
			    "changed_x_data",
			    GTK_SIGNAL_FUNC(guppi_canvas_item_request_update),
			    GTK_OBJECT(si));

  gtk_signal_connect_object(GTK_OBJECT(ss),
			    "changed_y_data",
			    GTK_SIGNAL_FUNC(guppi_canvas_item_request_update),
			    GTK_OBJECT(si));

}


/* $Id: guppi-boxplot-item.c,v 1.2 2000/05/03 17:06:48 trow Exp $ */
