/* This is -*- C -*- */
/* $Id: guppi-axis-item.c,v 1.7 2000/05/03 17:06:47 trow Exp $ */

/*
 * guppi-axis-item.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <guppi-rgb.h>
#include <guppi-alpha-template.h>

#include "guppi-axis-item.h"
#include "guppi-axis-calc.h"
#include "guppi-axis-state.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0,
  ARG_STATE
};

static void
guppi_axis_item_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_axis_item_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {
  case ARG_STATE:
    guppi_canvas_item_set_state(GUPPI_CANVAS_ITEM(obj),
				GUPPI_ITEM_STATE(GTK_VALUE_POINTER(*arg)));
    break;

  default:
    break;
  };
}

static void
guppi_axis_item_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_axis_item_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
rescaled_xy_cb(GuppiCanvasItem* gci, gchar xy)
{
  static void guppi_axis_item_calc_labels(GuppiAxisItem*);
  static void guppi_axis_item_calc_geometry(GuppiAxisItem*);

  GuppiAxisItem* ai = GUPPI_AXIS_ITEM(gci);
  GuppiAxisState* state = GUPPI_AXIS_STATE(guppi_canvas_item_state(gci));

  if ((guppi_axis_state_horizontal(state) && xy == 'y') ||
      (guppi_axis_state_vertical(state) && xy == 'x'))
    return;

  guppi_axis_item_calc_labels(ai);
  guppi_axis_item_calc_geometry(ai);
}

static void
rescaled_x_cb(GuppiCanvasItem* gci, double x0, double x1)
{
  rescaled_xy_cb(gci, 'x');
}

static void
rescaled_y_cb(GuppiCanvasItem* gci, double y0, double y1)
{
  rescaled_xy_cb(gci, 'y');
}

static void
guppi_axis_item_class_init(GuppiAxisItemClass* klass)
{
  static void guppi_axis_item_update(GnomeCanvasItem*, double affine[6],
				     ArtSVP*, gint);
  static void guppi_axis_item_render(GnomeCanvasItem*, GnomeCanvasBuf*);
  static void guppi_axis_item_state_set_hook(GuppiCanvasItem*,
					     GuppiItemState*);
  static void guppi_axis_item_calc_geometry(GuppiCanvasItem*);


  GtkObjectClass* object_class = (GtkObjectClass*)klass;
  GnomeCanvasItemClass* item_class = GNOME_CANVAS_ITEM_CLASS(klass);
  GuppiCanvasItemClass* gci_class = GUPPI_CANVAS_ITEM_CLASS(klass);

  parent_class = gtk_type_class(GUPPI_TYPE_CANVAS_ITEM);

  gtk_object_add_arg_type("GuppiAxisItem::state",
			  GTK_TYPE_POINTER,
			  GTK_ARG_WRITABLE,
			  ARG_STATE);

  object_class->get_arg = guppi_axis_item_get_arg;
  object_class->set_arg = guppi_axis_item_set_arg;
  object_class->destroy = guppi_axis_item_destroy;
  object_class->finalize = guppi_axis_item_finalize;

  item_class->update = guppi_axis_item_update;
  item_class->render = guppi_axis_item_render;

  gci_class->state_set_hook = guppi_axis_item_state_set_hook;
  gci_class->calc_geometry = guppi_axis_item_calc_geometry;

  gci_class->rescaled_x = rescaled_x_cb;
  gci_class->rescaled_y = rescaled_y_cb;
}

static void
guppi_axis_item_init(GuppiAxisItem* obj)
{

}

GtkType
guppi_axis_item_get_type(void)
{
  static GtkType guppi_axis_item_type = 0;
  if (!guppi_axis_item_type) {
    static const GtkTypeInfo guppi_axis_item_info = {
      "GuppiAxisItem",
      sizeof(GuppiAxisItem),
      sizeof(GuppiAxisItemClass),
      (GtkClassInitFunc)guppi_axis_item_class_init,
      (GtkObjectInitFunc)guppi_axis_item_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_axis_item_type = gtk_type_unique(GUPPI_TYPE_CANVAS_ITEM, &guppi_axis_item_info);
  }
  return guppi_axis_item_type;
}

static void
guppi_axis_item_calc_labels(GuppiAxisItem* gai)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(gai);
  GuppiAxisState* state;
  GList* old_label_list;
  GList* iter;

  g_return_if_fail(gai != NULL);

  state = GUPPI_AXIS_STATE(guppi_canvas_item_state(gci));

  old_label_list = gai->label_list;
  gai->label_list = NULL;

  if (guppi_axis_state_horizontal(state)) {

    gai->label_list =
      optimal_guppi_axis_numerical_labels(gci->x0, gci->x1, 6, 10);

    /* Manually rotate the x-axis labels */
    iter = gai->label_list;
    while (iter) {
      ((GuppiAxisLabel*)iter->data)->rot = -90;
      iter = g_list_next(iter);
    }
  
    guppi_axis_label_render_labels(gai->label_list, old_label_list);
    guppi_axis_label_list_free(old_label_list);
  } else if (guppi_axis_state_vertical(state)) {
    gai->label_list =
      optimal_guppi_axis_numerical_labels(gci->y0, gci->y1, 6, 10);
    guppi_axis_label_render_labels(gai->label_list, old_label_list);
    guppi_axis_label_list_free(old_label_list);
  }
}

static void
guppi_axis_item_label_size(GuppiAxisItem* gai, double* lw, double* lh)
{
  double lab_w=0, lab_h=0;
  GList* iter;
  GuppiAxisState* state;
  GuppiAxisLabel* label;

  if (lw) *lw = 0;
  if (lh) *lh = 0;

  g_return_if_fail(gai != NULL);

  state = GUPPI_AXIS_STATE(guppi_canvas_item_state(GUPPI_CANVAS_ITEM(gai)));

  if (gai->label_list == NULL)
    guppi_axis_item_calc_labels(gai);

  iter = gai->label_list;
  while (iter) {
    label = (GuppiAxisLabel*)iter->data;
    lab_w = MAX(lab_w, label->atemp->width);
    lab_h = MAX(lab_h, label->atemp->height);
    iter = g_list_next(iter);
  }

  lab_w += (gint)state->tick_size + (gint)state->rule_width + 1;
  lab_h += (gint)state->tick_size + (gint)state->rule_width + 1;

  if (lw) *lw = lab_w;
  if (lh) *lh = lab_h;
}

static void
guppi_axis_item_optimal_size(GuppiAxisItem* gai, double* optw, double* opth)
{
  g_return_if_fail(gai != NULL);
  guppi_axis_item_label_size(gai, optw, opth);
  if (optw) *optw += MIN(*optw*0.2, 10);
  if (opth) *opth += MIN(*opth*0.2, 10);
}

static void
guppi_axis_item_calc_geometry(GuppiAxisItem* gai)
{
  double w0=0, w=0, w1=0;
  double h0=0, h=0, h1=0;
  GuppiAxisState* state;

  g_return_if_fail(gai != NULL);
  
  state = GUPPI_AXIS_STATE0(guppi_canvas_item_state(GUPPI_CANVAS_ITEM(gai)));
  if (state == NULL)
    return;

  if (gai->label_list == NULL)
    guppi_axis_item_calc_labels(gai);

  /* Find the max label width/height */
  guppi_axis_item_optimal_size(gai, &w, &h);

  if (guppi_axis_state_horizontal(state)) {
    h0 = 0.8 * h;
    h1 = 1.2 * h;
    w = 0;
  } else if (guppi_axis_state_vertical(state)) {
    w0 = 0.8 * w;
    w1 = 1.2 * w;
    h = 0;
  }

  guppi_canvas_item_set_geometry(GUPPI_CANVAS_ITEM(gai),
				 w0, w, w1, 0.1,
				 h0, h, h1, 0.1);
}

static void
guppi_axis_item_update(GnomeCanvasItem* item, double affine[6],
		       ArtSVP* clip_path, gint flags)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(item);
  GuppiAxisItem* gai = GUPPI_AXIS_ITEM(item);
  GuppiAxisState* state;

  /* Chain our calls to update */
  if (parent_class && GNOME_CANVAS_ITEM_CLASS(parent_class)->update)
    GNOME_CANVAS_ITEM_CLASS(parent_class)->update(item,affine,clip_path,flags);

  state = GUPPI_AXIS_STATE(guppi_canvas_item_state(gci));

  if (gai->label_list == NULL)
    guppi_axis_item_calc_labels(gai);
  //guppi_canvas_item_calc_geometry_when_idle(gci);

  guppi_canvas_item_request_total_redraw(GUPPI_CANVAS_ITEM(item));
}

static void
guppi_axis_item_render(GnomeCanvasItem* item,
		       GnomeCanvasBuf* buf)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(item);
  GuppiAxisItem* gai = GUPPI_AXIS_ITEM(item);
  GuppiAxisState* state;
  double t, residue, residue2;
  gint i, j, xc, yc, xw, yw, offset;
  guchar alph;
  GList* iter;
  GuppiAxisLabel* lab;
  gpointer ptr, orig_ptr;

  /* Chain our calls to render */
  if (parent_class && GNOME_CANVAS_ITEM_CLASS(parent_class)->render)
    GNOME_CANVAS_ITEM_CLASS(parent_class)->render(item, buf);

  state = GUPPI_AXIS_STATE(guppi_canvas_item_state(gci));

  buf = guppi_canvas_item_clip_buf(gci, buf);

  xw = buf->rect.x1 - buf->rect.x0;
  yw = buf->rect.y1 - buf->rect.y0;

  /* First, draw our edge rules */
  if (guppi_axis_state_horizontal(state)) {
    j = state->type == AXIS_SOUTH ? gci->cy0 : gci->cy1-1;
    residue = state->rule_width;
    while (residue > 1e-4) {
      if (buf->rect.y0 <= j && j < buf->rect.y1) {
	ptr = buf->buf + (j - buf->rect.y0)*buf->buf_rowstride;
	alph = residue >= 1 ? 0xff : (guchar)(residue * 0xff);
	for (i=0; i<xw; ++i) {
	  PIXEL_RGBA(ptr, 0, 0, 0, alph);
	  ptr += 3;
	}
      }
      state->type == AXIS_SOUTH ? ++j : --j;
      --residue;
    }
  } else if (guppi_axis_state_vertical(state)) {
    j = state->type == AXIS_EAST ? gci->cx0 : gci->cx1-1;
    residue = state->rule_width;
    while (residue > 1e-4) {
      if (buf->rect.x0 <= j && j < buf->rect.x1) {
	ptr = buf->buf + (j - buf->rect.x0)*3;
	alph = residue >= 1 ? 0xff : (guchar)(residue * 0xff);
	for (i=0; i<yw; ++i) {
	  PIXEL_RGBA(ptr, 0, 0, 0, alph);
	  ptr += buf->buf_rowstride;
	}
      }
      state->type == AXIS_EAST ? ++j : --j;
      --residue;
    }
  }

  /* Next, we paint in our tick marks and labels */

  iter = gai->label_list;

  if (guppi_axis_state_horizontal(state)) {

    while (iter) {
      lab = (GuppiAxisLabel*)iter->data;
      t = lab->pos;
      guppi_canvas_item_plot2pixel(gci, t, 0, &xc, NULL);

      /* Draw our tick marks */
      j = state->type == AXIS_NORTH ?
	gci->cy1-1-(gint)floor(state->rule_width) :
	gci->cy0+(gint)floor(state->rule_width);

      residue = state->tick_size;
      while (residue > 1e-4) {

	if (buf->rect.y0 <= j && j < buf->rect.y1) {
	  orig_ptr = buf->buf + (j - buf->rect.y0)*buf->buf_rowstride +
	    (xc-buf->rect.x0)*3;
      
	  i=0;
	  residue2 = state->tick_width * MIN(residue, 1);
	  while (residue2 > 1e-4) {
	    if (i == 0 && buf->rect.x0 <= xc && xc < buf->rect.x1) {
	      alph = residue2 >= 1 ? 0xff : (guchar)(residue2 * 0xff);
	      PIXEL_RGBA(orig_ptr, 0, 0, 0, alph);
	    } else if (i > 0) {
	      alph = residue2 >= 2 ? 0xff : (guchar)((residue2/2)*0xff);
	      if (buf->rect.x0 <= xc+i && xc+i < buf->rect.x1) 
		PIXEL_RGBA(orig_ptr+3*i, 0, 0, 0, alph);
	      if (buf->rect.x0 <= xc-i && xc-i < buf->rect.x1) 
		PIXEL_RGBA(orig_ptr-3*i, 0, 0, 0, alph);
	    }
	    residue2 -= i ? 2 : 1;
	    ++i;
	  }
	}

	state->type == AXIS_SOUTH ? ++j : --j;
	--residue;
      }
      
      offset = (gint)state->tick_size + (gint)state->rule_width;
      
      if (lab->atemp) {
	xc -= lab->atemp->width/2;
	
	/* Don't draw if we are too obscured. */
	if ((gci->cx0-xc)/(double)lab->atemp->width < 0.50 &&
	    (xc+lab->atemp->width-gci->cx1)/(double)lab->atemp->width < 0.50) {
	  
	  /* Shift labels to avoid partial visibility. */
	  if (xc < gci->cx0)
	    xc = gci->cx0;
	  if (xc + lab->atemp->width >= gci->cx1)
	    xc = gci->cx1 - 1 - lab->atemp->width;
	  
	  guppi_alpha_template_print(lab->atemp,
				     xc,
				     state->type == AXIS_NORTH ?
				     gci->cy1-5-lab->atemp->height-offset :
				     gci->cy0+4+offset,
				     0xdf, 0, 0, 0xff,
				     buf->buf, buf->buf_rowstride,
				     buf->rect.x1 - buf->rect.x0,
				     buf->rect.y1 - buf->rect.y0,
				     buf->rect.x0, buf->rect.y0);
	}
      }
      iter = g_list_next(iter);
    }
  } else if (guppi_axis_state_vertical(state)) {

    while (iter) {
      lab = (GuppiAxisLabel*)iter->data;
      t = lab->pos;
      guppi_canvas_item_plot2pixel(gci, 0, t, NULL, &yc);

      /* Draw out tick marks */
      j = state->type == AXIS_EAST ?
	gci->cx0+(gint)state->rule_width :
	gci->cx1-1-(gint)state->rule_width;
      
      residue = state->tick_size;
      while (residue > 1e-4) {

	if (buf->rect.x0 <= j && j < buf->rect.x1) {
	  orig_ptr = buf->buf + (j-buf->rect.x0)*3 +
	    (yc-buf->rect.y0)*buf->buf_rowstride;
	
	  i=0;
	  residue2 = state->tick_width * MIN(residue, 1);
	  while (residue2 > 1e-4) {
	    if (i == 0 && buf->rect.y0 <= yc && yc < buf->rect.y1) {
	      alph = residue2 >= 1 ? 0xff : (guchar)(residue2 * 0xff);
	      PIXEL_RGBA(orig_ptr, 0, 0, 0, alph);
	    } else if (i > 0) {
	      alph = residue2 >= 2 ? 0xff : (guchar)((residue2/2)*0xff);
	      if (buf->rect.y0 <= yc+i && yc+i < buf->rect.y1) 
		PIXEL_RGBA(orig_ptr+i*buf->buf_rowstride, 0, 0, 0, alph);
	      if (buf->rect.y0 <= yc-i && yc-i < buf->rect.y1) 
		PIXEL_RGBA(orig_ptr-i*buf->buf_rowstride, 0, 0, 0, alph);
	    }
	    residue2 -= i ? 2 : 1;
	    ++i;
	  }
	}

	state->type == AXIS_EAST ? ++j : --j;
	--residue;
      }

      offset = (gint)state->tick_size + (gint)state->rule_width;

      if (lab->atemp) {
	yc -= lab->atemp->height/2;
	
	if ((gci->cy0-yc)/(double)lab->atemp->height < 0.50 &&
	    (yc+lab->atemp->height-gci->cy1)/(double)lab->atemp->height < 0.50) {
	  
	  if (yc < gci->cy0)
	    yc = gci->cy0;
	  if (yc + lab->atemp->height >= gci->cy1)
	    yc = gci->cy1 - 1 - lab->atemp->height;
	  
	  guppi_alpha_template_print(lab->atemp,
				     state->type == AXIS_WEST ?
				     gci->cx1-5-lab->atemp->width-offset :
				     gci->cx0+4+offset,
				     yc,
				     0xdf, 0, 0, 0xff,
				     buf->buf, buf->buf_rowstride,
				     buf->rect.x1 - buf->rect.x0,
				     buf->rect.y1 - buf->rect.y0,
				     buf->rect.x0, buf->rect.y0);
	}
      }
      
      iter = g_list_next(iter);
    }
  }

  g_free(buf);
}

static void
guppi_axis_item_state_set_hook(GuppiCanvasItem* item, GuppiItemState* state)
{

}



/* $Id: guppi-axis-item.c,v 1.7 2000/05/03 17:06:47 trow Exp $ */








