/* $Id: guppi-axis-calc.c,v 1.3 2000/04/14 14:55:59 trow Exp $ */

/*
 * guppi-axis-calc.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>.

 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <math.h>
#include <guppi-useful.h>
#include "guppi-axis-calc.h"

static const double base4_divisors[] = { 4, 2, 1, -1 };
static const double base8_divisors[] = { 8, 4, 2, 1, -1 };
static const double base10_divisors[] = { 10, 5, 4, 2, 1, -1 };
static const double base16_divisors[] = { 16, 8, 4, 2, 1, -1 };
static const double base32_divisors[] = { 32, 16, 8, 4, 2, 1, -1 };
static const double base64_divisors[] = { 64, 32, 16, 8, 4, 2, 1, -1 };

GList*
optimal_guppi_axis_numerical_labels(double pos_min, double pos_max,
				    gint goal, gint base)
{
  double width, mag, div, step, start, count, t;
  double delta_best=1e+8, step_best=0, start_best=0;
  gint i, count_best=0;
  const double* divisors = NULL;
  GList* label_list = NULL;

  g_return_val_if_fail(goal > 1, NULL);

  if (pos_min > pos_max) {
    t = pos_min;
    pos_min = pos_max;
    pos_max = t;
  }

  width = fabs(pos_max - pos_min);
  mag = ceil(log(width / goal) / log(base));
  
  switch (base) {
  case 4:
    divisors = base4_divisors;
    break;
  case 8:
    divisors = base8_divisors;
    break;
  case 10:
    divisors = base10_divisors;
    break;
  case 16:
    divisors = base16_divisors;
    break;
  case 32:
    divisors = base32_divisors;
    break;
  case 64:
    divisors = base64_divisors;
    break;
  default:
    g_warning("Illegal radix (base=%d)", base);
    return NULL;
  }

  g_assert(divisors != NULL);
  for(i=0; divisors[i] > 0; ++i) {
    div = divisors[i];
    step = pow(base, mag)/div;
    start = ceil(pos_min/step)*step;
    count = floor(width/step);
    if (pos_min <= start && start <= pos_max)
      ++count;
    
    if (fabs(count-goal) < delta_best) {
      delta_best = fabs(count-goal);
      step_best = step;
      start_best = start;
      count_best = (gint)count;
    }
  }

  if (step_best <= 0) {
    g_message("Search for nice axis points failed.  This shouldn't happen.");
    return NULL;
  }

  /* Create the label list */

  for (i=0; i < count_best; ++i) {
    GuppiAxisLabel* label = g_new0(GuppiAxisLabel, 1);

    label->pos = start_best + i*step_best;
    if (fabs(label->pos / step_best) < 1e-12)
      label->pos = 0;

    label->text = g_strdup_printf("%g", label->pos);
    
    label_list = g_list_prepend(label_list, label);
  }
  label_list = g_list_reverse(label_list);

  return label_list;
}

void
guppi_axis_label_list_free(GList* ll)
{
  GList* iter = ll;
  GuppiAxisLabel* label;

  while (iter != NULL) {

    label = (GuppiAxisLabel*)iter->data;
    g_assert(label != NULL);

    if (label->text) {
      g_free(label->text);
      label->text = NULL;

      if (label->atemp)
	gtk_object_unref(GTK_OBJECT(label->atemp));
    }

    g_free(label);

    iter = g_list_next(iter);
  }

  if (ll)
    g_list_free(ll);
}

void
guppi_axis_label_render_labels(GList* new_list, GList* old_list)
{
  GList* new_iter;
  GList* old_iter;
  GuppiAxisLabel* new_label;
  GuppiAxisLabel* old_label;
  GnomeFont* font;
  double size;

  g_return_if_fail(new_list != NULL);

  /* Steal already-rendered text alpha templates from old list */
  if (old_list) {
    new_iter = new_list;

    while (new_iter) {

      new_label = (GuppiAxisLabel*)new_iter->data;

      old_iter = old_list;
      while (old_iter) {
	old_label = (GuppiAxisLabel*)old_iter->data;

	if (new_label->font == old_label->font &&
	    new_label->size == old_label->size &&
	    new_label->rot == old_label->rot &&
	    !strcmp(new_label->text, old_label->text)) {
	  new_label->atemp = old_label->atemp;
	  old_label->atemp = NULL;
	  break;
	}
	old_iter = g_list_next(old_iter);
      }
      new_iter = g_list_next(new_iter);
    }
  }

  /* Go through and do the rendering for the remaining labels */
  new_iter = new_list;
  while (new_iter) {
    new_label = (GuppiAxisLabel*)new_iter->data;
    if (new_label->atemp == NULL) {

      font = new_label->font;
      if (font == NULL)
	font = guppi_default_font();

      size = new_label->size;
      if (size <= 0)
	size = guppi_default_axis_label_size();

      new_label->atemp = guppi_alpha_template_text_general(font,
							   size,
							   new_label->text,
							   new_label->rot,
							   TRUE, 0);
    }
    new_iter = g_list_next(new_iter);
  }
}


/* $Id: guppi-axis-calc.c,v 1.3 2000/04/14 14:55:59 trow Exp $ */
