/* $Id: guppi-scm-dataset.c,v 1.1 2000/01/17 05:44:54 trow Exp $ */

/*
 * guppi-scm-dataset.c
 *
 * Copyright (C) 1999, 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 * and Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-scm-dataset.h"

static long dataset_type_tag;

#define SCM_TO_DATASET(x) (GUPPI_DATASET(SCM_CDR(x)))
#define SCM_DATASET_P(x) (SCM_NIMP(x) && SCM_CAR(x) == dataset_type_tag)

gboolean
scm_dataset_p(SCM x)
{
  return SCM_DATASET_P(x);
}

SCM
dataset2scm(GuppiDataset* x)
{
  SCM smob;

  SCM_DEFER_INTS;
  SCM_NEWCELL(smob);
  SCM_SETCAR(smob, dataset_type_tag);
  SCM_SETCDR(smob, x);
  gtk_object_ref(GTK_OBJECT(x));
  SCM_ALLOW_INTS;

  return smob;
}

GuppiDataset*
scm2dataset(SCM x)
{
  return SCM_DATASET_P(x) ? SCM_TO_DATASET(x) : NULL;
}

static SCM
mark_dataset(SCM x)
{
  return SCM_BOOL_F;
}

static scm_sizet
free_dataset(SCM x)
{
  GuppiDataset* ds = SCM_TO_DATASET(x);

  SCM_DEFER_INTS;
  gtk_object_unref(GTK_OBJECT(ds));
  SCM_ALLOW_INTS;

  return 0;
}

static int
print_dataset(SCM x, SCM port, scm_print_state* state)
{
  static gchar buffer[256];
  GuppiDataset* d;

  d = SCM_TO_DATASET(x);

  g_snprintf(buffer, 256,
	     "<GuppiDataSet \"%s\">",
	     guppi_dataset_name(d));

  scm_puts(buffer, port);
  return 1;
}

/*****************************************************************************/

GUPPI_PROC(dsp, "dataset?",
	   1,0,0, (SCM x))
{
  return gh_bool2scm(SCM_DATASET_P(x));
}

GUPPI_PROC(ds_make, "make-dataset",
	   1,0,0, (SCM sname))
{
  GuppiDataset* ds;
  const gchar* name;
  gint len;

  SCM_ASSERT(gh_string_p(sname), sname, SCM_ARG1, str_ds_make);

  name = gh_scm2newstr(sname, &len);

  ds = guppi_dataset_new();
  guppi_dataset_set_name(ds, name);

  return dataset2scm(ds);
}

GUPPI_PROC(ds_name, "dataset-name",
	   1,0,0, (SCM sds))
{
  GuppiDataset* ds;
  
  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_name);

  ds = scm2dataset(sds);
  return gh_str02scm((gchar*)guppi_dataset_name(ds));
}

GUPPI_PROC(ds_source, "dataset-source",
	   1,0,0, (SCM sds))
{
  GuppiDataset* ds;
  
  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_source);

  ds = scm2dataset(sds);
  return gh_str02scm((gchar*)guppi_dataset_source(ds));
}

GUPPI_PROC(ds_ref, "dataset-reference",
	   1,0,0, (SCM sds))
{
  GuppiDataset* ds;
  
  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_ref);

  ds = scm2dataset(sds);
  return gh_str02scm((gchar*)guppi_dataset_reference(ds));
}

GUPPI_PROC(ds_cont, "dataset-contact",
	   1,0,0, (SCM sds))
{
  GuppiDataset* ds;
  
  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_cont);

  ds = scm2dataset(sds);
  return gh_str02scm((gchar*)guppi_dataset_contact(ds));
}

GUPPI_PROC(ds_url, "dataset-URL",
	   1,0,0, (SCM sds))
{
  GuppiDataset* ds;
  
  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_url);

  ds = scm2dataset(sds);
  return gh_str02scm((gchar*)guppi_dataset_URL(ds));
}

GUPPI_PROC(ds_notes, "dataset-notes",
	   1,0,0, (SCM sds))
{
  GuppiDataset* ds;
  
  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_notes);

  ds = scm2dataset(sds);
  return gh_str02scm((gchar*)guppi_dataset_notes(ds));
}

GUPPI_PROC(ds_setname, "dataset-set-name!",
	   2,0,0, (SCM sds, SCM stxt))
{
  GuppiDataset* ds;
  gchar* txt;
  gint len;

  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_setname);
  SCM_ASSERT(gh_string_p(stxt), stxt, SCM_ARG2, str_ds_setname);

  ds = scm2dataset(sds);
  txt = gh_scm2newstr(stxt, &len);
  guppi_dataset_set_name(ds, txt);
  g_free(txt);
  
  return sds;
}

GUPPI_PROC(ds_setsource, "dataset-set-source!",
	   2,0,0, (SCM sds, SCM stxt))
{
  GuppiDataset* ds;
  gchar* txt;
  gint len;

  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_setsource);
  SCM_ASSERT(gh_string_p(stxt), stxt, SCM_ARG2, str_ds_setsource);

  ds = scm2dataset(sds);
  txt = gh_scm2newstr(stxt, &len);
  guppi_dataset_set_source(ds, txt);
  g_free(txt);
  
  return sds;
}

GUPPI_PROC(ds_setref, "dataset-set-reference!",
	   2,0,0, (SCM sds, SCM stxt))
{
  GuppiDataset* ds;
  gchar* txt;
  gint len;

  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_setref);
  SCM_ASSERT(gh_string_p(stxt), stxt, SCM_ARG2, str_ds_setref);

  ds = scm2dataset(sds);
  txt = gh_scm2newstr(stxt, &len);
  guppi_dataset_set_reference(ds, txt);
  g_free(txt);
  
  return sds;
}

GUPPI_PROC(ds_setcont, "dataset-set-contact!",
	   2,0,0, (SCM sds, SCM stxt))
{
  GuppiDataset* ds;
  gchar* txt;
  gint len;

  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_setcont);
  SCM_ASSERT(gh_string_p(stxt), stxt, SCM_ARG2, str_ds_setcont);

  ds = scm2dataset(sds);
  txt = gh_scm2newstr(stxt, &len);
  guppi_dataset_set_contact(ds, txt);
  g_free(txt);
  
  return sds;
}

GUPPI_PROC(ds_seturl, "dataset-set-URL!",
	   2,0,0, (SCM sds, SCM stxt))
{
  GuppiDataset* ds;
  gchar* txt;
  gint len;

  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_seturl);
  SCM_ASSERT(gh_string_p(stxt), stxt, SCM_ARG2, str_ds_seturl);

  ds = scm2dataset(sds);
  txt = gh_scm2newstr(stxt, &len);
  guppi_dataset_set_URL(ds, txt);
  g_free(txt);
  
  return sds;
}

GUPPI_PROC(ds_setnotes, "dataset-set-notes!",
	   2,0,0, (SCM sds, SCM stxt))
{
  GuppiDataset* ds;
  gchar* txt;
  gint len;

  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_setnotes);
  SCM_ASSERT(gh_string_p(stxt), stxt, SCM_ARG2, str_ds_setnotes);

  ds = scm2dataset(sds);
  txt = gh_scm2newstr(stxt, &len);
  guppi_dataset_set_notes(ds, txt);
  g_free(txt);
  
  return sds;
}

static void
dataset_changed_cb(GuppiDataset* ds, gpointer user_data)
{
  g_return_if_fail(ds != NULL);
  g_return_if_fail(user_data != NULL);

  guppi_safe_call1((SCM)user_data, dataset2scm(ds));
}

GUPPI_PROC(ds_changed_connect, "dataset-changed-connect",
	   2,0,0, (SCM sds, SCM proc))
{
  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_changed_connect);
  SCM_ASSERT(gh_procedure_p(proc), proc, SCM_ARG2, str_ds_changed_connect);

  gtk_signal_connect(GTK_OBJECT(scm2dataset(sds)),
		     "changed",
		     GTK_SIGNAL_FUNC(dataset_changed_cb),
		     (gpointer)proc);

  return SCM_UNSPECIFIED;
}

GUPPI_PROC(ds_changed_disconn, "dataset-changed-disconnect",
	   2,0,0, (SCM sds, SCM proc))
{
  SCM_ASSERT(SCM_DATASET_P(sds), sds, SCM_ARG1, str_ds_changed_disconn);
  SCM_ASSERT(gh_procedure_p(proc), proc, SCM_ARG2, str_ds_changed_disconn);

  gtk_signal_disconnect_by_func(GTK_OBJECT(scm2dataset(sds)),
				GTK_SIGNAL_FUNC(dataset_changed_cb),
				(gpointer)proc);

  return SCM_UNSPECIFIED;
}

/*****************************************************************************/

void
guppi_scm_dataset_init(void)
{
  static gboolean initialized = FALSE;
  static struct scm_smobfuns dataset_fns = {
    mark_dataset, free_dataset, print_dataset, NULL
  };

  g_return_if_fail(!initialized);
  initialized = TRUE;

  dataset_type_tag = scm_newsmob(&dataset_fns);

#include "guppi-scm-dataset.x"

}

/* $Id: guppi-scm-dataset.c,v 1.1 2000/01/17 05:44:54 trow Exp $ */
