/* This is -*- C -*- */
/* $Id: guppi-data-catalog.c,v 1.3 2000/01/05 02:10:18 trow Exp $ */

/*
 * guppi-data-catalog.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-data-catalog.h"

static GtkObjectClass* parent_class = NULL;

enum {
  CHANGED,
  ADDED,
  REMOVED,
  DATA_CHANGED,
  LAST_SIGNAL
};
static guint gdc_signals[LAST_SIGNAL] = { 0 };

enum {
  ARG_0
};

static void
guppi_data_catalog_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_data_catalog_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_data_catalog_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_data_catalog_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_data_catalog_class_init(GuppiDataCatalogClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_OBJECT);

  gdc_signals[CHANGED] = 
    gtk_signal_new("changed",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataCatalogClass, changed),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  gdc_signals[ADDED] = 
    gtk_signal_new("added",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataCatalogClass, added),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_POINTER);

  gdc_signals[REMOVED] = 
    gtk_signal_new("removed",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataCatalogClass, removed),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_POINTER);

  gdc_signals[DATA_CHANGED] = 
    gtk_signal_new("data_changed",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataCatalogClass, data_changed),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_POINTER);

  gtk_object_class_add_signals(object_class, gdc_signals, LAST_SIGNAL);

  object_class->get_arg = guppi_data_catalog_get_arg;
  object_class->set_arg = guppi_data_catalog_set_arg;
  object_class->destroy = guppi_data_catalog_destroy;
  object_class->finalize = guppi_data_catalog_finalize;

}

static void
guppi_data_catalog_init(GuppiDataCatalog* obj)
{
  obj->data = NULL;
}

GtkType
guppi_data_catalog_get_type(void)
{
  static GtkType guppi_data_catalog_type = 0;
  if (!guppi_data_catalog_type) {
    static const GtkTypeInfo guppi_data_catalog_info = {
      "GuppiDataCatalog",
      sizeof(GuppiDataCatalog),
      sizeof(GuppiDataCatalogClass),
      (GtkClassInitFunc)guppi_data_catalog_class_init,
      (GtkObjectInitFunc)guppi_data_catalog_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_data_catalog_type = gtk_type_unique(GTK_TYPE_OBJECT, &guppi_data_catalog_info);
  }
  return guppi_data_catalog_type;
}

GtkObject*
guppi_data_catalog_new(void)
{
  return GTK_OBJECT(gtk_type_new(guppi_data_catalog_get_type()));
}

GuppiDataCatalog*
guppi_data_catalog(void)
{
  static GuppiDataCatalog* gdc = NULL;

  if (gdc == NULL) {
    gdc = GUPPI_DATA_CATALOG(guppi_data_catalog_new());
  }
  return gdc;
}

static void
data_change_converter_cb(GuppiData* data, gpointer user_data)
{
  GuppiDataCatalog* gdc;

  g_return_if_fail(data != NULL);
  gdc = GUPPI_DATA_CATALOG(user_data);
  g_return_if_fail(gdc != NULL);

  gtk_signal_emit(GTK_OBJECT(gdc), gdc_signals[CHANGED]);
  gtk_signal_emit(GTK_OBJECT(gdc), gdc_signals[DATA_CHANGED], data);
}

void
guppi_data_catalog_add(GuppiDataCatalog* gdc, GuppiData* d)
{
  if (gdc == NULL)
    gdc = guppi_data_catalog();
  
  g_return_if_fail(gdc != NULL);
  g_return_if_fail(d != NULL);

  if (g_list_find(gdc->data, d) != NULL) {
    g_warning("Attempt to double-add %s::%s to GuppiDataCatalog blocked",
	      guppi_data_dataset_name(d), guppi_data_label(d));
    return;
  }
    

  gdc->data = g_list_append(gdc->data, d);
  gtk_object_ref(GTK_OBJECT(d));

  gtk_signal_connect(GTK_OBJECT(d),
		     "changed",
		     GTK_SIGNAL_FUNC(data_change_converter_cb),
		     (gpointer)gdc);

  gtk_signal_emit(GTK_OBJECT(gdc), gdc_signals[CHANGED]);
  gtk_signal_emit(GTK_OBJECT(gdc), gdc_signals[ADDED], d);
}

void
guppi_data_catalog_remove(GuppiDataCatalog* gdc, GuppiData* d)
{
  if (gdc == NULL)
    gdc = guppi_data_catalog();

  g_return_if_fail(gdc != NULL);
  g_return_if_fail(d != NULL);

  if (g_list_find(gdc->data, d) != NULL) {
    gdc->data = g_list_remove(gdc->data, d);

    gtk_signal_disconnect_by_func(GTK_OBJECT(d),
				  GTK_SIGNAL_FUNC(data_change_converter_cb),
				  (gpointer)gdc);

    gtk_object_unref(GTK_OBJECT(d));
  }

  gtk_signal_emit(GTK_OBJECT(gdc), gdc_signals[CHANGED]);
  gtk_signal_emit(GTK_OBJECT(gdc), gdc_signals[REMOVED], d);
}

gint
guppi_data_catalog_size(const GuppiDataCatalog* gdc)
{
  if (gdc == NULL)
    gdc = guppi_data_catalog();

  g_return_val_if_fail(gdc != NULL, 0);

  return g_list_length(gdc->data);
}

GuppiData*
guppi_data_catalog_get(GuppiDataCatalog* gdc, gint i)
{
  GList* val;

  g_return_val_if_fail(i >= 0, NULL);
  
  if (gdc == NULL)
    gdc = guppi_data_catalog();

  g_return_val_if_fail(gdc != NULL, NULL);
  g_return_val_if_fail(i < guppi_data_catalog_size(gdc), NULL);

  val = g_list_nth(gdc->data, i);
  g_return_val_if_fail(val != NULL, NULL);

  gtk_object_ref(GTK_OBJECT(val->data));
  
  return GUPPI_DATA(val->data);
}

GuppiData**
guppi_data_catalog_get_all(GuppiDataCatalog* gdc)
{
  gint i, size;
  GList* iter;
  GuppiData** result;

  if (gdc == NULL)
    gdc = guppi_data_catalog();

  g_return_val_if_fail(gdc != NULL, NULL);

  size = guppi_data_catalog_size(gdc);
  if (size == 0)
    return NULL;

  result = g_new(GuppiData*, size+1);

  iter = gdc->data;
  for(i=0; i<size; ++i) {
    result[i] = GUPPI_DATA(iter->data);
    gtk_object_ref(GTK_OBJECT(result[i]));
    iter = g_list_next(iter);
  }
  result[size] = NULL;

  return result;
}

GuppiData**
guppi_data_catalog_get_by_type(GuppiDataCatalog* gdc, GtkType type)
{
  gint i, size;
  GList* iter;
  GuppiData** result;

  g_return_val_if_fail(type != 0, NULL);

  if (gdc == NULL)
    gdc = guppi_data_catalog();

  g_return_val_if_fail(gdc != NULL, NULL);

  /* First, sweep through and count */
  size = 0;
  iter = gdc->data;
  while (iter != NULL) {
    if (GTK_CHECK_TYPE(iter->data, type))
      ++size;
    iter = g_list_next(iter);
  }

  /* Next build and populate our vector */
  result = g_new(GuppiData*, size+1);
  i = 0;
  iter = gdc->data;
  while (iter != NULL) {
    if (GTK_CHECK_TYPE(iter->data, type)) {
      g_assert(i < size);
      result[i] = GUPPI_DATA(iter->data);
      gtk_object_ref(GTK_OBJECT(result[i]));
      ++i;
    }
    iter = g_list_next(iter);
  }
  result[size] = NULL;

  return result;
}



/* $Id: guppi-data-catalog.c,v 1.3 2000/01/05 02:10:18 trow Exp $ */
