/* $Id: guppi-data-browser.c,v 1.1.1.1 1999/12/03 07:02:40 trow Exp $ */

/*
 * guppi-data-browser.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gnome.h>
#include <string.h>
#include "guppi-data-browser.h"

static GtkObjectClass* parent_class = NULL;

static void
guppi_data_browser_class_init(GuppiDataBrowserClass* klass)
{
  GtkObjectClass* object_class;

  object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_CLIST);
}

static void
guppi_data_browser_init(GuppiDataBrowser* db)
{

}

GtkType
guppi_data_browser_get_type(void)
{
  static GtkType data_browser_type = 0;

  if (!data_browser_type) {
    static const GtkTypeInfo data_browser_info = {
      "GuppiDataBrowser",
      sizeof(GuppiDataBrowser),
      sizeof(GuppiDataBrowserClass),
      (GtkClassInitFunc)guppi_data_browser_class_init,
      (GtkObjectInitFunc)guppi_data_browser_init,
      NULL, NULL,
      (GtkClassInitFunc)NULL
    };
    data_browser_type = gtk_type_unique(GTK_TYPE_CLIST, &data_browser_info);
  }
  return data_browser_type;
}

static void
guppi_data_browser_optimize_columns(GuppiDataBrowser* db)
{
  gint i;
  GtkCList* c;

  c = GTK_CLIST(db);

  for(i=0; i<=db->data_count; ++i)
    gtk_clist_set_column_width(c, i,
			       gtk_clist_optimal_column_width(c, i));
}

static void
guppi_data_browser_build(GuppiDataBrowser* db)
{
  gsize j;
  dindex_t i;
  char numbuf[10];
  char** rows;

  rows = g_new0(char*, 1+db->data_count);
  for(j=1; j<=db->data_count; ++j)
    rows[j] = g_new0(char, 256);

  db->i0 = guppi_data_min_index(db->data[0]);
  db->i1 = guppi_data_max_index(db->data[0]);
  for(j=1; j<db->data_count; ++j) {
    db->i0 = MIN(db->i0, guppi_data_min_index(db->data[j]));
    db->i1 = MAX(db->i1, guppi_data_max_index(db->data[j]));
  }

  gtk_clist_freeze(GTK_CLIST(db));
  gtk_clist_clear(GTK_CLIST(db));

  gtk_clist_set_column_title(GTK_CLIST(db), 0, "index");
  gtk_clist_set_column_justification(GTK_CLIST(db), 0, GTK_JUSTIFY_RIGHT);

  for(j=0; j<db->data_count; ++j) 
    gtk_clist_set_column_title(GTK_CLIST(db), j+1, 
			       guppi_data_label(db->data[j]));

  gtk_clist_column_titles_show(GTK_CLIST(db));

  for(i=db->i0; i<=db->i1; ++i) {
    g_snprintf(numbuf, 10, "%d", i);
    rows[0] = numbuf;
    for(j=0; j<db->data_count; ++j) {
      if (guppi_data_in_bounds(db->data[j], i)) {
	guppi_data_get(db->data[j], i, rows[j+1], 256);
      } else
	rows[j+1][0] = '\0';
    }

    gtk_clist_append(GTK_CLIST(db), rows);
  }

  guppi_data_browser_optimize_columns(db);
  
  gtk_clist_thaw(GTK_CLIST(db));

  for(j=1; j<db->data_count; ++j)
    g_free(rows[j]);
  g_free(rows);
}

/*****************************************************************************/

/* Data editting callbacks */

/* Aiee!  Global data! */
static GtkWidget* edit_dialog = NULL;
static GtkWidget* index_label = NULL;
static GtkWidget* data_entry = NULL;
static GuppiData* current_edit_data = NULL;
static dindex_t current_edit_index;

/*
  One danger with doing this is that a GuppiData object will get freed
  while the dialog for editting it sits open, leaving a dangling pointer
  in current_edit_data.  We should have a callback that connected to that
  data and watches for it being deleted, so that we can close up the window
  in that case.  But for now this will do...

  Actually, we should adjust the reference count of the relevant
  GuppiData properly when we hold/release its pointer.  Again,
  something we can do when this all gets cleaned up later.  (This is
  the kind of thinking that led to all of this Y2K crap.)
*/

static void
guppi_data_browser_edit_dialog_clicked(GnomeDialog* gd, gint button)
{
  GtkWidget* error_dialog;
  char err_msg[1024];
  gchar* p;
  gboolean valid;

  if (button != 0)
    return;

  p = gtk_entry_get_text(GTK_ENTRY(data_entry));
  valid = guppi_data_validate(current_edit_data, p, err_msg, 1024);

  if (valid) {
    guppi_data_set(current_edit_data, current_edit_index, p);
  } else {
    error_dialog = gnome_error_dialog(err_msg);
    gtk_widget_show(error_dialog);
  }
  current_edit_data = NULL;
}

static void
guppi_data_browser_entry_activate(GtkWidget* w)
{
  guppi_data_browser_edit_dialog_clicked(GNOME_DIALOG(edit_dialog), 0);
  gnome_dialog_close(GNOME_DIALOG(edit_dialog));
}

static void
guppi_data_browser_set_edit_dialog_label()
{
  char buffer[256];
  g_snprintf(buffer, 256, "Editing \"%s\"\nindex = %d",
	     guppi_data_label(current_edit_data), current_edit_index);
  gtk_label_set_text(GTK_LABEL(index_label), buffer);
}

static void
guppi_data_browser_select_row_cb(GtkWidget* widget,
				 gint row, gint col,
				 GdkEventButton* event,
				 gpointer data)
{
  dindex_t i;
  GuppiDataBrowser* db = GUPPI_DATA_BROWSER(widget);
  char buffer[256];
  
  gtk_clist_unselect_row(GTK_CLIST(widget),row,col);
  if (col == 0)
    return;

  i = row - db->i0;
  if (! guppi_data_in_bounds(db->data[col-1], i))
    return;

  if (edit_dialog == NULL) {
    edit_dialog = gnome_dialog_new("Edit Data",
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
    gnome_dialog_close_hides(GNOME_DIALOG(edit_dialog), TRUE);
    gnome_dialog_set_close(GNOME_DIALOG(edit_dialog), TRUE);
    gtk_signal_connect(GTK_OBJECT(edit_dialog), "clicked",
		       GTK_SIGNAL_FUNC(guppi_data_browser_edit_dialog_clicked),
		       NULL);

    index_label = gtk_label_new("foo");
    data_entry = gtk_entry_new();

    /* Catch it if we hit return in the text entry widget, make that the
       same as clicking on OK */
    gtk_signal_connect(GTK_OBJECT(data_entry), "activate",
		       GTK_SIGNAL_FUNC(guppi_data_browser_entry_activate),
		       NULL);

    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(edit_dialog)->vbox),
		       index_label,
		       TRUE, TRUE, GNOME_PAD);
    gtk_box_pack_end(GTK_BOX(GNOME_DIALOG(edit_dialog)->vbox),
		     data_entry,
		     TRUE, TRUE, GNOME_PAD);

  }

  current_edit_data = db->data[col-1];
  current_edit_index = i;

  gtk_widget_show_all(edit_dialog);

  guppi_data_get(db->data[col-1], i, buffer, 256);
  gtk_entry_set_text(GTK_ENTRY(data_entry), buffer);
  gtk_entry_select_region(GTK_ENTRY(data_entry), 0, strlen(buffer));
  gtk_widget_grab_focus(GTK_WIDGET(data_entry));

  guppi_data_browser_set_edit_dialog_label();
}

/*****************************************************************************/

static void
guppi_data_browser_changed_one_cb(GuppiData* foo, dindex_t i, gpointer bar,
				  gpointer us)
{
  gsize j=0;
  GuppiDataBrowser* db = GUPPI_DATA_BROWSER(us);
  char buffer[256];

  while (j<db->data_count && db->data[j] != foo)
    ++j;
  if (j >= db->data_count) {
    g_warning("something bad happened here...");
    return;
  }

  /* Get the changed value and refresh the display */
  guppi_data_get(foo, i, buffer, 256);
  gtk_clist_set_text(GTK_CLIST(db),
		     i-db->i0, /* row */
		     j+1,      /* col */
  		     buffer);
  guppi_data_browser_optimize_columns(db);

  /* If we are currently editing this specific element, shut it down! */
  if (foo == current_edit_data &&  i == current_edit_index) {
    gnome_dialog_close(GNOME_DIALOG(edit_dialog));
    current_edit_data = NULL;
  }
    
}

static void
guppi_data_browser_changed_many_cb(GuppiData* foo, gpointer us)
{
  gsize j=0;
  GuppiDataBrowser* db = GUPPI_DATA_BROWSER(us);
  dindex_t i, i0, i1;
  char buffer[256];

  while (j<db->data_count && db->data[j] != foo)
    ++j;
  if (j >= db->data_count) {
    g_warning("Something got fucked up in guppi_data_browser_changed_many_cb()");
    return;
  }

  /* Shut down any edit window we have for this data */
  if (foo == current_edit_data) {
    gnome_dialog_close(GNOME_DIALOG(edit_dialog));
    current_edit_data = NULL;
  }

  i0 = guppi_data_min_index(foo);
  i1 = guppi_data_max_index(foo);

  /* If our changed thing spills outside of our displayed range, we need
     to rebuild everything in sight. */
  if (i0 < db->i0 || i1 > db->i1) {
    guppi_data_browser_build(db);
    return;
  }

  /* Otherwise, we can get away by just repainting our column */

  while (j<db->data_count && db->data[j] != foo)
    ++j;
  if (j >= db->data_count) {
    g_warning("Something got fucked up in guppi_data_browser_changed_label_cb()");
    return;
  }
  gtk_clist_freeze(GTK_CLIST(db));

  for(i=db->i0; i<=db->i1; ++i) {
    if (guppi_data_in_bounds(foo, i)) {
      guppi_data_get(foo, i, buffer, 256);
      gtk_clist_set_text(GTK_CLIST(db), i-db->i0, j+1, buffer);
    } else
      gtk_clist_set_text(GTK_CLIST(db), i-db->i0, j+1, "");
  }

  guppi_data_browser_optimize_columns(db);

  gtk_clist_thaw(GTK_CLIST(db));

}

static void
guppi_data_browser_changed_label_cb(GuppiData* foo, gpointer us)
{
  gsize j=0;
  GuppiDataBrowser* db = GUPPI_DATA_BROWSER(us);
  
  while (j<db->data_count && db->data[j] != foo)
    ++j;
  if (j >= db->data_count) {
    g_warning("Something got fucked up in guppi_data_browser_changed_label_cb()");
    return;
  }

  gtk_clist_set_column_title(GTK_CLIST(db), j+1,
			     guppi_data_label(foo));

  guppi_data_browser_optimize_columns(db);

  /* If we are currently editing this specific data set, change the label
     accordingly.  We are so cool.*/
  if (foo == current_edit_data)
    guppi_data_browser_set_edit_dialog_label();
}

/*****************************************************************************/

GtkWidget*
guppi_data_browser_new(GuppiData** vec)
{
  gint i;
  GuppiDataBrowser* db;

  g_return_val_if_fail(vec != NULL, NULL);

  db = GUPPI_DATA_BROWSER(gtk_type_new(guppi_data_browser_get_type()));

  db->data_count = 0;
  while (vec[db->data_count])
    ++db->data_count;

  db->data = g_new0(GuppiData*, db->data_count);
  for (i=0; i<db->data_count; ++i) {
    db->data[i] = vec[i];

    gtk_signal_connect(GTK_OBJECT(db->data[i]), "changed_one",
		       GTK_SIGNAL_FUNC(guppi_data_browser_changed_one_cb),
		       db);
    gtk_signal_connect(GTK_OBJECT(db->data[i]), "changed_many",
		       GTK_SIGNAL_FUNC(guppi_data_browser_changed_many_cb),
		       db);
    gtk_signal_connect(GTK_OBJECT(db->data[i]), "changed_label",
		       GTK_SIGNAL_FUNC(guppi_data_browser_changed_label_cb),
		       db);
  /* This would be a good place to attach any other signals... */
  }

  gtk_signal_connect(GTK_OBJECT(db),
		     "select_row",
		     GTK_SIGNAL_FUNC(guppi_data_browser_select_row_cb),
		     NULL);

  
  gtk_clist_construct(GTK_CLIST(db), 1+db->data_count, NULL); 
  gtk_clist_set_selection_mode(GTK_CLIST(db), GTK_SELECTION_SINGLE);

   guppi_data_browser_build(db);

  return GTK_WIDGET(db);
}


/* $Id: guppi-data-browser.c,v 1.1.1.1 1999/12/03 07:02:40 trow Exp $ */
