/* $Id: guppi-canvas-item.c,v 1.5 2000/01/21 06:14:38 trow Exp $ */

/*
 * guppi-canvas-item.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and 
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "rgb.h"
#include "guppi-canvas-item.h"

enum {
  RESIZED,
  RESCALED,
  RESCALED_X,
  RESCALED_Y,
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_X1, ARG_Y1, ARG_X2, ARG_Y2,
  ARG_XBND, ARG_YBND
};

static GtkObjectClass* parent_class = NULL;
static guint gci_signals[LAST_SIGNAL] = { 0 };

typedef void (*GuppiSignal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE)(GtkObject*, double arg1, double arg2, double arg3, double arg4, gpointer user_data);
typedef void (*GuppiSignal_NONE__DOUBLE_DOUBLE)(GtkObject*, double arg1, double arg2, gpointer user_data);

static void
guppi_marshal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE(GtkObject* object
,						GtkSignalFunc func,
						gpointer func_data,
						GtkArg* args)
{
  GuppiSignal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE rfunc;
  rfunc = (GuppiSignal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE)func;
  (*rfunc)(object, 
	   GTK_VALUE_DOUBLE(args[0]),
	   GTK_VALUE_DOUBLE(args[1]),
	   GTK_VALUE_DOUBLE(args[2]),
	   GTK_VALUE_DOUBLE(args[3]),
	   func_data);
}

static void
guppi_marshal_NONE__DOUBLE_DOUBLE(GtkObject* object,
				  GtkSignalFunc func,
				  gpointer func_data,
				  GtkArg* args)
{
  GuppiSignal_NONE__DOUBLE_DOUBLE rfunc;
  rfunc = (GuppiSignal_NONE__DOUBLE_DOUBLE)func;
  (*rfunc)(object, 
	   GTK_VALUE_DOUBLE(args[0]),
	   GTK_VALUE_DOUBLE(args[1]),
	   func_data);
}

static void
guppi_canvas_item_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(obj);

  switch (arg_id) {
  case ARG_X1:
    GTK_VALUE_DOUBLE(*arg) = gci->wx0;
    break;
  case ARG_Y1:
    GTK_VALUE_DOUBLE(*arg) = gci->wy0;
    break;
  case ARG_X2:
    GTK_VALUE_DOUBLE(*arg) = gci->wx1;
    break;
  case ARG_Y2:
    GTK_VALUE_DOUBLE(*arg) = gci->wy1;
    break;
  default:
    break;
  }
}

static void
guppi_canvas_item_destroy(GtkObject* obj)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(obj);
  static void guppi_canvas_item_remove_state(GuppiCanvasItem*);
  
  guppi_canvas_item_remove_state(gci);
}

static void
guppi_canvas_item_finalize(GtkObject* obj)
{

}


static void
guppi_canvas_item_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(obj);
  gboolean did_something = FALSE;

  switch (arg_id) {
  case ARG_X1:
    gci->wx0 = GTK_VALUE_DOUBLE(*arg);
    break;
  case ARG_Y1:
    gci->wy0 = GTK_VALUE_DOUBLE(*arg);
    break;
  case ARG_X2:
    gci->wx1 = GTK_VALUE_DOUBLE(*arg);
    break;
  case ARG_Y2:
    gci->wy1 = GTK_VALUE_DOUBLE(*arg);
    break;
  case ARG_XBND:
    guppi_canvas_item_set_x_bounder(gci,
				    GUPPI_BOUNDER(GTK_VALUE_POINTER(*arg)));
    break;
  case ARG_YBND:
    guppi_canvas_item_set_y_bounder(gci,
				    GUPPI_BOUNDER(GTK_VALUE_POINTER(*arg)));
    break;
  default:
    did_something = TRUE;
    break;
  }

  if (did_something)
    guppi_canvas_item_request_update(gci);
    
}

static void
guppi_canvas_item_update(GnomeCanvasItem* item, double affine[6],
			 ArtSVP* clip_path, gint flags)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(item);

  /* Chain our calls to update */
  if (parent_class && GNOME_CANVAS_ITEM_CLASS(parent_class)->update)
    GNOME_CANVAS_ITEM_CLASS(parent_class)->update(item,affine,clip_path,flags);

  // Convert from world to canvas coordinates, and remember.
  gnome_canvas_w2c(item->canvas, gci->wx0, gci->wy0, &gci->cx0, &gci->cy0);
  gnome_canvas_w2c(item->canvas, gci->wx1, gci->wy1, &gci->cx1, &gci->cy1);
  
  transform_set_pixel_bounds(gci->x_transform, gci->cx0, gci->cx1);
  transform_set_pixel_bounds(gci->y_transform, gci->cy0, gci->cy1);

  gnome_canvas_update_bbox(item, gci->wx0, gci->wy0, gci->wx1, gci->wy1);
}

static void
guppi_canvas_item_render(GnomeCanvasItem* item,
			 GnomeCanvasBuf* buf)
{
  /* Chain our calls to render */
  if (parent_class && GNOME_CANVAS_ITEM_CLASS(parent_class)->render)
    GNOME_CANVAS_ITEM_CLASS(parent_class)->render(item, buf);

  if (buf->is_bg) {
    gnome_canvas_buf_ensure_buf(buf);
    buf->is_bg = FALSE;
  }
}

static double
guppi_canvas_item_point(GnomeCanvasItem* item,
			double x, double y, gint cx, gint cy,
			GnomeCanvasItem** actual_item)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(item);

  double dist;

  if (gci->wx0 <= x && x <= gci->wx1 && gci->wy0 <= y && y <= gci->wy1) 
    dist = 0;
  else
    dist = MIN(MIN(fabs(x-gci->wx0), fabs(x-gci->wx1)),
	       MIN(fabs(y-gci->wy0), fabs(y-gci->wy1)));

  *actual_item = item;
  return dist;
}


static void
guppi_canvas_item_class_init(GuppiCanvasItemClass* klass)
{
  GtkObjectClass* object_class;
  GnomeCanvasItemClass* item_class;

  object_class = (GtkObjectClass*)klass;
  item_class = GNOME_CANVAS_ITEM_CLASS(klass);
  parent_class = gtk_type_class(GNOME_TYPE_CANVAS_ITEM);

  gci_signals[RESIZED] =
    gtk_signal_new("resized",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiCanvasItemClass, resized),
		   guppi_marshal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE,
		   GTK_TYPE_NONE, 4,
		   GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE,
		   GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE);

  gci_signals[RESCALED] = 
    gtk_signal_new("rescaled",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiCanvasItemClass, rescaled),
		   guppi_marshal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE,
		   GTK_TYPE_NONE, 4,
		   GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE,
		   GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE);

  gci_signals[RESCALED_X] =
    gtk_signal_new("rescaled_x",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiCanvasItemClass, rescaled_x),
		   guppi_marshal_NONE__DOUBLE_DOUBLE,
		   GTK_TYPE_NONE, 2,
		   GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE);

  gci_signals[RESCALED_Y] =
    gtk_signal_new("rescaled_y",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiCanvasItemClass, rescaled_y),
		   guppi_marshal_NONE__DOUBLE_DOUBLE,
		   GTK_TYPE_NONE, 2,
		   GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE);
    

  klass->resized = NULL;
  klass->rescaled = NULL;
  klass->rescaled_x = NULL;
  klass->rescaled_y = NULL;

  item_class->update = guppi_canvas_item_update;
  item_class->render = guppi_canvas_item_render;
  item_class->point = guppi_canvas_item_point;

  object_class->get_arg = guppi_canvas_item_get_arg;
  object_class->set_arg = guppi_canvas_item_set_arg;
  object_class->destroy = guppi_canvas_item_destroy;
  object_class->finalize = guppi_canvas_item_finalize;


  gtk_object_add_arg_type("GuppiCanvasItem::x1",
			  GTK_TYPE_DOUBLE,
			  GTK_ARG_READWRITE,
			  ARG_X1);
  gtk_object_add_arg_type("GuppiCanvasItem::y1",
			  GTK_TYPE_DOUBLE,
			  GTK_ARG_READWRITE,
			  ARG_Y1);
  gtk_object_add_arg_type("GuppiCanvasItem::x2",
			  GTK_TYPE_DOUBLE,
			  GTK_ARG_READWRITE,
			  ARG_X2);
  gtk_object_add_arg_type("GuppiCanvasItem::y2",
			  GTK_TYPE_DOUBLE,
			  GTK_ARG_READWRITE,
			  ARG_Y2);
  gtk_object_add_arg_type("GuppiCanvasItem::x-bounder",
			  GTK_TYPE_POINTER,
			  GTK_ARG_READWRITE,
			  ARG_XBND);
  gtk_object_add_arg_type("GuppiCanvasItem::y-bounder",
			  GTK_TYPE_POINTER,
			  GTK_ARG_READWRITE,
			  ARG_YBND);
}

static void
guppi_canvas_item_init(GuppiCanvasItem* gci)
{
  gci->x_transform  = transform_new();
  gci->y_transform  = transform_new();

  gci->x_bounder = NULL;
  gci->y_bounder = NULL;

  gci->x0 = gci->y0 = 0;
  gci->x1 = gci->y1 = 1;

  gci->full_x0 = gci->full_y0 = 0;
  gci->full_x1 = gci->full_y1 = 1;

  transform_set_plot_bounds(gci->x_transform, gci->x0, gci->x1);
  transform_set_plot_bounds(gci->y_transform, gci->y0, gci->y1);

  /* Hard-wired position for testing */
  gci->wx0 = 0;
  gci->wx1 = 100;
  gci->wy0 = 0;
  gci->wy1 = 100;

  gci->expecting_double_bound_change = FALSE;
}

GtkType
guppi_canvas_item_get_type(void)
{
  static GtkType gci_type = 0;

  if (!gci_type) {
    static const GtkTypeInfo gci_info = {
      "GuppiCanvasItem",
      sizeof(GuppiCanvasItem),
      sizeof(GuppiCanvasItemClass),
      (GtkClassInitFunc)guppi_canvas_item_class_init,
      (GtkObjectInitFunc)guppi_canvas_item_init,
      NULL, NULL,
      (GtkClassInitFunc)NULL
    };
    gci_type = gtk_type_unique(GNOME_TYPE_CANVAS_ITEM, &gci_info);
  }

  return gci_type;
}

/*****************************************************************************/

static void
guppi_canvas_item_remove_state(GuppiCanvasItem* gci)
{
  g_return_if_fail(gci != NULL);

  if (gci->state == NULL)
    return;

  gtk_signal_disconnect_by_data(GTK_OBJECT(gci->state), gci);
  gtk_object_unref(GTK_OBJECT(gci->state));
  gci->state = NULL;
}

static void 
state_update_cb(GuppiItemState* gis, gpointer data)
{
  g_return_if_fail(gis != NULL);
  g_return_if_fail(data != NULL);
  guppi_canvas_item_request_update(GUPPI_CANVAS_ITEM(data));
}

void
guppi_canvas_item_set_state(GuppiCanvasItem* gci, GuppiItemState* state)
{
  g_return_if_fail(gci != NULL);

  if (state == gci->state)
    return;

  if (gci->state)
    guppi_canvas_item_remove_state(gci);

  gci->state = state;

  gtk_signal_connect(GTK_OBJECT(gci->state),
		     "update",
		     GTK_SIGNAL_FUNC(state_update_cb),
		     gci);
}

/*****************************************************************************/

static void
guppi_canvas_item_set_x_scale_lowlevel(GuppiCanvasItem* gci,
				       double x0, double x1,
				       gboolean redraw)
{
  if (gci->x0 == x0 && gci->x1 == x1)
    return;

  gci->x0 = x0;
  gci->x1 = x1;

  transform_set_plot_bounds(gci->x_transform, gci->x0, gci->x1);

  gtk_signal_emit(GTK_OBJECT(gci), gci_signals[RESCALED_X], x0, x1);
 
  if (redraw)
    guppi_canvas_item_request_update(gci);
}

static void
guppi_canvas_item_set_y_scale_lowlevel(GuppiCanvasItem* gci,
				       double y0, double y1,
				       gboolean redraw)
{
  if (gci->y0 == y0 && gci->y1 == y1)
    return;

  gci->y0 = y0;
  gci->y1 = y1;

  transform_set_plot_bounds(gci->y_transform, gci->y0, gci->y1);

  gtk_signal_emit(GTK_OBJECT(gci), gci_signals[RESCALED_Y], y0, y1);
 
  if (redraw)
    guppi_canvas_item_request_update(gci);
}


void
guppi_canvas_item_set_x_scale(GuppiCanvasItem* gci, double x0, double x1)
{
  double w;

  g_return_if_fail(gci);

  if (gci->x_bounder) {

    /* Pass our request onto the bounder */
    w = gci->full_x1 - gci->full_x0;
    guppi_bounder_set(gci->x_bounder,
		      (x0 - gci->full_x0) / w,
		      (x1 - gci->full_x0) / w);

  } else {

    guppi_canvas_item_set_x_scale_lowlevel(gci, x0, x1, TRUE);

  }
}

void
guppi_canvas_item_set_y_scale(GuppiCanvasItem* gci, double y0, double y1)
{
  double w;

  g_return_if_fail(gci);

  if (gci->y_bounder) {

    /* Pass our request onto the bounder */
    w = gci->full_y1 - gci->full_y0;
    guppi_bounder_set(gci->y_bounder,
		      (y0 - gci->full_y0) / w,
		      (y1 - gci->full_y0) / w);

  } else {

    guppi_canvas_item_set_y_scale_lowlevel(gci, y0, y1, TRUE);

  }


}

void
guppi_canvas_item_set_scales(GuppiCanvasItem* gci,
			     double x0, double y0,
			     double x1, double y1)
{
  g_return_if_fail(gci);

  /* Our goal to avoid unnecessary redraw operations. */

  if (gci->x0 == x0 && gci->x1 == x1 && gci->y0 == y0 && gci->y1 == y1)
    return;

  if (gci->x0 == x0 && gci->x1 == x1) {
    guppi_canvas_item_set_y_scale(gci, y0, y1);
    return;
  }

  if (gci->y0 == y0 && gci->y1 == y1) {
    guppi_canvas_item_set_x_scale(gci, x0, x1);
    return;
  }

  if (gci->x_bounder && gci->y_bounder) {

    gci->expecting_double_bound_change = TRUE;
    guppi_canvas_item_set_x_scale(gci, x0, x1);
    guppi_canvas_item_set_y_scale(gci, y0, y1);

  } else if (gci->x_bounder) {

    guppi_canvas_item_set_y_scale_lowlevel(gci, y0, y1, FALSE);
    guppi_canvas_item_set_x_scale(gci, x0, x1);

  } else if (gci->y_bounder) {

    guppi_canvas_item_set_x_scale_lowlevel(gci, x0, x1, FALSE);
    guppi_canvas_item_set_y_scale(gci, y0, y1);

  } else {

    guppi_canvas_item_set_x_scale_lowlevel(gci, x0, x1, FALSE);
    guppi_canvas_item_set_y_scale_lowlevel(gci, y0, y1, TRUE);

  }

  gtk_signal_emit(GTK_OBJECT(gci), gci_signals[RESCALED], x0, y0, x1, y1);
  guppi_canvas_item_request_update(gci);
}

/*****************************************************************************/

void
guppi_canvas_item_rescale_around_point(GuppiCanvasItem* gci,
				       double x, double y,
				       double factor)
{
  g_return_if_fail(gci != NULL);
  g_return_if_fail(factor > 0);

  guppi_canvas_item_set_scales(gci,
			       factor * (gci->x0 - x) + x,
			       factor * (gci->y0 - y) + y,
			       factor * (gci->x1 - x) + x,
			       factor * (gci->y1 - y) + y);
}

void
guppi_canvas_item_recenter_around_point(GuppiCanvasItem* gci,
					double x, double y)
{
  double xc, yc;
  g_return_if_fail(gci != NULL);

  xc = (gci->x0 + gci->x1) / 2;
  yc = (gci->y0 + gci->y1) / 2;

  guppi_canvas_item_set_scales(gci,
			       gci->x0 + x - xc, gci->y0 + y - yc,
			       gci->x1 + x - xc, gci->y1 + y - yc);
  
}

/*****************************************************************************/

/*

  We have to follow our calls to set_{x,y}_scale() by a
  set_{x,y}_scale_lowlevel() to force updating in certain situations.
  If they are redundant, they will just return without doing anything,
  so they are harmless... no extra redraws or anything should result.

  Here is the problem:

  Say we start out with bounds [0,10] and scale [0,10].
  Thus the bounder is at [0,1].

  We increase the bounds to [0,20], and we automatically call
  set_scale to adjust our scale to [0,20].  This forwards a request to
  the bounder, requesting that it set itself to [0,1].  Since this is
  the same as before, the bounder drops the request as redundant.
  Since no bounder-changed signal is ever emitted, our set_*_scale
  call never results in a set_*_scale_lowevel call, even though our
  scale has certainly changed.

  By explicitly forcing a call to lowlevel, we should (hopefully!)
  avoid these problems.

  */

void
guppi_canvas_item_set_x_bounds(GuppiCanvasItem* gci, double x0, double x1)
{
  g_return_if_fail(gci != NULL);
  gci->full_x0 = x0;
  gci->full_x1 = x1;
  guppi_canvas_item_set_x_scale(gci, x0, x1);
  guppi_canvas_item_set_x_scale_lowlevel(gci, x0, x1, TRUE);

   guppi_canvas_item_request_update(gci);
}

void
guppi_canvas_item_set_y_bounds(GuppiCanvasItem* gci, double y0, double y1)
{
  g_return_if_fail(gci != NULL);
  gci->full_y0 = y0;
  gci->full_y1 = y1;
  guppi_canvas_item_set_y_scale(gci, y0, y1);
  guppi_canvas_item_set_y_scale_lowlevel(gci, y0, y1, TRUE);

  guppi_canvas_item_request_update(gci);
}

void
guppi_canvas_item_set_bounds(GuppiCanvasItem* gci,
			     double x0, double y0,
			     double x1, double y1)
{
  g_return_if_fail(gci != NULL);

  gci->full_x0 = x0;
  gci->full_x1 = x1;
  gci->full_y0 = y0;
  gci->full_y1 = y1;

  guppi_canvas_item_set_scales(gci, x0, y0, x1, y1);
  guppi_canvas_item_set_x_scale_lowlevel(gci, x0, x1, FALSE);
  guppi_canvas_item_set_y_scale_lowlevel(gci, y0, y1, TRUE);

  guppi_canvas_item_request_update(gci);
}

/*****************************************************************************/

static void
gci_x_bounder_cb(GuppiBounder* gb, gpointer user_data)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(user_data);
  double nx0, nx1;
  double w;

  g_return_if_fail(gb != NULL);
  g_return_if_fail(gci != NULL);

  w = gci->full_x1 - gci->full_x0;
  nx0 = gci->full_x0 + w * gb->t0;
  nx1 = gci->full_x0 + w * gb->t1;

  guppi_canvas_item_set_x_scale_lowlevel(gci, nx0, nx1,
					 !gci->expecting_double_bound_change);

  gci->expecting_double_bound_change = FALSE;
}

static void
gci_y_bounder_cb(GuppiBounder* gb, gpointer user_data)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(user_data);
  double ny0, ny1;
  double w;

  g_return_if_fail(gb != NULL);
  g_return_if_fail(gci != NULL);

  w = gci->full_y1 - gci->full_y0;
  ny0 = gci->full_y0 + w * gb->t0;
  ny1 = gci->full_y0 + w * gb->t1;

  guppi_canvas_item_set_y_scale_lowlevel(gci, ny0, ny1,
					 !gci->expecting_double_bound_change);

  gci->expecting_double_bound_change = FALSE;
}

void
guppi_canvas_item_set_x_bounder(GuppiCanvasItem* gci, GuppiBounder* gb)
{
  g_return_if_fail(gci != NULL);

  if (gci->x_bounder == gb)
    return;

  if (gci->x_bounder) {
    gtk_object_unref(GTK_OBJECT(gci->x_bounder));
    gtk_signal_disconnect(GTK_OBJECT(gci->x_bounder), gci->x_bnd_sh);
  }

  gci->x_bounder = gb;
    
  if (gb) {
    gtk_object_ref(GTK_OBJECT(gci->x_bounder));
    gci->x_bnd_sh = gtk_signal_connect(GTK_OBJECT(gci->x_bounder),
				       "changed",
				       GTK_SIGNAL_FUNC(gci_x_bounder_cb),
				       (gpointer)gci);
  }

  guppi_canvas_item_request_update(gci);

}

void
guppi_canvas_item_set_y_bounder(GuppiCanvasItem* gci, GuppiBounder* gb)
{
  g_return_if_fail(gci != NULL);

  if (gci->y_bounder == gb)
    return;

  if (gci->y_bounder) {
    gtk_object_unref(GTK_OBJECT(gci->y_bounder));
    gtk_signal_disconnect(GTK_OBJECT(gci->y_bounder), gci->y_bnd_sh);
  }

  gci->y_bounder = gb;
    
  if (gb) {
    gtk_object_ref(GTK_OBJECT(gci->y_bounder));
    gci->y_bnd_sh = gtk_signal_connect(GTK_OBJECT(gci->y_bounder),
				       "changed",
				       GTK_SIGNAL_FUNC(gci_y_bounder_cb),
				       (gpointer)gci);
  }

  guppi_canvas_item_request_update(gci);

}

/*****************************************************************************/

double
guppi_canvas_item_x_magnification(const GuppiCanvasItem* gci)
{
  g_return_val_if_fail(gci != NULL, 0);

  return fabs( (gci->full_x1 - gci->full_x0) / (gci->x1 - gci->x0) );
}

double
guppi_canvas_item_y_magnification(const GuppiCanvasItem* gci)
{
  g_return_val_if_fail(gci != NULL, 0);

  return fabs( (gci->full_y1 - gci->full_y0) / (gci->y1 - gci->y0) );
}

/*****************************************************************************/

void
guppi_canvas_item_plot2pixel(const GuppiCanvasItem* gci,
			     double x, double y,
			     gint* cx, gint* cy)
{
  g_return_if_fail(gci != NULL);

  *cx = transform_plot2pixel(gci->x_transform, x);
  *cy = transform_plot2pixel(gci->y_transform, y);
}

void
guppi_canvas_item_pixel2plot(const GuppiCanvasItem* gci,
			     gint cx, gint cy,
			     double* x, double* y)
{
  g_return_if_fail(gci != NULL);
  
  *x = transform_pixel2plot(gci->x_transform, cx);
  *y = transform_pixel2plot(gci->y_transform, cy);
}

void
guppi_canvas_item_world2plot(const GuppiCanvasItem* gci,
			     double wx, double wy,
			     double* x, double* y)
{
  gint cx, cy;

  g_return_if_fail(gci != NULL);


  gnome_canvas_w2c(GNOME_CANVAS_ITEM(gci)->canvas, wx, wy, &cx, &cy);
  guppi_canvas_item_pixel2plot(gci, cx, cy, x, y);
}

void
guppi_canvas_item_plot2world(const GuppiCanvasItem* gci,
			     double x, double y,
			     double* wx, double* wy)
{
  gint cx, cy;

  g_return_if_fail(gci != NULL);

  guppi_canvas_item_plot2pixel(gci, x, y, &cx, &cy);
  gnome_canvas_c2w(GNOME_CANVAS_ITEM(gci)->canvas, cy, cy, wx, wy);
}

void
guppi_canvas_item_request_redraw(GuppiCanvasItem* gci,
				 double x0, double y0,
				 double x1, double y1)
{
  gint cx0,cy0,cx1,cy1,t;

  g_return_if_fail(gci != NULL);

  guppi_canvas_item_plot2pixel(gci, x0, y0, &cx0, &cy0);
  guppi_canvas_item_plot2pixel(gci, x1, y1, &cx1, &cy1);

  if (cx0 > cx1) { t = cx0; cx0 = cx1; cx1 = t; }
  if (cy0 > cy1) { t = cy0; cy0 = cy1; cy1 = t; }

  gnome_canvas_request_redraw(GNOME_CANVAS_ITEM(gci)->canvas,
			      cx0-1, cy0-1,
			      cx1+1, cy1+1);
}

void
guppi_canvas_item_request_total_redraw(GuppiCanvasItem* gci)
{
  g_return_if_fail(gci != NULL);

  gnome_canvas_request_redraw(GNOME_CANVAS_ITEM(gci)->canvas,
			      gci->cx0, gci->cy0,
			      gci->cx1, gci->cy1);
}

void
guppi_canvas_item_request_update(GuppiCanvasItem* gci)
{
  g_return_if_fail(gci != NULL);
  gnome_canvas_item_request_update(GNOME_CANVAS_ITEM(gci));
}

/* $Id: guppi-canvas-item.c,v 1.5 2000/01/21 06:14:38 trow Exp $ */


