/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <panel-applet.h>
#include <gdk/gdkscreen.h>
#include <gdk/gdkx.h>
#include <gnome.h>

#include <libxklavier/xklavier_config.h>

#include "gswitchit_applet.h"

#include "../common/switchcuts.h"

#if 1
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#endif

#define GROUPS_SUBMENU_PATH "/popups/popup/groups"

static void GSwitchItAppletCmdProps( BonoboUIComponent * uic,
                                     GSwitchItApplet * sia,
                                     const gchar * verb );

static void GSwitchItAppletCmdAbout( BonoboUIComponent * uic,
                                     GSwitchItApplet * sia,
                                     const gchar * verb );

static void GSwitchItAppletCmdHelp( BonoboUIComponent * uic,
                                    GSwitchItApplet * sia,
                                    const gchar * verb );

static void GSwitchItAppletCmdSetGroup( BonoboUIComponent * uic,
                                        GSwitchItApplet * sia,
                                        const gchar * verb );

static void GSwitchItAppletCleanupGroupsSubmenu( GSwitchItApplet * sia );

static void GSwitchItAppletSetupGroupsSubmenu( GSwitchItApplet * sia );

static const BonoboUIVerb gswitchitAppletMenuVerbs[] = {
  BONOBO_UI_UNSAFE_VERB( "Props", GSwitchItAppletCmdProps ),
  BONOBO_UI_UNSAFE_VERB( "About", GSwitchItAppletCmdAbout ),
  BONOBO_UI_UNSAFE_VERB( "Help", GSwitchItAppletCmdHelp ),
  BONOBO_UI_VERB_END
};

static void GSwitchItAppletSetTooltip( GSwitchItApplet * sia,
                                       const char *str )
{
  GtkTooltips *tooltips;

  if( str == NULL )
    return;
  tooltips = gtk_tooltips_new(  );
  g_object_ref( G_OBJECT( tooltips ) );
  gtk_object_sink( GTK_OBJECT( tooltips ) );
  g_object_set_data_full( G_OBJECT( sia->applet ), "tooltips", tooltips,
                          ( GDestroyNotify ) g_object_unref );
  gtk_tooltips_set_tip( tooltips, sia->applet, str, NULL );
}

extern void GSwitchItAppletConfigChanged( GConfClient * client,
                                          guint cnxn_id,
                                          GConfEntry * entry,
                                          GSwitchItApplet * sia )
{
  XklDebug( 100, "Applet configuration changed in GConf - reiniting...\n" );

  XklPauseListen(  );

  GSwitchItConfigLoadAppletParams( &( sia->config ) );
  GSwitchItConfigActivateAppletParams( &( sia->config ) );

  GSwitchItAppletCleanupGroupsSubmenu( sia );
  GSwitchItAppletSetupGroupsSubmenu( sia );

  GSwitchItAppletRevalidate( sia );

  XklResumeListen(  );
}

static void GSwitchItAppletStateCallback( XklStateChange changeType,
                                          int group,
                                          Bool restore,
                                          GSwitchItApplet * sia )
{
  XklDebug( 150, "group is now %d, restore: %d\n", group, restore );

  if( !restore )
  {
    if( sia->config.doBeep )
      gdk_beep(  );
  }

  if( changeType == GROUP_CHANGED )
  {
    if( sia->config.commands[group] != NULL &&
        sia->config.commands[group][0] != '\0' )
    {
      XklDebug( 200, "do exec\n" );
      gnome_execute_shell( NULL, sia->config.commands[group] );
    }
    XklDebug( 200, "do repaint\n" );
    GSwitchItAppletRevalidateGroup( sia, group );
  }
}

GdkFilterReturn GSwitchItAppletFilterXEvt( GdkXEvent * xevent,
                                           GdkEvent * event,
                                           GSwitchItApplet * sia )
{
  int ignoredByXkl = XklFilterEvents( ( XEvent * ) xevent );
  return ignoredByXkl ? GDK_FILTER_CONTINUE : GDK_FILTER_REMOVE;
}

GdkFilterReturn GSwitchItAppletWmProtocolsFilter( GdkXEvent * xev,
                                                  GdkEvent * event,
                                                  gpointer data )
{
  XEvent *xevent = ( XEvent * ) xev;
  GdkWindow *win = event->any.window;
  GdkDisplay *display =
    gdk_x11_lookup_xdisplay( gdk_x11_drawable_get_xdisplay
                             ( GDK_DRAWABLE( win ) ) );

  if( ( Atom ) xevent->xclient.data.l[0] ==
      gdk_x11_get_xatom_by_name_for_display( display, "_NET_WM_PING" ) )
  {
    static long lastTimestamp = -1;
    long thisTimestamp = xevent->xclient.data.l[1];
    // allow only events we have not processed yet
    if( lastTimestamp == thisTimestamp )
      return GDK_FILTER_REMOVE;
    lastTimestamp = thisTimestamp;
  }

  return GDK_FILTER_CONTINUE;
}

void GSwitchItAppletRevalidate( GSwitchItApplet * sia )
{
  XklState *currentState;

  currentState = XklGetCurrentState(  );
  if( currentState->group >= 0 )
    GSwitchItAppletRevalidateGroup( sia, currentState->group );
}

static void GSwitchItAppletDropDrawings( GSwitchItApplet * sia )
{
  GtkWidget **da = sia->groupDrawingAreas;
  int i;
  for( i = XkbNumKbdGroups; --i >= 0; da++ )
  {
    if( *da == NULL )
      continue;
    g_object_unref( G_OBJECT( *da ) );
    *da = NULL;
  }
}

static void GSwitchItAppletPrepareDrawing( GSwitchItApplet * sia, int group )
{
  GdkPixbuf *image = sia->config.images[group];
  GdkPixbuf *scaled;
  PanelAppletOrient orient;
  int psize, xsize = 0, ysize = 0;
  double xyratio;

  if( image == NULL )
    return;

  orient = panel_applet_get_orient( PANEL_APPLET( sia->applet ) );
  psize = panel_applet_get_size( PANEL_APPLET( sia->applet ) ) - 4;
  XklDebug( 150, "Resizing the applet for size %d\n", psize );
  xyratio =
    1.0 * gdk_pixbuf_get_width( image ) / gdk_pixbuf_get_height( image );
  switch ( orient )
  {
    case PANEL_APPLET_ORIENT_UP:
    case PANEL_APPLET_ORIENT_DOWN:
      ysize = psize;
      xsize = psize * xyratio;
      break;
    case PANEL_APPLET_ORIENT_LEFT:
    case PANEL_APPLET_ORIENT_RIGHT:
      ysize = psize / xyratio;
      break;
  }

  scaled = gdk_pixbuf_scale_simple( image, xsize, ysize, GDK_INTERP_HYPER );
  sia->groupDrawingAreas[group] = gtk_image_new_from_pixbuf( scaled );
  g_object_unref( G_OBJECT( scaled ) );
  g_object_ref( G_OBJECT( sia->groupDrawingAreas[group] ) );
}

void GSwitchItAppletRevalidateGroup( GSwitchItApplet * sia, int group )
{
  const char *pname;
  GtkWidget **da;
  XklDebug( 200, "Revalidating for group %d\n", group );

  da = sia->groupDrawingAreas + group;
  if( *da == NULL )
  {
    XklDebug( 150, "Image for the group %d is not ready, let's prepare it!\n",
              group );
    GSwitchItAppletPrepareDrawing( sia, group );
  }

  if( *da != NULL )
  {
    GtkWidget *oc = gtk_bin_get_child( GTK_BIN( sia->border ) );
    gtk_widget_hide( oc );
    gtk_container_remove( GTK_CONTAINER( sia->border ), oc );
    gtk_container_add( GTK_CONTAINER( sia->border ), *da );
    gtk_widget_show( *da );
  }

  pname = sia->groupNames[group];
  GSwitchItAppletSetTooltip( sia, pname );
}

static void GSwitchItAppletRealize( GtkWidget * widget,
                                    GSwitchItApplet * sia )
{
  GSwitchItAppletRevalidate( sia );
}

static void GSwitchItAppletUnrealize( GtkWidget * widget,
                                      GSwitchItApplet * sia )
{
}

static void GSwitchItAppletChangePixelSize( PanelApplet * widget,
                                            guint size,
                                            GSwitchItApplet * sia )
{
  GSwitchItAppletDropDrawings( sia );
  GSwitchItAppletRevalidate( sia );
}

static gint GSwitchItAppletButtonPressed( GtkWidget * widget,
                                          GdkEventButton * event,
                                          GSwitchItConfig * sic )
{
  if( event->button == 1 && event->type == GDK_BUTTON_PRESS )
  {
    GSwitchItConfigLockNextGroup( sic );
    return GDK_FILTER_REMOVE;
  }
  return GDK_FILTER_CONTINUE;
}

void GSwitchItAppletCmdProps( BonoboUIComponent * uic,
                              GSwitchItApplet * sia, const gchar * verb )
{
  gnome_execute_shell( NULL, "gswitchit-properties-capplet" );
}

static void GSwitchItAppletCmdSetGroup( BonoboUIComponent * uic,
                                        GSwitchItApplet * sia,
                                        const gchar * verb )
{
  XklState st;
  Window cur;

  if( sscanf( verb, "Group_%d", &st.group ) != 1 )
  {
    XklDebug( 50, "Strange verb: [%s]\n", verb );
    return;
  }

  XklAllowOneSwitchToSecondaryGroup(  );
  cur = XklGetCurrentWindow(  );
  if( cur != ( Window ) NULL )
  {
    XklDebug( 150, "Enforcing the state %d for window %ld\n", st.group, cur );
    XklSaveState( XklGetCurrentWindow(  ), &st );
//    XSetInputFocus( GDK_DISPLAY(), cur, RevertToNone, CurrentTime );
  } else
  {
    XklDebug( 150, "??? Enforcing the state %d for unknown window\n",
              st.group );
    // strange situation - bad can happen
  }
  XklLockGroup( st.group );
}

void GSwitchItAppletCmdHelp( BonoboUIComponent * uic,
                             GSwitchItApplet * sia, const gchar * verb )
{
  GSwitchItHelp( NULL, "gswitchitApplet" );
}

static void GSwitchItAppletDestroyAbout( GtkWidget * w,
                                         GSwitchItApplet * sia )
{
  sia->aboutBox = NULL;
}

void GSwitchItAppletCmdAbout( BonoboUIComponent * uic,
                              GSwitchItApplet * sia, const gchar * verb )
{
  GtkWidget *href;
  const gchar *authors[] =
    { "Sergey V. Udaltsov<svu@users.sourceforge.net>", NULL };
  const gchar *documenters[] =
    { "Sergey V. Udaltsov<svu@users.sourceforge.net>", NULL };
  gchar *translatorCredits;
  GdkPixbuf *pixbuf = NULL;
  gchar *file;

  if( sia->aboutBox )
  {
    gtk_widget_show_now( sia->aboutBox );
    gdk_window_raise( sia->aboutBox->window );
    return;
  }

  file =
    gnome_program_locate_file( NULL, GNOME_FILE_DOMAIN_PIXMAP,
                               "gswitchit.png", TRUE, NULL );
  pixbuf = gdk_pixbuf_new_from_file( file, NULL );
  g_free( file );

  translatorCredits = _( "translator_credits" );
  translatorCredits =
    strcmp( translatorCredits,
            "translator_credits" ) != 0 ? translatorCredits : NULL;

  sia->aboutBox =
    gnome_about_new( _( PACKAGE ), VERSION,
                     _( "Copyright Sergey V. Udaltsov (C) 1999-2002" ),
                     _( "XKB toolkit for GNOME" ), authors, documenters,
                     translatorCredits, pixbuf );
  g_signal_connect( G_OBJECT( sia->aboutBox ), "destroy",
                    G_CALLBACK( GSwitchItAppletDestroyAbout ), sia );
  href =
    gnome_href_new( "http://gswitchit.sourceforge.net/",
                    _( "GSwitchIt web page @ Sourceforge.net" ) );
  gtk_box_pack_start( GTK_BOX( GTK_DIALOG( sia->aboutBox )->vbox ), href,
                      FALSE, FALSE, 0 );

  gtk_widget_show( sia->aboutBox );
}

static void GSwitchItAppletCleanupGroupsSubmenu( GSwitchItApplet * sia )
{
  int i;
  BonoboUIComponent *popup;

  popup = panel_applet_get_popup_component( PANEL_APPLET( sia->applet ) );

  for( i = XkbNumKbdGroups; --i >= 0; )
  {
    char path[80];
    g_snprintf( path, sizeof( path ), GROUPS_SUBMENU_PATH "/Group_%d", i );
    if( bonobo_ui_component_path_exists( popup, path, NULL ) )
    {
      bonobo_ui_component_rm( popup, path, NULL );
      XklDebug( 160, "Unregistered group menu item \'%s\'\n", path );
    }
  }

  XklDebug( 160, "Unregistered group submenu\n" );
}

static void GSwitchItAppletSetupGroupsSubmenu( GSwitchItApplet * sia )
{
  unsigned i, nGroups;
  char *pname;
  BonoboUIComponent *popup;

  popup = panel_applet_get_popup_component( PANEL_APPLET( sia->applet ) );

  XklDebug( 160, "Registered group submenu\n" );
  nGroups = XklGetNumGroups(  );
  pname = ( char * ) sia->groupNames;
  for( i = 0; i < nGroups; i++ )
  {
    char verb[40];
    BonoboUINode *node;
    g_snprintf( verb, sizeof( verb ), "Group_%d", i );

    node = bonobo_ui_node_new( "menuitem" );
    bonobo_ui_node_set_attr( node, "name", verb );
    bonobo_ui_node_set_attr( node, "verb", verb );
    bonobo_ui_node_set_attr( node, "label", pname );
    bonobo_ui_node_set_attr( node, "pixtype", "filename" );
    bonobo_ui_node_set_attr( node, "pixname", sia->config.imageFiles[i] );

    bonobo_ui_component_set_tree( popup, GROUPS_SUBMENU_PATH, node, NULL );

    bonobo_ui_component_add_verb( popup,
                                  verb,
                                  ( BonoboUIVerbFn )
                                  GSwitchItAppletCmdSetGroup, sia );

    XklDebug( 160, "Registered group menu item \'%s\' as \'%s\'\n", verb,
              pname );
    pname += sizeof( sia->groupNames[0] );
  }
}

static void GSwitchItAppletXkbConfigCallback( GSwitchItApplet * sia )
{
  const char *pname;
  XklState *currentState;

  XklDebug( 100, "XKB configuration changed on X Server - reiniting...\n" );

  GSwitchItConfigLoadGroupDescriptions( &( sia->config ), sia->groupNames );

  GSwitchItAppletCleanupGroupsSubmenu( sia );
  GSwitchItAppletSetupGroupsSubmenu( sia );

  // also, update tooltips
  currentState = XklGetCurrentState(  );
  if( currentState->group >= 0 )
  {
    pname = sia->groupNames[currentState->group];
    GSwitchItAppletSetTooltip( sia, pname );
  }
}

static void GSwitchItAppletWindowCallback( Window win, Window parent,
                                           GSwitchItApplet * sia )
{
  int dl = sia->config.defaultGroup;
  XklDebug( 150, "New window %ld created\n", win );
  if( dl != -1 )
  {
    XklDebug( 150, "Have to setup the layout %d\n", dl );
    XklAllowOneSwitchToSecondaryGroup(  );
    XklLockGroup( dl );
  }
}

static void GSwitchItAppletSetupMenu( GSwitchItApplet * sia )
{
  panel_applet_setup_menu_from_file( PANEL_APPLET( sia->applet ), NULL,
                                     "GNOME_GSwitchItApplet.xml",
                                     NULL, gswitchitAppletMenuVerbs, sia );
  GSwitchItAppletSetupGroupsSubmenu( sia );
}

static void GSwitchItAppletStartListen( GSwitchItApplet * sia )
{
  gdk_window_add_filter( NULL,
                         ( GdkFilterFunc ) GSwitchItAppletFilterXEvt, sia );
  gdk_window_add_filter( gdk_get_default_root_window(  ),
                         ( GdkFilterFunc ) GSwitchItAppletFilterXEvt, sia );
// WM_PROTOCOLS are handled in a special way - 
// otherwize gtk makes us crazy
  gdk_add_client_message_filter( gdk_atom_intern
                                 ( "WM_PROTOCOLS", FALSE ),
                                 GSwitchItAppletWmProtocolsFilter, NULL );

  XklStartListen(  );
}

static void GSwitchItAppletStopListen( GSwitchItApplet * sia )
{
  XklStopListen(  );

//!! no client message filter removal in gnome 2.2

  gdk_window_remove_filter( NULL,
                            ( GdkFilterFunc ) GSwitchItAppletFilterXEvt,
                            sia );
  gdk_window_remove_filter( gdk_get_default_root_window(  ),
                            ( GdkFilterFunc ) GSwitchItAppletFilterXEvt,
                            sia );
}

static void GSwitchItAppletTerm( PanelApplet * applet,
                                 GSwitchItApplet * sia );

static gboolean GSwitchItAppletInit( GSwitchItApplet * sia,
                                     PanelApplet * applet )
{
  GError *err = NULL;

  sia->applet = GTK_WIDGET( applet );

  sia->border = gtk_frame_new( NULL );
  gtk_frame_set_shadow_type( GTK_FRAME( sia->border ), GTK_SHADOW_IN );

  sia->defDrawingArea =
    gtk_image_new_from_stock( GTK_STOCK_STOP, GTK_ICON_SIZE_BUTTON );
  g_object_ref( G_OBJECT( sia->defDrawingArea ) );

  gtk_container_add( GTK_CONTAINER( sia->border ), sia->defDrawingArea );
  gtk_container_add( GTK_CONTAINER( sia->applet ), sia->border );

  GSwitchItAppletSetTooltip( sia, _( PACKAGE ) );

  gtk_widget_show_all( sia->applet );
  gtk_widget_realize( sia->applet );

  //GSwitchItInstallGlibLogAppender(  );

  if( XklInit( GDK_DISPLAY(  ) ) != 0 )
  {
    GSwitchItAppletSetTooltip( sia, _( "XKB initialization error" ) );
    return True;
  }

  XklConfigInit(  );
  if( !XklConfigLoadRegistry(  ) )
  {
    GSwitchItAppletSetTooltip( sia,
                               _
                               ( "Error loading XKB configuration registry" ) );
    return True;
  }

  XklRegisterStateCallback( ( XklStateCallback )
                            GSwitchItAppletStateCallback, ( void * ) sia );
  XklRegisterConfigCallback( ( XklConfigCallback )
                             GSwitchItAppletXkbConfigCallback,
                             ( void * ) sia );
  XklRegisterWindowCallback( ( XklWinCallback )
                             GSwitchItAppletWindowCallback, ( void * ) sia );

  GSwitchItConfigInit( &( sia->config ) );

  GSwitchItConfigLoadAppletParams( &( sia->config ) );
  GSwitchItConfigActivateAppletParams( &( sia->config ) );

  GSwitchItConfigStartListenSettings( &( sia->config ),
                                      ( GConfClientNotifyFunc )
                                      GSwitchItAppletConfigChanged, sia );

  GSwitchItConfigLoadGroupDescriptions( &( sia->config ), sia->groupNames );

  GSwitchItAppletRealize( sia->applet, sia );

  g_signal_connect( G_OBJECT( sia->applet ),
                    "change_size",
                    G_CALLBACK( GSwitchItAppletChangePixelSize ), sia );

  GSwitchItAppletStartListen( sia );

  gtk_widget_add_events( sia->applet, GDK_BUTTON_PRESS_MASK );

  g_signal_connect( G_OBJECT( sia->applet ),
                    "button_press_event",
                    G_CALLBACK( GSwitchItAppletButtonPressed ), sia );

  g_signal_connect( GTK_OBJECT( sia->applet ), "destroy",
                    G_CALLBACK( GSwitchItAppletTerm ), sia );

  gtk_object_set_data( GTK_OBJECT( sia->applet ), "sia", sia );

  GSwitchItAppletSetupMenu( sia );
}

void GSwitchItAppletTerm( PanelApplet * applet, GSwitchItApplet * sia )
{
  GSwitchItAppletStopListen( sia );

  GSwitchItConfigStopListenSettings( &( sia->config ) );

  XklRegisterStateCallback( NULL, NULL );
  XklRegisterConfigCallback( NULL, NULL );
  XklRegisterWindowCallback( NULL, NULL );
  XklConfigFreeRegistry(  );
  XklConfigTerm(  );
  XklTerm(  );

  GSwitchItConfigTerm( &( sia->config ) );

  GSwitchItAppletDropDrawings( sia );
  g_object_unref( G_OBJECT( sia->defDrawingArea ) );

  XklDebug( 100, "The applet successfully terminated\n" );
}

static gboolean GSwitchItAppletNew( PanelApplet * applet )
{
  GSwitchItApplet *sia;
  gboolean rv;
#if 0
  GLogLevelFlags fatal_mask;
  fatal_mask = G_LOG_LEVEL_WARNING | G_LOG_LEVEL_CRITICAL;
  g_log_set_always_fatal( fatal_mask );
#endif
  sia = g_new0( GSwitchItApplet, 1 );
  rv = GSwitchItAppletInit( sia, applet );

  XklDebug( 100, "The applet successfully started: %d\n", rv );

  return rv;
}

static gboolean GSwitchItAppletFactory( PanelApplet * applet,
                                        const gchar * iid, gpointer data )
{
#if 0
//shit...
//#define O_FLAGS ( O_CREAT | O_WRONLY | O_TRUNC | O_SYNC,)
#define O_FLAGS ( O_CREAT | O_WRONLY | O_TRUNC )
  int stdout2 = open( "/tmp/gswout",
                      O_FLAGS,
                      0666 );
  int stderr2 = open( "/tmp/gswerr",
                      O_FLAGS,
                      0666 );
  dup2( stdout2, 1 );
  dup2( stderr2, 2 );
  setlinebuf( stdout );
  setlinebuf( stderr );
#endif

  if( !strcmp( iid, "OAFIID:GNOME_GSwitchItApplet" ) )
    return GSwitchItAppletNew( applet );

  return TRUE;
}

PANEL_APPLET_BONOBO_FACTORY( "OAFIID:GNOME_GSwitchItApplet_Factory", PANEL_TYPE_APPLET, "command-line", "0", GSwitchItAppletFactory, NULL )     //data
