/* spconf.c
 *
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include "SRMessages.h"
#include "spconf.h"
#include "spui.h"
#include "libsrconf.h"
#include "gnopiconf.h"

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include "srintl.h"

void	spconf_changes_cb	(GConfClient	*client,
				guint		cnxn_id,
				GConfEntry	*entry,
				gpointer	user_data);

extern Speech 	*speech_setting;
static guint sp_notify_id;
SpeakerSettings speakers[]=
{
    {"accelerator",	DEFAULT_SPEECH_ENGINE_VOICE,	DEFAULT_SPEECH_VOLUME,	DEFAULT_SPEECH_RATE,	DEFAULT_SPEECH_PITCH},
    {"childcount",	DEFAULT_SPEECH_ENGINE_VOICE,	DEFAULT_SPEECH_VOLUME,	DEFAULT_SPEECH_RATE,	DEFAULT_SPEECH_PITCH},
    {"location",	DEFAULT_SPEECH_ENGINE_VOICE,	DEFAULT_SPEECH_VOLUME,	DEFAULT_SPEECH_RATE,	DEFAULT_SPEECH_PITCH},
    {"message",		DEFAULT_SPEECH_ENGINE_VOICE,	DEFAULT_SPEECH_VOLUME,	DEFAULT_SPEECH_RATE,	DEFAULT_SPEECH_PITCH},
    {"name",		DEFAULT_SPEECH_ENGINE_VOICE,	DEFAULT_SPEECH_VOLUME,	DEFAULT_SPEECH_RATE,	DEFAULT_SPEECH_PITCH},
    {"role",		DEFAULT_SPEECH_ENGINE_VOICE,	DEFAULT_SPEECH_VOLUME,	DEFAULT_SPEECH_RATE,	DEFAULT_SPEECH_PITCH},
    {"shortcut",	DEFAULT_SPEECH_ENGINE_VOICE,	DEFAULT_SPEECH_VOLUME,	DEFAULT_SPEECH_RATE,	DEFAULT_SPEECH_PITCH},
    {"state",		DEFAULT_SPEECH_ENGINE_VOICE,	DEFAULT_SPEECH_VOLUME,	DEFAULT_SPEECH_RATE,	DEFAULT_SPEECH_PITCH},
    {"system",		DEFAULT_SPEECH_ENGINE_VOICE,	DEFAULT_SPEECH_VOLUME,	DEFAULT_SPEECH_RATE,	DEFAULT_SPEECH_PITCH},
    {"text",		DEFAULT_SPEECH_ENGINE_VOICE,	DEFAULT_SPEECH_VOLUME,	DEFAULT_SPEECH_RATE,	DEFAULT_SPEECH_PITCH},
    {"value",		DEFAULT_SPEECH_ENGINE_VOICE,	DEFAULT_SPEECH_VOLUME,	DEFAULT_SPEECH_RATE,	DEFAULT_SPEECH_PITCH}
};

extern GConfClient *gnopernicus_client;

GnopernicusSpeakerListType 	*gnopernicus_speakers = NULL;
SpeakerSettingsListType  	*speakers_settings  = NULL;
DictionaryListType		*dictionary_list = NULL;
/**
 *
 * Create a new gconf client for Speech settings
 * <return> - gconf client
 *
**/
gboolean
spconf_gconf_client_init (void)
{
    GError *error = NULL;
    	 
    sru_return_val_if_fail (gnopiconf_client_add_dir (CONFIG_PATH SPEECH_PATH), FALSE);

    gconf_client_notify_add (gnopernicus_client,
			    CONFIG_PATH SPEECH_PARAMETER_SECTION,
			    spconf_changes_cb,
			    NULL,NULL,&error);
    if (error != NULL)
    {
        sru_warning (_("spconf:Failed to add notify:\n"));
        sru_warning (_("Recomanded to delete ~/.gconf and ~/.gconfd directories"));
        g_error_free (error);
        error = NULL;
        return FALSE;
    }
    
    return TRUE;
}

/**
 *
 * Load value for Speech settings
 *
**/
Speech* 
spconf_setting_init (gboolean set_struct)
{
    Speech *value = NULL;
    
    value = spconf_setting_new ();

    sru_assert (value);
    
    value->count_type       = spconf_count_get ();
    value->punctuation_type = spconf_punctuation_get ();
    value->text_echo_type   = spconf_text_echo_get ();
    value->modifiers_type   = spconf_modifiers_get ();
    value->cursors_type     = spconf_cursors_get ();
    value->spaces_type      = spconf_spaces_get ();
    value->dictionary	    = spconf_dictionary_get ();
    
    spconf_set_default_voices (FALSE); /* default for voices */
    
    dictionary_list =
	spconf_dictionary_free (dictionary_list);
    
    dictionary_list =
	    spconf_dictionary_load ();

    if (!set_struct)
	return value;

    spconf_setting_free (speech_setting);
    speech_setting = value;

    return value;
}

/**
 *
 * Create a new Speech structure				
 * <return> - new Speech structure
 *
**/
Speech* 
spconf_setting_new (void)
{ 
    Speech *new_speech = NULL ; 
    
    new_speech = ( Speech * ) g_new0 (Speech, 1 ) ; 
    
    if (!new_speech) 
	sru_error (_("Unable to allocate memory"));
    
    return new_speech ; 
} 

/**
 *
 * Load default value for Speech structure					
 *
**/
void 
spconf_load_default_settings (Speech* speech)
{   
    GConfValue *value = NULL;
    
    sru_return_if_fail (speech);
    
    value = gconf_client_get_default_from_schema (gnopernicus_client, 
						CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_COUNT, 
						NULL);
    g_free (speech->count_type);
    if (value)
    {
	speech->count_type = g_strdup ((gchar*)gconf_value_get_string (value));
	gconf_value_free (value);
	value = NULL;
    }
    else
	speech->count_type  = g_strdup (DEFAULT_SPEECH_COUNT_TYPE);
		
    value = gconf_client_get_default_from_schema (gnopernicus_client, 
						CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_PUNCTUATION, 
						NULL);
    g_free (speech->punctuation_type);
    if (value)
    {
	speech->punctuation_type = g_strdup ((gchar*)gconf_value_get_string (value));
	gconf_value_free (value);
	value = NULL;
    }
    else
	speech->punctuation_type  = g_strdup (DEFAULT_SPEECH_PUNCTUATION);

    value = gconf_client_get_default_from_schema (gnopernicus_client, 
						CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_TEXT_ECHO,
						NULL);
    g_free (speech->text_echo_type);
    if (value)
    {
	speech->text_echo_type = g_strdup ((gchar*)gconf_value_get_string (value));
	gconf_value_free (value);
	value = NULL;
    }
    else
	speech->text_echo_type  = g_strdup (DEFAULT_SPEECH_TEXT_ECHO);

    value = gconf_client_get_default_from_schema (gnopernicus_client, 
						CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_MODIFIERS,
						NULL);
    g_free (speech->modifiers_type);
    if (value)
    {
	speech->modifiers_type = g_strdup ((gchar*)gconf_value_get_string (value));
	gconf_value_free (value);
	value = NULL;
    }
    else
	speech->modifiers_type  = g_strdup (DEFAULT_SPEECH_MODIFIERS);

    value = gconf_client_get_default_from_schema (gnopernicus_client, 
						CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_CURSORS,
						NULL);
    g_free (speech->cursors_type);
    if (value)
    {
	speech->cursors_type = g_strdup ((gchar*)gconf_value_get_string (value));
	gconf_value_free (value);
	value = NULL;
    }
    else
	speech->cursors_type  = g_strdup (DEFAULT_SPEECH_CURSORS);

    value = gconf_client_get_default_from_schema (gnopernicus_client, 
						CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_SPACES,
						NULL);
    g_free (speech->spaces_type);
    if (value)
    {
	speech->spaces_type = g_strdup ((gchar*)gconf_value_get_string (value));
	gconf_value_free (value);
	value = NULL;
    }
    else
	speech->spaces_type  = g_strdup (DEFAULT_SPEECH_SPACES);

    value = gconf_client_get_default_from_schema (gnopernicus_client, 
						CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_DICTIONARY_ACTIVE,
						NULL);
    g_free (speech->dictionary);
    if (value)
    {
	speech->dictionary = g_strdup ((gchar*)gconf_value_get_string (value));
	gconf_value_free (value);
	value = NULL;
    }
    else
	speech->dictionary  = g_strdup (DEFAULT_SPEECH_DICTIONARY_ACTIVE);

    spconf_set_default_voices (TRUE);
    
    dictionary_list =
	spconf_dictionary_free (dictionary_list);
    
    dictionary_list =
	spconf_dictionary_load_default ();
}


/**
 *
 * Free Speech structure					
 *
**/
void 
spconf_setting_free (Speech* speech)
{
    if (!speech)
	return;
    
    g_free (speech->count_type);
    g_free (speech->punctuation_type);
    g_free (speech->text_echo_type);
    g_free (speech->modifiers_type);
    g_free (speech->cursors_type);
    g_free (speech->spaces_type);
    g_free (speech->dictionary);
    g_free (speech);
    speech = NULL;
    
    dictionary_list =
	spconf_dictionary_free (dictionary_list);

}

/**
 *
 * Terminate Speech configure					
 *
**/
void 
spconf_terminate (Speech *speech)
{
    spconf_setting_free (speech);
    
    gconf_client_notify_remove (gnopernicus_client, sp_notify_id);
    	
    if (gconf_client_dir_exists (gnopernicus_client, CONFIG_PATH SPEECH_PATH,NULL))
	gconf_client_remove_dir (gnopernicus_client, CONFIG_PATH SPEECH_PATH,NULL);
	
}

/**
 *
 * Set Methods 
 *
**/
void 
spconf_setting_set (const Speech *speech)
{
    spconf_cursors_set 	   (speech->cursors_type);
    spconf_spaces_set 	   (speech->spaces_type);
    spconf_modifiers_set   (speech->modifiers_type);
    spconf_count_set 	   (speech->count_type);
    spconf_punctuation_set (speech->punctuation_type);
    spconf_dictionary_set  (speech->dictionary);
    spconf_text_echo_set   (speech->text_echo_type);
}

void
spconf_dictionary_changes (void)
{
    gnopiconf_unset_key (CONFIG_PATH SPEECH_VOICE_KEY_PATH SPEECH_DICTIONARY_CHANGES);
    if (!gnopiconf_set_bool (TRUE, CONFIG_PATH SPEECH_VOICE_KEY_PATH SPEECH_DICTIONARY_CHANGES))
	sru_warning (_("spconf:Failed to set dictionary changes\n"));
}

void
spconf_speeach_voice_changes (const gchar *voice)
{
    gnopiconf_unset_key (CONFIG_PATH SPEECH_VOICE_KEY_PATH SPEECH_VOICE_CHANGES);
    if (!gnopiconf_set_string (voice, CONFIG_PATH SPEECH_VOICE_KEY_PATH SPEECH_VOICE_CHANGES))
	sru_warning (_("spconf:Failed to set voice changes\n"));
}

void
spconf_speeach_voice_removed (const gchar *voice)
{
    gnopiconf_unset_key (CONFIG_PATH SPEECH_VOICE_KEY_PATH SPEECH_VOICE_REMOVED);
    if (!gnopiconf_set_string (voice, CONFIG_PATH SPEECH_VOICE_KEY_PATH SPEECH_VOICE_REMOVED))
	sru_warning (_("spconf:Failed to set value\n"));
}

void 
spconf_count_set (const gchar *count)
{
    if (!gnopiconf_set_string (count, CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_COUNT))
    {
	sru_warning (_("spconf:Failed to set Count:%s\n"),count);
	sru_warning (_("Recomanded to delete ~/.gconf/apps/gnopernicus directories"));
    }
}

void 
spconf_punctuation_set (const gchar *punctuation)
{
    if (!gnopiconf_set_string (punctuation, CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_PUNCTUATION))
    {
	sru_warning (_("spconf:Failed to set Punctuation:%s\n"),punctuation);
	sru_warning (_("Recomanded to delete ~/.gconf/apps/gnopernicus directories"));
    }
}


void
spconf_play_voice (const gchar *gnopernicus_speaker)
{
    gnopiconf_unset_key (CONFIG_PATH SPEECH_VOICE_KEY_PATH SPEECH_VOICE_TEST);
    if (!gnopiconf_set_string (gnopernicus_speaker, CONFIG_PATH SPEECH_VOICE_KEY_PATH SPEECH_VOICE_TEST))
	sru_warning (_("spconf:Failed to set tesable voice%s\n"),gnopernicus_speaker);
}

void 
spconf_text_echo_set (const gchar *text_echo)
{
    if (!gnopiconf_set_string (text_echo, CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_TEXT_ECHO))
    {
	sru_warning (_("spconf:Failed to set text echo value:%s\n"),text_echo);
	sru_warning (_("Recomanded to delete ~/.gconf/apps/gnopernicus directories"));
    }
}

void 
spconf_modifiers_set (const gchar *modifiers)
{
    if (!gnopiconf_set_string (modifiers, CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_MODIFIERS))
    {
	sru_warning (_("spconf:Failed to set modifiers value:%s\n"),modifiers);
	sru_warning (_("Recomanded to delete ~/.gconf/apps/gnopernicus directories"));
    }
}

void 
spconf_cursors_set (const gchar *cursors)
{
    if (!gnopiconf_set_string (cursors, CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_CURSORS))
    {
	sru_warning (_("spconf:Failed to set cursors value:%s\n"),cursors);
	sru_warning (_("Recomanded to delete ~/.gconf/apps/gnopernicus directories"));
    }
}

void 
spconf_spaces_set (const gchar *spaces)
{
    if (!gnopiconf_set_string (spaces, CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_SPACES))
    {
	sru_warning (_("spconf:Failed to set spaces value:%s\n"),spaces);
	sru_warning (_("Recomanded to delete ~/.gconf/apps/gnopernicus directories"));
    }
}

void 
spconf_dictionary_set (const gchar *dictionary)
{
    if (!gnopiconf_set_string (dictionary, CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_DICTIONARY_ACTIVE))
    {
	sru_warning (_("spconf:Failed to set dictionary value:%s\n"),dictionary);
	sru_warning (_("Recomanded to delete ~/.gconf/apps/gnopernicus directories"));
    }
}

/**
 *
 * Get Methods
 *
**/

gchar*
spconf_count_get (void)
{
    return gnopiconf_get_string_with_default (CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_COUNT, 
					     DEFAULT_SPEECH_COUNT_TYPE);
}

gchar*
spconf_punctuation_get (void)
{
    return gnopiconf_get_string_with_default (CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_PUNCTUATION, 
					      DEFAULT_SPEECH_PUNCTUATION);						     
}

gint
spconf_voice_count_get (void)
{
    return gnopiconf_get_int_with_default ( CONFIG_PATH SPEECH_VOICE_PARAM_KEY_PATH SPEECH_VOICE_COUNT, 
					    DEFAULT_SPEECH_VOICE_COUNT);
}

gchar*
spconf_text_echo_get (void)
{
    return gnopiconf_get_string_with_default (CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_TEXT_ECHO, 
					      DEFAULT_SPEECH_TEXT_ECHO);						     
}

gchar*
spconf_modifiers_get (void)
{
    return gnopiconf_get_string_with_default (CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_MODIFIERS, 
					      DEFAULT_SPEECH_MODIFIERS);						     
}

gchar*
spconf_cursors_get (void)
{
    return gnopiconf_get_string_with_default (CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_CURSORS, 
					      DEFAULT_SPEECH_CURSORS);						     
}

gchar*
spconf_spaces_get (void)
{
    return gnopiconf_get_string_with_default (CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_SPACES, 
					      DEFAULT_SPEECH_SPACES);						     
}

gchar*
spconf_dictionary_get (void)
{
    return gnopiconf_get_string_with_default (CONFIG_PATH SPEECH_SETTING_KEY_PATH SPEECH_DICTIONARY_ACTIVE, 
					      DEFAULT_SPEECH_DICTIONARY_ACTIVE);						     
}

/*******************************************/
GnopernicusSpeakerListType*
spconf_gnopernicus_speakers_get (void)
{
    return gnopernicus_speakers;
}

GnopernicusSpeakerListType*
spconf_gnopernicus_speakers_clone (GnopernicusSpeakerListType* gs_list)
{
    GnopernicusSpeakerListType *elem;
    GnopernicusSpeakerListType *clone = NULL;
    
    if (!gs_list) 
	return NULL;
    
    for (elem = gs_list; elem ; elem = elem->next)
    {
	if (elem->data)
	    clone = g_slist_append (clone, g_strdup ((gchar*)elem->data));
    }
    return clone;
}

GnopernicusSpeakerListType*
spconf_gnopernicus_speakers_free (GnopernicusSpeakerListType *gs_list)
{
    if (gs_list)
	gs_list = spconf_free_slist_with_char_data (gs_list);
	
    return gs_list;
}

GnopernicusSpeakerListType*
spconf_gnopernicus_speakers_add (GnopernicusSpeakerListType *gs_list,
				 const gchar *gn_speaker)
{   
    GnopernicusSpeakerListType *elem;

    sru_return_val_if_fail (gn_speaker, NULL);
    
    elem = spconf_gnopernicus_speakers_find (gs_list, gn_speaker);
    
    if (elem)
	return gs_list;

    gs_list = g_slist_append (gs_list, g_strdup (gn_speaker));
					    
    return gs_list;
}

GnopernicusSpeakerListType*
spconf_gnopernicus_speakers_remove (GnopernicusSpeakerListType *gs_list,
				    const gchar *gn_speaker)
{
    GSList *elem = NULL;
    
    sru_return_val_if_fail (gs_list, NULL);
    sru_return_val_if_fail (gn_speaker, gs_list);

    elem = gs_list;
    while (elem)
    {
	if (elem->data && !strcmp (gn_speaker, (gchar*)elem->data))
	{
	    gs_list = g_slist_remove_link (gs_list, elem);
	    spconf_gnopernicus_speakers_save (gs_list);
	    g_free (elem->data);
	    g_slist_free (elem);
	    elem = NULL;
	    break;
	}
	elem = elem->next;
    }
    return gs_list;
}


GnopernicusSpeakerListType*
spconf_gnopernicus_speakers_find (GnopernicusSpeakerListType *gs_list,
				  const gchar *gn_speaker)
{   
    GnopernicusSpeakerListType *elem;

    sru_return_val_if_fail (gn_speaker, NULL);    
    if (!gs_list)
	return NULL;
        
    for (elem = gs_list; elem ; elem = elem->next)
    {
	if (!strcmp(elem->data, gn_speaker))
	    return elem;
    }
    
    return NULL;
}


GnopernicusSpeakerListType*
spconf_gnopernicus_speakers_load (GnopernicusSpeakerListType* gs_list)
{
    GnopernicusSpeakerListType *tmp_list = NULL;
    GConfValueType type;
    
    gs_list = spconf_gnopernicus_speakers_free (gs_list);
	
    tmp_list = gnopiconf_get_list_with_default (
		CONFIG_PATH SPEECH_VOICE_PARAM_KEY_PATH SPEECH_GNOPERNICUS_SPEAKERS,
		NULL, &type);
		
    if (!tmp_list)
    {
	gint iter;
	for (iter = 0 ; iter < G_N_ELEMENTS(speakers); iter++)
	    gs_list = g_slist_append (gs_list, g_strdup (speakers[iter].gnopernicus_speaker));

	spconf_gnopernicus_speakers_save (gs_list);
	
	speakers_settings =
	    spconf_speaker_settings_list_load_default ();
    }
    else
    {
	gs_list = tmp_list;
	speakers_settings =
	    spconf_speaker_settings_list_load_from_gconf (gs_list);
    }
    
    return gs_list;
}
/*
void
spconf_gnopernicus_speakers_debug ( GnopernicusSpeakerListType *gs)
{    
    GSList *elem;
    elem = gs;
    while (elem)
    {
     sru_debug ("%s",(gchar*)elem->data);
     elem=elem->next;
    }
}
*/
void
spconf_gnopernicus_speakers_save ( GnopernicusSpeakerListType *gs_list)
{    
    if (gs_list)
	gnopiconf_set_list (gs_list, GCONF_VALUE_STRING,
			    CONFIG_PATH SPEECH_VOICE_PARAM_KEY_PATH SPEECH_GNOPERNICUS_SPEAKERS);
			    
}


/*-----------------------------------------------------------------------*/
SpeakerSettings *
spconf_speaker_settings_load (const gchar *key)
{
    GSList *list = NULL;
    GConfValueType type;
    SpeakerSettings *speaker = NULL;
    gchar *path = NULL;
	
    path = g_strconcat (CONFIG_PATH, SPEECH_VOICE_PARAM_KEY_PATH, key, NULL);
    
    sru_return_val_if_fail (path, NULL);    
    
    list = gnopiconf_get_list_with_default (path, NULL, &type);
    g_free (path);
		
    if (type == GCONF_VALUE_STRING && list)
    {
	    GSList *iter = list;
	    speaker = spconf_speaker_settings_new ();
	    
	    if (speaker)
	    {
		speaker->gnopernicus_speaker = g_strdup (key);	    
		if (iter && iter->data)
		{
		    speaker->gs_speaker = iter->data;
		    iter->data = NULL;
		    iter = iter->next;
		    if (iter && iter->data)
		    {
			speaker->volume = atoi((gchar*)iter->data);
			iter = iter->next;
			if (iter && iter->data)
			{
			    speaker->rate   = atoi((gchar*)iter->data);
			    iter = iter->next;
			    if (iter && iter->data)
			        speaker->pitch  = atoi((gchar*)iter->data);
			}
		    }
		}		    
	    }
		
        list = spconf_free_slist_with_char_data (list);		
    }
    
    return speaker;
}

void
spconf_speaker_settings_save (SpeakerSettings *speaker_settings)
{
    gchar *path;
    GSList *list = NULL;
    
    path = g_strdup_printf ("%s%s", CONFIG_PATH SPEECH_VOICE_PARAM_KEY_PATH,
			    speaker_settings->gnopernicus_speaker);
    				    
    sru_assert (path);    
    list = spconf_create_speaker_item (speaker_settings);
    gnopiconf_set_list (list, GCONF_VALUE_STRING, path);
    g_free (path);	
    list = spconf_free_speaker_item (list);
    spconf_speeach_voice_changes (speaker_settings->gnopernicus_speaker);
}

SpeakerSettings*
spconf_speaker_settings_new (void)
{
    SpeakerSettings *new_speaker_table = NULL ; 
    
    new_speaker_table = ( SpeakerSettings * ) g_new0 (SpeakerSettings, 1 ) ; 
        
    if (!new_speaker_table) 
	sru_error (_("Unable to allocate memory"));
    
    return new_speaker_table ; 
}

SpeakerSettings*
spconf_speaker_settings_free (SpeakerSettings *speaker_settings)
{
    if (!speaker_settings) 
	return NULL;
    g_free (speaker_settings->gnopernicus_speaker);
    g_free (speaker_settings->gs_speaker);
    g_free (speaker_settings);
    speaker_settings = NULL;
    return speaker_settings;
}

SpeakerSettings*
spconf_speaker_settings_copy (SpeakerSettings *dest, 
			      SpeakerSettings *source)
{
    sru_return_val_if_fail (source, NULL);

    dest = spconf_speaker_settings_free (dest);
    dest = spconf_speaker_settings_new ();
	
    sru_return_val_if_fail (dest, NULL);
    
    dest->gnopernicus_speaker = g_strdup (source->gnopernicus_speaker);
    dest->gs_speaker = g_strdup (source->gs_speaker);
    dest->volume     = source->volume;
    dest->rate       = source->rate;
    dest->pitch      = source->pitch;

    return dest;
}


SpeakerSettingsListType*
spconf_speaker_settings_list_get (void)
{
    return speakers_settings;
}

SpeakerSettingsListType*
spconf_speaker_settings_list_clone (SpeakerSettingsListType *sp_list)
{
    SpeakerSettingsListType *elem;
    SpeakerSettingsListType *clone = NULL;
    
    sru_return_val_if_fail (sp_list, NULL);
    
    for (elem = sp_list; elem; elem = elem->next)
    {
	SpeakerSettings *item = (SpeakerSettings*)elem->data;
	SpeakerSettings *new_item = NULL;
	
	new_item = spconf_speaker_settings_new ();
	
	if (new_item)
	{
	    new_item->gnopernicus_speaker = g_strdup (item->gnopernicus_speaker);
	    new_item->gs_speaker = g_strdup (item->gs_speaker);
	    new_item->volume     = item->volume;
	    new_item->rate       = item->rate;
	    new_item->pitch      = item->pitch;
	    clone = g_slist_append (clone, new_item);
	}	
    }
    return clone;
}

SpeakerSettingsListType*
spconf_speaker_settings_list_find (SpeakerSettingsListType *sp_list,
		    	        const gchar *gnopernicus_speaker)
{
    GSList *elem = NULL;
    
    sru_return_val_if_fail (sp_list, NULL);
    sru_return_val_if_fail (gnopernicus_speaker, NULL);
    
    for (elem = sp_list; elem ; elem = elem->next)
    {
	SpeakerSettings *item = (SpeakerSettings*)elem->data;
	if (item)
	{
	    if (!strcmp (gnopernicus_speaker, 
			 item->gnopernicus_speaker))
		return elem;
	}
    }
    
    return NULL;
}

SpeakerSettingsListType*
spconf_speaker_settings_list_add (SpeakerSettingsListType *sp_list, 
			     const gchar 	  *gn_speaker,
			     const gchar 	  *gs_speaker,
			     gint  volume,
			     gint  rate,
			     gint  pitch)
{
    SpeakerSettings *speaker = NULL;
    SpeakerSettingsListType *elem = NULL;
    
    sru_return_val_if_fail (gs_speaker, sp_list);
    sru_return_val_if_fail (gn_speaker, sp_list);

	
    elem = spconf_speaker_settings_list_find (sp_list, gn_speaker);
    
    if (elem)
	return sp_list;
    					    
    speaker = spconf_speaker_settings_new ();
	
    sru_return_val_if_fail (speaker, sp_list);

    speaker->gnopernicus_speaker = g_strdup (gn_speaker);
    speaker->gs_speaker = g_strdup (gs_speaker);
    speaker->volume     = volume;
    speaker->rate       = rate;
    speaker->pitch      = pitch;
    
    sp_list = g_slist_append (sp_list, speaker);

    return sp_list;
}

SpeakerSettingsListType*
spconf_speaker_settings_list_copy (SpeakerSettingsListType *dest, 
				   SpeakerSettingsListType *source)
{
    SpeakerSettings *speaker_dest = NULL;
    SpeakerSettings *speaker_source = NULL;
    
    sru_return_val_if_fail (dest, NULL);
    sru_return_val_if_fail (source, NULL);
	
    speaker_dest   = (SpeakerSettings*) dest->data;
    speaker_source = (SpeakerSettings*) source->data;

    speaker_dest =
	spconf_speaker_settings_free (speaker_dest);
    speaker_dest = 
	spconf_speaker_settings_new ();
	
    if (speaker_dest)
    {
	speaker_dest->gnopernicus_speaker = g_strdup (speaker_source->gnopernicus_speaker);
	speaker_dest->gs_speaker = g_strdup (speaker_source->gs_speaker);
	speaker_dest->volume     = speaker_source->volume;
	speaker_dest->rate       = speaker_source->rate;
	speaker_dest->pitch      = speaker_source->pitch;
	dest->data 		 = speaker_dest;
    }
    return dest;
}



SpeakerSettingsListType*
spconf_speaker_settings_list_remove (SpeakerSettingsListType *sp_list,
				 const gchar *gn_speaker)
{
    SpeakerSettingsListType *elem = NULL;
    
    sru_return_val_if_fail (sp_list, 	sp_list);
    sru_return_val_if_fail (gn_speaker, sp_list);
    
    elem = sp_list;
    
    while (elem)
    {
	SpeakerSettings *item = (SpeakerSettings*)elem->data;
	if (item)
	{
	    if (!strcmp (gn_speaker, item->gnopernicus_speaker))
	    {
		gchar *path = NULL;
		sp_list = g_slist_remove_link (sp_list, elem);
		if (strcmp (NONE_ELEMENT, item->gnopernicus_speaker))
		{
		    path = g_strdup_printf ("%s%s", CONFIG_PATH SPEECH_VOICE_PARAM_KEY_PATH,
					    item->gnopernicus_speaker);
		    gnopiconf_unset_key (path);
		    spconf_speeach_voice_removed (item->gnopernicus_speaker);
		    g_free (path);
		}
		
		elem->data = 
		    spconf_speaker_settings_free ((SpeakerSettings*)elem->data);
		g_slist_free (elem);
		elem = NULL;
		break;
	    }
	}
	elem = elem->next;
    }
    return sp_list;
}


SpeakerSettingsListType*
spconf_speaker_settings_list_free (SpeakerSettingsListType *sp_list)
{
    SpeakerSettingsListType *elem = sp_list;
    if (!sp_list) 
	return NULL;
    
    while (elem)
    {
        elem->data = 
		spconf_speaker_settings_free ((SpeakerSettings*)elem->data);
	elem = elem->next;
    }
	
    g_slist_free (sp_list);
    sp_list = NULL;
    
    return sp_list;
}

SpeakerSettingsListType *
spconf_speaker_settings_list_load_default (void)
{
    gint iter;
    
    speakers_settings =
	    spconf_speaker_settings_list_free  (speakers_settings);
	
    for (iter = 0 ; iter < G_N_ELEMENTS (speakers); iter++)
    {
	SpeakerSettings *speaker = spconf_speaker_settings_new ();
	
	if (speaker)
	{
	    speaker->gnopernicus_speaker = g_strdup (speakers[iter].gnopernicus_speaker);
	    speaker->gs_speaker = g_strdup (speakers[iter].gs_speaker);
	    speaker->volume 	= speakers[iter].volume;
	    speaker->rate       = speakers[iter].rate;
	    speaker->pitch      = speakers[iter].pitch;
	    
	    spconf_speaker_settings_save (speaker);
	    
	    speakers_settings = 
		g_slist_append (speakers_settings, speaker);
	}
    }
    
    return speakers_settings;
}

SpeakerSettingsListType *
spconf_speaker_settings_list_load_from_gconf (GnopernicusSpeakerListType *gnopernicus_speakers)
{
    SpeakerSettingsListType *elem = NULL;

    speakers_settings = 
	    spconf_speaker_settings_list_free  (speakers_settings);

	
    for (elem = gnopernicus_speakers; elem ; elem = elem->next)
    {
	SpeakerSettings *speaker =
	    spconf_speaker_settings_load ((gchar*)elem->data);
	if (speaker)
	{
	    speakers_settings = 
			g_slist_append (speakers_settings, speaker);
	}
	else
	    continue;
    }

    return speakers_settings;
}

void
spconf_speaker_settings_list_save (SpeakerSettingsListType *sp_list)
{
    SpeakerSettingsListType *elem = NULL;
        
    sru_return_if_fail (sp_list);
    
    for (elem = sp_list; elem ; elem = elem->next)
    {
	SpeakerSettings *item = (SpeakerSettings*)elem->data;
	if (item)
	    spconf_speaker_settings_save (item);
    }
}


void
spconf_speakers_list_free (void) 
{
    gnopernicus_speakers =
	spconf_gnopernicus_speakers_free (gnopernicus_speakers);
    speakers_settings =
	spconf_speaker_settings_list_free  (speakers_settings);
}


GSList*
spconf_free_slist_with_char_data (GSList *list)
{
    GSList *elem = NULL;
    if (!list) 
	return NULL;
    
    for (elem = list; elem ; elem = elem->next)
	g_free ((gchar*)elem->data);
	
    g_slist_free (list);
    list = NULL;
    return list;
}

void
spconf_load_voices (void)
{
    gint count = spconf_voice_count_get ();
    gint iter;
    GList *voice_list = NULL;
    
    if (count == -1) 
	return;
    
    for (iter = 0 ; iter < count ; iter++)
    {
	gchar *path = NULL;
	gchar *voice;
	path = g_strdup_printf ("%svoice_%d_name", CONFIG_PATH SPEECH_VOICE_PARAM_KEY_PATH, iter);
	voice = gnopiconf_get_string_with_default ( path, "");
	if (voice)
	    voice_list = g_list_append (voice_list, voice);
	g_free (path);
    }
    if (!voice_list)
	voice_list = g_list_append (voice_list, g_strdup (NONE_ELEMENT));
    
    spui_engine_voice_list_set 	(voice_list);
    
    while (voice_list)
    {
	g_free ((gchar*)voice_list->data);
	voice_list->data = NULL;
	voice_list = g_list_next (voice_list);
    }
    
    voice_list = g_list_first (voice_list);
    g_list_free (voice_list);
    voice_list = NULL;    
}

/*---------------------------*/
GSList*
spconf_create_speaker_item (SpeakerSettings *item)
{
    GSList *list = NULL;
    sru_return_val_if_fail (item, NULL);
    list = g_slist_append (list, g_strdup (item->gs_speaker));
    list = g_slist_append (list, g_strdup_printf ("%d", item->volume));
    list = g_slist_append (list, g_strdup_printf ("%d", item->rate));
    list = g_slist_append (list, g_strdup_printf ("%d", item->pitch));
    return list;
}


GSList*
spconf_free_speaker_item (GSList *item)
{
     return spconf_free_slist_with_char_data (item);
}

void
spconf_set_default_voices (gboolean force)
{
    GnopernicusSpeakerListType *list = NULL;
    GnopernicusSpeakerListType *gs = NULL;
    GConfValueType type;
    gint iter;
    
    gs = spconf_gnopernicus_speakers_free (gs);

    list = gnopiconf_get_list_with_default (
		CONFIG_PATH SPEECH_VOICE_PARAM_KEY_PATH SPEECH_GNOPERNICUS_SPEAKERS,
		NULL, &type);
	
    if (!list || force)
    {
	if (force && list)
	    list = spconf_free_speaker_item (list);
	    
	for (iter = 0 ; iter < G_N_ELEMENTS(speakers); iter++)
	{
	    SpeakerSettings *speaker = spconf_speaker_settings_new ();
	    
	    gs = g_slist_append (gs, g_strdup (speakers[iter].gnopernicus_speaker));
	    
	    if (speaker)
	    {
		speaker->gnopernicus_speaker = g_strdup (speakers[iter].gnopernicus_speaker);
		speaker->gs_speaker = g_strdup (speakers[iter].gs_speaker);
		speaker->volume     = speakers[iter].volume;
		speaker->rate       = speakers[iter].rate;
		speaker->pitch      = speakers[iter].pitch;

		spconf_speaker_settings_save (speaker);	    
		
		spconf_speaker_settings_free (speaker);
	    }
	}

	spconf_gnopernicus_speakers_save (gs);
	gs = spconf_free_speaker_item (gs);
    }
    else
	list = spconf_free_speaker_item (list);
    
}

DictionaryListType*
spconf_dictionary_load_default (void)
{
    DictionaryListType *list = NULL;
    list = g_slist_append (list, g_strdup (DEFAULT_SPEECH_DICTIONARY_ENTRY));
    spconf_dictionary_save (list);
    return list;
}

DictionaryListType*
spconf_dictionary_load (void)
{
    DictionaryListType *list = NULL;
    GConfValueType type;
    
    list = gnopiconf_get_list_with_default (CONFIG_PATH SPEECH_DICTIONARY_PATH SPEECH_DICTIONARY_LIST, 
				    	    NULL, &type);
    if (!list)
	list = spconf_dictionary_load_default ();
	
    return list;
}

gboolean
spconf_dictionary_save (DictionaryListType *list)
{
    sru_return_val_if_fail (list, FALSE);

    if (!gnopiconf_set_list (list, 
			    GCONF_VALUE_STRING, 
			    CONFIG_PATH SPEECH_DICTIONARY_PATH SPEECH_DICTIONARY_LIST))
	return FALSE;
	
    spconf_dictionary_changes ();
    
    return TRUE;
}

DictionaryListType*
spconf_dictionary_free (DictionaryListType *list)
{
    DictionaryListType *elem = NULL;
    if (!list)
	return NULL;
    for (elem = list; elem ; elem = elem->next)
    {
	g_free ((gchar*)elem->data);
	elem->data = NULL;
    }
    g_slist_free (list);
    list = NULL;
    return list;
}

gboolean
spconf_dictionary_split_entry (const gchar *entry,
			       gchar **word,
			       gchar **replace)
{
    gchar *separator = NULL;
    
    *word = NULL;
    *replace = NULL;

    sru_return_val_if_fail (entry, FALSE);
    
    separator = g_utf8_strchr (entry, -1, '<');
    if (separator && *(g_utf8_find_next_char (separator, NULL)) == '>')
    {
	*word = g_strndup (entry, separator - entry);
	separator = g_utf8_find_next_char (separator, NULL);
	*replace  = g_strdup (g_utf8_find_next_char (separator,NULL));
	return TRUE;
    }
    return FALSE;
}

gchar*
spconf_dictionary_create_entry (const gchar *word,
			        const gchar *replace)
{
    sru_return_val_if_fail (word, NULL);
    sru_return_val_if_fail (replace, NULL);
    
    return g_strdup_printf ("%s<>%s", word, replace);
}

DictionaryListType*
spconf_dictionary_find_word (DictionaryListType *list,
			    const gchar *word)
{
    DictionaryListType *elem;

    sru_return_val_if_fail (word, NULL);
    sru_return_val_if_fail (list, NULL);
	
    for (elem = list; elem ; elem = elem->next)
    {
	gchar *word1 = NULL;
	gchar *word2 = NULL;

	spconf_dictionary_split_entry ((gchar*)elem->data, &word1, &word2);
	
	if (!strcmp (word1, word))
	{
	    g_free (word1);
	    g_free (word2);
	    return elem;
	}
	
	g_free (word1);
	g_free (word2);
    }
    
    return elem;
}

DictionaryListType*
spconf_dictionary_modify_word (DictionaryListType *list,
			       const gchar *word,
			       const gchar *replace)
{
    DictionaryListType *elem = NULL;
    sru_return_val_if_fail (replace, list);
    sru_return_val_if_fail (word, list);
    sru_return_val_if_fail (list, list);

    if ((elem = spconf_dictionary_find_word (list, word)) == NULL)
	return list;

    g_free (elem->data);
    elem->data = 
	spconf_dictionary_create_entry (word, replace);
    
    return list;
}

DictionaryListType*
spconf_dictionary_add_word (DictionaryListType *list,
			    const gchar *word,
			    const gchar *replace)
{
    sru_return_val_if_fail (replace, list);
    sru_return_val_if_fail (word, list);
    sru_return_val_if_fail (list, list);

    if (spconf_dictionary_find_word (list, word))
	return list;

    list = g_slist_append (list,
	    spconf_dictionary_create_entry (word, replace));
    return list;
}

DictionaryListType*
spconf_dictionary_remove_word (DictionaryListType *list,
			      const gchar *word)
{
    DictionaryListType *elem = NULL;
    sru_return_val_if_fail (word, list);
    sru_return_val_if_fail (list, list);

    if ((elem = spconf_dictionary_find_word (list, word)) == NULL)
	return list;
    list = g_slist_remove_link (list, elem);
    spconf_dictionary_free (elem);
    return list;
}

/**
 *
 * Speech settings callback function
 * <client> - gconf client
 * <cnxn_id> - notify id
 * <entry> - entry structure
 * <user_data> - not used
 *
**/
void
spconf_changes_cb			(GConfClient	*client,
					guint		cnxn_id,
					GConfEntry	*entry,
					gpointer	user_data)
{        
    GnopernicusSpeakerListType *gs_list = NULL;
    SpeakerSettingsListType    *sp_list = NULL;
    gchar *key = NULL;
    gchar *path = NULL;

    if (! entry || !entry->value)
	return;
	
    key = (gchar*) gconf_entry_get_key (entry);

    sru_debug  ("spconf:spconf_changes_cb:Entry key:%s",gconf_entry_get_key(entry));
    sru_debug  ("spconf:spconf_changes_cb:Entry value type:%i",entry->value->type);
    switch (entry->value->type)
    {
    	case GCONF_VALUE_INT:    sru_debug ("spconf:spconf_changes_cb:Entry value:%in",gconf_value_get_int(entry->value));break;
    	case GCONF_VALUE_FLOAT:  sru_debug ("spconf:spconf_changes_cb:Entry value:%f",gconf_value_get_float(entry->value));break;
    	case GCONF_VALUE_STRING: sru_debug ("spconf:spconf_changes_cb:Entry value:%s",gconf_value_get_string(entry->value));break;
    	case GCONF_VALUE_BOOL:   sru_debug ("spconf:spconf_changes_cb:Entry value:%i",gconf_value_get_bool(entry->value));break;
	default:break;
    }

    path = (key + strlen (CONFIG_PATH SPEECH_PARAMETER_SECTION) + 1);
    gs_list = spconf_gnopernicus_speakers_find (gnopernicus_speakers, path);
    if (gs_list)
    {
	    SpeakerSettings *speaker = NULL;
	    
	    sp_list =
		spconf_speaker_settings_list_find (speakers_settings, path);
		
	    if (!sp_list) 
		return;
		
	    speaker = spconf_speaker_settings_load (path);
	
	    if (!speaker)
		return;
	    
	    spconf_speaker_settings_free ((SpeakerSettings *)sp_list->data);
	    
	    sp_list->data = speaker;
	    
	    spui_modify_speaker_values ((SpeakerSettings *)sp_list->data);
    }
}
