﻿use utf8;  # NB: This file includes non-ASCII string constants in UTF-8 encoding

package Font::TTF::OTTags;

=head1 NAME

Font::TTF::OTTags - Utilities for TrueType/OpenType tags

=head1 SYNOPSIS

  use Font::TTF::OTTags qw( %tttags %ttnames %iso639 readtagsfile );
  
  # Look at built-in stuff:
  $script_tag = $tttags{'SCRIPT'}{'Cypriot Syllabary'};  # returns 'cprt'
  $lang_name = $ttnames{'LANGUAGE'}{'ALT '};             # returns 'Altai'
  
  # Map iso639-2 tag to/from OT lang tag
  @isotags = $iso639{'ALT '};    # returns [ 'atv', 'alt' ]
  $lang_tag = $iso639{'atv'};    # returns 'ALT '
    
  # Read latest tags file to add to built-in definitions
  readtagsfile ("C:\\Program Files\\Microsoft VOLT\\TAGS.txt");

First-level keys to %tttags and %ttnames are:

=over

=item SCRIPT

retrieve script tag or name

=item LANGUAGE 

retrieve language tag or name

=item FEATURE 

retrieve feature tag or name

=back

Built-in data has been derived from the 2014-07-11 draft of the
3rd edition of ISO/IEC 14496-22 
(Information technology - Coding of audio-visual objects - Part 22: Open Font Format)
which, when finalized and approved, will replace the second edition (ISO/IEC 14496-22:2009).

=head1 METHODS

=cut

use strict;
use vars qw( %tttags %ttnames %iso639 @EXPORT_OK @ISA );
require Exporter;
@ISA = qw( Exporter );
@EXPORT_OK = qw( %tttags %ttnames %iso639 readtagsfile);


# All data below derived Microsoft OpenType specification 1.6

%tttags = (

'SCRIPT' => {
    "Arabic" => 'arab',
    "Armenian" => 'armn',
    "Avestan" => 'avst',
    "Balinese" => 'bali',
    "Bamum" => 'bamu',
    "Batak" => 'batk',
    "Bengali" => 'beng',
    "Bengali v.2" => 'bng2',
    "Bopomofo" => 'bopo',
    "Brahmi" => 'brah',
    "Braille" => 'brai',
    "Buginese" => 'bugi',
    "Buhid" => 'buhd',
    "Byzantine Music" => 'byzm',
    "Canadian Syllabics" => 'cans',
    "Carian" => 'cari',
    "Chakma" => 'cakm',
    "Cham" => 'cham',
    "Cherokee" => 'cher',
    "CJK Ideographic" => 'hani',
    "Coptic" => 'copt',
    "Cypriot Syllabary" => 'cprt',
    "Cyrillic" => 'cyrl',
    "Default" => 'DFLT',
    "Deseret" => 'dsrt',
    "Devanagari" => 'deva',
    "Devanagari v.2" => 'dev2',
    "Egyptian hieroglyphs" => 'egyp',
    "Ethiopic" => 'ethi',
    "Georgian" => 'geor',
    "Glagolitic" => 'glag',
    "Gothic" => 'goth',
    "Greek" => 'grek',
    "Gujarati" => 'gujr',
    "Gujarati v.2" => 'gjr2',
    "Gurmukhi" => 'guru',
    "Gurmukhi v.2" => 'gur2',
    "Hangul" => 'hang',
    "Hangul Jamo" => 'jamo',
    "Hanunoo" => 'hano',
    "Hebrew" => 'hebr',
    "Hiragana" => 'kana',
    "Imperial Aramaic" => 'armi',
    "Inscriptional Pahlavi" => 'phli',
    "Inscriptional Parthian" => 'prti',
    "Javanese" => 'java',
    "Kaithi" => 'kthi',
    "Kannada" => 'knda',
    "Kannada v.2" => 'knd2',
    "Katakana" => 'kana',
    "Kayah Li" => 'kali',
    "Kharosthi" => 'khar',
    "Khmer" => 'khmr',
    "Lao" => 'lao ',
    "Latin" => 'latn',
    "Lepcha" => 'lepc',
    "Limbu" => 'limb',
    "Linear B" => 'linb',
    "Lisu (Fraser)" => 'lisu',
    "Lycian" => 'lyci',
    "Lydian" => 'lydi',
    "Malayalam" => 'mlym',
    "Malayalam v.2" => 'mlm2',
    "Mandaic, Mandaean" => 'mand',
    "Mathematical Alphanumeric Symbols" => 'math',
    "Meitei Mayek (Meithei, Meetei)" => 'mtei',
    "Meroitic Cursive" => 'merc',
    "Meroitic Hieroglyphs" => 'mero',
    "Miao" => 'plrd',
    "Mongolian" => 'mong',
    "Musical Symbols" => 'musc',
    "Myanmar" => 'mymr',
    "Myanmar v.2" => 'mym2',
    "New Tai Lue" => 'talu',
    "N'Ko" => 'nko ',
    "Ogham" => 'ogam',
    "Ol Chiki" => 'olck',
    "Old Italic" => 'ital',
    "Old Persian Cuneiform" => 'xpeo',
    "Old South Arabian" => 'sarb',
    "Old Turkic, Orkhon Runic" => 'orkh',
    "Odia (formerly Oriya)" => 'orya',
    "Odia (formerly Oriya) v.2" => 'ory2',
    "Osmanya" => 'osma',
    "Phags-pa" => 'phag',
    "Phoenician" => 'phnx',
    "Rejang" => 'rjng',
    "Runic" => 'runr',
    "Samaritan" => 'samr',
    "Saurashtra" => 'saur',
    "Sharada" => 'shrd',
    "Shavian" => 'shaw',
    "Sinhala" => 'sinh',
    "Sora Sompeng" => 'sora',
    "Sumero-Akkadian Cuneiform" => 'xsux',
    "Sundanese" => 'sund',
    "Syloti Nagri" => 'sylo',
    "Syriac" => 'syrc',
    "Tagalog" => 'tglg',
    "Tagbanwa" => 'tagb',
    "Tai Le" => 'tale',
    "Tai Tham (Lanna)" => 'lana',
    "Tai Viet" => 'tavt',
    "Takri" => 'takr',
    "Tamil" => 'taml',
    "Tamil v.2" => 'tml2',
    "Telugu" => 'telu',
    "Telugu v.2" => 'tel2',
    "Thaana" => 'thaa',
    "Thai" => 'thai',
    "Tibetan" => 'tibt',
    "Tifinagh" => 'tfng',
    "Ugaritic Cuneiform" => 'ugar',
    "Vai" => 'vai ',
    "Yi" => 'yi  ',
    },

'LANGUAGE' => {
    "Aari" => 'ARI ',
    "Abaza" => 'ABA ',
    "Abkhazian" => 'ABK ',
    "Achi" => 'ACR ',
    "Acholi" => 'ACH ',
    "Adyghe" => 'ADY ',
    "Afar" => 'AFR ',
    "Afrikaans" => 'AFK ',
    "Agaw" => 'AGW ',
    "Aiton" => 'AIO ',
    "Akan" => 'AKA ',
    "Albanian" => 'SQI ',
    "Alsatian" => 'ALS ',
    "Altai" => 'ALT ',
    "Amharic" => 'AMH ',
    "Anglo-Saxon" => 'ANG ',
    "Arabic" => 'ARA ',
    "Aragonese" => 'ARG ',
    "Arakwal" => 'RKW ',
    "Armenian East" => 'HYE0',
    "Armenian" => 'HYE ',
    "Aromanian" => 'RUP ',
    "Arpitan" => 'FRP ',
    "Assamese" => 'ASM ',
    "Asturian" => 'AST ',
    "Athapaskan" => 'ATH ',
    "Avar" => 'AVR ',
    "Awadhi" => 'AWA ',
    "Aymara" => 'AYM ',
    "Azerbaijani" => 'AZE ',
    "Badaga" => 'BAD ',
    "Baghelkhandi" => 'BAG ',
    "Bagri" => 'BGQ ',
    "Balante" => 'BLN ',
    "Balinese" => 'BAN ',
    "Balkar" => 'BAL ',
    "Balti" => 'BLT ',
    "Baluchi" => 'BLI ',
    "Bambara (Bamanankan)" => 'BMB ',
    "Bamileke" => 'BML ',
    "Banda" => 'BAD0',
    "Bandjalang" => 'BDY ',
    "Baoulé" => 'BAU ',
    "Bashkir" => 'BSH ',
    "Basque" => 'EUQ ',
    "Batak Simalungun" => 'BTS ',
    "Batak Toba" => 'BBC ',
    "Bavarian" => 'BAR ',
    "Belarusian" => 'BEL ',
    "Bemba" => 'BEM ',
    "Bench" => 'BCH ',
    "Bengali" => 'BEN ',
    "Berber" => 'BBR ',
    "Beti" => 'BTI ',
    "Bhili" => 'BHI ',
    "Bhojpuri" => 'BHO ',
    "Bible Cree" => 'BCR ',
    "Bikol" => 'BIK ',
    "Bilen" => 'BIL ',
    "Bishnupriya Manipuri" => 'BPY ',
    "Bislama" => 'BIS ',
    "Blackfoot" => 'BKF ',
    "Bodo" => 'BRX ',
    "Bosnian" => 'BOS ',
    "Bouyei" => 'PCC ',
    "Brahui" => 'BRH ',
    "Braj Bhasha" => 'BRI ',
    "Breton" => 'BRE ',
    "Bugis" => 'BUG ',
    "Bulgarian" => 'BGR ',
    "Burmese" => 'BRM ',
    "Cajun French" => 'FRC ',
    "Carrier" => 'CRR ',
    "Catalan" => 'CAT ',
    "Cebuano" => 'CEB ',
    "Central Yupik" => 'ESU ',
    "Chaha Gurage" => 'CHG ',
    "Chamorro" => 'CHA ',
    "Chattisgarhi" => 'CHH ',
    "Chechen" => 'CHE ',
    "Cherokee" => 'CHR ',
    "Cheyenne" => 'CHY ',
    "Chichewa (Chewa, Nyanja)" => 'CHI ',
    "Chiga" => 'CGG ',
    "Chin" => 'QIN ',
    "Chinese Phonetic" => 'ZHP ',
    "Chinese Simplified" => 'ZHS ',
    "Chinese Traditional" => 'ZHT ',
    "Chinese, Hong Kong SAR" => 'ZHH ',
    "Chipewyan" => 'CHP ',
    "Chittagonian" => 'CTG ',
    "Choctaw" => 'CHO ',
    "Chukchi" => 'CHK ',
    "Church Slavonic" => 'CSL ',
    "Chuukese" => 'CHK0',
    "Chuvash" => 'CHU ',
    "Comorian" => 'CMR ',
    "Coptic" => 'COP ',
    "Cornish" => 'COR ',
    "Corsican" => 'COS ',
    "Cree" => 'CRE ',
    "Creoles" => 'CPP ',
    "Crimean Tatar" => 'CRT ',
    "Croatian" => 'HRV ',
    "Czech" => 'CSY ',
    "Dan" => 'DNJ ',
    "Dangme" => 'DNG ',
    "Danish" => 'DAN ',
    "Dargwa" => 'DAR ',
    "Dari" => 'DRI ',
    "Dayi" => 'DAX ',
    "Dehong Dai" => 'TDD ',
    "Dhangu" => 'DHG ',
    "Dhuwal" => 'DUJ ',
    "Dimli" => 'DIQ ',
    "Dinka" => 'DNK ',
    "Divehi (Dhivehi, Maldivian) (deprecated)" => 'DHV ',
    "Divehi (Dhivehi, Maldivian)" => 'DIV ',
    "Djambarrpuyngu" => 'DJR0',
    "Dogri" => 'DGO ',
    "Dogri" => 'DGR ',
    "Dungan" => 'DUN ',
    "Dutch (Flemish)" => 'FLE ',
    "Dutch" => 'NLD ',
    "Dzongkha" => 'DZN ',
    "Eastern Cree" => 'ECR ',
    "Eastern Maninkakan" => 'EMK ',
    "Eastern Pwo Karen" => 'KJP ',
    "Ebira" => 'EBI ',
    "Edo" => 'EDO ',
    "Efik" => 'EFI ',
    "English" => 'ENG ',
    "Erzya" => 'ERZ ',
    "Esperanto" => 'NTO ',
    "Estonian" => 'ETI ',
    "Even" => 'EVN ',
    "Evenki" => 'EVK ',
    "Ewe" => 'EWE ',
    "Fang" => 'FAN0',
    "Fanti" => 'FAT ',
    "Faroese" => 'FOS ',
    "Fijian" => 'FJI ',
    "Filipino" => 'PIL ',
    "Finnish" => 'FIN ',
    "Fon" => 'FON ',
    "Forest Nenets" => 'FNE ',
    "French Antillean" => 'FAN ',
    "French" => 'FRA ',
    "Frisian" => 'FRI ',
    "Friulian" => 'FRL ',
    "Fulah" => 'FUL ',
    "Futa" => 'FTA ',
    "Ga" => 'GAD ',
    "Gagauz" => 'GAG ',
    "Galician" => 'GAL ',
    "Ganda" => 'LUG ',
    "Garhwali" => 'GAW ',
    "Garo" => 'GRO ',
    "Garshuni" => 'GAR ',
    "Ge'ez" => 'GEZ ',
    "Georgian" => 'KAT ',
    "German" => 'DEU ',
    "Gilaki" => 'GLK ',
    "Gilyak" => 'GIL ',
    "Githabul" => 'GIH ',
    "Gogo" => 'GOG ',
    "Gondi" => 'GON ',
    "Greek" => 'ELL ',
    "Greenlandic" => 'GRN ',
    "Guarani" => 'GUA ',
    "Gujarati" => 'GUJ ',
    "Gumatj" => 'GNN ',
    "Gumuz" => 'GMZ ',
    "Gupapuyngu" => 'GUF ',
    "Gusii" => 'GUZ ',
    "Haitian (Haitian Creole)" => 'HAI ',
    "Halam" => 'HAL ',
    "Hammer-Banna" => 'HBN ',
    "Harari" => 'HRI ',
    "Harauti" => 'HAR ',
    "Haryanvi" => 'BGC ',
    "Hausa" => 'HAU ',
    "Hawaiian" => 'HAW ',
    "Haya" => 'HAY ',
    "Hazaragi" => 'HAZ ',
    "Hebrew" => 'IWR ',
    "Herero" => 'HER ',
    "High Mari" => 'HMA ',
    "Hiligaynon" => 'HIL ',
    "Hindi" => 'HIN ',
    "Hindko" => 'HND ',
    "Hiri Motu" => 'HMO ',
    "Hmong Daw" => 'MWW ',
    "Hmong" => 'HMN ',
    "Ho" => 'HO  ',
    "Hungarian" => 'HUN ',
    "Iban" => 'IBA ',
    "Ibibio" => 'IBB ',
    "Icelandic" => 'ISL ',
    "Ido" => 'IDO ',
    "Igbo" => 'IBO ',
    "Ijo languages" => 'IJO ',
    "Ilokano" => 'ILO ',
    "Inari Sami" => 'ISM ',
    "Indonesian" => 'IND ',
    "Ingush" => 'ING ',
    "Interlingua" => 'INA ',
    "Interlingue" => 'ILE ',
    "Inuktitut" => 'INU ',
    "Inupiat" => 'IPK ',
    "Irish Traditional" => 'IRT ',
    "Irish" => 'IRI ',
    "Italian" => 'ITA ',
    "Jamaican Creole" => 'JAM ',
    "Japanese" => 'JAN ',
    "Javanese" => 'JAV ',
    "Jula" => 'JUL ',
    "Kabardian" => 'KAB ',
    "Kabuverdianu (Crioulo)" => 'KEA ',
    "Kabyle" => 'KAB0',
    "Kachchi" => 'KAC ',
    "Kalenjin" => 'KAL ',
    "Kalmyk" => 'KLM ',
    "Kamba" => 'KMB ',
    "Kanauji" => 'BJJ ',
    "Kannada" => 'KAN ',
    "Kanuri" => 'KNR ',
    "Kaqchikel" => 'CAK ',
    "Karachay" => 'KAR ',
    "Karaim" => 'KRM ',
    "Karakalpak" => 'KRK ',
    "Karelian" => 'KRL ',
    "Karen" => 'KRN ',
    "Kashmiri" => 'KSH ',
    "Kashubian" => 'CSB ',
    "Kazakh" => 'KAZ ',
    "Kebena" => 'KEB ',
    "Kekchi" => 'KEK ',
    "Khakass" => 'KHA ',
    "Khamti Shan" => 'KHT ',
    "Khanty-Kazim" => 'KHK ',
    "Khanty-Shurishkar" => 'KHS ',
    "Khanty-Vakhi" => 'KHV ',
    "Khasi" => 'KSI ',
    "Khmer" => 'KHM ',
    "Khowar" => 'KHW ',
    "Khutsuri Georgian" => 'KGE ',
    "Kikongo" => 'KON ',
    "Kikuyu (Gikuyu)" => 'KIK ',
    "Kildin Sami" => 'KSM ',
    "Kinyarwanda" => 'RUA ',
    "Kirghiz (Kyrgyz)" => 'KIR ',
    "Kiribati (Gilbertese)" => 'GIL0',
    "Kirmanjki" => 'KIU ',
    "Kisii" => 'KIS ',
    "Kituba" => 'MKW ',
    "Kodagu" => 'KOD ',
    "Kokni" => 'KKN ',
    "Komi" => 'KOM ',
    "Komi-Permyak" => 'KOP ',
    "Komi-Zyrian" => 'KOZ ',
    "Komo" => 'KMO ',
    "Komso" => 'KMS ',
    "Kongo" => 'KON0',
    "Konkani" => 'KOK ',
    "Koorete" => 'KRT ',
    "Korean Old Hangul" => 'KOH ',
    "Korean" => 'KOR ',
    "Koryak" => 'KYK ',
    "Kosraean" => 'KOS ',
    "Kpelle (Guinea)" => 'GKP ',
    "Kpelle (Liberia)" => 'XPE ',
    "Kpelle" => 'KPL ',
    "Krio" => 'KRI ',
    "Kuanyama" => 'KUA ',
    "Kui" => 'KUI ',
    "Kulvi" => 'KUL ',
    "Kumaoni" => 'KMN ',
    "Kumyk" => 'KUM ',
    "Kurdish" => 'KUR ',
    "Kurukh" => 'KUU ',
    "Kuy" => 'KUY ',
    "K'iche'" => 'QUC ',
    "L-Cree" => 'LCR ',
    "Ladakhi" => 'LDK ',
    "Ladin" => 'LAD ',
    "Ladino" => 'JUD ',
    "Lahuli" => 'LAH ',
    "Lak" => 'LAK ',
    "Laki" => 'LKI ',
    "Lambani" => 'LAM ',
    "Lampung" => 'LJP ',
    "Lao" => 'LAO ',
    "Latin" => 'LAT ',
    "Latvian" => 'LVI ',
    "Laz" => 'LAZ ',
    "Lezgi" => 'LEZ ',
    "Ligurian" => 'LIJ ',
    "Limbu" => 'LMB ',
    "Limburgish" => 'LIM ',
    "Lingala" => 'LIN ',
    "Lisu" => 'LIS ',
    "Lithuanian" => 'LTH ',
    "Lojban" => 'JBO ',
    "Loma" => 'LOM ',
    "Lombard" => 'LMO ',
    "Lomwe" => 'LMW ',
    "Low Mari" => 'LMA ',
    "Low Saxon" => 'NDS ',
    "Lower Sorbian" => 'LSB ',
    "Luba-Katanga" => 'LUB ',
    "Luba-Lulua" => 'LUA ',
    "Lule Sami" => 'LSM ',
    "Luo" => 'LUO ',
    "Luri" => 'LRC ',
    "Luxembourgish" => 'LTZ ',
    "Luyia" => 'LUH ',
    "Lü" => 'XBD ',
    "Macedonian" => 'MKD ',
    "Madura" => 'MAD ',
    "Magahi" => 'MAG ',
    "Maithili" => 'MTH ',
    "Majang" => 'MAJ ',
    "Makasar" => 'MKR ',
    "Makhuwa" => 'MAK ',
    "Makonde" => 'KDE ',
    "Malagasy" => 'MLG ',
    "Malay" => 'MLY ',
    "Malayalam Reformed" => 'MLR ',
    "Malayalam Traditional" => 'MAL ',
    "Male" => 'MLE ',
    "Malinke" => 'MLN ',
    "Maltese" => 'MTS ',
    "Mam" => 'MAM ',
    "Manchu" => 'MCH ',
    "Mandar" => 'MDR ',
    "Mandinka" => 'MND ',
    "Maninka" => 'MNK ',
    "Manipuri" => 'MNI ',
    "Mansi" => 'MAN ',
    "Manx" => 'MNX ',
    "Maori" => 'MRI ',
    "Mapudungun" => 'MAP ',
    "Marathi" => 'MAR ',
    "Marshallese" => 'MAH ',
    "Marwari" => 'MAW ',
    "Mayan" => 'MYN ',
    "Mazanderani" => 'MZN ',
    "Mbundu" => 'MBN ',
    "Me'en" => 'MEN ',
    "Mende" => 'MDE ',
    "Meru" => 'MER ',
    "Mewati" => 'WTM ',
    "Minangkabau" => 'MIN ',
    "Minjangbal" => 'XJB ',
    "Mirandese" => 'MWL ',
    "Mizo" => 'MIZ ',
    "Mohawk" => 'MOH ',
    "Moksha" => 'MOK ',
    "Moldavian" => 'MOL ',
    "Mon" => 'MON ',
    "Mongolian" => 'MNG ',
    "Moose Cree" => 'MCR ',
    "Morisyen" => 'MFE ',
    "Moroccan" => 'MOR ',
    "Mossi" => 'MOS ',
    "Mundari" => 'MUN ',
    "Muscogee" => 'MUS ',
    "N'Ko" => 'NKO ',
    "N-Cree" => 'NCR ',
    "Naga-Assamese" => 'NAG ',
    "Nagari" => 'NGR ',
    "Nahuatl" => 'NAH ',
    "Nanai" => 'NAN ',
    "Naskapi" => 'NAS ',
    "Nauruan" => 'NAU ',
    "Navajo" => 'NAV ',
    "Ndau" => 'NDC ',
    "Ndebele" => 'NDB ',
    "Ndonga" => 'NDG ',
    "Neapolitan" => 'NAP ',
    "Nepali" => 'NEP ',
    "Newari" => 'NEW ',
    "Ngbaka" => 'NGA ',
    "Nigerian Fulfulde" => 'FUV ',
    "Nimadi" => 'NOE ',
    "Nisi" => 'NIS ',
    "Niuean" => 'NIU ',
    "Nogai" => 'NOG ',
    "Norfolk" => 'PIH ',
    "Northern Sami" => 'NSM ',
    "Northern Thai" => 'NTA ',
    "Norway House Cree" => 'NHC ',
    "Norwegian Nynorsk (Nynorsk, Norwegian)" => 'NYN ',
    "Norwegian" => 'NOR ',
    "Novial" => 'NOV ',
    "Nyamwezi" => 'NYM ',
    "Nyankole" => 'NKL ',
    "Occitan" => 'OCI ',
    "Odia (formerly Oriya)" => 'ORI ',
    "Oji-Cree" => 'OCR ',
    "Ojibway" => 'OJB ',
    "Old Irish" => 'SGA ',
    "Oromo" => 'ORO ',
    "Ossetian" => 'OSS ',
    "Pa'o Karen" => 'BLK ',
    "Palauan" => 'PAU ',
    "Palaung" => 'PLG ',
    "Palestinian Aramaic" => 'PAA ',
    "Pali" => 'PAL ',
    "Palpa" => 'PAP ',
    "Pampangan" => 'PAM ',
    "Pangasinan" => 'PAG ',
    "Papiamentu" => 'PAP0',
    "Pashto" => 'PAS ',
    "Pennsylvania German" => 'PDC ',
    "Persian" => 'FAR ',
    "Phake" => 'PHK ',
    "Phonetic transcription - Americanist conventions" => 'APPH',
    "Phonetic transcription - IPA conventions" => 'IPPH',
    "Picard" => 'PCD ',
    "Piemontese" => 'PMS ',
    "Pocomchi" => 'POH ',
    "Pohnpeian" => 'PON ',
    "Polish" => 'PLK ',
    "Polytonic Greek" => 'PGR ',
    "Portuguese" => 'PTG ',
    "Provencal" => 'PRO ',
    "Punjabi" => 'PAN ',
    "Quechua (Bolivia)" => 'QUH ',
    "Quechua (Ecuador)" => 'QVI ',
    "Quechua (Peru)" => 'QWH ',
    "Quechua" => 'QUZ ',
    "R-Cree" => 'RCR ',
    "Rajasthani" => 'RAJ ',
    "Rakhine" => 'ARK ',
    "Rarotongan" => 'RAR ',
    "Rejang" => 'REJ ',
    "Riang" => 'RIA ',
    "Ripuarian" => 'KSH0',
    "Ritarungo" => 'RIT ',
    "Romanian" => 'ROM ',
    "Romansh" => 'RMS ',
    "Romany" => 'ROY ',
    "Rotuman" => 'RTM ',
    "Rundi" => 'RUN ',
    "Russian Buriat" => 'RBU ',
    "Russian" => 'RUS ',
    "Rusyn" => 'RSY ',
    "Sadri" => 'SAD ',
    "Sakha" => 'YAK ',
    "Samoan" => 'SMO ',
    "Samogitian" => 'SGS ',
    "San Blas Kuna" => 'CUK ',
    "Sango" => 'SGO ',
    "Sanskrit" => 'SAN ',
    "Santali" => 'SAT ',
    "Sardinian" => 'SRD ',
    "Sasak" => 'SAS ',
    "Saterland Frisian" => 'STQ ',
    "Sayisi" => 'SAY ',
    "Scots" => 'SCO ',
    "Scottish Gaelic (Gaelic)" => 'GAE ',
    "Sekota" => 'SEK ',
    "Selkup" => 'SEL ',
    "Sena" => 'SNA ',
    "Seraiki" => 'SRK ',
    "Serbian" => 'SRB ',
    "Serer" => 'SRR ',
    "Shan" => 'SHN ',
    "Shona" => 'SNA0',
    "Sibe" => 'SIB ',
    "Sicilian" => 'SCN ',
    "Sidamo" => 'SID ',
    "Silesian" => 'SZL ',
    "Silte Gurage" => 'SIG ',
    "Sindhi" => 'SND ',
    "Sinhala (Sinhalese)" => 'SNH ',
    "Skolt Sami" => 'SKS ',
    "Slavey" => 'SLA ',
    "Slovak" => 'SKY ',
    "Slovenian" => 'SLV ',
    "Sodo Gurage" => 'SOG ',
    "Soga" => 'XOG ',
    "Somali" => 'SML ',
    "Songe" => 'SOP ',
    "Soninke" => 'SNK ',
    "Sotho, Northern" => 'NSO ',
    "Sotho, Southern" => 'SOT ',
    "South Slavey" => 'SSL ',
    "Southern Kiwai" => 'KJD ',
    "Southern Sami" => 'SSM ',
    "Spanish" => 'ESP ',
    "Standard Morrocan Tamazigh" => 'ZGH ',
    "Sukuma" => 'SUK ',
    "Sundanese" => 'SUN ',
    "Suri" => 'SUR ',
    "Sutu" => 'SXT ',
    "Svan" => 'SVA ',
    "Swadaya Aramaic" => 'SWA ',
    "Swahili" => 'SWK ',
    "Swati" => 'SWZ ',
    "Swedish" => 'SVE ',
    "Sylheti" => 'SYL ',
    "Syriac" => 'SYR ',
    "S'gaw Karen" => 'KSW ',
    "TH-Cree" => 'TCR ',
    "Tabasaran" => 'TAB ',
    "Tachelhit" => 'SHI ',
    "Tagalog" => 'TGL ',
    "Tahitian" => 'THT ',
    "Tajik" => 'TAJ ',
    "Tamashek" => 'TMH ',
    "Tamazight" => 'TZM ',
    "Tamil" => 'TAM ',
    "Tarifit" => 'RIF ',
    "Tatar" => 'TAT ',
    "Telugu" => 'TEL ',
    "Temne" => 'TMN ',
    "Tetum" => 'TET ',
    "Thai" => 'THA ',
    "Tibetan" => 'TIB ',
    "Tigre" => 'TGR ',
    "Tigrinya" => 'TGY ',
    "Tiv" => 'TIV ',
    "Todo" => 'TOD ',
    "Tok Pisin" => 'TPI ',
    "Toma" => 'TOD0',
    "Tonga" => 'TNG ',
    "Tongan" => 'TGN ',
    "Torki" => 'AZB ',
    "Tsonga" => 'TSG ',
    "Tswana" => 'TNA ',
    "Tulu" => 'TUL ',
    "Tumbuka" => 'TUM ',
    "Tundra Nenets" => 'TNE ',
    "Turkish" => 'TRK ',
    "Turkmen" => 'TKM ',
    "Turoyo Aramaic" => 'TUA ',
    "Tuvalu" => 'TVL ',
    "Tuvin" => 'TUV ',
    "Twi" => 'TWI ',
    "Tzotzil" => 'TZO ',
    "Tày" => 'TYZ ',
    "Udmurt" => 'UDM ',
    "Ukrainian" => 'UKR ',
    "Umbundu" => 'UMB ',
    "Upper Saxon" => 'SXU ',
    "Upper Sorbian" => 'USB ',
    "Urdu" => 'URD ',
    "Uyghur" => 'UYG ',
    "Uzbek" => 'UZB ',
    "Venda" => 'VEN ',
    "Venetian" => 'VEC ',
    "Vietnamese" => 'VIT ',
    "Vlax Romani" => 'RMY ',
    "Volapük" => 'VOL ',
    "Võro" => 'VRO ',
    "Wa" => 'WA  ',
    "Wagdi" => 'WAG ',
    "Walloon" => 'WLN ',
    "Waray-Waray" => 'WAR ',
    "Wayuu" => 'GUC ',
    "Welsh" => 'WEL ',
    "West-Cree" => 'WCR ',
    "Western Kayah" => 'KYU ',
    "Western Panjabi" => 'PNB ',
    "Western Pwo Karen" => 'PWO ',
    "Wolof" => 'WLF ',
    "Woods Cree" => 'DCR ',
    "Xhosa" => 'XHS ',
    "Y-Cree" => 'YCR ',
    "Yao" => 'YAO ',
    "Yapese" => 'YAP ',
    "Yi Classic" => 'YIC ',
    "Yi Modern" => 'YIM ',
    "Yiddish" => 'JII ',
    "Yoruba" => 'YBA ',
    "Zamboanga Chavacano" => 'CBK ',
    "Zande" => 'ZND ',
    "Zarma" => 'DJR ',
    "Zazaki" => 'ZZA ',
    "Zealandic" => 'ZEA ',
    "Zhuang" => 'ZHA ',
    "Zulu" => 'ZUL ',
    },

'FEATURE' => {
    "Above-base Forms" => 'abvf',
    "Above-base Mark Positioning" => 'abvm',
    "Above-base Substitutions" => 'abvs',
    "Access All Alternates" => 'aalt',
    "Akhands" => 'akhn',
    "Alternate Annotation Forms" => 'nalt',
    "Alternate Half Widths" => 'halt',
    "Alternate Vertical Half Metrics" => 'vhal',
    "Alternate Vertical Metrics" => 'valt',
    "Alternative Fractions" => 'afrc',
    "Below-base Forms" => 'blwf',
    "Below-base Mark Positioning" => 'blwm',
    "Below-base Substitutions" => 'blws',
    "Capital Spacing" => 'cpsp',
    "Case-Sensitive Forms" => 'case',
    "Centered CJK Punctuation" => 'cpct',
    "Character Variants 01" => 'cv01',
    "Character Variants 02" => 'cv02',
    "Character Variants 03" => 'cv03',
    "Character Variants 04" => 'cv04',
    "Character Variants 05" => 'cv05',
    "Character Variants 06" => 'cv06',
    "Character Variants 07" => 'cv07',
    "Character Variants 08" => 'cv08',
    "Character Variants 09" => 'cv09',
    "Character Variants 10" => 'cv10',
    "Character Variants 11" => 'cv11',
    "Character Variants 12" => 'cv12',
    "Character Variants 13" => 'cv13',
    "Character Variants 14" => 'cv14',
    "Character Variants 15" => 'cv15',
    "Character Variants 16" => 'cv16',
    "Character Variants 17" => 'cv17',
    "Character Variants 18" => 'cv18',
    "Character Variants 19" => 'cv19',
    "Character Variants 20" => 'cv20',
    "Character Variants 21" => 'cv21',
    "Character Variants 22" => 'cv22',
    "Character Variants 23" => 'cv23',
    "Character Variants 24" => 'cv24',
    "Character Variants 25" => 'cv25',
    "Character Variants 26" => 'cv26',
    "Character Variants 27" => 'cv27',
    "Character Variants 28" => 'cv28',
    "Character Variants 29" => 'cv29',
    "Character Variants 30" => 'cv30',
    "Character Variants 31" => 'cv31',
    "Character Variants 32" => 'cv32',
    "Character Variants 33" => 'cv33',
    "Character Variants 34" => 'cv34',
    "Character Variants 35" => 'cv35',
    "Character Variants 36" => 'cv36',
    "Character Variants 37" => 'cv37',
    "Character Variants 38" => 'cv38',
    "Character Variants 39" => 'cv39',
    "Character Variants 40" => 'cv40',
    "Character Variants 41" => 'cv41',
    "Character Variants 42" => 'cv42',
    "Character Variants 43" => 'cv43',
    "Character Variants 44" => 'cv44',
    "Character Variants 45" => 'cv45',
    "Character Variants 46" => 'cv46',
    "Character Variants 47" => 'cv47',
    "Character Variants 48" => 'cv48',
    "Character Variants 49" => 'cv49',
    "Character Variants 50" => 'cv50',
    "Character Variants 51" => 'cv51',
    "Character Variants 52" => 'cv52',
    "Character Variants 53" => 'cv53',
    "Character Variants 54" => 'cv54',
    "Character Variants 55" => 'cv55',
    "Character Variants 56" => 'cv56',
    "Character Variants 57" => 'cv57',
    "Character Variants 58" => 'cv58',
    "Character Variants 59" => 'cv59',
    "Character Variants 60" => 'cv60',
    "Character Variants 61" => 'cv61',
    "Character Variants 62" => 'cv62',
    "Character Variants 63" => 'cv63',
    "Character Variants 64" => 'cv64',
    "Character Variants 65" => 'cv65',
    "Character Variants 66" => 'cv66',
    "Character Variants 67" => 'cv67',
    "Character Variants 68" => 'cv68',
    "Character Variants 69" => 'cv69',
    "Character Variants 70" => 'cv70',
    "Character Variants 71" => 'cv71',
    "Character Variants 72" => 'cv72',
    "Character Variants 73" => 'cv73',
    "Character Variants 74" => 'cv74',
    "Character Variants 75" => 'cv75',
    "Character Variants 76" => 'cv76',
    "Character Variants 77" => 'cv77',
    "Character Variants 78" => 'cv78',
    "Character Variants 79" => 'cv79',
    "Character Variants 80" => 'cv80',
    "Character Variants 81" => 'cv81',
    "Character Variants 82" => 'cv82',
    "Character Variants 83" => 'cv83',
    "Character Variants 84" => 'cv84',
    "Character Variants 85" => 'cv85',
    "Character Variants 86" => 'cv86',
    "Character Variants 87" => 'cv87',
    "Character Variants 88" => 'cv88',
    "Character Variants 89" => 'cv89',
    "Character Variants 90" => 'cv90',
    "Character Variants 91" => 'cv91',
    "Character Variants 92" => 'cv92',
    "Character Variants 93" => 'cv93',
    "Character Variants 94" => 'cv94',
    "Character Variants 95" => 'cv95',
    "Character Variants 96" => 'cv96',
    "Character Variants 97" => 'cv97',
    "Character Variants 98" => 'cv98',
    "Character Variants 99" => 'cv99',
    "Conjunct Form After Ro" => 'cfar',
    "Conjunct Forms" => 'cjct',
    "Contextual Alternates" => 'calt',
    "Contextual Ligatures" => 'clig',
    "Contextual Swash" => 'cswh',
    "Cursive Positioning" => 'curs',
    "Denominators" => 'dnom',
    "Discretionary Ligatures" => 'dlig',
    "Distances" => 'dist',
    "Dotless Forms" => 'dtls',
    "Expert Forms" => 'expt',
    "Final Glyph on Line Alternates" => 'falt',
    "Flattened ascent forms" => 'flac',
    "Fractions" => 'frac',
    "Full Widths" => 'fwid',
    "Glyph Composition / Decomposition" => 'ccmp',
    "Halant Forms" => 'haln',
    "Half Forms" => 'half',
    "Half Widths" => 'hwid',
    "Hangul" => 'hngl',
    "Historical Forms" => 'hist',
    "Historical Ligatures" => 'hlig',
    "Hojo Kanji Forms (JIS X 0212-1990 Kanji Forms)" => 'hojo',
    "Horizontal Kana Alternates" => 'hkna',
    "Initial Forms" => 'init',
    "Isolated Forms" => 'isol',
    "Italics" => 'ital',
    "JIS2004 Forms" => 'jp04',
    "JIS78 Forms" => 'jp78',
    "JIS83 Forms" => 'jp83',
    "JIS90 Forms" => 'jp90',
    "Justification Alternates" => 'jalt',
    "Kerning" => 'kern',
    "Leading Jamo Forms" => 'ljmo',
    "Left Bounds" => 'lfbd',
    "Left-to-right glyph alternates" => 'ltra',
    "Left-to-right mirrored forms" => 'ltrm',
    "Lining Figures" => 'lnum',
    "Localized Forms" => 'locl',
    "Mark Positioning via Substitution" => 'mset',
    "Mark Positioning" => 'mark',
    "Mark to Mark Positioning" => 'mkmk',
    "Math script style alternates" => 'ssty',
    "Mathematical Greek" => 'mgrk',
    "Medial Forms #2" => 'med2',
    "Medial Forms" => 'medi',
    "NLC Kanji Forms" => 'nlck',
    "Nukta Forms" => 'nukt',
    "Numerators" => 'numr',
    "Oldstyle Figures" => 'onum',
    "Optical Bounds" => 'opbd',
    "Optical size" => 'size',
    "Ordinals" => 'ordn',
    "Ornaments" => 'ornm',
    "Petite Capitals From Capitals" => 'c2pc',
    "Petite Capitals" => 'pcap',
    "Post-base Forms" => 'pstf',
    "Post-base Substitutions" => 'psts',
    "Pre-Base Forms" => 'pref',
    "Pre-base Substitutions" => 'pres',
    "Proportional Alternate Vertical Metrics" => 'vpal',
    "Proportional Alternate Widths" => 'palt',
    "Proportional Figures" => 'pnum',
    "Proportional Kana" => 'pkna',
    "Proportional Widths" => 'pwid',
    "Quarter Widths" => 'qwid',
    "Rakar Forms" => 'rkrf',
    "Randomize" => 'rand',
    "Reph Forms" => 'rphf',
    "Required Contextual Alternates" => 'rclt',
    "Required Ligatures" => 'rlig',
    "Right Bounds" => 'rtbd',
    "Right-to-left alternates" => 'rtla',
    "Right-to-left mirrored forms" => 'rtlm',
    "Ruby Notation Forms" => 'ruby',
    "Scientific Inferiors" => 'sinf',
    "Simplified Forms" => 'smpl',
    "Slashed Zero" => 'zero',
    "Small Capitals From Capitals" => 'c2sc',
    "Small Capitals" => 'smcp',
    "Standard Ligatures" => 'liga',
    "Stretching Glyph Decomposition" => 'stch',
    "Stylistic Alternates" => 'salt',
    "Stylistic Set 1" => 'ss01',
    "Stylistic Set 10" => 'ss10',
    "Stylistic Set 11" => 'ss11',
    "Stylistic Set 12" => 'ss12',
    "Stylistic Set 13" => 'ss13',
    "Stylistic Set 14" => 'ss14',
    "Stylistic Set 15" => 'ss15',
    "Stylistic Set 16" => 'ss16',
    "Stylistic Set 17" => 'ss17',
    "Stylistic Set 18" => 'ss18',
    "Stylistic Set 19" => 'ss19',
    "Stylistic Set 2" => 'ss02',
    "Stylistic Set 20" => 'ss20',
    "Stylistic Set 3" => 'ss03',
    "Stylistic Set 4" => 'ss04',
    "Stylistic Set 5" => 'ss05',
    "Stylistic Set 6" => 'ss06',
    "Stylistic Set 7" => 'ss07',
    "Stylistic Set 8" => 'ss08',
    "Stylistic Set 9" => 'ss09',
    "Subscript" => 'subs',
    "Superscript" => 'sups',
    "Swash" => 'swsh',
    "Tabular Figures" => 'tnum',
    "Terminal Forms #2" => 'fin2',
    "Terminal Forms #3" => 'fin3',
    "Terminal Forms" => 'fina',
    "Third Widths" => 'twid',
    "Titling" => 'titl',
    "Traditional Forms" => 'trad',
    "Traditional Name Forms" => 'tnam',
    "Trailing Jamo Forms" => 'tjmo',
    "Unicase" => 'unic',
    "Vattu Variants" => 'vatu',
    "Vertical Alternates and Rotation" => 'vrt2',
    "Vertical Kana Alternates" => 'vkna',
    "Vertical Kerning" => 'vkrn',
    "Vertical Writing" => 'vert',
    "Vowel Jamo Forms" => 'vjmo',
    },

);

%iso639 = (
    'ABA ' => 'abq',
    'ABK ' => 'abk',
    'ACH ' => 'ach',
    'ACR ' => 'acr',
    'ADY ' => 'ady',
    'AFK ' => 'afr',
    'AFR ' => 'aar',
    'AGW ' => 'ahg',
    'AIO ' => 'aio',
    'AKA ' => 'aka',
    'ALS ' => 'gsw',
    'ALT ' => 'atv alt',
    'AMH ' => 'amh',
    'ANG ' => 'ang',
    'ARA ' => 'ara',
    'ARG ' => 'arg',
    'ARI ' => 'aiw',
    'ARK ' => 'mhv rmz rki',
    'ASM ' => 'asm',
    'AST ' => 'ast',
    'ATH ' => 'apk apj apl apm apw nav bea sek bcr caf crx clc gwi haa chp dgr scs xsl srs ing hoi koy hup ktw mvb wlk coq ctc gce tol tuu kkz tgx tht aht tfn taa tau tcb kuu tce ttm txc',
    'AVR ' => 'ava',
    'AWA ' => 'awa',
    'AYM ' => 'aym',
    'AZB ' => 'azb',
    'AZE ' => 'aze',
    'BAD ' => 'bfq',
    'BAD0' => 'bad',
    'BAG ' => 'bfy',
    'BAL ' => 'krc',
    'BAN ' => 'ban',
    'BAR ' => 'bar',
    'BAU ' => 'bci',
    'BBC ' => 'bbc',
    'BCH ' => 'bcq',
    'BDY ' => 'bdy',
    'BEL ' => 'bel',
    'BEM ' => 'bem',
    'BEN ' => 'ben',
    'BGC ' => 'bgc',
    'BGQ ' => 'bgq',
    'BGR ' => 'bul',
    'BHI ' => 'bhi bhb',
    'BHO ' => 'bho',
    'BIK ' => 'bik bhk bcl bto cts bln',
    'BIL ' => 'byn',
    'BIS ' => 'bis',
    'BJJ ' => 'bjj',
    'BKF ' => 'bla',
    'BLI ' => 'bal',
    'BLK ' => 'blk',
    'BLN ' => 'bjt ble',
    'BLT ' => 'bft',
    'BMB ' => 'bam',
    'BOS ' => 'bos',
    'BPY ' => 'bpy',
    'BRE ' => 'bre',
    'BRH ' => 'brh',
    'BRI ' => 'bra',
    'BRM ' => 'mya',
    'BRX ' => 'brx',
    'BSH ' => 'bak',
    'BTI ' => 'btb',
    'BTS ' => 'bts',
    'BUG ' => 'bug',
    'CAK ' => 'cak',
    'CAT ' => 'cat',
    'CBK ' => 'cbk',
    'CEB ' => 'ceb',
    'CGG ' => 'cgg',
    'CHA ' => 'cha',
    'CHE ' => 'che',
    'CHG ' => 'sgw',
    'CHH ' => 'hne',
    'CHI ' => 'nya',
    'CHK ' => 'ckt',
    'CHK0' => 'chk',
    'CHO ' => 'cho',
    'CHP ' => 'chp',
    'CHR ' => 'chr',
    'CHU ' => 'chv',
    'CHY ' => 'chy',
    'CMR ' => 'swb wlc wni zdj',
    'COP ' => 'cop',
    'COR ' => 'cor',
    'COS ' => 'cos',
    'CPP ' => 'cpp',
    'CRE ' => 'cre',
    'CRR ' => 'crx caf',
    'CRT ' => 'crh',
    'CSB ' => 'csb',
    'CSL ' => 'chu',
    'CSY ' => 'ces',
    'CTG ' => 'ctg',
    'CUK ' => 'cuk',
    'DAN ' => 'dan',
    'DAR ' => 'dar',
    'DAX ' => 'dax',
    'DCR ' => 'cwd',
    'DEU ' => 'deu',
    'DGO ' => 'dgo',
    'DGR ' => 'doi',
    'DHG ' => 'dhg',
    'DHV ' => 'div',
    'DIQ ' => 'diq',
    'DIV ' => 'div',
    'DJR ' => 'dje',
    'DJR0' => 'djr',
    'DNG ' => 'ada',
    'DNJ ' => 'dnj',
    'DNK ' => 'din',
    'DRI ' => 'prs',
    'DUJ ' => 'duj',
    'DUN ' => 'dng',
    'DZN ' => 'dzo',
    'EBI ' => 'igb',
    'ECR ' => 'crj crl',
    'EDO ' => 'bin',
    'EFI ' => 'efi',
    'ELL ' => 'ell',
    'EMK ' => 'emk',
    'ENG ' => 'eng',
    'ERZ ' => 'myv',
    'ESP ' => 'spa',
    'ESU ' => 'esu',
    'ETI ' => 'est',
    'EUQ ' => 'eus',
    'EVK ' => 'evn',
    'EVN ' => 'eve',
    'EWE ' => 'ewe',
    'FAN ' => 'acf',
    'FAN0' => 'fan',
    'FAR ' => 'fas',
    'FAT ' => 'fat',
    'FIN ' => 'fin',
    'FJI ' => 'fij',
    'FLE ' => 'vls',
    'FNE ' => 'enf',
    'FON ' => 'fon',
    'FOS ' => 'fao',
    'FRA ' => 'fra',
    'FRC ' => 'frc',
    'FRI ' => 'fry',
    'FRL ' => 'fur',
    'FRP ' => 'frp',
    'FTA ' => 'fuf',
    'FUL ' => 'ful',
    'FUV ' => 'fuv',
    'GAD ' => 'gaa',
    'GAE ' => 'gla',
    'GAG ' => 'gag',
    'GAL ' => 'glg',
    'GAW ' => 'gbm',
    'GEZ ' => 'gez',
    'GIH ' => 'gih',
    'GIL ' => 'niv',
    'GIL0' => 'gil',
    'GKP ' => 'gkp',
    'GLK ' => 'glk',
    'GMZ ' => 'guk',
    'GNN ' => 'gnn',
    'GOG ' => 'gog',
    'GON ' => 'gon gno ggo',
    'GRN ' => 'kal',
    'GRO ' => 'grt',
    'GUA ' => 'grn',
    'GUC ' => 'guc',
    'GUF ' => 'guf',
    'GUJ ' => 'guj',
    'GUZ ' => 'guz',
    'HAI ' => 'hat',
    'HAL ' => 'flm',
    'HAR ' => 'hoj',
    'HAU ' => 'hau',
    'HAW ' => 'haw',
    'HAY ' => 'hay',
    'HAZ ' => 'haz',
    'HBN ' => 'amf',
    'HER ' => 'her',
    'HIL ' => 'hil',
    'HIN ' => 'hin',
    'HMA ' => 'mrj',
    'HMN ' => 'hmn',
    'HMO ' => 'hmo',
    'HND ' => 'hno hnd',
    'HO  ' => 'hoc',
    'HRI ' => 'har',
    'HRV ' => 'hrv',
    'HUN ' => 'hun',
    'HYE ' => 'hye',
    'HYE0' => 'hye',
    'IBA ' => 'iba',
    'IBB ' => 'ibb',
    'IBO ' => 'ibo',
    'IDO ' => 'ido',
    'IJO ' => 'ijc',
    'ILE ' => 'ile',
    'ILO ' => 'ilo',
    'INA ' => 'ina',
    'IND ' => 'ind',
    'ING ' => 'inh',
    'INU ' => 'iku',
    'IPK ' => 'ipk',
    'IRI ' => 'gle',
    'IRT ' => 'gle',
    'ISL ' => 'isl',
    'ISM ' => 'smn',
    'ITA ' => 'ita',
    'IWR ' => 'heb',
    'JAM ' => 'jam',
    'JAN ' => 'jpn',
    'JAV ' => 'jav',
    'JBO ' => 'jbo',
    'JII ' => 'yid',
    'JUD ' => 'lad',
    'JUL ' => 'dyu',
    'KAB ' => 'kbd',
    'KAB0' => 'kab',
    'KAC ' => 'kfr',
    'KAL ' => 'kln',
    'KAN ' => 'kan',
    'KAR ' => 'krc',
    'KAT ' => 'kat',
    'KAZ ' => 'kaz',
    'KDE ' => 'kde',
    'KEA ' => 'kea',
    'KEB ' => 'ktb',
    'KEK ' => 'kek',
    'KGE ' => 'kat',
    'KHA ' => 'kjh',
    'KHK ' => 'kca',
    'KHM ' => 'khm',
    'KHS ' => 'kca',
    'KHT ' => 'kht',
    'KHV ' => 'kca',
    'KHW ' => 'khw',
    'KIK ' => 'kik',
    'KIR ' => 'kir',
    'KIS ' => 'kqs kss',
    'KIU ' => 'kiu',
    'KJD ' => 'kjd',
    'KJP ' => 'kjp',
    'KKN ' => 'kex',
    'KLM ' => 'xal',
    'KMB ' => 'kam',
    'KMN ' => 'kfy',
    'KMO ' => 'kmw',
    'KMS ' => 'kxc',
    'KNR ' => 'kau',
    'KOD ' => 'kfa',
    'KOH ' => 'okm',
    'KOK ' => 'kok',
    'KOM ' => 'kom',
    'KON ' => 'ktu',
    'KON0' => 'kon',
    'KOP ' => 'koi',
    'KOR ' => 'kor',
    'KOS ' => 'kos',
    'KOZ ' => 'kpv',
    'KPL ' => 'kpe',
    'KRI ' => 'kri',
    'KRK ' => 'kaa',
    'KRL ' => 'krl',
    'KRM ' => 'kdr',
    'KRN ' => 'kar',
    'KRT ' => 'kqy',
    'KSH ' => 'kas',
    'KSH0' => 'ksh',
    'KSI ' => 'kha',
    'KSM ' => 'sjd',
    'KSW ' => 'ksw',
    'KUA ' => 'kua',
    'KUI ' => 'kxu',
    'KUL ' => 'kfx',
    'KUM ' => 'kum',
    'KUR ' => 'kur',
    'KUU ' => 'kru',
    'KUY ' => 'kdt',
    'KYK ' => 'kpy',
    'KYU ' => 'kyu',
    'LAD ' => 'lld',
    'LAH ' => 'bfu',
    'LAK ' => 'lbe',
    'LAM ' => 'lmn',
    'LAO ' => 'lao',
    'LAT ' => 'lat',
    'LAZ ' => 'lzz',
    'LCR ' => 'crm',
    'LDK ' => 'lbj',
    'LEZ ' => 'lez',
    'LIJ ' => 'lij',
    'LIM ' => 'lim',
    'LIN ' => 'lin',
    'LIS ' => 'lis',
    'LJP ' => 'ljp',
    'LKI ' => 'lki',
    'LMA ' => 'mhr',
    'LMB ' => 'lif',
    'LMO ' => 'lmo',
    'LMW ' => 'ngl',
    'LOM ' => 'lom',
    'LRC ' => 'lrc luz bqi zum',
    'LSB ' => 'dsb',
    'LSM ' => 'smj',
    'LTH ' => 'lit',
    'LTZ ' => 'ltz',
    'LUA ' => 'lua',
    'LUB ' => 'lub',
    'LUG ' => 'lug',
    'LUH ' => 'luy',
    'LUO ' => 'luo',
    'LVI ' => 'lav',
    'MAD ' => 'mad',
    'MAG ' => 'mag',
    'MAH ' => 'mah',
    'MAJ ' => 'mpe',
    'MAK ' => 'vmw',
    'MAL ' => 'mal',
    'MAM ' => 'mam',
    'MAN ' => 'mns',
    'MAP ' => 'arn',
    'MAR ' => 'mar',
    'MAW ' => 'mwr dhd rwr mve wry mtr swv',
    'MBN ' => 'kmb',
    'MCH ' => 'mnc',
    'MCR ' => 'crm',
    'MDE ' => 'men',
    'MDR ' => 'mdr',
    'MEN ' => 'mym',
    'MER ' => 'mer',
    'MFE ' => 'mfe',
    'MIN ' => 'min',
    'MIZ ' => 'lus',
    'MKD ' => 'mkd',
    'MKR ' => 'mak',
    'MKW ' => 'mkw',
    'MLE ' => 'mdy',
    'MLG ' => 'mlg',
    'MLN ' => 'mlq',
    'MLR ' => 'mal',
    'MLY ' => 'msa',
    'MND ' => 'mnk',
    'MNG ' => 'mon',
    'MNI ' => 'mni',
    'MNK ' => 'man mnk myq mku msc emk mwk mlq',
    'MNX ' => 'glv',
    'MOH ' => 'mho',
    'MOK ' => 'mdf',
    'MOL ' => 'mol',
    'MON ' => 'mnw',
    'MOS ' => 'mos',
    'MRI ' => 'mri',
    'MTH ' => 'mai',
    'MTS ' => 'mlt',
    'MUN ' => 'unr',
    'MUS ' => 'mus',
    'MWL ' => 'mwl',
    'MWW ' => 'mww',
    'MYN ' => 'myn',
    'MZN ' => 'mzn',
    'NAG ' => 'nag',
    'NAH ' => 'nah',
    'NAN ' => 'gld',
    'NAP ' => 'nap',
    'NAS ' => 'nsk',
    'NAU ' => 'nau',
    'NAV ' => 'nav',
    'NCR ' => 'csw',
    'NDB ' => 'nbl nde',
    'NDC ' => 'ndc',
    'NDG ' => 'ndo',
    'NDS ' => 'nds',
    'NEP ' => 'nep',
    'NEW ' => 'new',
    'NGA ' => 'nga',
    'NHC ' => 'csw',
    'NIS ' => 'dap',
    'NIU ' => 'niu',
    'NKL ' => 'nyn',
    'NKO ' => 'ngo',
    'NLD ' => 'nld',
    'NOE ' => 'noe',
    'NOG ' => 'nog',
    'NOR ' => 'nob',
    'NOV ' => 'nov',
    'NSM ' => 'sme',
    'NSO ' => 'nso',
    'NTA ' => 'nod',
    'NTO ' => 'epo',
    'NYM ' => 'nym',
    'NYN ' => 'nno',
    'OCI ' => 'oci',
    'OCR ' => 'ojs',
    'OJB ' => 'oji',
    'ORI ' => 'ori',
    'ORO ' => 'orm',
    'OSS ' => 'oss',
    'PAA ' => 'sam',
    'PAG ' => 'pag',
    'PAL ' => 'pli',
    'PAM ' => 'pam',
    'PAN ' => 'pan',
    'PAP ' => 'plp',
    'PAP0' => 'pap',
    'PAS ' => 'pus',
    'PAU ' => 'pau',
    'PCC ' => 'pcc',
    'PCD ' => 'pcd',
    'PDC ' => 'pdc',
    'PGR ' => 'ell',
    'PHK ' => 'phk',
    'PIH ' => 'pih',
    'PIL ' => 'fil',
    'PLG ' => 'pce rbb pll',
    'PLK ' => 'pol',
    'PMS ' => 'pms',
    'PNB ' => 'pnb',
    'POH ' => 'poh',
    'PON ' => 'pon',
    'PRO ' => 'pro',
    'PTG ' => 'por',
    'PWO ' => 'pwo',
    'QIN ' => 'bgr cnh cnw czt sez tcp csy ctd flm pck tcz zom cmr dao hlt cka cnk mrh mwg cbl cnb csh',
    'QUC ' => 'quc',
    'QUH ' => 'quh',
    'QUZ ' => 'quz',
    'QVI ' => 'qvi',
    'QWH ' => 'qwh',
    'RAJ ' => 'raj',
    'RAR ' => 'rar',
    'RBU ' => 'bxr',
    'RCR ' => 'atj',
    'REJ ' => 'rej',
    'RIA ' => 'ria',
    'RIF ' => 'rif',
    'RIT ' => 'rit',
    'RKW ' => 'rkw',
    'RMS ' => 'roh',
    'RMY ' => 'rmy',
    'ROM ' => 'ron',
    'ROY ' => 'rom',
    'RSY ' => 'rue',
    'RTM ' => 'rtm',
    'RUA ' => 'kin',
    'RUN ' => 'run',
    'RUP ' => 'rup',
    'RUS ' => 'rus',
    'SAD ' => 'sck',
    'SAN ' => 'san',
    'SAS ' => 'sas',
    'SAT ' => 'sat',
    'SAY ' => 'chp',
    'SCN ' => 'scn',
    'SCO ' => 'sco',
    'SEK ' => 'xan',
    'SEL ' => 'sel',
    'SGA ' => 'sga',
    'SGO ' => 'sag',
    'SGS ' => 'sgs',
    'SHI ' => 'shi',
    'SHN ' => 'shn',
    'SIB ' => 'sjo',
    'SID ' => 'sid',
    'SIG ' => 'xst',
    'SKS ' => 'sms',
    'SKY ' => 'slk',
    'SLA ' => 'scs',
    'SLV ' => 'slv',
    'SML ' => 'som',
    'SMO ' => 'smo',
    'SNA ' => 'she',
    'SNA0' => 'sna',
    'SND ' => 'snd',
    'SNH ' => 'sin',
    'SNK ' => 'snk',
    'SOG ' => 'gru',
    'SOP ' => 'sop',
    'SOT ' => 'sot',
    'SQI ' => 'gsw',
    'SRB ' => 'srp',
    'SRD ' => 'srd',
    'SRK ' => 'skr',
    'SRR ' => 'srr',
    'SSL ' => 'xsl',
    'SSM ' => 'sma',
    'STQ ' => 'stq',
    'SUK ' => 'suk',
    'SUN ' => 'sun',
    'SUR ' => 'suq',
    'SVA ' => 'sva',
    'SVE ' => 'swe',
    'SWA ' => 'aii',
    'SWK ' => 'swa',
    'SWZ ' => 'ssw',
    'SXT ' => 'ngo',
    'SXU ' => 'sxu',
    'SYL ' => 'syl',
    'SYR ' => 'syr',
    'SZL ' => 'szl',
    'TAB ' => 'tab',
    'TAJ ' => 'tgk',
    'TAM ' => 'tam',
    'TAT ' => 'tat',
    'TCR ' => 'cwd',
    'TDD ' => 'tdd',
    'TEL ' => 'tel',
    'TET ' => 'tet',
    'TGL ' => 'tgl',
    'TGN ' => 'ton',
    'TGR ' => 'tig',
    'TGY ' => 'tir',
    'THA ' => 'tha',
    'THT ' => 'tah',
    'TIB ' => 'bod',
    'TIV ' => 'tiv',
    'TKM ' => 'tuk',
    'TMH ' => 'tmh',
    'TMN ' => 'tem',
    'TNA ' => 'tsn',
    'TNE ' => 'enh',
    'TNG ' => 'toi',
    'TOD ' => 'xal',
    'TOD0' => 'tod',
    'TPI ' => 'tpi',
    'TRK ' => 'tur',
    'TSG ' => 'tso',
    'TUA ' => 'tru',
    'TUL ' => 'tcy',
    'TUM ' => 'tum',
    'TUV ' => 'tyv',
    'TVL ' => 'tvl',
    'TWI ' => 'twi',
    'TYZ ' => 'tyz',
    'TZM ' => 'tzm',
    'TZO ' => 'tzo',
    'UDM ' => 'udm',
    'UKR ' => 'ukr',
    'UMB ' => 'umb',
    'URD ' => 'urd',
    'USB ' => 'hsb',
    'UYG ' => 'uig',
    'UZB ' => 'uzb uzn uzs',
    'VEC ' => 'vec',
    'VEN ' => 'ven',
    'VIT ' => 'vie',
    'VOL ' => 'vol',
    'VRO ' => 'vro',
    'WA  ' => 'wbm',
    'WAG ' => 'wbr',
    'WAR ' => 'war',
    'WCR ' => 'crk',
    'WEL ' => 'cym',
    'WLF ' => 'wol',
    'WLN ' => 'wln',
    'WTM ' => 'wtm',
    'XBD ' => 'khb',
    'XHS ' => 'xho',
    'XJB ' => 'xjb',
    'XOG ' => 'xog',
    'XPE ' => 'xpe',
    'YAK ' => 'sah',
    'YAO ' => 'yao',
    'YAP ' => 'yap',
    'YBA ' => 'yor',
    'YCR ' => 'cre',
    'YIM ' => 'iii',
    'ZEA ' => 'zea',
    'ZGH ' => 'zgh',
    'ZHA ' => 'zha',
    'ZHH ' => 'zho',
    'ZHP ' => 'zho',
    'ZHS ' => 'zho',
    'ZHT ' => 'zho',
    'ZND ' => 'zne',
    'ZUL ' => 'zul',
    'ZZA ' => 'zza',
);

{
    foreach my $s (qw ( SCRIPT LANGUAGE FEATURE ) )
    {
        map { $ttnames{$s}{$tttags{$s}{$_}} = $_ }  keys %{$tttags{$s}};
    }
    
    # For ISO639 info, the raw data is a space-separated list of ISO639
    # language IDs. We convert that list to an array.
    
    foreach my $tag (keys %iso639)
    {
        my $list = $iso639{$tag};
        $iso639{$tag} = [ split(' ', $list) ];
        # Also set the reverse mapping:
        map { $iso639{$_} = $tag } @{$iso639{$tag}};
    }
}


=head2 readtagsfile ( filename )

Read a file in the syntax of Tags.txt (included with Microsoft VOLT) to obtain additional/replacement tag definitions.

Returns 0 if cannot open the file; else 1.

=cut

sub readtagsfile
{
    my $fname = shift;
    open (TAGS, $fname) or return 0;
    my ($what, $name, $tag);
    while (<TAGS>)
    {
        ($what, $name, $tag) = (m/"([^"]*)", "([^"]*)", "([^"]*)"/);  #"
        $ttnames{$what}{$tag} = $name;
        $tttags{$what}{$name} = $tag;
    }
    close TAGS;
    return 1;   
}

1;

=head1 AUTHOR

Bob Hallissy. L<http://scripts.sil.org/FontUtils>.


=head1 LICENSING

Copyright (c) 1998-2014, SIL International (http://www.sil.org) 

This module is released under the terms of the Artistic License 2.0. 
For details, see the full text of the license in the file LICENSE.



=cut